<?php
/**
 * Custom Twitter Feed Builder
 *
 * @since 2.0
 */
namespace TwitterFeed\Builder;


use TwitterFeed\CTF_Feed_Locator;
use TwitterFeed\CTF_Settings;
use TwitterFeed\Builder\Tabs\CTF_Styling_Tab;
use TwitterFeed\Builder\CTF_Feed_Saver;
use TwitterFeed\V2\CtfOauthConnect;

class CTF_Feed_Builder {
	private static $instance;
	public static function instance() {
		if ( null === self::$instance) {
			require CTF_PLUGIN_DIR . 'vendor/autoload.php';
			self::$instance = new self();
			return self::$instance;

		}
	}


	/**
	 * Constructor.
	 *
	 * @since 2.0
	 */
	function __construct(){
		$this->init();
	}

	/**
	 * Init the Builder.
	 *
	 * @since 2.0
	*/
	function init(){
		if( is_admin() ){
			add_action('admin_menu', [$this, 'register_menu']);
			// add ajax listeners
			CTF_Feed_Saver_Manager::hooks();
			CTF_Feed_Builder::hooks();
			#echo json_encode(CTF_Feed_Saver::settings_defaults());

		}
	}

	/**
	 * Mostly AJAX related hooks
	 *
	 * @since 2.0
	 */
	public static function hooks() {
		add_action( 'wp_ajax_ctf_dismiss_onboarding', array( 'TwitterFeed\Builder\CTF_Feed_Builder', 'after_dismiss_onboarding' ) );
		add_action( 'wp_ajax_ctf_other_plugins_modal', array( 'TwitterFeed\Builder\CTF_Feed_Builder', 'sb_other_plugins_modal' ) );
	}

	/**
	 * Check users capabilities and maybe nonce before AJAX actions
	 *
	 * @param $check_nonce
	 * @param string $action
	 *
	 * @since 2.0.6
	 */
	public static function check_privilege( $check_nonce = false, $action = 'ctf-admin' ) {
		$cap = current_user_can( 'manage_twitter_feed_options' ) ? 'manage_twitter_feed_options' : 'manage_options';
		$cap = apply_filters( 'ctf_settings_pages_capability', $cap );

		if ( ! current_user_can( $cap ) ) {
			wp_die ( 'You did not do this the right way!' );
		}

		if ( $check_nonce ) {
			$nonce = ! empty( $_POST[ $check_nonce ] ) ? $_POST[ $check_nonce ] : false;

			if ( ! wp_verify_nonce( $nonce, $action ) ) {
				wp_die ( 'You did not do this the right way!' );
			}
		}
	}

	/**
	 * Register Menu.
	 *
	 * @since 2.0
	 */
	function register_menu(){
	 	$cap = current_user_can( 'manage_twitter_feed_options' ) ? 'manage_twitter_feed_options' : 'manage_options';
    	$cap = apply_filters( 'ctf_settings_pages_capability', $cap );

		$feed_builder = add_submenu_page(
	        'custom-twitter-feeds',
	        __( 'All Feeds', 'custom-twitter-feeds' ),
	        __( 'All Feeds', 'custom-twitter-feeds' ),
	        $cap,
	        'ctf-feed-builder',
	        [$this, 'feed_builder'],
	        0
	    );
	    add_action( 'load-' . $feed_builder, [$this,'builder_enqueue_admin_scripts']);
	}

	/**
	 * Enqueue Builder CSS & Script.
	 *
	 * Loads only for builder pages
	 *
	 * @since 2.0
	 */
    public function builder_enqueue_admin_scripts(){
        if(get_current_screen()):
        	$screen = get_current_screen();
			if ( strpos($screen->id, 'ctf-feed-builder')  !== false ) :

		        $installed_plugins = get_plugins();

		        $newly_retrieved_source_connection_data = [];
		        $license_key = get_option( 'ctf_license_key', '' );
				$ctf_options = get_option( 'ctf_options', array() );

				$active_extensions = [
					//Fake
					[
						'type' => 'feedLayout',
						'active' => false,
						'ids' => ['layout']
					],
					[
						'type' => 'advancedFilters',
						'active' => false,
						'ids' => ['includewords','excludewords','includeanyall','filterandor','excludeanyall','remove_by_id']
					],
					[
						'type' => 'mediaExtension',
						'active' => false,
						'ids' => ['include_media','include_retweeter','include_twittercards']
					],
					[
						'type' => 'autoscrollExtension',
						'active' => false,
						'ids' => ['autoscrolldistance']
					],
					[
						'type' => 'lightboxExtension',
						'active' => false,
						'ids' => ['disablelightbox']
					]

				];

				$upgrade_url = 'https://smashballoon.com/custom-twitter-feeds/demo?utm_campaign=twitter-free&utm_source=lite-upgrade-bar';

		        $ctf_builder = array(
					'ajax_handler'		=> 	admin_url( 'admin-ajax.php' ),
					'pluginType' 		=> 'free',
					'activeExtensions'	=> $active_extensions,
					'builderUrl'		=> admin_url( 'admin.php?page=ctf-feed-builder' ),
					'nonce'				=> wp_create_nonce( 'ctf-admin' ),
					'upgradeUrl'		=> $upgrade_url,
					'pluginUrl'			=> trailingslashit( CTF_PLUGIN_URL ),
					'ctfRebrand'		=> ctf_should_rebrand_to_x(),
					'adminPostURL'		=> 	admin_url( 'post.php' ),
					'adminSettingsURL'	=> 	admin_url('admin.php?page=ctf-settings'),
					'widgetsPageURL'	=> 	admin_url( 'widgets.php' ),
					'supportPageUrl'    => admin_url( 'admin.php?page=ctf-support' ),
					'genericText'       => self::get_generic_text(),
					'translatedText' 	=> $this->get_translated_text(),
					'welcomeScreen' => array(
						'mainHeading' => __( 'All Feeds', 'custom-twitter-feeds' ),
						'createFeed' => __( 'Create your Feed', 'custom-twitter-feeds' ),
						'createFeedDescription' => __( 'Connect your Twitter account and choose a feed type', 'custom-twitter-feeds' ),
						'customizeFeed' => __( 'Customize your feed type', 'custom-twitter-feeds' ),
						'customizeFeedDescription' => __( 'Choose layouts, color schemes, filters and more', 'custom-twitter-feeds' ),
						'embedFeed' => __( 'Embed your feed', 'custom-twitter-feeds' ),
						'embedFeedDescription' => __( 'Easily add the feed anywhere on your website', 'custom-twitter-feeds' ),
						'customizeImgPath' => CTF_BUILDER_URL . 'assets/img/welcome-1.png',
						'embedImgPath' => CTF_BUILDER_URL . 'assets/img/welcome-2.png',
					),
					'pluginsInfo' => [
						'social_wall' => [
							'installed' => isset( $installed_plugins['social-wall/social-wall.php'] ) ? true : false,
							'activated' => is_plugin_active( 'social-wall/social-wall.php' ),
							'settingsPage' => admin_url( 'admin.php?page=sbsw' ),
						]
					],
					'allFeedsScreen' => array(
						'mainHeading' => __( 'All Feeds', 'custom-twitter-feeds' ),
						'columns' => array(
							'nameText' => __( 'Name', 'custom-twitter-feeds' ),
							'shortcodeText' => __( 'Shortcode', 'custom-twitter-feeds' ),
							'instancesText' => __( 'Instances', 'custom-twitter-feeds' ),
							'actionsText' => __( 'Actions', 'custom-twitter-feeds' ),
						),
						'bulkActions' => __( 'Bulk Actions', 'custom-twitter-feeds' ),
						'legacyFeeds' => array(
							'heading' => __( 'Legacy Feeds', 'custom-twitter-feeds' ),
							'toolTip' => __( 'What are Legacy Feeds?', 'custom-twitter-feeds' ),
							'toolTipExpanded' => array(
								__( 'Legacy feeds are older feeds from before the version 2 update. You can edit settings for these feeds by using the "Settings" button to the right. These settings will apply to all legacy feeds, just like the settings before version 2, and work in the same way that they used to.', 'custom-twitter-feeds' ),
								__( 'You can also create a new feed, which will now have it\'s own individual settings. Modifying settings for new feeds will not affect other feeds.', 'custom-twitter-feeds' ),
							),
							'toolTipExpandedAction' => array(
								__( 'Legacy feeds represent shortcodes of old feeds found on your website before the version 2 update.', 'custom-twitter-feeds' ),
								__( 'To edit Legacy feed settings, you will need to use the "Settings" button above or edit their shortcode settings directly. To delete them, simply remove the shortcode wherever it is being used on your site.', 'custom-twitter-feeds' ),
							),
							'show' => __( 'Show Legacy Feeds', 'custom-twitter-feeds' ),
							'hide' => __( 'Hide Legacy Feeds', 'custom-twitter-feeds' ),
						),
						'socialWallLinks' => CTF_Feed_Builder::get_social_wall_links(),
						'onboarding' => $this->get_onboarding_text()
					),
					'dialogBoxPopupScreen'  => array(
						'deleteSourceCustomizer' => array(
							'heading' =>  __( 'Delete "#"?', 'custom-twitter-feeds' ),
							'description' => __( 'You are going to delete this source. To retrieve it, you will need to add it again. Are you sure you want to continue?', 'custom-twitter-feeds' ),
						),
						'deleteSingleFeed' => array(
							'heading' =>  __( 'Delete "#"?', 'custom-twitter-feeds' ),
							'description' => __( 'You are going to delete this feed. You will lose all the settings. Are you sure you want to continue?', 'custom-twitter-feeds' ),
						),
						'deleteMultipleFeeds' => array(
							'heading' =>  __( 'Delete Feeds?', 'custom-twitter-feeds' ),
							'description' => __( 'You are going to delete these feeds. You will lose all the settings. Are you sure you want to continue?', 'custom-twitter-feeds' ),
						),
						'backAllToFeed' => array(
							'heading' =>  __( 'Are you Sure?', 'custom-twitter-feeds' ),
							'description' => __( 'Are you sure you want to leave this page, all unsaved settings will be lost, please make sure to save before leaving.', 'custom-twitter-feeds' ),
						),
						'unsavedFeedSources' => array(
							'heading' =>  __( 'You have unsaved changes', 'custom-twitter-feeds' ),
							'description' => __( 'If you exit without saving, all the changes you made will be reverted.', 'custom-twitter-feeds' ),
							'customButtons' => array(
								'confirm' => [
									'text' => __( 'Save and Exit', 'custom-twitter-feeds' ),
									'color' => 'blue'
								],
								'cancel' => [
									'text' => __( 'Exit without Saving', 'custom-twitter-feeds' ),
									'color' => 'red'
								]
							)
                        ),
                        'createMoreFeeds' => array(
							'icon' =>  'twitterSmash',
							'heading' =>  __( 'Create more than one feed<br/>with Twitter Feeds Pro', 'custom-twitter-feeds' ),
							'description' => __( 'Due to new Twitter API limitations, we can no longer support creation of multiple feeds. To continue creating more than a single feed, upgrade to Pro.', 'custom-twitter-feeds' ),
							'customButtons' => array(
								'confirm' => [
									'text' => __( 'Upgrade to Pro', 'custom-twitter-feeds' ),
									'color' => 'blue'
								],
								'cancel' => [
									'text' => __( 'Close', 'custom-twitter-feeds' ),
									'color' => 'gray'
								]
							)
						)

					),
					 'selectFeedTypeScreen' => array(
						'mainHeading' => __( 'Create a Twitter Feed', 'custom-twitter-feeds' ),
						'feedTypeHeading' => __( 'Select Feed Type', 'custom-twitter-feeds' ),
						'mainDescription' => __( 'Select one or more feed types. You can add or remove them later.', 'custom-twitter-feeds' ),
						'updateHeading' => __( 'Update Feed Type', 'custom-twitter-feeds' ),
						'advancedHeading' => __( 'Advanced Feeds', 'custom-twitter-feeds' ),
						'anotherFeedTypeHeading' => __( 'Add Another Feed Type', 'custom-twitter-feeds' ),
			        ),
					 'selectFeedTemplateScreen' => array(
						'feedTemplateHeading' => __( 'Start with a template', 'custom-twitter-feeds' ),
						'feedTemplateDescription' => __( 'Select a starting point for your feed. You can customize this later.', 'custom-twitter-feeds' ),
						'updateHeading' => __( 'Select another template', 'custom-twitter-feeds' ),
						'updateHeadingWarning' => __( 'Changing a template will override your layout, header and button settings', 'custom-twitter-feeds' ),
						'updateHeadingWarning2' => __( 'Changing a template might override your customizations', 'custom-twitter-feeds' )
			        ),
					'mainFooterScreen' => array(
						'heading' => sprintf( __( 'Upgrade to the %sAll Access Bundle%s to get all of our Pro Plugins', 'custom-twitter-feeds' ), '<strong>', '</strong>' ),
						'description' => __( 'Includes all Smash Balloon plugins for one low price: Instagram, Facebook, Twitter, YouTube, and Social Wall', 'custom-twitter-feeds' ),
						'promo' => sprintf( __( '%sBonus%s Lite users get %s50&#37; Off%s automatically applied at checkout', 'custom-twitter-feeds' ), '<span class="ctf-bld-ft-bns">', '</span>', '<strong>', '</strong>' ),
					),
					'embedPopupScreen' => array(
						'heading' => __( 'Embed Feed', 'custom-twitter-feeds' ),
						'description' => __( 'Add the unique shortcode to any page, post, or widget:', 'custom-twitter-feeds' ),
						'description_2' => __( 'Or use the built in WordPress block or widget', 'custom-twitter-feeds' ),
						'addPage' => __( 'Add to a Page', 'custom-twitter-feeds' ),
						'addWidget' => __( 'Add to a Widget', 'custom-twitter-feeds' ),
						'selectPage' => __( 'Select Page', 'custom-twitter-feeds' ),
					),
					'connectAccountScreen' => self::connect_account_secreen(),


			        'links' => self::get_links_with_utm(),
			        'pluginsInfo' => [
						'social_wall' => [
							'installed' => isset( $installed_plugins['social-wall/social-wall.php'] ) ? true : false,
							'activated' => is_plugin_active( 'social-wall/social-wall.php' ),
							'settingsPage' => admin_url( 'admin.php?page=sbsw' ),
						]
					],
					'selectSourceScreen' => self::select_source_screen_text(),
			        'feedTypes' => $this->get_feed_types(),
			        'advancedFeedTypes' => $this->get_advanced_feed_types(),
			        'feedTemplates' => $this->get_feed_templates(),
					'socialInfo' => $this->get_smashballoon_info(),
			        'svgIcons' => $this->builder_svg_icons(),
					'installPluginsPopup' => $this->install_plugins_popup(),
			        'feeds' => self::get_feed_list(),
			        'itemsPerPage'			=> CTF_Db::RESULTS_PER_PAGE,
			        'feedsCount' 			=> CTF_Db::feeds_count(),

			        'sources' => [],
					'sourceConnectionURLs' => [],
					'appCredentials' => self::get_app_credentials( $ctf_options ),
					'legacyFeeds' => $this->get_legacy_feed_list(),
			        'extensionsPopup' => [
			        	'hashtag' => [
							'heading' 		=> __('Upgrade to Plus to get Hashtag Feeds', 'custom-twitter-feeds'),
							'description' 	=> __( 'Create incredibly powerful hashtag based feeds using your selected hashtags using powerful hashtag queries.', 'custom-twitter-feeds' ),
							'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_235_1820)"><g opacity="0.8"><g filter="url(#filter0_d_235_1820)"><rect x="54" y="105" width="288" height="113" rx="4" fill="white"/></g><rect x="122" y="124" width="188" height="14" rx="1" fill="#DCDDE1"/><rect x="122" y="150" width="160" height="14" rx="1" fill="#DCDDE1"/><circle cx="86.5" cy="137.5" r="17.5" fill="#DCDDE1"/><path d="M132.574 184.785H130.343L130.745 182.844C130.755 182.787 130.753 182.728 130.738 182.672C130.723 182.616 130.696 182.564 130.659 182.519C130.622 182.475 130.575 182.439 130.523 182.413C130.471 182.388 130.414 182.375 130.355 182.374H129.552C129.459 182.372 129.368 182.402 129.295 182.459C129.222 182.516 129.17 182.597 129.15 182.687L128.74 184.785H126.57L126.972 182.844C126.982 182.788 126.979 182.731 126.965 182.676C126.951 182.621 126.926 182.57 126.89 182.525C126.855 182.481 126.811 182.444 126.76 182.418C126.71 182.392 126.655 182.377 126.598 182.374H125.794C125.701 182.372 125.611 182.402 125.537 182.459C125.464 182.516 125.413 182.597 125.392 182.687L124.991 184.785H122.579C122.485 184.783 122.394 184.814 122.32 184.873C122.247 184.932 122.196 185.015 122.178 185.107L122.013 185.91C122.001 185.969 122.002 186.031 122.017 186.089C122.032 186.147 122.059 186.202 122.098 186.248C122.136 186.294 122.185 186.331 122.24 186.357C122.295 186.382 122.354 186.394 122.415 186.393H124.645L123.986 189.608H121.575C121.481 189.606 121.389 189.637 121.316 189.696C121.242 189.754 121.192 189.837 121.173 189.929L121.008 190.733C120.996 190.792 120.997 190.853 121.012 190.911C121.027 190.97 121.055 191.024 121.093 191.071C121.132 191.117 121.18 191.154 121.235 191.179C121.29 191.204 121.35 191.216 121.41 191.215H123.656L123.255 193.156C123.245 193.214 123.247 193.274 123.263 193.331C123.279 193.388 123.307 193.441 123.345 193.485C123.383 193.53 123.431 193.566 123.485 193.591C123.539 193.615 123.597 193.627 123.656 193.626H124.46C124.55 193.624 124.636 193.592 124.705 193.535C124.774 193.478 124.822 193.4 124.842 193.313L125.26 191.215H127.43L127.028 193.156C127.018 193.212 127.021 193.269 127.035 193.324C127.049 193.379 127.074 193.431 127.109 193.475C127.145 193.519 127.189 193.556 127.239 193.582C127.29 193.608 127.345 193.623 127.402 193.626H128.206C128.298 193.628 128.389 193.598 128.462 193.541C128.536 193.484 128.587 193.403 128.607 193.313L129.029 191.215H131.441C131.535 191.217 131.626 191.186 131.7 191.127C131.773 191.068 131.824 190.986 131.842 190.894L132.003 190.09C132.015 190.031 132.014 189.97 131.999 189.911C131.984 189.853 131.957 189.798 131.918 189.752C131.88 189.706 131.831 189.669 131.776 189.644C131.721 189.619 131.662 189.606 131.601 189.608H129.359L130.014 186.393H132.425C132.519 186.395 132.611 186.363 132.684 186.305C132.758 186.246 132.808 186.163 132.827 186.071L132.992 185.267C133.004 185.207 133.003 185.145 132.987 185.085C132.971 185.025 132.942 184.97 132.902 184.923C132.861 184.877 132.811 184.84 132.754 184.816C132.697 184.792 132.635 184.781 132.574 184.785ZM127.751 189.608H125.581L126.24 186.393H128.41L127.751 189.608Z" fill="#59AB46"/><rect x="138" y="182" width="50" height="12" rx="1" fill="#D6ECD1"/><path d="M212.574 184.785H210.343L210.745 182.844C210.755 182.787 210.753 182.728 210.738 182.672C210.723 182.616 210.696 182.564 210.659 182.519C210.622 182.475 210.575 182.439 210.523 182.413C210.471 182.388 210.414 182.375 210.355 182.374H209.552C209.459 182.372 209.368 182.402 209.295 182.459C209.222 182.516 209.17 182.597 209.15 182.687L208.74 184.785H206.57L206.972 182.844C206.982 182.788 206.979 182.731 206.965 182.676C206.951 182.621 206.926 182.57 206.89 182.525C206.855 182.481 206.811 182.444 206.76 182.418C206.71 182.392 206.655 182.377 206.598 182.374H205.794C205.701 182.372 205.611 182.402 205.537 182.459C205.464 182.516 205.413 182.597 205.392 182.687L204.991 184.785H202.579C202.485 184.783 202.394 184.814 202.32 184.873C202.247 184.932 202.196 185.015 202.178 185.107L202.013 185.91C202.001 185.969 202.002 186.031 202.017 186.089C202.032 186.147 202.059 186.202 202.098 186.248C202.136 186.294 202.185 186.331 202.24 186.357C202.295 186.382 202.354 186.394 202.415 186.393H204.645L203.986 189.608H201.575C201.481 189.606 201.389 189.637 201.316 189.696C201.242 189.754 201.192 189.837 201.173 189.929L201.008 190.733C200.996 190.792 200.997 190.853 201.012 190.911C201.027 190.97 201.055 191.024 201.093 191.071C201.132 191.117 201.18 191.154 201.235 191.179C201.29 191.204 201.35 191.216 201.41 191.215H203.656L203.255 193.156C203.245 193.214 203.247 193.274 203.263 193.331C203.279 193.388 203.307 193.441 203.345 193.485C203.383 193.53 203.431 193.566 203.485 193.591C203.539 193.615 203.597 193.627 203.656 193.626H204.46C204.55 193.624 204.636 193.592 204.705 193.535C204.774 193.478 204.822 193.4 204.842 193.313L205.26 191.215H207.43L207.028 193.156C207.018 193.212 207.021 193.269 207.035 193.324C207.049 193.379 207.074 193.431 207.109 193.475C207.145 193.519 207.189 193.556 207.239 193.582C207.29 193.608 207.345 193.623 207.402 193.626H208.206C208.298 193.628 208.389 193.598 208.462 193.541C208.536 193.484 208.587 193.403 208.607 193.313L209.029 191.215H211.441C211.535 191.217 211.626 191.186 211.7 191.127C211.773 191.068 211.824 190.986 211.842 190.894L212.003 190.09C212.015 190.031 212.014 189.97 211.999 189.911C211.984 189.853 211.957 189.798 211.918 189.752C211.88 189.706 211.831 189.669 211.776 189.644C211.721 189.619 211.662 189.606 211.601 189.608H209.359L210.014 186.393H212.425C212.519 186.395 212.611 186.363 212.684 186.305C212.758 186.246 212.808 186.163 212.827 186.071L212.992 185.267C213.004 185.207 213.003 185.145 212.987 185.085C212.971 185.025 212.942 184.97 212.902 184.923C212.861 184.877 212.811 184.84 212.754 184.816C212.697 184.792 212.635 184.781 212.574 184.785ZM207.751 189.608H205.581L206.24 186.393H208.41L207.751 189.608Z" fill="#59AB46"/><rect x="218" y="182" width="50" height="12" rx="1" fill="#D6ECD1"/></g><g opacity="0.8"><g filter="url(#filter1_d_235_1820)"><rect x="54" y="225" width="288" height="113" rx="4" fill="white"/></g><rect x="122" y="244" width="188" height="14" rx="1" fill="#DCDDE1"/><circle cx="86.5" cy="257.5" r="17.5" fill="#DCDDE1"/></g><g filter="url(#filter2_d_235_1820)"><rect x="53" y="45" width="289" height="53" fill="white"/></g><circle cx="75.1713" cy="68.1716" r="8" transform="rotate(-45 75.1713 68.1716)" stroke="#0096CC" stroke-width="2"/><line x1="80.8282" y1="73.8284" x2="87.1922" y2="80.1924" stroke="#0096CC" stroke-width="2"/><rect x="106" y="63" width="119" height="16" rx="2" fill="#B5E5FF"/><rect opacity="0.6" x="14" y="150" width="372" height="114" fill="url(#paint0_linear_235_1820)"/></g><defs><filter id="filter0_d_235_1820" x="45.1412" y="100.571" width="305.718" height="130.718" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.32203" operator="erode" in="SourceAlpha" result="effect1_dropShadow_235_1820"/><feOffset dy="4.42938"/><feGaussianBlur stdDeviation="6.0904"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_235_1820"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_235_1820" result="shape"/></filter><filter id="filter1_d_235_1820" x="45.1412" y="220.571" width="305.718" height="130.718" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.32203" operator="erode" in="SourceAlpha" result="effect1_dropShadow_235_1820"/><feOffset dy="4.42938"/><feGaussianBlur stdDeviation="6.0904"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_235_1820"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_235_1820" result="shape"/></filter><filter id="filter2_d_235_1820" x="44.8915" y="40.9457" width="305.217" height="69.217" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.04069" operator="erode" in="SourceAlpha" result="effect1_dropShadow_235_1820"/><feOffset dy="4.05426"/><feGaussianBlur stdDeviation="5.5746"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_235_1820"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_235_1820" result="shape"/></filter><linearGradient id="paint0_linear_235_1820" x1="200" y1="150" x2="200" y2="264" gradientUnits="userSpaceOnUse"><stop stop-color="#F3F4F5" stop-opacity="0"/><stop offset="0.925689" stop-color="#F3F4F5"/></linearGradient><clipPath id="clip0_235_1820"><rect width="396" height="264" fill="white"/></clipPath></defs></svg>',
							'popupContentBtn' 	=> '<div class="ctf-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Lite users get 50% OFF', 'custom-twitter-feeds' ) .'</div>',
							'bullets'       => [
								'heading' => __( 'And get much more!', 'custom-twitter-feeds' ),
								'content' => [
									__( 'Display photos, videos & GIFs', 'custom-twitter-feeds' ),
									__( 'Combine mulitple feeds into one', 'custom-twitter-feeds' ),
									__( 'Unlimited feeds, 12 daily updates', 'custom-twitter-feeds' ),
									__( 'Auto load tweets on scroll', 'custom-twitter-feeds' ),
									__( 'Powerful Tweet filters', 'custom-twitter-feeds' ),
									__( 'View tweets in a lightbox', 'custom-twitter-feeds' ),
									__( 'Multiple tweet layout options', 'custom-twitter-feeds' ),
									__( 'Fast & Friendly Support', 'custom-twitter-feeds' ),
									__( '30 Day Money Back Guarantee', 'custom-twitter-feeds' )
								]
							],
							'buyUrl' 		=> sprintf('https://smashballoon.com/custom-twitter-feeds/demo?utm_campaign=twitter-free&utm_source=feed-type&utm_medium=hashtag')
						],
			        	'search' => [
							'heading' 		=> __('Upgrade to Elite to get Search Feeds', 'custom-twitter-feeds'),
							'description' 	=> __( 'Create incredibly advanced search based feeds using a wide selection of search terms and operators to form feeds using powerful search queries.', 'custom-twitter-feeds' ),
							'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_235_1820)"><g opacity="0.8"><g filter="url(#filter0_d_235_1820)"><rect x="54" y="105" width="288" height="113" rx="4" fill="white"/></g><rect x="122" y="124" width="188" height="14" rx="1" fill="#DCDDE1"/><rect x="122" y="150" width="160" height="14" rx="1" fill="#DCDDE1"/><circle cx="86.5" cy="137.5" r="17.5" fill="#DCDDE1"/><path d="M132.574 184.785H130.343L130.745 182.844C130.755 182.787 130.753 182.728 130.738 182.672C130.723 182.616 130.696 182.564 130.659 182.519C130.622 182.475 130.575 182.439 130.523 182.413C130.471 182.388 130.414 182.375 130.355 182.374H129.552C129.459 182.372 129.368 182.402 129.295 182.459C129.222 182.516 129.17 182.597 129.15 182.687L128.74 184.785H126.57L126.972 182.844C126.982 182.788 126.979 182.731 126.965 182.676C126.951 182.621 126.926 182.57 126.89 182.525C126.855 182.481 126.811 182.444 126.76 182.418C126.71 182.392 126.655 182.377 126.598 182.374H125.794C125.701 182.372 125.611 182.402 125.537 182.459C125.464 182.516 125.413 182.597 125.392 182.687L124.991 184.785H122.579C122.485 184.783 122.394 184.814 122.32 184.873C122.247 184.932 122.196 185.015 122.178 185.107L122.013 185.91C122.001 185.969 122.002 186.031 122.017 186.089C122.032 186.147 122.059 186.202 122.098 186.248C122.136 186.294 122.185 186.331 122.24 186.357C122.295 186.382 122.354 186.394 122.415 186.393H124.645L123.986 189.608H121.575C121.481 189.606 121.389 189.637 121.316 189.696C121.242 189.754 121.192 189.837 121.173 189.929L121.008 190.733C120.996 190.792 120.997 190.853 121.012 190.911C121.027 190.97 121.055 191.024 121.093 191.071C121.132 191.117 121.18 191.154 121.235 191.179C121.29 191.204 121.35 191.216 121.41 191.215H123.656L123.255 193.156C123.245 193.214 123.247 193.274 123.263 193.331C123.279 193.388 123.307 193.441 123.345 193.485C123.383 193.53 123.431 193.566 123.485 193.591C123.539 193.615 123.597 193.627 123.656 193.626H124.46C124.55 193.624 124.636 193.592 124.705 193.535C124.774 193.478 124.822 193.4 124.842 193.313L125.26 191.215H127.43L127.028 193.156C127.018 193.212 127.021 193.269 127.035 193.324C127.049 193.379 127.074 193.431 127.109 193.475C127.145 193.519 127.189 193.556 127.239 193.582C127.29 193.608 127.345 193.623 127.402 193.626H128.206C128.298 193.628 128.389 193.598 128.462 193.541C128.536 193.484 128.587 193.403 128.607 193.313L129.029 191.215H131.441C131.535 191.217 131.626 191.186 131.7 191.127C131.773 191.068 131.824 190.986 131.842 190.894L132.003 190.09C132.015 190.031 132.014 189.97 131.999 189.911C131.984 189.853 131.957 189.798 131.918 189.752C131.88 189.706 131.831 189.669 131.776 189.644C131.721 189.619 131.662 189.606 131.601 189.608H129.359L130.014 186.393H132.425C132.519 186.395 132.611 186.363 132.684 186.305C132.758 186.246 132.808 186.163 132.827 186.071L132.992 185.267C133.004 185.207 133.003 185.145 132.987 185.085C132.971 185.025 132.942 184.97 132.902 184.923C132.861 184.877 132.811 184.84 132.754 184.816C132.697 184.792 132.635 184.781 132.574 184.785ZM127.751 189.608H125.581L126.24 186.393H128.41L127.751 189.608Z" fill="#59AB46"/><rect x="138" y="182" width="50" height="12" rx="1" fill="#D6ECD1"/><path d="M212.574 184.785H210.343L210.745 182.844C210.755 182.787 210.753 182.728 210.738 182.672C210.723 182.616 210.696 182.564 210.659 182.519C210.622 182.475 210.575 182.439 210.523 182.413C210.471 182.388 210.414 182.375 210.355 182.374H209.552C209.459 182.372 209.368 182.402 209.295 182.459C209.222 182.516 209.17 182.597 209.15 182.687L208.74 184.785H206.57L206.972 182.844C206.982 182.788 206.979 182.731 206.965 182.676C206.951 182.621 206.926 182.57 206.89 182.525C206.855 182.481 206.811 182.444 206.76 182.418C206.71 182.392 206.655 182.377 206.598 182.374H205.794C205.701 182.372 205.611 182.402 205.537 182.459C205.464 182.516 205.413 182.597 205.392 182.687L204.991 184.785H202.579C202.485 184.783 202.394 184.814 202.32 184.873C202.247 184.932 202.196 185.015 202.178 185.107L202.013 185.91C202.001 185.969 202.002 186.031 202.017 186.089C202.032 186.147 202.059 186.202 202.098 186.248C202.136 186.294 202.185 186.331 202.24 186.357C202.295 186.382 202.354 186.394 202.415 186.393H204.645L203.986 189.608H201.575C201.481 189.606 201.389 189.637 201.316 189.696C201.242 189.754 201.192 189.837 201.173 189.929L201.008 190.733C200.996 190.792 200.997 190.853 201.012 190.911C201.027 190.97 201.055 191.024 201.093 191.071C201.132 191.117 201.18 191.154 201.235 191.179C201.29 191.204 201.35 191.216 201.41 191.215H203.656L203.255 193.156C203.245 193.214 203.247 193.274 203.263 193.331C203.279 193.388 203.307 193.441 203.345 193.485C203.383 193.53 203.431 193.566 203.485 193.591C203.539 193.615 203.597 193.627 203.656 193.626H204.46C204.55 193.624 204.636 193.592 204.705 193.535C204.774 193.478 204.822 193.4 204.842 193.313L205.26 191.215H207.43L207.028 193.156C207.018 193.212 207.021 193.269 207.035 193.324C207.049 193.379 207.074 193.431 207.109 193.475C207.145 193.519 207.189 193.556 207.239 193.582C207.29 193.608 207.345 193.623 207.402 193.626H208.206C208.298 193.628 208.389 193.598 208.462 193.541C208.536 193.484 208.587 193.403 208.607 193.313L209.029 191.215H211.441C211.535 191.217 211.626 191.186 211.7 191.127C211.773 191.068 211.824 190.986 211.842 190.894L212.003 190.09C212.015 190.031 212.014 189.97 211.999 189.911C211.984 189.853 211.957 189.798 211.918 189.752C211.88 189.706 211.831 189.669 211.776 189.644C211.721 189.619 211.662 189.606 211.601 189.608H209.359L210.014 186.393H212.425C212.519 186.395 212.611 186.363 212.684 186.305C212.758 186.246 212.808 186.163 212.827 186.071L212.992 185.267C213.004 185.207 213.003 185.145 212.987 185.085C212.971 185.025 212.942 184.97 212.902 184.923C212.861 184.877 212.811 184.84 212.754 184.816C212.697 184.792 212.635 184.781 212.574 184.785ZM207.751 189.608H205.581L206.24 186.393H208.41L207.751 189.608Z" fill="#59AB46"/><rect x="218" y="182" width="50" height="12" rx="1" fill="#D6ECD1"/></g><g opacity="0.8"><g filter="url(#filter1_d_235_1820)"><rect x="54" y="225" width="288" height="113" rx="4" fill="white"/></g><rect x="122" y="244" width="188" height="14" rx="1" fill="#DCDDE1"/><circle cx="86.5" cy="257.5" r="17.5" fill="#DCDDE1"/></g><g filter="url(#filter2_d_235_1820)"><rect x="53" y="45" width="289" height="53" fill="white"/></g><circle cx="75.1713" cy="68.1716" r="8" transform="rotate(-45 75.1713 68.1716)" stroke="#0096CC" stroke-width="2"/><line x1="80.8282" y1="73.8284" x2="87.1922" y2="80.1924" stroke="#0096CC" stroke-width="2"/><rect x="106" y="63" width="119" height="16" rx="2" fill="#B5E5FF"/><rect opacity="0.6" x="14" y="150" width="372" height="114" fill="url(#paint0_linear_235_1820)"/></g><defs><filter id="filter0_d_235_1820" x="45.1412" y="100.571" width="305.718" height="130.718" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.32203" operator="erode" in="SourceAlpha" result="effect1_dropShadow_235_1820"/><feOffset dy="4.42938"/><feGaussianBlur stdDeviation="6.0904"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_235_1820"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_235_1820" result="shape"/></filter><filter id="filter1_d_235_1820" x="45.1412" y="220.571" width="305.718" height="130.718" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.32203" operator="erode" in="SourceAlpha" result="effect1_dropShadow_235_1820"/><feOffset dy="4.42938"/><feGaussianBlur stdDeviation="6.0904"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_235_1820"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_235_1820" result="shape"/></filter><filter id="filter2_d_235_1820" x="44.8915" y="40.9457" width="305.217" height="69.217" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.04069" operator="erode" in="SourceAlpha" result="effect1_dropShadow_235_1820"/><feOffset dy="4.05426"/><feGaussianBlur stdDeviation="5.5746"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_235_1820"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_235_1820" result="shape"/></filter><linearGradient id="paint0_linear_235_1820" x1="200" y1="150" x2="200" y2="264" gradientUnits="userSpaceOnUse"><stop stop-color="#F3F4F5" stop-opacity="0"/><stop offset="0.925689" stop-color="#F3F4F5"/></linearGradient><clipPath id="clip0_235_1820"><rect width="396" height="264" fill="white"/></clipPath></defs></svg>',
							'popupContentBtn' 	=> '<div class="ctf-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Lite users get 50% OFF', 'custom-twitter-feeds' ) .'</div>',
							'bullets'       => [
								'heading' => __( 'And get much more!', 'custom-twitter-feeds' ),
								'content' => [
									__( 'Display photos, videos & GIFs', 'custom-twitter-feeds' ),
									__( 'Combine mulitple feeds into one', 'custom-twitter-feeds' ),
									__( 'Unlimited feeds, 12 daily updates', 'custom-twitter-feeds' ),
									__( 'Auto load tweets on scroll', 'custom-twitter-feeds' ),
									__( 'Powerful Tweet filters', 'custom-twitter-feeds' ),
									__( 'View tweets in a lightbox', 'custom-twitter-feeds' ),
									__( 'Multiple tweet layout options', 'custom-twitter-feeds' ),
									__( 'Fast & Friendly Support', 'custom-twitter-feeds' ),
									__( '30 Day Money Back Guarantee', 'custom-twitter-feeds' )
								]
							],
							'buyUrl' 		=> sprintf('https://smashballoon.com/custom-twitter-feeds/demo/?utm_campaign=twitter-free&utm_source=feed-type&utm_medium=search')
						],
						'mentionstimeline' => [
							'heading' 		=> __( 'Mentions Currently Unavailable', 'custom-twitter-feeds' ),
							'description' 	=> __( 'Display tweets in a single feed from other public accounts that mention your Twitter handle. Great for displaying user generated content such as testimonials or reviews.', 'custom-twitter-feeds' ),
							'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g filter="url(#filter0_d_235_9853)"><rect x="38" y="69.681" width="233.43" height="137.698" fill="white"/></g><path fill-rule="evenodd" clip-rule="evenodd" d="M60.6153 83.0218C62.9048 82.5664 65.2779 82.8001 67.4346 83.6934C69.5912 84.5867 71.4345 86.0995 72.7314 88.0404C74.0283 89.9814 74.7205 92.2633 74.7205 94.5976V96.3462C74.7205 97.4669 74.2753 98.5417 73.4829 99.3342C72.6904 100.127 71.6156 100.572 70.4949 100.572C69.3742 100.572 68.2994 100.127 67.5069 99.3342C67.3592 99.1865 67.2236 99.029 67.1007 98.8632C65.9844 99.9578 64.4828 100.572 62.9179 100.572C61.3334 100.572 59.8139 99.9424 58.6935 98.822C57.5731 97.7016 56.9437 96.1821 56.9437 94.5976C56.9437 93.0132 57.5731 91.4936 58.6935 90.3733C59.8139 89.2529 61.3334 88.6235 62.9179 88.6235C64.5023 88.6235 66.0219 89.2529 67.1423 90.3733C68.2626 91.4936 68.8921 93.0132 68.8921 94.5976V96.3462C68.8921 96.7713 69.0609 97.179 69.3615 97.4795C69.6621 97.7801 70.0698 97.949 70.4949 97.949C70.92 97.949 71.3277 97.7801 71.6283 97.4795C71.9289 97.179 72.0977 96.7713 72.0977 96.3462V94.5976C72.0977 92.782 71.5593 91.0072 70.5506 89.4976C69.5419 87.988 68.1083 86.8114 66.4309 86.1166C64.7535 85.4218 62.9077 85.24 61.127 85.5942C59.3463 85.9484 57.7106 86.8227 56.4268 88.1065C55.1429 89.3903 54.2686 91.026 53.9144 92.8067C53.5602 94.5875 53.742 96.4332 54.4368 98.1106C55.1316 99.788 56.3082 101.222 57.8178 102.23C59.3275 103.239 61.1023 103.777 62.9179 103.777L62.9192 103.777C64.5305 103.779 66.1129 103.355 67.5061 102.548C68.1328 102.185 68.9352 102.399 69.2982 103.026C69.6613 103.652 69.4476 104.455 68.8209 104.818C67.0264 105.857 64.9894 106.402 62.9166 106.4L62.9179 105.089V106.4C62.9175 106.4 62.917 106.4 62.9166 106.4C60.5827 106.4 58.3013 105.708 56.3607 104.411C54.4198 103.114 52.907 101.271 52.0137 99.1143C51.1203 96.9577 50.8866 94.5845 51.342 92.2951C51.7974 90.0056 52.9215 87.9025 54.5722 86.2519C56.2228 84.6013 58.3258 83.4772 60.6153 83.0218ZM66.2693 94.5976C66.2693 93.7088 65.9162 92.8564 65.2877 92.2279C64.6592 91.5994 63.8067 91.2463 62.9179 91.2463C62.029 91.2463 61.1766 91.5994 60.5481 92.2279C59.9196 92.8564 59.5665 93.7088 59.5665 94.5976C59.5665 95.4865 59.9196 96.3389 60.5481 96.9674C61.1766 97.5959 62.029 97.949 62.9179 97.949C63.8067 97.949 64.6592 97.5959 65.2877 96.9674C65.9162 96.3389 66.2693 95.4865 66.2693 94.5976Z" fill="#0096CC"/><rect x="89.1445" y="86.7292" width="66.8817" height="15.7369" rx="1.3114" fill="#BFE8FF"/><rect x="53.7363" y="128.694" width="200.645" height="17.0483" rx="2.62281" fill="#DCDDE1"/><rect x="53.7363" y="160.168" width="120.649" height="17.0483" rx="2.62281" fill="#DCDDE1"/><g filter="url(#filter1_d_235_9853)"><rect x="103.459" y="40.0995" width="233.43" height="137.698" transform="rotate(5 103.459 40.0995)" fill="white"/></g><path fill-rule="evenodd" clip-rule="evenodd" d="M124.825 55.3606C127.145 55.1065 129.489 55.5462 131.56 56.624C133.63 57.7019 135.335 59.3696 136.458 61.4162C137.58 63.4628 138.071 65.7963 137.868 68.1218L137.715 69.8637C137.618 70.9801 137.08 72.012 136.222 72.7324C135.363 73.4528 134.254 73.8026 133.137 73.7049C132.021 73.6073 130.989 73.0701 130.269 72.2116C130.134 72.0516 130.013 71.8829 129.905 71.707C128.698 72.7002 127.148 73.1809 125.589 73.0446C124.011 72.9065 122.552 72.147 121.533 70.9332C120.515 69.7195 120.02 68.1508 120.158 66.5724C120.297 64.994 121.056 63.5351 122.27 62.5166C123.484 61.4982 125.052 61.0036 126.631 61.1417C128.209 61.2798 129.668 62.0392 130.686 63.253C131.705 64.4667 132.199 66.0354 132.061 67.6138L131.909 69.3557C131.872 69.7792 132.005 70.2 132.278 70.5257C132.551 70.8513 132.943 71.0551 133.366 71.0921C133.789 71.1292 134.21 70.9965 134.536 70.7232C134.862 70.45 135.065 70.0586 135.102 69.6351L135.255 67.8932C135.413 66.0845 135.031 64.2695 134.158 62.6777C133.285 61.0859 131.959 59.7888 130.349 58.9505C128.738 58.1121 126.915 57.7702 125.111 57.9678C123.306 58.1655 121.6 58.8939 120.209 60.0609C118.818 61.228 117.805 62.7813 117.297 64.5243C116.789 66.2674 116.809 68.122 117.355 69.8535C117.901 71.5851 118.948 73.1159 120.364 74.2523C121.78 75.3887 123.501 76.0798 125.31 76.238L125.311 76.2381C126.916 76.3801 128.529 76.0961 129.988 75.4134C130.644 75.1064 131.424 75.3892 131.731 76.0451C132.039 76.7011 131.756 77.4818 131.1 77.7888C129.221 78.6681 127.145 79.0334 125.08 78.8507L125.196 77.5444L125.081 78.8508C125.081 78.8508 125.08 78.8508 125.08 78.8507C122.755 78.6471 120.543 77.7587 118.722 76.2978C116.902 74.8367 115.555 72.8685 114.854 70.6422C114.152 68.4159 114.126 66.0315 114.779 63.7904C115.432 61.5493 116.735 59.5522 118.523 58.0518C120.312 56.5513 122.505 55.6147 124.825 55.3606ZM129.449 67.3852C129.526 66.4997 129.249 65.6198 128.677 64.9389C128.106 64.258 127.287 63.832 126.402 63.7545C125.517 63.677 124.637 63.9545 123.956 64.5258C123.275 65.0971 122.849 65.9156 122.771 66.801C122.694 67.6865 122.971 68.5664 123.543 69.2473C124.114 69.9282 124.932 70.3543 125.818 70.4317C126.703 70.5092 127.583 70.2317 128.264 69.6604C128.945 69.0891 129.371 68.2707 129.449 67.3852Z" fill="#59AB46"/><rect x="152.924" y="61.5405" width="66.8817" height="15.7369" rx="1.3114" transform="rotate(5 152.924 61.5405)" fill="#D6ECD1"/><rect x="113.994" y="100.26" width="200.645" height="17.0483" rx="2.62281" transform="rotate(5 113.994 100.26)" fill="#DCDDE1"/><rect x="111.25" y="131.614" width="120.649" height="17.0483" rx="2.62281" transform="rotate(5 111.25 131.614)" fill="#DCDDE1"/><defs><filter id="filter0_d_235_9853" x="29.8915" y="65.6267" width="249.647" height="153.915" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.04069" operator="erode" in="SourceAlpha" result="effect1_dropShadow_235_9853"/><feOffset dy="4.05426"/><feGaussianBlur stdDeviation="5.5746"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_235_9853"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_235_9853" result="shape"/></filter><filter id="filter1_d_235_9853" x="83.3485" y="36.0453" width="260.76" height="173.735" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.04069" operator="erode" in="SourceAlpha" result="effect1_dropShadow_235_9853"/><feOffset dy="4.05426"/><feGaussianBlur stdDeviation="5.5746"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_235_9853"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_235_9853" result="shape"/></filter></defs></svg>',
							'popupContentBtn' 	=> '<div class="ctf-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Lite users get 50% OFF', 'custom-twitter-feeds' ) .'</div>',
							'bullets'       => [
								'heading' => __( 'And get much more!', 'custom-twitter-feeds' ),
								'content' => [
									__( 'Display photos, videos & GIFs', 'custom-twitter-feeds' ),
									__( 'Combine mulitple feeds into one', 'custom-twitter-feeds' ),
									__( 'Unlimited feeds, 12 daily updates', 'custom-twitter-feeds' ),
									__( 'Auto load tweets on scroll', 'custom-twitter-feeds' ),
									__( 'Powerful Tweet filters', 'custom-twitter-feeds' ),
									__( 'View tweets in a lightbox', 'custom-twitter-feeds' ),
									__( 'Multiple tweet layout options', 'custom-twitter-feeds' ),
									__( 'Fast & Friendly Support', 'custom-twitter-feeds' ),
									__( '30 Day Money Back Guarantee', 'custom-twitter-feeds' )
								]
							],
							'buyUrl' 		=> sprintf('https://smashballoon.com/custom-twitter-feeds/demo/?utm_campaign=twitter-free&utm_source=feed-type&utm_medium=hashtag')
						],
						'lists' => [
							'heading' 		=> __( 'Upgrade to Elite to get List Feeds', 'custom-twitter-feeds' ),
							'description' 	=> __( 'Twitter Lists are great for curating content. Display these curated lists easily on your WordPress site with our built-in Twitter List feature.', 'custom-twitter-feeds' ),
							'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_235_10306)"><g filter="url(#filter0_d_235_10306)"><rect x="73" y="33" width="250" height="244" fill="white"/></g><path d="M73 37C73 34.7909 74.7909 33 77 33H319C321.209 33 323 34.7909 323 37V133H73V37Z" fill="#FEF4EF"/><path d="M191.75 76.125C191.75 76.6223 191.552 77.0992 191.201 77.4508C190.849 77.8025 190.372 78 189.875 78C189.378 78 188.901 77.8025 188.549 77.4508C188.198 77.0992 188 76.6223 188 76.125C188 75.6277 188.198 75.1508 188.549 74.7992C188.901 74.4475 189.378 74.25 189.875 74.25C190.372 74.25 190.849 74.4475 191.201 74.7992C191.552 75.1508 191.75 75.6277 191.75 76.125Z" fill="#E34F0E"/><path d="M189.875 85.5C190.372 85.5 190.849 85.3025 191.201 84.9508C191.552 84.5992 191.75 84.1223 191.75 83.625C191.75 83.1277 191.552 82.6508 191.201 82.2992C190.849 81.9475 190.372 81.75 189.875 81.75C189.378 81.75 188.901 81.9475 188.549 82.2992C188.198 82.6508 188 83.1277 188 83.625C188 84.1223 188.198 84.5992 188.549 84.9508C188.901 85.3025 189.378 85.5 189.875 85.5Z" fill="#E34F0E"/><path d="M191.75 91.125C191.75 91.6223 191.552 92.0992 191.201 92.4508C190.849 92.8025 190.372 93 189.875 93C189.378 93 188.901 92.8025 188.549 92.4508C188.198 92.0992 188 91.6223 188 91.125C188 90.6277 188.198 90.1508 188.549 89.7992C188.901 89.4475 189.378 89.25 189.875 89.25C190.372 89.25 190.849 89.4475 191.201 89.7992C191.552 90.1508 191.75 90.6277 191.75 91.125Z" fill="#E34F0E"/><path d="M196.75 75.5C196.418 75.5 196.101 75.6317 195.866 75.8661C195.632 76.1005 195.5 76.4185 195.5 76.75C195.5 77.0815 195.632 77.3995 195.866 77.6339C196.101 77.8683 196.418 78 196.75 78H206.75C207.082 78 207.399 77.8683 207.634 77.6339C207.868 77.3995 208 77.0815 208 76.75C208 76.4185 207.868 76.1005 207.634 75.8661C207.399 75.6317 207.082 75.5 206.75 75.5H196.75Z" fill="#E34F0E"/><path d="M195.5 84.25C195.5 83.9185 195.632 83.6005 195.866 83.3661C196.101 83.1317 196.418 83 196.75 83H206.75C207.082 83 207.399 83.1317 207.634 83.3661C207.868 83.6005 208 83.9185 208 84.25C208 84.5815 207.868 84.8995 207.634 85.1339C207.399 85.3683 207.082 85.5 206.75 85.5H196.75C196.418 85.5 196.101 85.3683 195.866 85.1339C195.632 84.8995 195.5 84.5815 195.5 84.25Z" fill="#E34F0E"/><path d="M196.75 90.5C196.418 90.5 196.101 90.6317 195.866 90.8661C195.632 91.1005 195.5 91.4185 195.5 91.75C195.5 92.0815 195.632 92.3995 195.866 92.6339C196.101 92.8683 196.418 93 196.75 93H206.75C207.082 93 207.399 92.8683 207.634 92.6339C207.868 92.3995 208 92.0815 208 91.75C208 91.4185 207.868 91.1005 207.634 90.8661C207.399 90.6317 207.082 90.5 206.75 90.5H196.75Z" fill="#E34F0E"/><path d="M180.5 70.5C180.5 69.1739 181.027 67.9021 181.964 66.9645C182.902 66.0268 184.174 65.5 185.5 65.5H210.5C211.826 65.5 213.098 66.0268 214.036 66.9645C214.973 67.9021 215.5 69.1739 215.5 70.5V95.5C215.5 96.8261 214.973 98.0979 214.036 99.0355C213.098 99.9732 211.826 100.5 210.5 100.5H185.5C184.174 100.5 182.902 99.9732 181.964 99.0355C181.027 98.0979 180.5 96.8261 180.5 95.5V70.5ZM185.5 68C184.837 68 184.201 68.2634 183.732 68.7322C183.263 69.2011 183 69.837 183 70.5V95.5C183 96.163 183.263 96.7989 183.732 97.2678C184.201 97.7366 184.837 98 185.5 98H210.5C211.163 98 211.799 97.7366 212.268 97.2678C212.737 96.7989 213 96.163 213 95.5V70.5C213 69.837 212.737 69.2011 212.268 68.7322C211.799 68.2634 211.163 68 210.5 68H185.5Z" fill="#E34F0E"/><circle cx="108" cy="161" r="13" fill="#DCDDE1"/><circle cx="108" cy="199" r="13" fill="#DCDDE1"/><circle cx="108" cy="237" r="13" fill="#DCDDE1"/><rect x="137" y="155" width="59" height="14" rx="1" fill="#DCDDE1"/><rect x="137" y="193" width="113" height="14" rx="1" fill="#DCDDE1"/><rect x="137" y="231" width="68" height="14" rx="1" fill="#DCDDE1"/><rect opacity="0.6" y="150" width="400" height="114" fill="url(#paint0_linear_235_10306)"/></g><defs><filter id="filter0_d_235_10306" x="64.8915" y="28.9457" width="266.217" height="260.217" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.04069" operator="erode" in="SourceAlpha" result="effect1_dropShadow_235_10306"/><feOffset dy="4.05426"/><feGaussianBlur stdDeviation="5.5746"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_235_10306"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_235_10306" result="shape"/></filter><linearGradient id="paint0_linear_235_10306" x1="200" y1="150" x2="200" y2="264" gradientUnits="userSpaceOnUse"><stop stop-color="#F3F4F5" stop-opacity="0"/><stop offset="0.925689" stop-color="#F3F4F5"/></linearGradient><clipPath id="clip0_235_10306"><rect width="396" height="264" fill="white"/></clipPath></defs></svg>',
							'popupContentBtn' 	=> '<div class="ctf-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Lite users get 50% OFF', 'custom-twitter-feeds' ) .'</div>',
							'bullets'       => [
								'heading' => __( 'And get much more!', 'custom-twitter-feeds' ),
								'content' => [
									__( 'Display photos, videos & GIFs', 'custom-twitter-feeds' ),
									__( 'Combine mulitple feeds into one', 'custom-twitter-feeds' ),
									__( 'Unlimited feeds, 12 daily updates', 'custom-twitter-feeds' ),
									__( 'Auto load tweets on scroll', 'custom-twitter-feeds' ),
									__( 'Powerful Tweet filters', 'custom-twitter-feeds' ),
									__( 'View tweets in a lightbox', 'custom-twitter-feeds' ),
									__( 'Multiple tweet layout options', 'custom-twitter-feeds' ),
									__( 'Fast & Friendly Support', 'custom-twitter-feeds' ),
									__( '30 Day Money Back Guarantee', 'custom-twitter-feeds' )
								]
							],
							'buyUrl' 		=> sprintf('https://smashballoon.com/custom-twitter-feeds/demo/?utm_campaign=twitter-free&utm_source=feed-type&utm_medium=list')
						],
			        	'socialwall' => [
							//Combine all your social media channels into one Social Wall
							'heading'         => __( 'Combine all your social media channels into one', 'custom-twitter-feeds' ) . ' <span>' . __( 'Social Wall', 'custom-twitter-feeds' ) . '</span>',
							'description' 	=> __( '<span class="sb-social-wall">A dash of Instagram, a sprinkle of Facebook, a spoonful of Twitter, and a dollop of YouTube, all in the same feed.</span>', 'custom-twitter-feeds' ),
							'popupContentBtn' 	=> '<div class="ctf-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Lite users get 50% OFF', 'custom-twitter-feeds' ) .'</div>',
							'img' 			=> '<svg width="397" height="264" viewBox="0 0 397 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0)"><g filter="url(#filter0_ddd)"><rect x="18.957" y="63" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip1)"><path d="M18.957 63H132.769V176.812H18.957V63Z" fill="#0068A0"/><rect x="56.957" y="106" width="105" height="105" rx="9" fill="#005B8C"/></g><path d="M36.0293 165.701C31.4649 165.701 27.7305 169.427 27.7305 174.017C27.7305 178.166 30.7678 181.61 34.7347 182.232V176.423H32.6268V174.017H34.7347V172.183C34.7347 170.1 35.9712 168.954 37.8716 168.954C38.7762 168.954 39.7222 169.112 39.7222 169.112V171.162H38.6766C37.6475 171.162 37.3239 171.801 37.3239 172.456V174.017H39.6309L39.2575 176.423H37.3239V182.232C39.2794 181.924 41.0602 180.926 42.3446 179.419C43.629 177.913 44.3325 175.996 44.3281 174.017C44.3281 169.427 40.5936 165.701 36.0293 165.701Z" fill="#006BFA"/><rect x="53.1016" y="169.699" width="41.2569" height="9.95855" rx="1.42265" fill="#D0D1D7"/><g filter="url(#filter1_ddd)"><rect x="18.957" y="201" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip2)"><path d="M18.957 201H132.769V314.812H18.957V201Z" fill="#EC352F"/><circle cx="23.957" cy="243" r="59" fill="#FE544F"/></g><g filter="url(#filter2_ddd)"><rect x="139.957" y="23" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip3)"><path d="M139.957 23H253.769V136.812H139.957V23Z" fill="#8C8F9A"/><circle cx="127.457" cy="142.5" r="78.5" fill="#D0D1D7"/></g><path d="M157.026 129.493C154.537 129.493 152.553 131.516 152.553 133.967C152.553 136.456 154.537 138.44 157.026 138.44C159.477 138.44 161.5 136.456 161.5 133.967C161.5 131.516 159.477 129.493 157.026 129.493ZM157.026 136.884C155.431 136.884 154.109 135.601 154.109 133.967C154.109 132.372 155.392 131.088 157.026 131.088C158.621 131.088 159.905 132.372 159.905 133.967C159.905 135.601 158.621 136.884 157.026 136.884ZM162.706 129.338C162.706 128.754 162.239 128.287 161.655 128.287C161.072 128.287 160.605 128.754 160.605 129.338C160.605 129.921 161.072 130.388 161.655 130.388C162.239 130.388 162.706 129.921 162.706 129.338ZM165.662 130.388C165.584 128.987 165.273 127.743 164.262 126.731C163.25 125.72 162.005 125.409 160.605 125.331C159.166 125.253 154.848 125.253 153.408 125.331C152.008 125.409 150.802 125.72 149.752 126.731C148.74 127.743 148.429 128.987 148.351 130.388C148.274 131.827 148.274 136.145 148.351 137.585C148.429 138.985 148.74 140.191 149.752 141.241C150.802 142.253 152.008 142.564 153.408 142.642C154.848 142.719 159.166 142.719 160.605 142.642C162.005 142.564 163.25 142.253 164.262 141.241C165.273 140.191 165.584 138.985 165.662 137.585C165.74 136.145 165.74 131.827 165.662 130.388ZM163.795 139.102C163.523 139.88 162.9 140.463 162.161 140.774C160.994 141.241 158.271 141.124 157.026 141.124C155.742 141.124 153.019 141.241 151.891 140.774C151.113 140.463 150.53 139.88 150.219 139.102C149.752 137.974 149.868 135.25 149.868 133.967C149.868 132.722 149.752 129.999 150.219 128.832C150.53 128.093 151.113 127.509 151.891 127.198C153.019 126.731 155.742 126.848 157.026 126.848C158.271 126.848 160.994 126.731 162.161 127.198C162.9 127.47 163.484 128.093 163.795 128.832C164.262 129.999 164.145 132.722 164.145 133.967C164.145 135.25 164.262 137.974 163.795 139.102Z" fill="url(#paint0_linear)"/><rect x="174.102" y="129.699" width="41.2569" height="9.95855" rx="1.42265" fill="#D0D1D7"/><g filter="url(#filter3_ddd)"><rect x="139.957" y="161" width="114" height="109" rx="2.8453" fill="white"/></g><rect x="148.957" y="194" width="91" height="8" rx="1.42265" fill="#D0D1D7"/><rect x="148.957" y="208" width="51" height="8" rx="1.42265" fill="#D0D1D7"/><path d="M164.366 172.062C163.788 172.324 163.166 172.497 162.521 172.579C163.181 172.182 163.691 171.552 163.931 170.794C163.308 171.169 162.618 171.432 161.891 171.582C161.298 170.937 160.466 170.562 159.521 170.562C157.758 170.562 156.318 172.002 156.318 173.779C156.318 174.034 156.348 174.282 156.401 174.514C153.731 174.379 151.353 173.097 149.771 171.154C149.493 171.627 149.336 172.182 149.336 172.767C149.336 173.884 149.898 174.874 150.768 175.437C150.236 175.437 149.741 175.287 149.306 175.062V175.084C149.306 176.644 150.416 177.949 151.886 178.242C151.414 178.371 150.918 178.389 150.438 178.294C150.642 178.934 151.041 179.493 151.579 179.894C152.117 180.295 152.767 180.517 153.438 180.529C152.301 181.43 150.891 181.916 149.441 181.909C149.186 181.909 148.931 181.894 148.676 181.864C150.101 182.779 151.796 183.312 153.611 183.312C159.521 183.312 162.768 178.407 162.768 174.154C162.768 174.012 162.768 173.877 162.761 173.734C163.391 173.284 163.931 172.714 164.366 172.062Z" fill="#1B90EF"/><g filter="url(#filter4_ddd)"><rect x="260.957" y="63" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip4)"><rect x="260.957" y="63" width="113.812" height="113.812" fill="#D72C2C"/><path d="M283.359 103.308L373.461 193.41H208.793L283.359 103.308Z" fill="#DF5757"/></g><path d="M276.37 176.456L280.677 173.967L276.37 171.477V176.456ZM285.963 169.958C286.071 170.348 286.145 170.871 286.195 171.535C286.253 172.199 286.278 172.772 286.278 173.27L286.328 173.967C286.328 175.784 286.195 177.12 285.963 177.975C285.755 178.722 285.274 179.203 284.527 179.411C284.137 179.519 283.423 179.593 282.328 179.643C281.249 179.701 280.262 179.726 279.349 179.726L278.029 179.776C274.552 179.776 272.386 179.643 271.531 179.411C270.784 179.203 270.303 178.722 270.096 177.975C269.988 177.585 269.913 177.062 269.863 176.398C269.805 175.734 269.78 175.162 269.78 174.664L269.73 173.967C269.73 172.149 269.863 170.813 270.096 169.958C270.303 169.212 270.784 168.73 271.531 168.523C271.921 168.415 272.635 168.34 273.73 168.29C274.809 168.232 275.797 168.207 276.71 168.207L278.029 168.158C281.506 168.158 283.672 168.29 284.527 168.523C285.274 168.73 285.755 169.212 285.963 169.958Z" fill="#EB2121"/><rect x="295.102" y="169.699" width="41.2569" height="9.95855" rx="1.42265" fill="#D0D1D7"/><g filter="url(#filter5_ddd)"><rect x="260.957" y="201" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip5)"><rect x="260.957" y="201" width="113.812" height="113.812" fill="#59AB46"/><circle cx="374.457" cy="235.5" r="44.5" fill="#468737"/></g><g clip-path="url(#clip6)"><path d="M139.957 228H253.957V296C253.957 296.552 253.509 297 252.957 297H140.957C140.405 297 139.957 296.552 139.957 296V228Z" fill="#0068A0"/><circle cx="227.957" cy="245" r="34" fill="#004D77"/></g></g><defs><filter id="filter0_ddd" x="0.462572" y="53.0414" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter1_ddd" x="0.462572" y="191.041" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter2_ddd" x="121.463" y="13.0414" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter3_ddd" x="121.463" y="151.041" width="150.989" height="145.989" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter4_ddd" x="242.463" y="53.0414" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter5_ddd" x="242.463" y="191.041" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><linearGradient id="paint0_linear" x1="154.502" y1="158.603" x2="191.208" y2="121.133" gradientUnits="userSpaceOnUse"><stop stop-color="white"/><stop offset="0.147864" stop-color="#F6640E"/><stop offset="0.443974" stop-color="#BA03A7"/><stop offset="0.733337" stop-color="#6A01B9"/><stop offset="1" stop-color="#6B01B9"/></linearGradient><clipPath id="clip0"><rect width="396" height="264" fill="white" transform="translate(0.957031)"/></clipPath><clipPath id="clip1"><path d="M18.957 65.3711C18.957 64.0616 20.0186 63 21.3281 63H130.398C131.708 63 132.769 64.0616 132.769 65.3711V156.895H18.957V65.3711Z" fill="white"/></clipPath><clipPath id="clip2"><path d="M18.957 203.371C18.957 202.062 20.0186 201 21.3281 201H130.398C131.708 201 132.769 202.062 132.769 203.371V294.895H18.957V203.371Z" fill="white"/></clipPath><clipPath id="clip3"><path d="M139.957 25.3711C139.957 24.0616 141.019 23 142.328 23H251.398C252.708 23 253.769 24.0616 253.769 25.3711V116.895H139.957V25.3711Z" fill="white"/></clipPath><clipPath id="clip4"><path d="M260.957 65.3711C260.957 64.0616 262.019 63 263.328 63H372.398C373.708 63 374.769 64.0616 374.769 65.3711V156.895H260.957V65.3711Z" fill="white"/></clipPath><clipPath id="clip5"><path d="M260.957 203.371C260.957 202.062 262.019 201 263.328 201H372.398C373.708 201 374.769 202.062 374.769 203.371V294.895H260.957V203.371Z" fill="white"/></clipPath><clipPath id="clip6"><path d="M139.957 228H253.957V296C253.957 296.552 253.509 297 252.957 297H140.957C140.405 297 139.957 296.552 139.957 296V228Z" fill="white"/></clipPath></defs></svg>',
							'demoUrl' 		=> 'https://smashballoon.com/social-wall/demo/?utm_campaign=twitter-free&utm_source=feed-type&utm_medium=social-wall&utm_content=learn-more',
							'buyUrl' 		=> sprintf('https://smashballoon.com/social-wall/pricing/?edd_license_key=%s&upgrade=true&utm_campaign=twitter-free&utm_source=feed-type&utm_medium=social-wall&utm_content=upgrade', $license_key),
							'bullets'       => [
								'heading' => __( 'Upgrade to the All Access Bundle and get:', 'custom-twitter-feeds' ),
								'content' => [
									__( 'Custom Facebook Feed Pro', 'custom-twitter-feeds' ),
									__( 'All Pro Facebook Extensions', 'custom-twitter-feeds' ),
									__( 'Custom Twitter Feeds Pro', 'custom-twitter-feeds' ),
									__( 'Instagram Feed Pro', 'custom-twitter-feeds' ),
									__( 'YouTube Feeds Pro', 'custom-twitter-feeds' ),
									__( 'Social Wall Pro', 'custom-twitter-feeds' ),
								]
							],
						],

						//Other Types
						'feedLayout' => [
							'heading' 		=> __( 'Upgrade to Pro to get Advanced Layouts', 'custom-twitter-feeds' ),
							'description' 	=> __( 'Create beautiful and responsive full width grids of your Twitter content. Create multiple grids on the same page or throughout your site, or use responsive rotating carousels/slideshows of your Tweets instead.', 'custom-twitter-feeds' ),
							'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_1081_60249)" filter="url(#filter0_d_1081_60249)"><rect width="233.535" height="135.012" transform="translate(100.297 56.9052) rotate(5)" fill="white"/><rect x="127.309" y="132.337" width="80.2777" height="75.4124" rx="1.21633" transform="rotate(5 127.309 132.337)" fill="#FCE1D5"/><rect x="218.141" y="126.853" width="80.2777" height="80.2777" rx="1.21633" transform="rotate(5 218.141 126.853)" fill="#BFE8FF"/><circle cx="143.137" cy="90.9873" r="12.1633" transform="rotate(5 143.137 90.9873)" fill="#DCDDE1"/><rect x="167.795" y="88.2607" width="66.8981" height="9.73063" rx="1.21633" transform="rotate(5 167.795 88.2607)" fill="#DCDDE1"/></g><g clip-path="url(#clip1_1081_60249)" filter="url(#filter1_d_1081_60249)"><rect width="233.535" height="135.012" transform="translate(64 43.5536)" fill="white"/><rect x="117.518" y="66.6638" width="126.498" height="85.143" rx="1.21633" fill="#43A6DB"/><rect x="127.248" y="111.668" width="102.172" height="8.5143" rx="1.21633" fill="#F7FDFF"/><rect x="127.248" y="128.697" width="77.845" height="8.5143" rx="1.21633" fill="#F7FDFF"/><circle cx="137.587" cy="89.1659" r="10.3388" fill="#F7FDFF"/><rect x="252.531" y="66.6638" width="126.498" height="85.143" rx="1.21633" fill="#F6966B"/><rect x="262.262" y="111.668" width="102.172" height="8.5143" rx="1.21633" fill="#FEF4EF"/><rect x="262.262" y="128.697" width="77.845" height="8.5143" rx="1.21633" fill="#FEF4EF"/><circle cx="272.601" cy="89.1659" r="10.3388" fill="#FEF4EF"/><rect x="-18.7109" y="66.6638" width="126.498" height="85.143" rx="1.21633" fill="#96CE89"/><rect x="-8.98047" y="111.668" width="102.172" height="8.5143" rx="1.21633" fill="#E6F4E3"/><rect x="-8.98047" y="128.697" width="77.845" height="8.5143" rx="1.21633" fill="#E6F4E3"/></g><defs><filter id="filter0_d_1081_60249" x="80.4208" y="52.8509" width="260.631" height="171.07" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.04069" operator="erode" in="SourceAlpha" result="effect1_dropShadow_1081_60249"/><feOffset dy="4.05426"/><feGaussianBlur stdDeviation="5.5746"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1081_60249"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1081_60249" result="shape"/></filter><filter id="filter1_d_1081_60249" x="55.8915" y="39.4993" width="249.752" height="151.229" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.04069" operator="erode" in="SourceAlpha" result="effect1_dropShadow_1081_60249"/><feOffset dy="4.05426"/><feGaussianBlur stdDeviation="5.5746"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1081_60249"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1081_60249" result="shape"/></filter><clipPath id="clip0_1081_60249"><rect width="233.535" height="135.012" fill="white" transform="translate(100.297 56.9052) rotate(5)"/></clipPath><clipPath id="clip1_1081_60249"><rect width="233.535" height="135.012" fill="white" transform="translate(64 43.5536)"/></clipPath></defs></svg>',
							'popupContentBtn' 	=> '<div class="ctf-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Lite users get 50% OFF', 'custom-twitter-feeds' ) .'</div>',
							'bullets'       => [
								'heading' => __( 'And get much more!', 'custom-twitter-feeds' ),
								'content' => [
									__( 'Display photos, videos & GIFs', 'custom-twitter-feeds' ),
									__( 'Combine mulitple feeds into one', 'custom-twitter-feeds' ),
									__( 'Unlimited feeds, 12 daily updates', 'custom-twitter-feeds' ),
									__( 'Auto load tweets on scroll', 'custom-twitter-feeds' ),
									__( 'Powerful Tweet filters', 'custom-twitter-feeds' ),
									__( 'View tweets in a lightbox', 'custom-twitter-feeds' ),
									__( 'Multiple tweet layout options', 'custom-twitter-feeds' ),
									__( 'Fast & Friendly Support', 'custom-twitter-feeds' ),
									__( '30 Day Money Back Guarantee', 'custom-twitter-feeds' )
								]
							],
							'buyUrl' 		=> sprintf('https://smashballoon.com/custom-twitter-feeds/demo/?utm_campaign=twitter-free&utm_source=feed-type&utm_medium=hashtag')
						],
						'advancedFilters' => [
							'heading' 		=> __( 'Upgrade to Pro to get Advanced Filters', 'custom-twitter-feeds' ),
							'description' 	=> __( 'Use our filter settings to only show Tweets in your feed which contain specific words or hashtags, or do the opposite and hide any which contain them.', 'custom-twitter-feeds' ),
							'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g filter="url(#filter0_d_605_67392)"><rect x="104" y="71" width="233.43" height="137.698" fill="white"/></g><rect x="162.148" y="91.0482" width="66.8817" height="15.7369" rx="1.3114" fill="#F9BBA0"/><rect x="119.738" y="130.013" width="200.645" height="17.0483" rx="2.62281" fill="#DCDDE1"/><rect x="119.738" y="161.487" width="120.649" height="17.0483" rx="2.62281" fill="#DCDDE1"/><circle cx="134" cy="98" r="14" fill="#F9BBA0"/><g filter="url(#filter1_d_605_67392)"><rect x="48" y="45" width="233.43" height="137.698" fill="white"/></g><rect x="106.148" y="65.0482" width="66.8817" height="15.7369" rx="1.3114" fill="#B6DDAD"/><rect x="63.7383" y="104.013" width="200.645" height="17.0483" rx="2.62281" fill="#DCDDE1"/><rect x="63.7383" y="135.487" width="120.649" height="17.0483" rx="2.62281" fill="#DCDDE1"/><circle cx="78" cy="72" r="14" fill="#B6DDAD"/><circle cx="255.751" cy="43.7509" r="18.7509" fill="#0096CC"/><g clip-path="url(#clip0_605_67392)"><path d="M255.245 41.217C254.708 41.217 254.192 41.4306 253.812 41.8108C253.432 42.1909 253.218 42.7065 253.218 43.2442C253.218 43.7818 253.432 44.2974 253.812 44.6775C254.192 45.0577 254.708 45.2713 255.245 45.2713C255.783 45.2713 256.299 45.0577 256.679 44.6775C257.059 44.2974 257.272 43.7818 257.272 43.2442C257.272 42.7065 257.059 42.1909 256.679 41.8108C256.299 41.4306 255.783 41.217 255.245 41.217ZM255.245 46.6227C254.349 46.6227 253.49 46.2667 252.856 45.6331C252.223 44.9995 251.867 44.1402 251.867 43.2442C251.867 42.3481 252.223 41.4888 252.856 40.8552C253.49 40.2216 254.349 39.8656 255.245 39.8656C256.141 39.8656 257.001 40.2216 257.634 40.8552C258.268 41.4888 258.624 42.3481 258.624 43.2442C258.624 44.1402 258.268 44.9995 257.634 45.6331C257.001 46.2667 256.141 46.6227 255.245 46.6227ZM255.245 38.1763C251.867 38.1763 248.981 40.2778 247.812 43.2442C248.981 46.2105 251.867 48.312 255.245 48.312C258.624 48.312 261.509 46.2105 262.678 43.2442C261.509 40.2778 258.624 38.1763 255.245 38.1763Z" fill="white"/></g><circle cx="311.751" cy="71.7509" r="18.7509" fill="#D72C2C"/><g clip-path="url(#clip1_605_67392)"><path d="M311.127 69.217L313.269 71.3522V71.2441C313.269 70.7065 313.055 70.1909 312.675 69.8107C312.295 69.4305 311.779 69.217 311.241 69.217H311.127ZM308.221 69.7575L309.268 70.8049C309.235 70.9468 309.214 71.0887 309.214 71.2441C309.214 71.7817 309.428 72.2973 309.808 72.6775C310.188 73.0577 310.704 73.2712 311.241 73.2712C311.39 73.2712 311.539 73.251 311.681 73.2172L312.728 74.2645C312.275 74.4875 311.775 74.6227 311.241 74.6227C310.345 74.6227 309.486 74.2667 308.852 73.6331C308.219 72.9995 307.863 72.1401 307.863 71.2441C307.863 70.7103 307.998 70.2103 308.221 69.7575ZM304.484 66.0209L306.025 67.5615L306.329 67.8656C305.214 68.744 304.336 69.8927 303.809 71.2441C304.978 74.2105 307.863 76.3119 311.241 76.3119C312.289 76.3119 313.289 76.1092 314.201 75.7443L314.492 76.0281L316.465 78.0012L317.323 77.143L305.342 65.1627L304.484 66.0209ZM311.241 67.8656C312.137 67.8656 312.997 68.2215 313.63 68.8551C314.264 69.4887 314.62 70.3481 314.62 71.2441C314.62 71.6766 314.532 72.0955 314.377 72.4739L316.357 74.4537C317.37 73.6091 318.181 72.5009 318.674 71.2441C317.505 68.2777 314.62 66.1763 311.241 66.1763C310.295 66.1763 309.39 66.3452 308.539 66.6493L310.005 68.1021C310.39 67.9534 310.802 67.8656 311.241 67.8656Z" fill="white"/></g><defs><filter id="filter0_d_605_67392" x="95.8915" y="66.9457" width="249.647" height="153.915" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.04069" operator="erode" in="SourceAlpha" result="effect1_dropShadow_605_67392"/><feOffset dy="4.05426"/><feGaussianBlur stdDeviation="5.5746"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_605_67392"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_605_67392" result="shape"/></filter><filter id="filter1_d_605_67392" x="39.8915" y="40.9457" width="249.647" height="153.915" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.04069" operator="erode" in="SourceAlpha" result="effect1_dropShadow_605_67392"/><feOffset dy="4.05426"/><feGaussianBlur stdDeviation="5.5746"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_605_67392"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_605_67392" result="shape"/></filter><clipPath id="clip0_605_67392"><rect width="16.217" height="16.217" fill="white" transform="translate(247.137 35.1356)"/></clipPath><clipPath id="clip1_605_67392"><rect width="16.217" height="16.217" fill="white" transform="translate(303.133 63.1356)"/></clipPath></defs></svg>',
							'popupContentBtn' 	=> '<div class="ctf-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Lite users get 50% OFF', 'custom-twitter-feeds' ) .'</div>',
							'bullets'       => [
								'heading' => __( 'And get much more!', 'custom-twitter-feeds' ),
								'content' => [
									__( 'Display photos, videos & GIFs', 'custom-twitter-feeds' ),
									__( 'Combine mulitple feeds into one', 'custom-twitter-feeds' ),
									__( 'Unlimited feeds, 12 daily updates', 'custom-twitter-feeds' ),
									__( 'Auto load tweets on scroll', 'custom-twitter-feeds' ),
									__( 'Powerful Tweet filters', 'custom-twitter-feeds' ),
									__( 'View tweets in a lightbox', 'custom-twitter-feeds' ),
									__( 'Multiple tweet layout options', 'custom-twitter-feeds' ),
									__( 'Fast & Friendly Support', 'custom-twitter-feeds' ),
									__( '30 Day Money Back Guarantee', 'custom-twitter-feeds' )
								]
							],
							'buyUrl' 		=> sprintf('https://smashballoon.com/custom-twitter-feeds/demo/?utm_campaign=twitter-free&utm_source=feed-type&utm_medium=hashtag')
						],

						'mediaExtension' => [
							'heading' 		=> __( 'Upgrade to Pro to add Media, Replies & Twitter Cards', 'custom-twitter-feeds' ),
							'description' 	=> __( 'Display photos, videos, and gifs in your Tweets, show replies to other Tweets, and display beautiful Twitter cards to showcase your links.', 'custom-twitter-feeds' ),
							'img' 			=> '<svg width="384" height="264" viewBox="0 0 384 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g filter="url(#filter0_d_605_69520)"><rect x="61" y="61.5052" width="190.262" height="125.322" rx="3.41788" transform="rotate(-2 61 61.5052)" fill="white"/></g><g clip-path="url(#clip0_605_69520)"><rect x="68.0703" y="68.0983" width="176.59" height="111.651" rx="2.27859" transform="rotate(-2 68.0703 68.0983)" fill="#E2F5FF"/><path d="M120.926 177.971L248.449 173.518L244.552 61.9354L117.029 66.3886L120.926 177.971Z" fill="#86D0F9"/><path d="M152.809 176.858L248.451 173.518L246.105 106.341L153.879 109.561C151.992 109.627 150.516 111.21 150.582 113.096L152.809 176.858Z" fill="#E34F0E"/><mask id="path-5-inside-1_605_69520" fill="white"><path d="M131.614 157.648C131.862 164.734 129.608 171.68 125.248 177.27C120.887 182.86 114.699 186.738 107.767 188.223C100.834 189.708 93.6003 188.707 87.3325 185.394C81.0646 182.081 76.1626 176.667 73.4854 170.103L89.9485 163.389C91.0491 166.088 93.0644 168.313 95.6411 169.675C98.2179 171.037 101.192 171.449 104.042 170.838C106.892 170.228 109.436 168.634 111.228 166.335C113.021 164.037 113.947 161.182 113.846 158.269L131.614 157.648Z"/></mask><path d="M131.614 157.648C131.862 164.734 129.608 171.68 125.248 177.27C120.887 182.86 114.699 186.738 107.767 188.223C100.834 189.708 93.6003 188.707 87.3325 185.394C81.0646 182.081 76.1626 176.667 73.4854 170.103L89.9485 163.389C91.0491 166.088 93.0644 168.313 95.6411 169.675C98.2179 171.037 101.192 171.449 104.042 170.838C106.892 170.228 109.436 168.634 111.228 166.335C113.021 164.037 113.947 161.182 113.846 158.269L131.614 157.648Z" stroke="#59AB46" stroke-width="36.9005" mask="url(#path-5-inside-1_605_69520)"/></g><g filter="url(#filter1_d_605_69520)"><path d="M140.928 128.126C140.989 127.246 141.753 126.582 142.633 126.644L310.757 138.4C311.637 138.461 312.301 139.225 312.239 140.105L307.225 211.817C307.163 212.697 306.4 213.361 305.52 213.299L137.395 201.543C136.515 201.481 135.852 200.718 135.913 199.838L140.928 128.126Z" fill="white"/></g><rect x="187.566" y="148.202" width="102.239" height="12.7799" rx="1.04748" transform="rotate(4 187.566 148.202)" fill="#BFE8FF"/><rect x="185.785" y="173.699" width="69.4907" height="13.5786" rx="2.09495" transform="rotate(4 185.785 173.699)" fill="#BFE8FF"/><g clip-path="url(#clip1_605_69520)"><path d="M151.329 148.916L158.468 142.752C159.093 142.212 160.078 142.65 160.078 143.488L160.078 146.736C166.593 146.81 171.76 148.116 171.76 154.29C171.76 156.782 170.154 159.251 168.379 160.542C167.826 160.945 167.036 160.439 167.241 159.786C169.08 153.905 166.368 152.343 160.078 152.253L160.078 155.818C160.078 156.658 159.092 157.094 158.468 156.555L151.329 150.39C150.88 150.002 150.88 149.305 151.329 148.916Z" fill="#0096CC"/></g><defs><filter id="filter0_d_605_69520" x="52.0028" y="50.4252" width="212.514" height="149.88" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.41788" operator="erode" in="SourceAlpha" result="effect1_dropShadow_605_69520"/><feOffset dy="4.55717"/><feGaussianBlur stdDeviation="6.26611"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_605_69520"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_605_69520" result="shape"/></filter><filter id="filter1_d_605_69520" x="127.005" y="120.973" width="194.143" height="104.474" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="3.23831"/><feGaussianBlur stdDeviation="4.45268"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.15 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_605_69520"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_605_69520" result="shape"/></filter><clipPath id="clip0_605_69520"><rect x="68.0703" y="68.0983" width="176.59" height="111.651" rx="2.27859" transform="rotate(-2 68.0703 68.0983)" fill="white"/></clipPath><clipPath id="clip1_605_69520"><rect width="20.7673" height="20.7673" fill="white" transform="translate(150.992 141.216)"/></clipPath></defs></svg>',
							'popupContentBtn' 	=> '<div class="ctf-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Lite users get 50% OFF', 'custom-twitter-feeds' ) .'</div>',
							'bullets'       => [
								'heading' => __( 'And get much more!', 'custom-twitter-feeds' ),
								'content' => [
									__( 'Display photos, videos & GIFs', 'custom-twitter-feeds' ),
									__( 'Combine mulitple feeds into one', 'custom-twitter-feeds' ),
									__( 'Unlimited feeds, 12 daily updates', 'custom-twitter-feeds' ),
									__( 'Auto load tweets on scroll', 'custom-twitter-feeds' ),
									__( 'Powerful Tweet filters', 'custom-twitter-feeds' ),
									__( 'View tweets in a lightbox', 'custom-twitter-feeds' ),
									__( 'Multiple tweet layout options', 'custom-twitter-feeds' ),
									__( 'Fast & Friendly Support', 'custom-twitter-feeds' ),
									__( '30 Day Money Back Guarantee', 'custom-twitter-feeds' )
								]
							],
							'buyUrl' 		=> sprintf('https://smashballoon.com/custom-twitter-feeds/demo/?utm_campaign=twitter-free&utm_source=feed-type&utm_medium=hashtag')
						],

						'autoscrollExtension' => [
							'heading' 		=> __( 'Upgrade to Pro to add<br/>Infinite Scroll', 'custom-twitter-feeds' ),
							'description' 	=> __( 'Automatically load more posts as the user reaches the end of the feed .', 'custom-twitter-feeds' ),
							'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_605_69687)"><path fill-rule="evenodd" clip-rule="evenodd" d="M205.405 -24C204.002 -24 202.865 -22.8626 202.865 -21.4595V59.8378C202.865 61.2409 204.002 62.3784 205.405 62.3784H303.216C304.619 62.3784 305.757 61.2409 305.757 59.8378V-21.4595C305.757 -22.8626 304.619 -24 303.216 -24H205.405ZM88.5405 -10.027C87.1374 -10.027 86 -8.88959 86 -7.48648V73.8108C86 75.2139 87.1374 76.3513 88.5405 76.3513H186.351C187.754 76.3513 188.892 75.2139 188.892 73.8108V-7.48649C188.892 -8.88959 187.754 -10.027 186.351 -10.027H88.5405ZM88.5405 90.3243C87.1374 90.3243 86 91.4618 86 92.8649V174.162C86 175.565 87.1374 176.703 88.5405 176.703H186.351C187.754 176.703 188.892 175.565 188.892 174.162V92.8649C188.892 91.4618 187.754 90.3243 186.351 90.3243H88.5405ZM202.865 78.8919C202.865 77.4888 204.002 76.3514 205.405 76.3514H303.216C304.619 76.3514 305.757 77.4888 305.757 78.8919V160.189C305.757 161.592 304.619 162.73 303.216 162.73H205.405C204.002 162.73 202.865 161.592 202.865 160.189V78.8919Z" fill="#DCDDE1"/><rect x="86" y="194" width="219" height="43" rx="4" fill="#0096CC"/><path d="M118.999 212.146C118.711 212.146 118.52 211.954 118.52 211.667V207.834C118.52 207.546 118.711 207.354 118.999 207.354C119.286 207.354 119.478 207.546 119.478 207.834V211.667C119.478 211.954 119.286 212.146 118.999 212.146Z" fill="white" stroke="white" stroke-width="0.887334"/><path opacity="0.3" d="M118.999 223.646C118.711 223.646 118.52 223.454 118.52 223.167V219.333C118.52 219.046 118.711 218.854 118.999 218.854C119.286 218.854 119.478 219.046 119.478 219.333V223.167C119.478 223.454 119.286 223.646 118.999 223.646Z" fill="white" stroke="white" stroke-width="0.887334"/><path opacity="0.3" d="M120.913 212.625C120.818 212.625 120.77 212.625 120.674 212.577C120.482 212.433 120.386 212.194 120.53 211.954L122.447 208.648C122.591 208.456 122.83 208.361 123.07 208.504C123.261 208.648 123.357 208.888 123.213 209.127L121.297 212.433C121.201 212.529 121.057 212.625 120.913 212.625Z" fill="white" stroke="white" stroke-width="0.887334"/><path opacity="0.3" d="M115.163 222.592C115.068 222.592 115.02 222.592 114.924 222.544C114.732 222.4 114.636 222.16 114.78 221.921L116.697 218.615C116.841 218.423 117.08 218.327 117.32 218.471C117.511 218.615 117.607 218.854 117.463 219.094L115.547 222.4C115.451 222.496 115.307 222.592 115.163 222.592Z" fill="white" stroke="white" stroke-width="0.887334"/><path opacity="0.93" d="M117.08 212.625C116.936 212.625 116.793 212.529 116.697 212.385L114.78 209.079C114.636 208.888 114.732 208.6 114.924 208.456C115.116 208.313 115.403 208.408 115.547 208.6L117.463 211.906C117.607 212.098 117.511 212.385 117.32 212.529C117.224 212.625 117.176 212.625 117.08 212.625Z" fill="white" stroke="white" stroke-width="0.887334"/><path opacity="0.3" d="M122.83 222.592C122.686 222.592 122.543 222.496 122.447 222.352L120.53 219.046C120.386 218.854 120.482 218.567 120.674 218.423C120.866 218.279 121.153 218.375 121.297 218.567L123.213 221.873C123.357 222.065 123.261 222.352 123.07 222.496C122.974 222.544 122.926 222.592 122.83 222.592Z" fill="white" stroke="white" stroke-width="0.887334"/><path opacity="0.65" d="M115.168 215.979H111.335C111.047 215.979 110.855 215.787 110.855 215.5C110.855 215.212 111.047 215.021 111.335 215.021H115.168C115.455 215.021 115.647 215.212 115.647 215.5C115.647 215.787 115.455 215.979 115.168 215.979Z" fill="white" stroke="white" stroke-width="0.887334"/><path opacity="0.3" d="M126.668 215.979H122.835C122.547 215.979 122.355 215.787 122.355 215.5C122.355 215.212 122.547 215.021 122.835 215.021H126.668C126.955 215.021 127.147 215.212 127.147 215.5C127.147 215.787 126.955 215.979 126.668 215.979Z" fill="white" stroke="white" stroke-width="0.887334"/><path opacity="0.86" d="M115.692 214.015C115.597 214.015 115.549 214.015 115.453 213.967L112.147 212.05C111.955 211.906 111.859 211.667 112.003 211.427C112.147 211.236 112.386 211.14 112.626 211.284L115.932 213.2C116.124 213.344 116.219 213.584 116.076 213.823C115.98 213.967 115.836 214.015 115.692 214.015Z" fill="white" stroke="white" stroke-width="0.887334"/><path opacity="0.3" d="M125.657 219.765C125.561 219.765 125.513 219.765 125.418 219.717L122.111 217.8C121.92 217.656 121.824 217.417 121.968 217.177C122.111 216.986 122.351 216.89 122.591 217.034L125.897 218.95C126.088 219.094 126.184 219.333 126.041 219.573C125.945 219.717 125.801 219.765 125.657 219.765Z" fill="white" stroke="white" stroke-width="0.887334"/><path opacity="0.44" d="M112.339 219.765C112.196 219.765 112.052 219.669 111.956 219.525C111.812 219.333 111.908 219.046 112.1 218.902L115.406 216.985C115.598 216.842 115.885 216.937 116.029 217.129C116.173 217.321 116.077 217.608 115.885 217.752L112.579 219.669C112.531 219.765 112.435 219.765 112.339 219.765Z" fill="white" stroke="white" stroke-width="0.887334"/><path opacity="0.3" d="M122.308 214.015C122.164 214.015 122.021 213.919 121.925 213.775C121.781 213.583 121.877 213.296 122.068 213.152L125.375 211.235C125.566 211.092 125.854 211.188 125.998 211.379C126.141 211.571 126.045 211.858 125.854 212.002L122.548 213.919C122.452 214.015 122.404 214.015 122.308 214.015Z" fill="white" stroke="white" stroke-width="0.887334"/><path d="M138.371 221.5H145.559V219.805H140.387V210.227H138.371V221.5ZM150.824 221.672C153.379 221.672 154.934 220 154.934 217.234V217.219C154.934 214.469 153.371 212.797 150.824 212.797C148.285 212.797 146.715 214.477 146.715 217.219V217.234C146.715 220 148.262 221.672 150.824 221.672ZM150.824 220.094C149.473 220.094 148.699 219.039 148.699 217.242V217.227C148.699 215.43 149.473 214.367 150.824 214.367C152.168 214.367 152.941 215.43 152.941 217.227V217.242C152.941 219.039 152.176 220.094 150.824 220.094ZM159.215 221.641C160.34 221.641 161.246 221.156 161.715 220.328H161.848V221.5H163.77V215.664C163.77 213.859 162.535 212.797 160.34 212.797C158.309 212.797 156.934 213.75 156.746 215.188L156.738 215.258H158.574L158.582 215.227C158.777 214.664 159.355 214.344 160.246 214.344C161.293 214.344 161.848 214.812 161.848 215.664V216.398L159.652 216.531C157.574 216.656 156.41 217.547 156.41 219.07V219.086C156.41 220.641 157.59 221.641 159.215 221.641ZM158.332 218.992V218.977C158.332 218.273 158.84 217.875 159.926 217.805L161.848 217.68V218.367C161.848 219.383 160.98 220.148 159.809 220.148C158.949 220.148 158.332 219.719 158.332 218.992ZM169.184 221.641C170.387 221.641 171.309 221.094 171.777 220.164H171.91V221.5H173.863V209.656H171.91V214.32H171.777C171.332 213.398 170.355 212.82 169.184 212.82C167.02 212.82 165.684 214.508 165.684 217.227V217.242C165.684 219.945 167.043 221.641 169.184 221.641ZM169.793 220C168.465 220 167.66 218.961 167.66 217.242V217.227C167.66 215.508 168.473 214.461 169.793 214.461C171.113 214.461 171.941 215.516 171.941 217.227V217.242C171.941 218.953 171.121 220 169.793 220ZM177.309 211.531C177.941 211.531 178.465 211.023 178.465 210.383C178.465 209.75 177.941 209.234 177.309 209.234C176.668 209.234 176.145 209.75 176.145 210.383C176.145 211.023 176.668 211.531 177.309 211.531ZM176.332 221.5H178.277V212.961H176.332V221.5ZM180.668 221.5H182.613V216.523C182.613 215.242 183.332 214.43 184.488 214.43C185.668 214.43 186.207 215.102 186.207 216.43V221.5H188.152V215.977C188.152 213.945 187.113 212.797 185.207 212.797C183.973 212.797 183.152 213.352 182.746 214.266H182.613V212.961H180.668V221.5ZM194.121 224.5C196.613 224.5 198.191 223.258 198.191 221.289V212.961H196.246V214.367H196.145C195.66 213.398 194.738 212.82 193.559 212.82C191.355 212.82 190.012 214.523 190.012 216.992V217.008C190.012 219.43 191.355 221.102 193.52 221.102C194.699 221.102 195.629 220.602 196.121 219.688H196.254V221.32C196.254 222.414 195.48 223.031 194.152 223.031C193.066 223.031 192.418 222.648 192.285 222.102L192.277 222.078H190.332L190.324 222.102C190.52 223.539 191.871 224.5 194.121 224.5ZM194.121 219.555C192.77 219.555 192.004 218.516 192.004 217.008V216.992C192.004 215.484 192.77 214.438 194.121 214.438C195.465 214.438 196.285 215.484 196.285 216.992V217.008C196.285 218.516 195.473 219.555 194.121 219.555ZM204.84 221.5H206.785V216.523C206.785 215.242 207.504 214.43 208.66 214.43C209.84 214.43 210.379 215.102 210.379 216.43V221.5H212.324V215.977C212.324 213.945 211.285 212.797 209.379 212.797C208.145 212.797 207.324 213.352 206.918 214.266H206.785V212.961H204.84V221.5ZM218.277 221.672C220.598 221.672 221.738 220.336 222.012 219.234L222.035 219.164H220.184L220.16 219.219C219.973 219.609 219.371 220.141 218.316 220.141C216.996 220.141 216.168 219.25 216.145 217.719H222.113V217.078C222.113 214.5 220.613 212.797 218.184 212.797C215.754 212.797 214.199 214.555 214.199 217.242V217.25C214.199 219.969 215.738 221.672 218.277 221.672ZM218.207 214.32C219.293 214.32 220.082 215.016 220.215 216.414H216.16C216.309 215.055 217.121 214.32 218.207 214.32ZM225.574 221.5H227.59L229.246 215.312H229.379L231.043 221.5H233.082L235.395 212.961H233.465L232.051 219.375H231.918L230.262 212.961H228.41L226.77 219.375H226.637L225.215 212.961H223.262L225.574 221.5ZM241.355 224.336H243.301V220.141H243.434C243.879 221.062 244.855 221.641 246.027 221.641C248.191 221.641 249.535 219.961 249.535 217.242V217.227C249.535 214.523 248.168 212.82 246.027 212.82C244.832 212.82 243.902 213.367 243.434 214.297H243.301V212.961H241.355V224.336ZM245.418 220C244.098 220 243.277 218.945 243.277 217.234V217.219C243.277 215.508 244.098 214.461 245.418 214.461C246.746 214.461 247.551 215.508 247.551 217.227V217.242C247.551 218.953 246.746 220 245.418 220ZM255.184 221.672C257.738 221.672 259.293 220 259.293 217.234V217.219C259.293 214.469 257.73 212.797 255.184 212.797C252.645 212.797 251.074 214.477 251.074 217.219V217.234C251.074 220 252.621 221.672 255.184 221.672ZM255.184 220.094C253.832 220.094 253.059 219.039 253.059 217.242V217.227C253.059 215.43 253.832 214.367 255.184 214.367C256.527 214.367 257.301 215.43 257.301 217.227V217.242C257.301 219.039 256.535 220.094 255.184 220.094ZM264.473 221.672C266.574 221.672 268.098 220.586 268.098 218.961V218.945C268.098 217.703 267.309 216.992 265.637 216.617L264.254 216.312C263.332 216.102 262.973 215.797 262.973 215.297V215.289C262.973 214.648 263.598 214.227 264.465 214.227C265.379 214.227 265.941 214.664 266.082 215.203L266.09 215.227H267.918V215.211C267.793 213.828 266.551 212.797 264.473 212.797C262.457 212.797 261.051 213.859 261.051 215.398V215.406C261.051 216.664 261.824 217.43 263.449 217.789L264.832 218.102C265.77 218.312 266.145 218.641 266.145 219.141V219.156C266.145 219.797 265.473 220.234 264.504 220.234C263.535 220.234 262.965 219.82 262.762 219.227L262.754 219.211H260.824V219.227C260.988 220.672 262.301 221.672 264.473 221.672ZM273.355 221.539C273.73 221.539 274.09 221.508 274.371 221.453V219.953C274.137 219.977 273.988 219.984 273.715 219.984C272.902 219.984 272.566 219.625 272.566 218.773V214.469H274.371V212.961H272.566V210.867H270.59V212.961H269.262V214.469H270.59V219.234C270.59 220.883 271.395 221.539 273.355 221.539ZM279.426 221.672C281.527 221.672 283.051 220.586 283.051 218.961V218.945C283.051 217.703 282.262 216.992 280.59 216.617L279.207 216.312C278.285 216.102 277.926 215.797 277.926 215.297V215.289C277.926 214.648 278.551 214.227 279.418 214.227C280.332 214.227 280.895 214.664 281.035 215.203L281.043 215.227H282.871V215.211C282.746 213.828 281.504 212.797 279.426 212.797C277.41 212.797 276.004 213.859 276.004 215.398V215.406C276.004 216.664 276.777 217.43 278.402 217.789L279.785 218.102C280.723 218.312 281.098 218.641 281.098 219.141V219.156C281.098 219.797 280.426 220.234 279.457 220.234C278.488 220.234 277.918 219.82 277.715 219.227L277.707 219.211H275.777V219.227C275.941 220.672 277.254 221.672 279.426 221.672Z" fill="white"/></g><rect opacity="0.6" x="69" width="257" height="68" fill="url(#paint0_linear_605_69687)"/><defs><linearGradient id="paint0_linear_605_69687" x1="197.5" y1="0" x2="197.5" y2="68" gradientUnits="userSpaceOnUse"><stop stop-color="#F3F4F5"/><stop offset="1" stop-color="#F3F4F5" stop-opacity="0"/></linearGradient><clipPath id="clip0_605_69687"><rect width="396" height="264" fill="white"/></clipPath></defs></svg>',
							'popupContentBtn' 	=> '<div class="ctf-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Lite users get 50% OFF', 'custom-twitter-feeds' ) .'</div>',
							'bullets'       => [
								'heading' => __( 'And get much more!', 'custom-twitter-feeds' ),
								'content' => [
									__( 'Display photos, videos & GIFs', 'custom-twitter-feeds' ),
									__( 'Combine mulitple feeds into one', 'custom-twitter-feeds' ),
									__( 'Unlimited feeds, 12 daily updates', 'custom-twitter-feeds' ),
									__( 'Auto load tweets on scroll', 'custom-twitter-feeds' ),
									__( 'Powerful Tweet filters', 'custom-twitter-feeds' ),
									__( 'View tweets in a lightbox', 'custom-twitter-feeds' ),
									__( 'Multiple tweet layout options', 'custom-twitter-feeds' ),
									__( 'Fast & Friendly Support', 'custom-twitter-feeds' ),
									__( '30 Day Money Back Guarantee', 'custom-twitter-feeds' )
								]
							],
							'buyUrl' 		=> sprintf('https://smashballoon.com/custom-twitter-feeds/demo/?utm_campaign=twitter-free&utm_source=feed-type&utm_medium=hashtag')
						],

						'lightboxExtension' => [
							'heading' 		=> __( 'Upgrade to Pro to enable the popup Lightbox', 'custom-twitter-feeds' ),
							'description' 	=> __( 'View the photos and videos from your feed in a beautiful pop-up lightbox which allows users to experience your content without leaving your site.', 'custom-twitter-feeds' ),
							'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M50.6018 136.122L48.2658 134.019L39.3153 143.959L49.2558 152.91L51.3591 150.574L43.7704 143.726L50.6018 136.122Z" fill="#8C8F9A"/><path d="M347.2 119.809L345.097 122.145L352.686 128.993L345.854 136.597L348.19 138.7L357.141 128.76L347.2 119.809Z" fill="#8C8F9A"/><g clip-path="url(#clip0_605_69824)" filter="url(#filter0_d_605_69824)"><rect width="261.925" height="173.162" transform="translate(62.6836 52.392) rotate(-3)" fill="white"/><rect x="112.469" y="187.874" width="93.129" height="5.82056" rx="1.45514" transform="rotate(-3 112.469 187.874)" fill="#D0D1D7"/><rect x="113.152" y="200.952" width="53.8402" height="5.82056" rx="1.45514" transform="rotate(-3 113.152 200.952)" fill="#D0D1D7"/><circle cx="94.1758" cy="195.21" r="8.73084" transform="rotate(-3 94.1758 195.21)" fill="#DCDDE1"/><g clip-path="url(#clip1_605_69824)"><rect x="62.6836" y="52.3918" width="262.169" height="173.162" transform="rotate(-3 62.6836 52.3918)" fill="#E34F0E"/><rect x="191.141" y="20.5734" width="271.58" height="334.479" rx="55.73" transform="rotate(2.99107 191.141 20.5734)" fill="#DCDDE1"/><circle cx="141.742" cy="201.742" r="113.935" transform="rotate(-3 141.742 201.742)" fill="#0096CC"/></g></g><defs><filter id="filter0_d_605_69824" x="53.8248" y="34.2545" width="288.346" height="204.35" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.32203" operator="erode" in="SourceAlpha" result="effect1_dropShadow_605_69824"/><feOffset dy="4.42938"/><feGaussianBlur stdDeviation="6.0904"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_605_69824"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_605_69824" result="shape"/></filter><clipPath id="clip0_605_69824"><rect width="261.925" height="173.162" fill="white" transform="translate(62.6836 52.392) rotate(-3)"/></clipPath><clipPath id="clip1_605_69824"><rect width="262.15" height="121.608" fill="white" transform="translate(62.6836 52.3919) rotate(-3)"/></clipPath></defs></svg>',
							'popupContentBtn' 	=> '<div class="ctf-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Lite users get 50% OFF', 'custom-twitter-feeds' ) .'</div>',
							'bullets'       => [
								'heading' => __( 'And get much more!', 'custom-twitter-feeds' ),
								'content' => [
									__( 'Display photos, videos & GIFs', 'custom-twitter-feeds' ),
									__( 'Combine mulitple feeds into one', 'custom-twitter-feeds' ),
									__( 'Unlimited feeds, 12 daily updates', 'custom-twitter-feeds' ),
									__( 'Auto load tweets on scroll', 'custom-twitter-feeds' ),
									__( 'Powerful Tweet filters', 'custom-twitter-feeds' ),
									__( 'View tweets in a lightbox', 'custom-twitter-feeds' ),
									__( 'Multiple tweet layout options', 'custom-twitter-feeds' ),
									__( 'Fast & Friendly Support', 'custom-twitter-feeds' ),
									__( '30 Day Money Back Guarantee', 'custom-twitter-feeds' )
								]
							],
							'buyUrl' 		=> sprintf('https://smashballoon.com/custom-twitter-feeds/demo/?utm_campaign=twitter-free&utm_source=feed-type&utm_medium=hashtag')
						],

						'feedTemplates' =>[
						    'heading' 		=> __( 'Upgrade to Pro to get one-click templates!', 'custom-twitter-feeds' ),
						    'description' 	=> __( 'Choose from a selection of feed templates which make it easier and quicker to create the perfect feed for your site.', 'custom-twitter-feeds' ),
						    'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink"><g clip-path="url(#clip0_5232_97155)"><g filter="url(#filter0_dd_5232_97155)"><rect x="175.881" y="35.4429" width="208.613" height="158.297" rx="5.31509" fill="white"/><g clip-path="url(#clip1_5232_97155)"><rect x="181.195" y="40.7581" width="197.983" height="147.667" rx="2.18766" fill="#F3F4F5"/><g filter="url(#filter1_dd_5232_97155)"><g clip-path="url(#clip2_5232_97155)"><rect x="198.697" y="57.1658" width="51.8985" height="82.117" rx="0.7284" fill="white"/><rect x="198.697" y="57.1658" width="51.8985" height="65.9202" fill="url(#pattern0)"/><path d="M200.97 125.923V125.682H202.238V125.923H201.746V127.271H201.461V125.923H200.97ZM202.772 126.573V127.271H202.492V125.682H202.766V126.282H202.78C202.808 126.215 202.851 126.162 202.91 126.123C202.969 126.083 203.044 126.064 203.135 126.064C203.218 126.064 203.29 126.081 203.351 126.116C203.413 126.15 203.461 126.201 203.494 126.268C203.528 126.335 203.545 126.416 203.545 126.512V127.271H203.264V126.556C203.264 126.476 203.244 126.413 203.202 126.369C203.162 126.324 203.104 126.301 203.03 126.301C202.98 126.301 202.936 126.312 202.897 126.334C202.858 126.355 202.827 126.386 202.805 126.426C202.783 126.467 202.772 126.515 202.772 126.573ZM203.854 127.271V126.079H204.134V127.271H203.854ZM203.995 125.91C203.95 125.91 203.912 125.895 203.88 125.866C203.848 125.836 203.832 125.8 203.832 125.758C203.832 125.716 203.848 125.68 203.88 125.65C203.912 125.62 203.95 125.605 203.995 125.605C204.04 125.605 204.078 125.62 204.11 125.65C204.142 125.68 204.158 125.716 204.158 125.758C204.158 125.8 204.142 125.836 204.11 125.866C204.078 125.895 204.04 125.91 203.995 125.91ZM205.387 126.394L205.131 126.422C205.124 126.396 205.111 126.372 205.093 126.349C205.076 126.327 205.052 126.308 205.022 126.294C204.992 126.28 204.955 126.273 204.912 126.273C204.853 126.273 204.804 126.286 204.764 126.311C204.725 126.337 204.705 126.37 204.706 126.41C204.705 126.445 204.718 126.473 204.744 126.494C204.77 126.516 204.814 126.534 204.874 126.548L205.078 126.591C205.19 126.616 205.274 126.654 205.329 126.707C205.384 126.76 205.412 126.829 205.413 126.914C205.412 126.989 205.39 127.056 205.347 127.113C205.304 127.17 205.244 127.214 205.168 127.246C205.091 127.278 205.003 127.295 204.904 127.295C204.758 127.295 204.64 127.264 204.552 127.203C204.463 127.141 204.409 127.056 204.392 126.946L204.666 126.92C204.679 126.974 204.705 127.014 204.746 127.042C204.786 127.069 204.838 127.083 204.903 127.083C204.97 127.083 205.023 127.069 205.064 127.042C205.105 127.014 205.125 126.98 205.125 126.94C205.125 126.906 205.112 126.878 205.085 126.855C205.06 126.833 205.019 126.816 204.964 126.804L204.761 126.761C204.647 126.738 204.562 126.698 204.507 126.641C204.452 126.584 204.425 126.512 204.426 126.425C204.425 126.352 204.445 126.288 204.486 126.235C204.526 126.18 204.583 126.138 204.655 126.109C204.728 126.079 204.812 126.064 204.908 126.064C205.047 126.064 205.157 126.094 205.237 126.153C205.318 126.213 205.368 126.293 205.387 126.394ZM206.476 127.271L206.139 126.079H206.426L206.635 126.917H206.646L206.86 126.079H207.144L207.358 126.913H207.369L207.576 126.079H207.863L207.525 127.271H207.233L207.009 126.466H206.993L206.77 127.271H206.476ZM208.574 127.295C208.455 127.295 208.352 127.27 208.265 127.22C208.178 127.17 208.112 127.099 208.065 127.007C208.019 126.915 207.996 126.807 207.996 126.682C207.996 126.56 208.019 126.452 208.065 126.359C208.112 126.266 208.178 126.194 208.262 126.142C208.347 126.09 208.446 126.064 208.56 126.064C208.633 126.064 208.702 126.076 208.768 126.099C208.833 126.123 208.891 126.159 208.941 126.208C208.992 126.257 209.032 126.32 209.061 126.396C209.09 126.471 209.104 126.561 209.104 126.666V126.752H208.127V126.563H208.835C208.835 126.509 208.823 126.461 208.8 126.419C208.777 126.377 208.746 126.343 208.705 126.319C208.664 126.295 208.617 126.283 208.564 126.283C208.506 126.283 208.456 126.297 208.412 126.325C208.369 126.352 208.335 126.388 208.311 126.433C208.287 126.478 208.275 126.527 208.274 126.58V126.745C208.274 126.814 208.287 126.874 208.312 126.924C208.338 126.973 208.373 127.011 208.418 127.037C208.464 127.063 208.517 127.076 208.578 127.076C208.619 127.076 208.656 127.07 208.689 127.059C208.722 127.047 208.751 127.029 208.775 127.007C208.8 126.984 208.818 126.956 208.831 126.922L209.093 126.952C209.076 127.021 209.045 127.081 208.998 127.133C208.952 127.184 208.893 127.224 208.821 127.253C208.749 127.281 208.667 127.295 208.574 127.295ZM209.892 127.295C209.772 127.295 209.669 127.27 209.582 127.22C209.496 127.17 209.429 127.099 209.383 127.007C209.336 126.915 209.313 126.807 209.313 126.682C209.313 126.56 209.336 126.452 209.383 126.359C209.43 126.266 209.495 126.194 209.58 126.142C209.664 126.09 209.763 126.064 209.877 126.064C209.95 126.064 210.02 126.076 210.085 126.099C210.151 126.123 210.208 126.159 210.259 126.208C210.309 126.257 210.349 126.32 210.378 126.396C210.407 126.471 210.422 126.561 210.422 126.666V126.752H209.445V126.563H210.152C210.152 126.509 210.14 126.461 210.117 126.419C210.095 126.377 210.063 126.343 210.022 126.319C209.982 126.295 209.935 126.283 209.881 126.283C209.823 126.283 209.773 126.297 209.729 126.325C209.686 126.352 209.652 126.388 209.628 126.433C209.604 126.478 209.592 126.527 209.591 126.58V126.745C209.591 126.814 209.604 126.874 209.629 126.924C209.655 126.973 209.69 127.011 209.736 127.037C209.781 127.063 209.834 127.076 209.895 127.076C209.936 127.076 209.973 127.07 210.006 127.059C210.04 127.047 210.068 127.029 210.093 127.007C210.117 126.984 210.135 126.956 210.148 126.922L210.41 126.952C210.393 127.021 210.362 127.081 210.315 127.133C210.269 127.184 210.21 127.224 210.138 127.253C210.066 127.281 209.984 127.295 209.892 127.295ZM210.937 126.897L210.936 126.558H210.981L211.41 126.079H211.738L211.211 126.666H211.153L210.937 126.897ZM210.681 127.271V125.682H210.962V127.271H210.681ZM211.429 127.271L211.041 126.729L211.231 126.531L211.765 127.271H211.429ZM212.511 127.271V126.079H212.792V127.271H212.511ZM212.652 125.91C212.608 125.91 212.569 125.895 212.537 125.866C212.505 125.836 212.489 125.8 212.489 125.758C212.489 125.716 212.505 125.68 212.537 125.65C212.569 125.62 212.608 125.605 212.652 125.605C212.697 125.605 212.735 125.62 212.767 125.65C212.799 125.68 212.815 125.716 212.815 125.758C212.815 125.8 212.799 125.836 212.767 125.866C212.735 125.895 212.697 125.91 212.652 125.91ZM213.384 126.573V127.271H213.103V126.079H213.371V126.282H213.385C213.412 126.215 213.456 126.162 213.516 126.123C213.577 126.083 213.651 126.064 213.74 126.064C213.823 126.064 213.894 126.081 213.955 126.117C214.017 126.152 214.065 126.203 214.098 126.269C214.132 126.336 214.149 126.417 214.149 126.512V127.271H213.868V126.556C213.868 126.476 213.847 126.414 213.806 126.369C213.765 126.324 213.708 126.301 213.636 126.301C213.587 126.301 213.543 126.312 213.505 126.334C213.467 126.355 213.437 126.386 213.415 126.426C213.394 126.467 213.384 126.515 213.384 126.573ZM215.236 127.271H214.929L215.489 125.682H215.844L216.404 127.271H216.097L215.673 126.008H215.66L215.236 127.271ZM215.247 126.648H216.085V126.879H215.247V126.648ZM216.623 127.271V126.079H216.895V126.278H216.908C216.93 126.209 216.967 126.156 217.02 126.119C217.073 126.081 217.134 126.062 217.202 126.062C217.217 126.062 217.235 126.063 217.254 126.065C217.274 126.066 217.29 126.067 217.303 126.07V126.328C217.291 126.324 217.272 126.321 217.246 126.318C217.221 126.314 217.196 126.312 217.172 126.312C217.121 126.312 217.075 126.323 217.034 126.345C216.994 126.367 216.962 126.397 216.939 126.436C216.916 126.475 216.904 126.52 216.904 126.571V127.271H216.623ZM217.982 127.295C217.863 127.295 217.761 127.268 217.676 127.216C217.591 127.164 217.525 127.092 217.479 127C217.434 126.907 217.411 126.8 217.411 126.68C217.411 126.559 217.434 126.452 217.481 126.359C217.527 126.266 217.593 126.194 217.678 126.142C217.763 126.09 217.864 126.064 217.981 126.064C218.077 126.064 218.163 126.082 218.237 126.117C218.312 126.153 218.372 126.202 218.417 126.267C218.461 126.331 218.487 126.406 218.493 126.492H218.224C218.213 126.435 218.188 126.387 218.147 126.349C218.106 126.31 218.052 126.29 217.984 126.29C217.927 126.29 217.877 126.306 217.833 126.337C217.79 126.367 217.756 126.411 217.732 126.469C217.708 126.526 217.696 126.595 217.696 126.675C217.696 126.756 217.708 126.826 217.732 126.885C217.755 126.943 217.789 126.987 217.832 127.019C217.875 127.05 217.926 127.066 217.984 127.066C218.026 127.066 218.063 127.058 218.095 127.042C218.129 127.026 218.156 127.003 218.178 126.973C218.201 126.943 218.216 126.907 218.224 126.864H218.493C218.486 126.948 218.461 127.023 218.418 127.088C218.375 127.153 218.317 127.203 218.243 127.24C218.169 127.276 218.082 127.295 217.982 127.295ZM219.026 126.573V127.271H218.745V125.682H219.02V126.282H219.034C219.062 126.215 219.105 126.162 219.164 126.123C219.223 126.083 219.298 126.064 219.389 126.064C219.471 126.064 219.544 126.081 219.605 126.116C219.667 126.15 219.714 126.201 219.748 126.268C219.782 126.335 219.799 126.416 219.799 126.512V127.271H219.518V126.556C219.518 126.476 219.497 126.413 219.456 126.369C219.415 126.324 219.358 126.301 219.284 126.301C219.234 126.301 219.19 126.312 219.15 126.334C219.112 126.355 219.081 126.386 219.059 126.426C219.037 126.467 219.026 126.515 219.026 126.573ZM220.107 127.271V126.079H220.388V127.271H220.107ZM220.249 125.91C220.204 125.91 220.166 125.895 220.134 125.866C220.102 125.836 220.086 125.8 220.086 125.758C220.086 125.716 220.102 125.68 220.134 125.65C220.166 125.62 220.204 125.605 220.249 125.605C220.294 125.605 220.332 125.62 220.363 125.65C220.395 125.68 220.411 125.716 220.411 125.758C220.411 125.8 220.395 125.836 220.363 125.866C220.332 125.895 220.294 125.91 220.249 125.91ZM221.294 126.079V126.297H220.609V126.079H221.294ZM220.778 125.794H221.059V126.913C221.059 126.95 221.065 126.979 221.076 127C221.088 127.019 221.103 127.033 221.123 127.04C221.142 127.047 221.163 127.051 221.186 127.051C221.204 127.051 221.22 127.05 221.234 127.047C221.249 127.044 221.261 127.042 221.268 127.04L221.316 127.26C221.301 127.265 221.279 127.27 221.251 127.277C221.224 127.283 221.19 127.287 221.151 127.288C221.08 127.29 221.017 127.279 220.96 127.256C220.904 127.232 220.859 127.195 220.826 127.146C220.794 127.096 220.778 127.034 220.778 126.959V125.794ZM222.066 127.295C221.946 127.295 221.843 127.27 221.756 127.22C221.67 127.17 221.603 127.099 221.557 127.007C221.51 126.915 221.487 126.807 221.487 126.682C221.487 126.56 221.51 126.452 221.557 126.359C221.604 126.266 221.669 126.194 221.754 126.142C221.838 126.09 221.937 126.064 222.051 126.064C222.124 126.064 222.194 126.076 222.259 126.099C222.325 126.123 222.383 126.159 222.433 126.208C222.483 126.257 222.523 126.32 222.552 126.396C222.581 126.471 222.596 126.561 222.596 126.666V126.752H221.619V126.563H222.326C222.326 126.509 222.314 126.461 222.292 126.419C222.269 126.377 222.237 126.343 222.196 126.319C222.156 126.295 222.109 126.283 222.055 126.283C221.997 126.283 221.947 126.297 221.904 126.325C221.86 126.352 221.826 126.388 221.802 126.433C221.778 126.478 221.766 126.527 221.765 126.58V126.745C221.765 126.814 221.778 126.874 221.803 126.924C221.829 126.973 221.864 127.011 221.91 127.037C221.955 127.063 222.009 127.076 222.07 127.076C222.11 127.076 222.147 127.07 222.181 127.059C222.214 127.047 222.242 127.029 222.267 127.007C222.291 126.984 222.309 126.956 222.322 126.922L222.584 126.952C222.568 127.021 222.536 127.081 222.489 127.133C222.443 127.184 222.384 127.224 222.313 127.253C222.241 127.281 222.158 127.295 222.066 127.295ZM223.375 127.295C223.256 127.295 223.154 127.268 223.069 127.216C222.984 127.164 222.918 127.092 222.872 127C222.827 126.907 222.804 126.8 222.804 126.68C222.804 126.559 222.827 126.452 222.874 126.359C222.92 126.266 222.986 126.194 223.071 126.142C223.156 126.09 223.257 126.064 223.374 126.064C223.47 126.064 223.556 126.082 223.63 126.117C223.705 126.153 223.765 126.202 223.81 126.267C223.854 126.331 223.88 126.406 223.886 126.492H223.617C223.606 126.435 223.581 126.387 223.54 126.349C223.499 126.31 223.445 126.29 223.377 126.29C223.32 126.29 223.27 126.306 223.226 126.337C223.183 126.367 223.149 126.411 223.125 126.469C223.101 126.526 223.089 126.595 223.089 126.675C223.089 126.756 223.101 126.826 223.125 126.885C223.148 126.943 223.182 126.987 223.225 127.019C223.268 127.05 223.319 127.066 223.377 127.066C223.419 127.066 223.456 127.058 223.488 127.042C223.522 127.026 223.549 127.003 223.571 126.973C223.594 126.943 223.609 126.907 223.617 126.864H223.886C223.879 126.948 223.854 127.023 223.811 127.088C223.768 127.153 223.71 127.203 223.636 127.24C223.562 127.276 223.475 127.295 223.375 127.295ZM224.733 126.079V126.297H224.048V126.079H224.733ZM224.217 125.794H224.498V126.913C224.498 126.95 224.504 126.979 224.515 127C224.527 127.019 224.543 127.033 224.562 127.04C224.581 127.047 224.602 127.051 224.626 127.051C224.643 127.051 224.659 127.05 224.674 127.047C224.689 127.044 224.7 127.042 224.708 127.04L224.755 127.26C224.74 127.265 224.719 127.27 224.691 127.277C224.663 127.283 224.63 127.287 224.59 127.288C224.52 127.29 224.456 127.279 224.4 127.256C224.343 127.232 224.299 127.195 224.266 127.146C224.233 127.096 224.217 127.034 224.217 126.959V125.794ZM225.75 126.77V126.079H226.031V127.271H225.758V127.059H225.746C225.719 127.126 225.675 127.181 225.613 127.223C225.552 127.266 225.477 127.287 225.388 127.287C225.309 127.287 225.24 127.269 225.18 127.235C225.121 127.2 225.074 127.149 225.041 127.082C225.007 127.015 224.99 126.933 224.99 126.838V126.079H225.271V126.795C225.271 126.87 225.292 126.93 225.333 126.975C225.375 127.019 225.429 127.042 225.496 127.042C225.538 127.042 225.578 127.031 225.616 127.011C225.655 126.991 225.687 126.961 225.712 126.921C225.737 126.881 225.75 126.83 225.75 126.77ZM226.342 127.271V126.079H226.614V126.278H226.626C226.648 126.209 226.685 126.156 226.738 126.119C226.791 126.081 226.852 126.062 226.92 126.062C226.936 126.062 226.953 126.063 226.972 126.065C226.992 126.066 227.008 126.067 227.021 126.07V126.328C227.009 126.324 226.991 126.321 226.965 126.318C226.939 126.314 226.915 126.312 226.891 126.312C226.84 126.312 226.794 126.323 226.753 126.345C226.713 126.367 226.681 126.397 226.657 126.436C226.634 126.475 226.622 126.52 226.622 126.571V127.271H226.342ZM227.708 127.295C227.589 127.295 227.486 127.27 227.399 127.22C227.312 127.17 227.246 127.099 227.199 127.007C227.153 126.915 227.13 126.807 227.13 126.682C227.13 126.56 227.153 126.452 227.199 126.359C227.246 126.266 227.312 126.194 227.396 126.142C227.481 126.09 227.58 126.064 227.694 126.064C227.767 126.064 227.836 126.076 227.902 126.099C227.967 126.123 228.025 126.159 228.075 126.208C228.126 126.257 228.166 126.32 228.195 126.396C228.224 126.471 228.238 126.561 228.238 126.666V126.752H227.261V126.563H227.969C227.969 126.509 227.957 126.461 227.934 126.419C227.911 126.377 227.88 126.343 227.839 126.319C227.798 126.295 227.751 126.283 227.698 126.283C227.64 126.283 227.59 126.297 227.546 126.325C227.503 126.352 227.469 126.388 227.445 126.433C227.421 126.478 227.409 126.527 227.408 126.58V126.745C227.408 126.814 227.421 126.874 227.446 126.924C227.471 126.973 227.507 127.011 227.552 127.037C227.598 127.063 227.651 127.076 227.712 127.076C227.753 127.076 227.79 127.07 227.823 127.059C227.856 127.047 227.885 127.029 227.909 127.007C227.934 126.984 227.952 126.956 227.965 126.922L228.227 126.952C228.21 127.021 228.179 127.081 228.132 127.133C228.086 127.184 228.027 127.224 227.955 127.253C227.883 127.281 227.801 127.295 227.708 127.295ZM229.504 127.292C229.41 127.292 229.327 127.268 229.253 127.22C229.179 127.172 229.12 127.102 229.077 127.011C229.034 126.919 229.013 126.808 229.013 126.677C229.013 126.544 229.035 126.433 229.078 126.342C229.122 126.25 229.181 126.181 229.256 126.134C229.33 126.087 229.413 126.064 229.505 126.064C229.575 126.064 229.632 126.076 229.677 126.099C229.722 126.123 229.758 126.151 229.784 126.184C229.811 126.217 229.831 126.247 229.846 126.276H229.857V125.682H230.139V127.271H229.863V127.083H229.846C229.831 127.112 229.81 127.143 229.783 127.176C229.755 127.208 229.719 127.235 229.674 127.258C229.629 127.281 229.572 127.292 229.504 127.292ZM229.583 127.062C229.642 127.062 229.693 127.046 229.735 127.014C229.777 126.981 229.808 126.936 229.83 126.878C229.852 126.82 229.863 126.752 229.863 126.675C229.863 126.598 229.852 126.531 229.83 126.474C229.809 126.417 229.777 126.373 229.735 126.342C229.694 126.31 229.643 126.294 229.583 126.294C229.52 126.294 229.468 126.311 229.426 126.343C229.384 126.376 229.352 126.421 229.331 126.478C229.31 126.536 229.299 126.601 229.299 126.675C229.299 126.75 229.31 126.816 229.331 126.875C229.353 126.933 229.385 126.978 229.427 127.012C229.469 127.045 229.521 127.062 229.583 127.062ZM230.803 127.295C230.728 127.295 230.66 127.282 230.599 127.255C230.539 127.228 230.491 127.187 230.456 127.134C230.422 127.081 230.404 127.015 230.404 126.937C230.404 126.87 230.417 126.814 230.441 126.77C230.466 126.726 230.5 126.691 230.543 126.664C230.586 126.638 230.634 126.618 230.688 126.605C230.743 126.591 230.799 126.581 230.857 126.574C230.926 126.567 230.983 126.561 231.027 126.555C231.07 126.549 231.102 126.539 231.121 126.527C231.141 126.514 231.152 126.494 231.152 126.467V126.463C231.152 126.404 231.134 126.359 231.1 126.327C231.065 126.295 231.015 126.279 230.95 126.279C230.881 126.279 230.826 126.294 230.786 126.324C230.746 126.354 230.719 126.389 230.705 126.43L230.443 126.393C230.464 126.32 230.498 126.26 230.545 126.211C230.593 126.162 230.651 126.125 230.72 126.101C230.789 126.076 230.865 126.064 230.948 126.064C231.006 126.064 231.063 126.071 231.12 126.084C231.177 126.097 231.229 126.12 231.276 126.151C231.323 126.181 231.361 126.223 231.389 126.276C231.418 126.328 231.432 126.394 231.432 126.474V127.271H231.162V127.108H231.153C231.136 127.141 231.112 127.172 231.081 127.201C231.05 127.229 231.012 127.252 230.965 127.27C230.919 127.287 230.865 127.295 230.803 127.295ZM230.876 127.089C230.932 127.089 230.981 127.078 231.023 127.056C231.064 127.033 231.096 127.003 231.118 126.965C231.141 126.928 231.152 126.888 231.152 126.844V126.703C231.143 126.71 231.128 126.717 231.107 126.723C231.087 126.73 231.063 126.735 231.037 126.74C231.012 126.744 230.986 126.748 230.961 126.752C230.935 126.756 230.913 126.759 230.895 126.761C230.853 126.767 230.815 126.776 230.782 126.789C230.749 126.802 230.723 126.82 230.704 126.844C230.685 126.866 230.675 126.896 230.675 126.932C230.675 126.984 230.694 127.023 230.732 127.049C230.769 127.076 230.818 127.089 230.876 127.089ZM231.737 127.271V126.079H232.018V127.271H231.737ZM231.878 125.91C231.833 125.91 231.795 125.895 231.763 125.866C231.731 125.836 231.715 125.8 231.715 125.758C231.715 125.716 231.731 125.68 231.763 125.65C231.795 125.62 231.833 125.605 231.878 125.605C231.923 125.605 231.961 125.62 231.993 125.65C232.025 125.68 232.041 125.716 232.041 125.758C232.041 125.8 232.025 125.836 231.993 125.866C231.961 125.895 231.923 125.91 231.878 125.91ZM232.609 125.682V127.271H232.328V125.682H232.609ZM233.095 127.718C233.056 127.718 233.021 127.715 232.988 127.709C232.956 127.703 232.931 127.696 232.911 127.689L232.977 127.47C233.017 127.482 233.054 127.487 233.086 127.487C233.118 127.486 233.146 127.476 233.171 127.457C233.195 127.438 233.216 127.405 233.233 127.36L233.258 127.296L232.825 126.079H233.123L233.398 126.979H233.41L233.686 126.079H233.985L233.507 127.416C233.485 127.479 233.456 127.533 233.419 127.578C233.382 127.623 233.337 127.658 233.284 127.682C233.231 127.706 233.168 127.718 233.095 127.718ZM235.011 127.271L234.674 126.079H234.96L235.17 126.917H235.181L235.395 126.079H235.678L235.892 126.913H235.904L236.11 126.079H236.397L236.06 127.271H235.767L235.544 126.466H235.528L235.304 127.271H235.011ZM237.109 127.295C236.989 127.295 236.886 127.27 236.799 127.22C236.713 127.17 236.646 127.099 236.6 127.007C236.553 126.915 236.53 126.807 236.53 126.682C236.53 126.56 236.553 126.452 236.6 126.359C236.647 126.266 236.713 126.194 236.797 126.142C236.881 126.09 236.98 126.064 237.094 126.064C237.168 126.064 237.237 126.076 237.302 126.099C237.368 126.123 237.426 126.159 237.476 126.208C237.527 126.257 237.567 126.32 237.596 126.396C237.624 126.471 237.639 126.561 237.639 126.666V126.752H236.662V126.563H237.37C237.369 126.509 237.358 126.461 237.335 126.419C237.312 126.377 237.28 126.343 237.239 126.319C237.199 126.295 237.152 126.283 237.098 126.283C237.041 126.283 236.99 126.297 236.947 126.325C236.903 126.352 236.869 126.388 236.845 126.433C236.821 126.478 236.809 126.527 236.809 126.58V126.745C236.809 126.814 236.821 126.874 236.847 126.924C236.872 126.973 236.907 127.011 236.953 127.037C236.999 127.063 237.052 127.076 237.113 127.076C237.154 127.076 237.191 127.07 237.224 127.059C237.257 127.047 237.286 127.029 237.31 127.007C237.334 126.984 237.353 126.956 237.365 126.922L237.627 126.952C237.611 127.021 237.579 127.081 237.533 127.133C237.487 127.184 237.428 127.224 237.356 127.253C237.284 127.281 237.202 127.295 237.109 127.295ZM238.805 127.295C238.73 127.295 238.662 127.282 238.601 127.255C238.541 127.228 238.494 127.187 238.458 127.134C238.424 127.081 238.406 127.015 238.406 126.937C238.406 126.87 238.419 126.814 238.444 126.77C238.469 126.726 238.502 126.691 238.545 126.664C238.588 126.638 238.637 126.618 238.691 126.605C238.745 126.591 238.801 126.581 238.859 126.574C238.929 126.567 238.985 126.561 239.029 126.555C239.072 126.549 239.104 126.539 239.124 126.527C239.144 126.514 239.154 126.494 239.154 126.467V126.463C239.154 126.404 239.136 126.359 239.102 126.327C239.067 126.295 239.017 126.279 238.952 126.279C238.883 126.279 238.829 126.294 238.788 126.324C238.748 126.354 238.722 126.389 238.708 126.43L238.445 126.393C238.466 126.32 238.5 126.26 238.548 126.211C238.595 126.162 238.654 126.125 238.722 126.101C238.791 126.076 238.867 126.064 238.95 126.064C239.008 126.064 239.065 126.071 239.122 126.084C239.179 126.097 239.231 126.12 239.278 126.151C239.325 126.181 239.363 126.223 239.391 126.276C239.42 126.328 239.435 126.394 239.435 126.474V127.271H239.165V127.108H239.155C239.138 127.141 239.114 127.172 239.083 127.201C239.053 127.229 239.014 127.252 238.968 127.27C238.921 127.287 238.867 127.295 238.805 127.295ZM238.878 127.089C238.935 127.089 238.984 127.078 239.025 127.056C239.066 127.033 239.098 127.003 239.12 126.965C239.143 126.928 239.155 126.888 239.155 126.844V126.703C239.146 126.71 239.131 126.717 239.11 126.723C239.089 126.73 239.066 126.735 239.04 126.74C239.014 126.744 238.988 126.748 238.963 126.752C238.938 126.756 238.916 126.759 238.897 126.761C238.855 126.767 238.818 126.776 238.784 126.789C238.751 126.802 238.725 126.82 238.706 126.844C238.687 126.866 238.677 126.896 238.677 126.932C238.677 126.984 238.696 127.023 238.734 127.049C238.772 127.076 238.82 127.089 238.878 127.089ZM239.739 127.271V126.079H240.011V126.278H240.024C240.046 126.209 240.083 126.156 240.136 126.119C240.189 126.081 240.25 126.062 240.318 126.062C240.333 126.062 240.351 126.063 240.37 126.065C240.39 126.066 240.406 126.067 240.419 126.07V126.328C240.407 126.324 240.388 126.321 240.362 126.318C240.337 126.314 240.312 126.312 240.288 126.312C240.237 126.312 240.191 126.323 240.15 126.345C240.11 126.367 240.078 126.397 240.055 126.436C240.032 126.475 240.02 126.52 240.02 126.571V127.271H239.739ZM241.106 127.295C240.986 127.295 240.883 127.27 240.796 127.22C240.71 127.17 240.643 127.099 240.597 127.007C240.55 126.915 240.527 126.807 240.527 126.682C240.527 126.56 240.55 126.452 240.597 126.359C240.644 126.266 240.71 126.194 240.794 126.142C240.878 126.09 240.977 126.064 241.091 126.064C241.165 126.064 241.234 126.076 241.299 126.099C241.365 126.123 241.423 126.159 241.473 126.208C241.524 126.257 241.563 126.32 241.592 126.396C241.621 126.471 241.636 126.561 241.636 126.666V126.752H240.659V126.563H241.367C241.366 126.509 241.354 126.461 241.332 126.419C241.309 126.377 241.277 126.343 241.236 126.319C241.196 126.295 241.149 126.283 241.095 126.283C241.038 126.283 240.987 126.297 240.944 126.325C240.9 126.352 240.866 126.388 240.842 126.433C240.818 126.478 240.806 126.527 240.806 126.58V126.745C240.806 126.814 240.818 126.874 240.844 126.924C240.869 126.973 240.904 127.011 240.95 127.037C240.995 127.063 241.049 127.076 241.11 127.076C241.151 127.076 241.188 127.07 241.221 127.059C241.254 127.047 241.283 127.029 241.307 127.007C241.331 126.984 241.35 126.956 241.362 126.922L241.624 126.952C241.608 127.021 241.576 127.081 241.53 127.133C241.483 127.184 241.425 127.224 241.353 127.253C241.281 127.281 241.198 127.295 241.106 127.295ZM201.631 129.079V129.297H200.926V129.079H201.631ZM201.102 130.271V128.967C201.102 128.887 201.119 128.82 201.152 128.767C201.186 128.713 201.231 128.673 201.287 128.647C201.343 128.621 201.406 128.608 201.475 128.608C201.524 128.608 201.567 128.611 201.604 128.619C201.642 128.627 201.67 128.634 201.688 128.64L201.632 128.857C201.621 128.854 201.606 128.85 201.587 128.847C201.569 128.842 201.549 128.84 201.527 128.84C201.475 128.84 201.438 128.853 201.416 128.878C201.395 128.903 201.384 128.939 201.384 128.985V130.271H201.102ZM202.356 130.295C202.236 130.295 202.133 130.27 202.046 130.22C201.96 130.17 201.893 130.099 201.847 130.007C201.8 129.915 201.777 129.807 201.777 129.682C201.777 129.56 201.8 129.452 201.847 129.359C201.894 129.266 201.959 129.194 202.044 129.142C202.128 129.09 202.227 129.064 202.341 129.064C202.414 129.064 202.484 129.076 202.549 129.099C202.615 129.123 202.672 129.159 202.723 129.208C202.773 129.257 202.813 129.32 202.842 129.396C202.871 129.471 202.886 129.561 202.886 129.666V129.752H201.909V129.563H202.616C202.616 129.509 202.604 129.461 202.581 129.419C202.559 129.377 202.527 129.343 202.486 129.319C202.446 129.295 202.399 129.283 202.345 129.283C202.287 129.283 202.237 129.297 202.193 129.325C202.15 129.352 202.116 129.388 202.092 129.433C202.068 129.478 202.056 129.527 202.055 129.58V129.745C202.055 129.814 202.068 129.874 202.093 129.924C202.119 129.973 202.154 130.011 202.2 130.037C202.245 130.063 202.298 130.076 202.359 130.076C202.4 130.076 202.437 130.07 202.47 130.059C202.504 130.047 202.532 130.029 202.557 130.007C202.581 129.984 202.599 129.956 202.612 129.922L202.874 129.952C202.857 130.021 202.826 130.081 202.779 130.133C202.733 130.184 202.674 130.224 202.602 130.253C202.53 130.281 202.448 130.295 202.356 130.295ZM203.488 130.295C203.413 130.295 203.345 130.282 203.284 130.255C203.224 130.228 203.176 130.187 203.141 130.134C203.107 130.081 203.089 130.015 203.089 129.937C203.089 129.87 203.102 129.814 203.126 129.77C203.151 129.726 203.185 129.691 203.228 129.664C203.271 129.638 203.319 129.618 203.373 129.605C203.428 129.591 203.484 129.581 203.542 129.574C203.611 129.567 203.668 129.561 203.712 129.555C203.755 129.549 203.787 129.539 203.806 129.527C203.826 129.514 203.836 129.494 203.836 129.467V129.463C203.836 129.404 203.819 129.359 203.785 129.327C203.75 129.295 203.7 129.279 203.635 129.279C203.566 129.279 203.511 129.294 203.471 129.324C203.431 129.354 203.404 129.389 203.39 129.43L203.128 129.393C203.149 129.32 203.183 129.26 203.23 129.211C203.278 129.162 203.336 129.125 203.405 129.101C203.474 129.076 203.55 129.064 203.633 129.064C203.691 129.064 203.748 129.071 203.805 129.084C203.862 129.097 203.914 129.12 203.961 129.151C204.008 129.181 204.045 129.223 204.074 129.276C204.103 129.328 204.117 129.394 204.117 129.474L204.117 130.271H203.847V130.108H203.838C203.821 130.141 203.797 130.172 203.766 130.201C203.735 130.229 203.697 130.252 203.65 130.27C203.604 130.287 203.55 130.295 203.488 130.295ZM203.561 130.089C203.617 130.089 203.666 130.078 203.708 130.056C203.749 130.033 203.781 130.003 203.803 129.965C203.826 129.928 203.837 129.888 203.837 129.844V129.703C203.828 129.71 203.813 129.717 203.792 129.723C203.772 129.73 203.748 129.735 203.722 129.74C203.697 129.744 203.671 129.748 203.646 129.752C203.62 129.756 203.598 129.759 203.58 129.761C203.538 129.767 203.5 129.776 203.467 129.789C203.434 129.802 203.408 129.82 203.389 129.844C203.37 129.866 203.36 129.896 203.36 129.932C203.36 129.984 203.379 130.023 203.417 130.049C203.454 130.076 203.503 130.089 203.561 130.089ZM205.017 129.079V129.297H204.332V129.079H205.017ZM204.501 128.794H204.782V129.913C204.782 129.95 204.787 129.979 204.799 130C204.811 130.019 204.826 130.033 204.845 130.04C204.865 130.047 204.886 130.051 204.909 130.051C204.927 130.051 204.943 130.05 204.957 130.047C204.972 130.044 204.984 130.042 204.991 130.04L205.039 130.26C205.024 130.265 205.002 130.27 204.974 130.277C204.947 130.283 204.913 130.287 204.873 130.288C204.803 130.29 204.74 130.279 204.683 130.256C204.627 130.232 204.582 130.195 204.549 130.146C204.516 130.096 204.5 130.034 204.501 129.959V128.794ZM206.033 129.77V129.079H206.314V130.271H206.042V130.059H206.03C206.003 130.126 205.958 130.181 205.897 130.223C205.836 130.266 205.76 130.287 205.671 130.287C205.593 130.287 205.524 130.269 205.464 130.235C205.404 130.2 205.358 130.149 205.324 130.082C205.291 130.015 205.274 129.933 205.274 129.838V129.079H205.555V129.795C205.555 129.87 205.575 129.93 205.617 129.975C205.658 130.019 205.712 130.042 205.78 130.042C205.821 130.042 205.861 130.031 205.9 130.011C205.939 129.991 205.971 129.961 205.995 129.921C206.021 129.881 206.033 129.83 206.033 129.77ZM206.625 130.271V129.079H206.897V129.278H206.91C206.932 129.209 206.969 129.156 207.022 129.119C207.075 129.081 207.136 129.062 207.204 129.062C207.219 129.062 207.237 129.063 207.256 129.065C207.276 129.066 207.292 129.067 207.305 129.07V129.328C207.293 129.324 207.274 129.321 207.248 129.318C207.223 129.314 207.198 129.312 207.174 129.312C207.123 129.312 207.077 129.323 207.036 129.345C206.996 129.367 206.964 129.397 206.941 129.436C206.918 129.475 206.906 129.52 206.906 129.571V130.271H206.625ZM207.513 130.271V129.079H207.794V130.271H207.513ZM207.654 128.91C207.61 128.91 207.572 128.895 207.54 128.866C207.508 128.836 207.492 128.8 207.492 128.758C207.492 128.716 207.508 128.68 207.54 128.65C207.572 128.62 207.61 128.605 207.654 128.605C207.7 128.605 207.738 128.62 207.769 128.65C207.801 128.68 207.817 128.716 207.817 128.758C207.817 128.8 207.801 128.836 207.769 128.866C207.738 128.895 207.7 128.91 207.654 128.91ZM208.386 129.573V130.271H208.105V129.079H208.373V129.282H208.387C208.415 129.215 208.458 129.162 208.518 129.123C208.579 129.083 208.654 129.064 208.743 129.064C208.825 129.064 208.897 129.081 208.958 129.117C209.019 129.152 209.067 129.203 209.1 129.269C209.135 129.336 209.151 129.417 209.151 129.512V130.271H208.87V129.556C208.87 129.476 208.849 129.414 208.808 129.369C208.767 129.324 208.71 129.301 208.638 129.301C208.589 129.301 208.545 129.312 208.507 129.334C208.469 129.355 208.439 129.386 208.418 129.426C208.396 129.467 208.386 129.515 208.386 129.573ZM209.977 130.743C209.876 130.743 209.789 130.729 209.717 130.702C209.644 130.675 209.586 130.639 209.542 130.593C209.498 130.548 209.468 130.497 209.451 130.442L209.704 130.381C209.715 130.404 209.732 130.427 209.753 130.45C209.775 130.473 209.804 130.492 209.841 130.507C209.878 130.523 209.925 130.53 209.981 130.53C210.061 130.53 210.127 130.511 210.179 130.472C210.231 130.434 210.258 130.371 210.258 130.283V130.057H210.244C210.229 130.086 210.208 130.116 210.18 130.146C210.153 130.177 210.116 130.202 210.071 130.223C210.026 130.244 209.969 130.254 209.901 130.254C209.809 130.254 209.726 130.233 209.652 130.19C209.578 130.146 209.519 130.082 209.475 129.996C209.431 129.909 209.409 129.801 209.409 129.671C209.409 129.541 209.431 129.43 209.475 129.34C209.519 129.25 209.578 129.181 209.652 129.134C209.727 129.087 209.81 129.064 209.901 129.064C209.971 129.064 210.029 129.076 210.074 129.099C210.119 129.123 210.155 129.151 210.182 129.184C210.209 129.217 210.23 129.247 210.244 129.276H210.259V129.079H210.536V130.291C210.536 130.393 210.512 130.477 210.463 130.544C210.415 130.61 210.348 130.66 210.264 130.693C210.18 130.726 210.084 130.743 209.977 130.743ZM209.979 130.034C210.039 130.034 210.089 130.019 210.131 129.99C210.173 129.961 210.205 129.92 210.227 129.865C210.248 129.811 210.259 129.746 210.259 129.67C210.259 129.595 210.248 129.529 210.227 129.473C210.205 129.416 210.174 129.373 210.132 129.342C210.091 129.31 210.04 129.294 209.979 129.294C209.916 129.294 209.864 129.311 209.822 129.343C209.78 129.376 209.749 129.42 209.728 129.477C209.706 129.534 209.696 129.598 209.696 129.67C209.696 129.743 209.706 129.807 209.728 129.862C209.749 129.916 209.781 129.958 209.823 129.989C209.865 130.019 209.917 130.034 209.979 130.034ZM211.753 130.295C211.677 130.295 211.609 130.282 211.549 130.255C211.489 130.228 211.441 130.187 211.406 130.134C211.371 130.081 211.354 130.015 211.354 129.937C211.354 129.87 211.366 129.814 211.391 129.77C211.416 129.726 211.45 129.691 211.493 129.664C211.536 129.638 211.584 129.618 211.638 129.605C211.692 129.591 211.748 129.581 211.806 129.574C211.876 129.567 211.933 129.561 211.976 129.555C212.02 129.549 212.051 129.539 212.071 129.527C212.091 129.514 212.101 129.494 212.101 129.467V129.463C212.101 129.404 212.084 129.359 212.049 129.327C212.015 129.295 211.965 129.279 211.899 129.279C211.831 129.279 211.776 129.294 211.736 129.324C211.696 129.354 211.669 129.389 211.655 129.43L211.393 129.393C211.413 129.32 211.448 129.26 211.495 129.211C211.543 129.162 211.601 129.125 211.67 129.101C211.739 129.076 211.815 129.064 211.898 129.064C211.955 129.064 212.013 129.071 212.069 129.084C212.126 129.097 212.178 129.12 212.225 129.151C212.272 129.181 212.31 129.223 212.339 129.276C212.368 129.328 212.382 129.394 212.382 129.474V130.271H212.112V130.108H212.103C212.086 130.141 212.062 130.172 212.031 130.201C212 130.229 211.962 130.252 211.915 130.27C211.869 130.287 211.815 130.295 211.753 130.295ZM211.826 130.089C211.882 130.089 211.931 130.078 211.972 130.056C212.014 130.033 212.046 130.003 212.068 129.965C212.091 129.928 212.102 129.888 212.102 129.844V129.703C212.093 129.71 212.078 129.717 212.057 129.723C212.036 129.73 212.013 129.735 211.987 129.74C211.961 129.744 211.936 129.748 211.91 129.752C211.885 129.756 211.863 129.759 211.844 129.761C211.802 129.767 211.765 129.776 211.732 129.789C211.699 129.802 211.673 129.82 211.653 129.844C211.634 129.866 211.625 129.896 211.625 129.932C211.625 129.984 211.644 130.023 211.681 130.049C211.719 130.076 211.767 130.089 211.826 130.089ZM213.25 129.079H213.531V130.34C213.531 130.425 213.515 130.496 213.482 130.552C213.45 130.608 213.403 130.65 213.341 130.677C213.28 130.705 213.205 130.718 213.117 130.718C213.107 130.718 213.097 130.718 213.088 130.717C213.079 130.717 213.07 130.717 213.059 130.717V130.488C213.067 130.488 213.074 130.489 213.08 130.489C213.087 130.489 213.093 130.489 213.1 130.489C213.154 130.489 213.192 130.476 213.215 130.45C213.239 130.424 213.25 130.386 213.25 130.336V129.079ZM213.39 128.91C213.345 128.91 213.306 128.895 213.274 128.866C213.243 128.836 213.227 128.8 213.227 128.758C213.227 128.716 213.243 128.68 213.274 128.65C213.306 128.62 213.345 128.605 213.39 128.605C213.435 128.605 213.473 128.62 213.504 128.65C213.536 128.68 213.552 128.716 213.552 128.758C213.552 128.8 213.536 128.836 213.504 128.866C213.473 128.895 213.435 128.91 213.39 128.91ZM213.842 130.271V129.079H214.123V130.271H213.842ZM213.983 128.91C213.939 128.91 213.9 128.895 213.868 128.866C213.836 128.836 213.82 128.8 213.82 128.758C213.82 128.716 213.836 128.68 213.868 128.65C213.9 128.62 213.939 128.605 213.983 128.605C214.028 128.605 214.067 128.62 214.098 128.65C214.13 128.68 214.146 128.716 214.146 128.758C214.146 128.8 214.13 128.836 214.098 128.866C214.067 128.895 214.028 128.91 213.983 128.91ZM214.952 130.743C214.851 130.743 214.764 130.729 214.692 130.702C214.62 130.675 214.561 130.639 214.517 130.593C214.473 130.548 214.443 130.497 214.426 130.442L214.679 130.381C214.69 130.404 214.707 130.427 214.728 130.45C214.75 130.473 214.779 130.492 214.816 130.507C214.853 130.523 214.9 130.53 214.957 130.53C215.036 130.53 215.102 130.511 215.155 130.472C215.207 130.434 215.233 130.371 215.233 130.283V130.057H215.219C215.204 130.086 215.183 130.116 215.155 130.146C215.128 130.177 215.091 130.202 215.046 130.223C215.001 130.244 214.944 130.254 214.876 130.254C214.784 130.254 214.701 130.233 214.627 130.19C214.553 130.146 214.494 130.082 214.45 129.996C214.406 129.909 214.385 129.801 214.385 129.671C214.385 129.541 214.406 129.43 214.45 129.34C214.494 129.25 214.553 129.181 214.628 129.134C214.702 129.087 214.785 129.064 214.877 129.064C214.947 129.064 215.004 129.076 215.049 129.099C215.094 129.123 215.131 129.151 215.158 129.184C215.185 129.217 215.205 129.247 215.219 129.276H215.234V129.079H215.511V130.291C215.511 130.393 215.487 130.477 215.439 130.544C215.39 130.61 215.323 130.66 215.239 130.693C215.155 130.726 215.059 130.743 214.952 130.743ZM214.954 130.034C215.014 130.034 215.064 130.019 215.106 129.99C215.148 129.961 215.18 129.92 215.202 129.865C215.224 129.811 215.234 129.746 215.234 129.67C215.234 129.595 215.224 129.529 215.202 129.473C215.181 129.416 215.149 129.373 215.107 129.342C215.066 129.31 215.015 129.294 214.954 129.294C214.892 129.294 214.839 129.311 214.798 129.343C214.756 129.376 214.724 129.42 214.703 129.477C214.682 129.534 214.671 129.598 214.671 129.67C214.671 129.743 214.682 129.807 214.703 129.862C214.725 129.916 214.756 129.958 214.798 129.989C214.841 130.019 214.893 130.034 214.954 130.034ZM216.763 129.394L216.507 129.422C216.5 129.396 216.487 129.372 216.469 129.349C216.452 129.327 216.428 129.308 216.398 129.294C216.368 129.28 216.331 129.273 216.288 129.273C216.229 129.273 216.18 129.286 216.14 129.311C216.101 129.337 216.081 129.37 216.082 129.41C216.081 129.445 216.094 129.473 216.12 129.494C216.146 129.516 216.19 129.534 216.25 129.548L216.454 129.591C216.566 129.616 216.65 129.654 216.705 129.707C216.76 129.76 216.788 129.829 216.789 129.914C216.788 129.989 216.766 130.056 216.723 130.113C216.68 130.17 216.62 130.214 216.544 130.246C216.467 130.278 216.379 130.295 216.28 130.295C216.134 130.295 216.017 130.264 215.928 130.203C215.839 130.141 215.786 130.056 215.768 129.946L216.042 129.92C216.055 129.974 216.081 130.014 216.122 130.042C216.162 130.069 216.214 130.083 216.279 130.083C216.346 130.083 216.399 130.069 216.44 130.042C216.481 130.014 216.501 129.98 216.501 129.94C216.501 129.906 216.488 129.878 216.461 129.855C216.436 129.833 216.395 129.816 216.34 129.804L216.137 129.761C216.023 129.738 215.938 129.698 215.883 129.641C215.828 129.584 215.801 129.512 215.802 129.425C215.801 129.352 215.821 129.288 215.862 129.235C215.902 129.18 215.959 129.138 216.032 129.109C216.104 129.079 216.189 129.064 216.284 129.064C216.423 129.064 216.533 129.094 216.614 129.153C216.694 129.213 216.744 129.293 216.763 129.394ZM217.385 130.295C217.31 130.295 217.242 130.282 217.181 130.255C217.121 130.228 217.074 130.187 217.038 130.134C217.004 130.081 216.986 130.015 216.986 129.937C216.986 129.87 216.999 129.814 217.024 129.77C217.049 129.726 217.082 129.691 217.125 129.664C217.168 129.638 217.217 129.618 217.271 129.605C217.325 129.591 217.381 129.581 217.439 129.574C217.509 129.567 217.565 129.561 217.609 129.555C217.652 129.549 217.684 129.539 217.704 129.527C217.724 129.514 217.734 129.494 217.734 129.467V129.463C217.734 129.404 217.716 129.359 217.682 129.327C217.647 129.295 217.597 129.279 217.532 129.279C217.463 129.279 217.409 129.294 217.368 129.324C217.328 129.354 217.302 129.389 217.288 129.43L217.025 129.393C217.046 129.32 217.08 129.26 217.128 129.211C217.175 129.162 217.234 129.125 217.302 129.101C217.371 129.076 217.447 129.064 217.53 129.064C217.588 129.064 217.645 129.071 217.702 129.084C217.759 129.097 217.811 129.12 217.858 129.151C217.905 129.181 217.943 129.223 217.971 129.276C218 129.328 218.015 129.394 218.015 129.474V130.271H217.745V130.108H217.735C217.718 130.141 217.694 130.172 217.663 130.201C217.633 130.229 217.594 130.252 217.548 130.27C217.501 130.287 217.447 130.295 217.385 130.295ZM217.458 130.089C217.515 130.089 217.564 130.078 217.605 130.056C217.646 130.033 217.678 130.003 217.7 129.965C217.723 129.928 217.735 129.888 217.735 129.844V129.703C217.726 129.71 217.711 129.717 217.69 129.723C217.669 129.73 217.646 129.735 217.62 129.74C217.594 129.744 217.568 129.748 217.543 129.752C217.518 129.756 217.496 129.759 217.477 129.761C217.435 129.767 217.398 129.776 217.364 129.789C217.331 129.802 217.305 129.82 217.286 129.844C217.267 129.866 217.257 129.896 217.257 129.932C217.257 129.984 217.276 130.023 217.314 130.049C217.352 130.076 217.4 130.089 217.458 130.089ZM218.565 130.271L218.228 129.079H218.515L218.724 129.917H218.735L218.949 129.079H219.232L219.447 129.913H219.458L219.665 129.079H219.952L219.614 130.271H219.322L219.098 129.466H219.082L218.858 130.271H218.565ZM220.746 130.271V128.682H221.027V129.276H221.039C221.053 129.247 221.073 129.217 221.1 129.184C221.126 129.151 221.162 129.123 221.207 129.099C221.252 129.076 221.309 129.064 221.379 129.064C221.471 129.064 221.554 129.087 221.628 129.134C221.703 129.181 221.762 129.25 221.805 129.342C221.849 129.433 221.871 129.544 221.871 129.677C221.871 129.808 221.85 129.919 221.807 130.011C221.764 130.102 221.705 130.172 221.631 130.22C221.557 130.268 221.474 130.292 221.38 130.292C221.312 130.292 221.255 130.281 221.21 130.258C221.165 130.235 221.129 130.208 221.101 130.176C221.074 130.143 221.054 130.112 221.039 130.083H221.022V130.271H220.746ZM221.021 129.675C221.021 129.752 221.032 129.82 221.054 129.878C221.076 129.936 221.108 129.981 221.149 130.014C221.191 130.046 221.242 130.062 221.302 130.062C221.364 130.062 221.416 130.045 221.458 130.012C221.499 129.978 221.531 129.933 221.552 129.875C221.574 129.816 221.585 129.75 221.585 129.675C221.585 129.601 221.574 129.536 221.553 129.478C221.532 129.421 221.5 129.376 221.458 129.343C221.416 129.311 221.364 129.294 221.302 129.294C221.242 129.294 221.191 129.31 221.149 129.342C221.107 129.373 221.075 129.417 221.053 129.474C221.032 129.531 221.021 129.598 221.021 129.675ZM222.892 129.77V129.079H223.172V130.271H222.9V130.059H222.888C222.861 130.126 222.817 130.181 222.755 130.223C222.694 130.266 222.619 130.287 222.529 130.287C222.451 130.287 222.382 130.269 222.322 130.235C222.262 130.2 222.216 130.149 222.182 130.082C222.149 130.015 222.132 129.933 222.132 129.838V129.079H222.413V129.795C222.413 129.87 222.433 129.93 222.475 129.975C222.516 130.019 222.571 130.042 222.638 130.042C222.679 130.042 222.719 130.031 222.758 130.011C222.797 129.991 222.829 129.961 222.854 129.921C222.879 129.881 222.892 129.83 222.892 129.77ZM223.483 130.271V129.079H223.764V130.271H223.483ZM223.624 128.91C223.58 128.91 223.542 128.895 223.51 128.866C223.477 128.836 223.461 128.8 223.461 128.758C223.461 128.716 223.477 128.68 223.51 128.65C223.542 128.62 223.58 128.605 223.624 128.605C223.669 128.605 223.708 128.62 223.739 128.65C223.771 128.68 223.787 128.716 223.787 128.758C223.787 128.8 223.771 128.836 223.739 128.866C223.708 128.895 223.669 128.91 223.624 128.91ZM224.356 128.682V130.271H224.075V128.682H224.356ZM225.109 130.292C225.015 130.292 224.931 130.268 224.857 130.22C224.783 130.172 224.725 130.102 224.682 130.011C224.639 129.919 224.618 129.808 224.618 129.677C224.618 129.544 224.639 129.433 224.683 129.342C224.727 129.25 224.786 129.181 224.86 129.134C224.935 129.087 225.018 129.064 225.109 129.064C225.179 129.064 225.237 129.076 225.282 129.099C225.327 129.123 225.362 129.151 225.389 129.184C225.415 129.217 225.436 129.247 225.45 129.276H225.462V128.682H225.743V130.271H225.467V130.083H225.45C225.436 130.112 225.415 130.143 225.387 130.176C225.36 130.208 225.324 130.235 225.279 130.258C225.234 130.281 225.177 130.292 225.109 130.292ZM225.187 130.062C225.247 130.062 225.297 130.046 225.339 130.014C225.381 129.981 225.413 129.936 225.435 129.878C225.456 129.82 225.467 129.752 225.467 129.675C225.467 129.598 225.456 129.531 225.435 129.474C225.413 129.417 225.382 129.373 225.34 129.342C225.299 129.31 225.248 129.294 225.187 129.294C225.125 129.294 225.072 129.311 225.03 129.343C224.988 129.376 224.957 129.421 224.936 129.478C224.914 129.536 224.904 129.601 224.904 129.675C224.904 129.75 224.914 129.816 224.936 129.875C224.957 129.933 224.989 129.978 225.031 130.012C225.074 130.045 225.126 130.062 225.187 130.062ZM226.065 130.271V129.079H226.346V130.271H226.065ZM226.206 128.91C226.161 128.91 226.123 128.895 226.091 128.866C226.059 128.836 226.043 128.8 226.043 128.758C226.043 128.716 226.059 128.68 226.091 128.65C226.123 128.62 226.161 128.605 226.206 128.605C226.251 128.605 226.289 128.62 226.321 128.65C226.353 128.68 226.369 128.716 226.369 128.758C226.369 128.8 226.353 128.836 226.321 128.866C226.289 128.895 226.251 128.91 226.206 128.91ZM226.937 129.573V130.271H226.656V129.079H226.925V129.282H226.939C226.966 129.215 227.01 129.162 227.07 129.123C227.13 129.083 227.205 129.064 227.294 129.064C227.376 129.064 227.448 129.081 227.509 129.117C227.571 129.152 227.618 129.203 227.652 129.269C227.686 129.336 227.703 129.417 227.702 129.512V130.271H227.421V129.556C227.421 129.476 227.401 129.414 227.359 129.369C227.318 129.324 227.262 129.301 227.189 129.301C227.14 129.301 227.097 129.312 227.058 129.334C227.021 129.355 226.991 129.386 226.969 129.426C226.948 129.467 226.937 129.515 226.937 129.573ZM228.528 130.743C228.427 130.743 228.341 130.729 228.268 130.702C228.196 130.675 228.138 130.639 228.094 130.593C228.05 130.548 228.019 130.497 228.002 130.442L228.255 130.381C228.266 130.404 228.283 130.427 228.305 130.45C228.326 130.473 228.356 130.492 228.392 130.507C228.43 130.523 228.476 130.53 228.533 130.53C228.612 130.53 228.678 130.511 228.731 130.472C228.783 130.434 228.809 130.371 228.809 130.283V130.057H228.795C228.781 130.086 228.759 130.116 228.731 130.146C228.704 130.177 228.668 130.202 228.622 130.223C228.577 130.244 228.52 130.254 228.452 130.254C228.361 130.254 228.277 130.233 228.203 130.19C228.129 130.146 228.07 130.082 228.026 129.996C227.983 129.909 227.961 129.801 227.961 129.671C227.961 129.541 227.983 129.43 228.026 129.34C228.07 129.25 228.129 129.181 228.204 129.134C228.278 129.087 228.361 129.064 228.453 129.064C228.523 129.064 228.58 129.076 228.625 129.099C228.671 129.123 228.707 129.151 228.734 129.184C228.761 129.217 228.781 129.247 228.795 129.276H228.811V129.079H229.088V130.291C229.088 130.393 229.063 130.477 229.015 130.544C228.966 130.61 228.9 130.66 228.815 130.693C228.731 130.726 228.635 130.743 228.528 130.743ZM228.53 130.034C228.59 130.034 228.641 130.019 228.683 129.99C228.724 129.961 228.756 129.92 228.778 129.865C228.8 129.811 228.811 129.746 228.811 129.67C228.811 129.595 228.8 129.529 228.778 129.473C228.757 129.416 228.725 129.373 228.683 129.342C228.642 129.31 228.591 129.294 228.53 129.294C228.468 129.294 228.416 129.311 228.374 129.343C228.332 129.376 228.3 129.42 228.279 129.477C228.258 129.534 228.247 129.598 228.247 129.67C228.247 129.743 228.258 129.807 228.279 129.862C228.301 129.916 228.333 129.958 228.374 129.989C228.417 130.019 228.469 130.034 228.53 130.034Z" fill="#434960"/><g clip-path="url(#clip3_5232_97155)"><path d="M202.887 134.273V134.292C203.145 134.292 203.354 134.501 203.354 134.759C203.354 135.035 203.19 135.302 202.978 135.533C202.77 135.759 202.519 135.948 202.341 136.074C202.162 135.948 201.911 135.759 201.704 135.533C201.492 135.302 201.327 135.035 201.327 134.759C201.327 134.501 201.537 134.292 201.795 134.292V134.273L201.794 134.292C201.87 134.292 201.943 134.31 202.01 134.345C202.077 134.379 202.134 134.429 202.177 134.491L202.177 134.491L202.326 134.703L202.341 134.724L202.356 134.703L202.505 134.491L202.505 134.491C202.548 134.43 202.605 134.379 202.671 134.345C202.738 134.31 202.812 134.292 202.887 134.292L202.887 134.273ZM202.887 134.273C203.155 134.273 203.372 134.491 203.372 134.759C203.372 135.326 202.7 135.844 202.341 136.097L202.49 134.481C202.534 134.417 202.594 134.365 202.663 134.328C202.732 134.292 202.809 134.273 202.887 134.273ZM202.341 136.341L202.33 136.334C202.118 136.187 201.813 135.97 201.561 135.702C201.31 135.434 201.109 135.113 201.109 134.759C201.109 134.577 201.181 134.403 201.31 134.274C201.438 134.145 201.613 134.073 201.795 134.073L202.341 136.341ZM202.341 136.341L202.351 136.334M202.341 136.341L202.351 136.334M202.351 136.334C202.564 136.187 202.868 135.97 203.12 135.702C203.372 135.434 203.573 135.113 203.573 134.759C203.573 134.669 203.555 134.58 203.52 134.496C203.486 134.413 203.435 134.338 203.372 134.274C203.308 134.21 203.233 134.16 203.149 134.125C203.066 134.091 202.977 134.073 202.887 134.073C202.777 134.073 202.668 134.1 202.571 134.151M202.351 136.334L202.571 134.151M202.571 134.151C202.48 134.197 202.402 134.264 202.341 134.345M202.571 134.151L202.341 134.345M202.341 134.345C202.28 134.264 202.201 134.197 202.111 134.151C202.013 134.1 201.905 134.073 201.795 134.073L202.341 134.345Z" fill="#434960" stroke="#434960" stroke-width="0.03642"/></g><path d="M205.752 136.045H205.028L205.375 135.652C205.607 135.405 205.732 135.247 205.732 135.017C205.732 134.752 205.52 134.525 205.176 134.525C204.842 134.525 204.595 134.747 204.595 135.073H204.847C204.847 134.88 204.967 134.757 205.158 134.757C205.344 134.757 205.462 134.887 205.462 135.05C205.462 135.226 205.352 135.354 205.176 135.542L204.638 136.126V136.274H205.752V136.045ZM207.212 135.894V135.67H206.978V134.551H206.779L205.889 135.713V135.894H206.72V136.274H206.978V135.894H207.212ZM206.18 135.67L206.72 134.941V135.67H206.18Z" fill="#434960"/><g clip-path="url(#clip4_5232_97155)"><path d="M212.536 133.969C212.376 133.969 212.218 134 212.071 134.061C211.924 134.122 211.79 134.212 211.677 134.324C211.45 134.552 211.322 134.861 211.322 135.182C211.321 135.463 211.418 135.735 211.596 135.951L211.353 136.194C211.336 136.211 211.325 136.232 211.321 136.256C211.316 136.279 211.319 136.304 211.328 136.326C211.338 136.348 211.354 136.366 211.375 136.379C211.395 136.391 211.419 136.397 211.443 136.396H212.536C212.857 136.396 213.166 136.268 213.394 136.041C213.621 135.813 213.749 135.504 213.749 135.182C213.749 134.861 213.621 134.552 213.394 134.324C213.166 134.097 212.857 133.969 212.536 133.969ZM212.536 136.153H211.736L211.849 136.041C211.871 136.018 211.884 135.987 211.884 135.955C211.884 135.923 211.871 135.892 211.849 135.869C211.69 135.711 211.591 135.502 211.568 135.278C211.546 135.055 211.602 134.83 211.727 134.644C211.852 134.457 212.037 134.319 212.252 134.253C212.467 134.188 212.698 134.199 212.906 134.285C213.113 134.371 213.285 134.526 213.391 134.724C213.497 134.922 213.531 135.151 213.487 135.371C213.444 135.592 213.325 135.79 213.151 135.933C212.978 136.075 212.76 136.153 212.536 136.153Z" fill="#434960"/></g><path d="M216.44 135.716C216.44 135.328 216.108 135.096 215.711 135.183L216.218 134.551H215.925L215.405 135.193C215.252 135.382 215.178 135.517 215.178 135.729C215.178 136.042 215.443 136.297 215.807 136.297C216.177 136.297 216.44 136.042 216.44 135.716ZM215.448 135.721C215.448 135.481 215.609 135.359 215.81 135.359C216.014 135.359 216.172 135.497 216.172 135.723C216.172 135.938 216.014 136.08 215.81 136.08C215.609 136.08 215.448 135.938 215.448 135.721Z" fill="#434960"/></g></g><g filter="url(#filter2_dd_5232_97155)"><g clip-path="url(#clip5_5232_97155)"><rect x="198.697" y="142.014" width="51.8985" height="68.4688" rx="0.7284" fill="white"/><rect x="198.697" y="142.014" width="51.8985" height="48.4386" fill="url(#pattern1)"/></g></g><g filter="url(#filter3_dd_5232_97155)"><g clip-path="url(#clip6_5232_97155)"><rect x="254.238" y="57.1658" width="51.8985" height="67.6354" rx="0.7284" fill="white"/><rect x="254.238" y="57.1658" width="51.8985" height="48.4386" fill="url(#pattern2)"/><rect x="273.357" y="74.4639" width="13.8396" height="13.8396" rx="6.9198" fill="white"/><path d="M278.639 79.675C278.639 79.5396 278.781 79.4516 278.902 79.5121L282.865 81.4937C283 81.5609 283 81.7524 282.865 81.8195L278.902 83.8011C278.781 83.8616 278.639 83.7736 278.639 83.6382V79.675Z" fill="black"/><path d="M257.444 108.637C257.437 108.57 257.406 108.517 257.353 108.479C257.299 108.441 257.23 108.422 257.145 108.422C257.085 108.422 257.033 108.432 256.99 108.45C256.947 108.468 256.915 108.492 256.892 108.523C256.869 108.554 256.857 108.59 256.857 108.63C256.857 108.663 256.864 108.692 256.879 108.716C256.895 108.74 256.916 108.761 256.942 108.778C256.969 108.794 256.998 108.808 257.03 108.82C257.062 108.831 257.094 108.841 257.127 108.849L257.276 108.886C257.336 108.9 257.394 108.919 257.449 108.942C257.505 108.966 257.555 108.996 257.599 109.032C257.643 109.069 257.678 109.112 257.704 109.164C257.73 109.215 257.743 109.275 257.743 109.344C257.743 109.437 257.719 109.519 257.672 109.59C257.624 109.66 257.555 109.715 257.465 109.755C257.376 109.794 257.267 109.814 257.14 109.814C257.016 109.814 256.909 109.795 256.818 109.756C256.728 109.718 256.657 109.662 256.605 109.589C256.555 109.515 256.527 109.426 256.523 109.32H256.806C256.811 109.376 256.828 109.422 256.858 109.458C256.888 109.495 256.927 109.523 256.975 109.541C257.023 109.559 257.078 109.568 257.138 109.568C257.2 109.568 257.255 109.559 257.302 109.54C257.35 109.521 257.387 109.494 257.414 109.461C257.441 109.427 257.455 109.387 257.455 109.341C257.455 109.3 257.442 109.266 257.419 109.239C257.395 109.211 257.361 109.189 257.319 109.171C257.276 109.152 257.226 109.135 257.17 109.121L256.989 109.074C256.858 109.041 256.754 108.99 256.678 108.921C256.603 108.853 256.565 108.761 256.565 108.648C256.565 108.554 256.59 108.472 256.641 108.402C256.692 108.331 256.762 108.277 256.85 108.238C256.938 108.198 257.037 108.179 257.149 108.179C257.261 108.179 257.36 108.198 257.445 108.238C257.53 108.277 257.597 108.331 257.646 108.4C257.695 108.469 257.72 108.548 257.721 108.637H257.444ZM258.537 109.813C258.417 109.813 258.314 109.788 258.227 109.739C258.141 109.688 258.074 109.618 258.028 109.526C257.981 109.434 257.958 109.325 257.958 109.201C257.958 109.078 257.981 108.971 258.028 108.878C258.075 108.785 258.141 108.712 258.225 108.661C258.309 108.608 258.408 108.582 258.522 108.582C258.596 108.582 258.665 108.594 258.73 108.618C258.796 108.641 258.854 108.678 258.904 108.727C258.955 108.776 258.994 108.838 259.023 108.914C259.052 108.99 259.067 109.08 259.067 109.185V109.271H258.09V109.081H258.798C258.797 109.028 258.785 108.98 258.763 108.938C258.74 108.895 258.708 108.862 258.667 108.838C258.627 108.813 258.58 108.801 258.526 108.801C258.469 108.801 258.418 108.815 258.375 108.843C258.331 108.871 258.297 108.907 258.273 108.952C258.249 108.996 258.237 109.045 258.236 109.098V109.264C258.236 109.333 258.249 109.392 258.275 109.442C258.3 109.491 258.335 109.529 258.381 109.555C258.426 109.581 258.48 109.594 258.541 109.594C258.582 109.594 258.619 109.589 258.652 109.577C258.685 109.565 258.713 109.548 258.738 109.525C258.762 109.502 258.78 109.474 258.793 109.441L259.055 109.47C259.039 109.539 259.007 109.6 258.96 109.652C258.914 109.703 258.855 109.743 258.784 109.771C258.712 109.799 258.629 109.813 258.537 109.813ZM259.854 109.813C259.734 109.813 259.631 109.788 259.544 109.739C259.458 109.688 259.391 109.618 259.345 109.526C259.298 109.434 259.275 109.325 259.275 109.201C259.275 109.078 259.298 108.971 259.345 108.878C259.392 108.785 259.458 108.712 259.542 108.661C259.626 108.608 259.725 108.582 259.839 108.582C259.913 108.582 259.982 108.594 260.047 108.618C260.113 108.641 260.171 108.678 260.221 108.727C260.272 108.776 260.312 108.838 260.341 108.914C260.369 108.99 260.384 109.08 260.384 109.185V109.271H259.407V109.081H260.115C260.114 109.028 260.103 108.98 260.08 108.938C260.057 108.895 260.025 108.862 259.984 108.838C259.944 108.813 259.897 108.801 259.843 108.801C259.786 108.801 259.735 108.815 259.692 108.843C259.648 108.871 259.614 108.907 259.59 108.952C259.566 108.996 259.554 109.045 259.554 109.098V109.264C259.554 109.333 259.566 109.392 259.592 109.442C259.617 109.491 259.652 109.529 259.698 109.555C259.744 109.581 259.797 109.594 259.858 109.594C259.899 109.594 259.936 109.589 259.969 109.577C260.002 109.565 260.031 109.548 260.055 109.525C260.079 109.502 260.098 109.474 260.11 109.441L260.372 109.47C260.356 109.539 260.324 109.6 260.278 109.652C260.232 109.703 260.173 109.743 260.101 109.771C260.029 109.799 259.947 109.813 259.854 109.813ZM261.803 108.598V108.815H261.117V108.598H261.803ZM261.287 108.312H261.567V109.431C261.567 109.469 261.573 109.498 261.584 109.518C261.596 109.538 261.612 109.551 261.631 109.559C261.65 109.566 261.671 109.569 261.695 109.569C261.712 109.569 261.728 109.568 261.743 109.566C261.758 109.563 261.769 109.561 261.777 109.559L261.824 109.778C261.809 109.783 261.788 109.789 261.76 109.795C261.732 109.801 261.699 109.805 261.659 109.806C261.589 109.808 261.525 109.798 261.469 109.774C261.412 109.75 261.368 109.714 261.335 109.664C261.302 109.614 261.286 109.552 261.287 109.478V108.312ZM262.366 109.091V109.79H262.085V108.201H262.36V108.8H262.374C262.402 108.733 262.445 108.68 262.503 108.641C262.562 108.602 262.637 108.582 262.728 108.582C262.811 108.582 262.883 108.6 262.945 108.634C263.006 108.669 263.054 108.72 263.088 108.786C263.122 108.853 263.139 108.935 263.139 109.031V109.79H262.858V109.074C262.858 108.994 262.837 108.932 262.796 108.887C262.755 108.842 262.697 108.82 262.623 108.82C262.574 108.82 262.529 108.831 262.49 108.852C262.451 108.874 262.421 108.904 262.398 108.945C262.377 108.985 262.366 109.034 262.366 109.091ZM263.975 109.813C263.855 109.813 263.752 109.788 263.665 109.739C263.579 109.688 263.512 109.618 263.466 109.526C263.419 109.434 263.396 109.325 263.396 109.201C263.396 109.078 263.419 108.971 263.466 108.878C263.513 108.785 263.578 108.712 263.663 108.661C263.747 108.608 263.846 108.582 263.96 108.582C264.033 108.582 264.103 108.594 264.168 108.618C264.234 108.641 264.291 108.678 264.342 108.727C264.392 108.776 264.432 108.838 264.461 108.914C264.49 108.99 264.505 109.08 264.505 109.185V109.271H263.528V109.081H264.235C264.235 109.028 264.223 108.98 264.2 108.938C264.178 108.895 264.146 108.862 264.105 108.838C264.065 108.813 264.018 108.801 263.964 108.801C263.906 108.801 263.856 108.815 263.812 108.843C263.769 108.871 263.735 108.907 263.711 108.952C263.687 108.996 263.675 109.045 263.674 109.098V109.264C263.674 109.333 263.687 109.392 263.712 109.442C263.738 109.491 263.773 109.529 263.819 109.555C263.864 109.581 263.917 109.594 263.978 109.594C264.019 109.594 264.056 109.589 264.089 109.577C264.123 109.565 264.151 109.548 264.176 109.525C264.2 109.502 264.218 109.474 264.231 109.441L264.493 109.47C264.476 109.539 264.445 109.6 264.398 109.652C264.352 109.703 264.293 109.743 264.221 109.771C264.149 109.799 264.067 109.813 263.975 109.813ZM265.932 108.598V108.815H265.227V108.598H265.932ZM265.403 109.79V108.485C265.403 108.405 265.42 108.338 265.453 108.285C265.487 108.232 265.532 108.192 265.588 108.166C265.644 108.139 265.707 108.126 265.776 108.126C265.824 108.126 265.868 108.13 265.905 108.138C265.943 108.145 265.971 108.152 265.989 108.159L265.933 108.376C265.921 108.372 265.906 108.369 265.888 108.365C265.87 108.361 265.85 108.359 265.828 108.359C265.775 108.359 265.739 108.372 265.717 108.397C265.696 108.422 265.685 108.457 265.685 108.504V109.79H265.403ZM266.656 109.813C266.537 109.813 266.434 109.788 266.347 109.739C266.26 109.688 266.194 109.618 266.147 109.526C266.101 109.434 266.078 109.325 266.078 109.201C266.078 109.078 266.101 108.971 266.147 108.878C266.194 108.785 266.26 108.712 266.344 108.661C266.429 108.608 266.528 108.582 266.642 108.582C266.715 108.582 266.784 108.594 266.85 108.618C266.915 108.641 266.973 108.678 267.023 108.727C267.074 108.776 267.114 108.838 267.143 108.914C267.172 108.99 267.186 109.08 267.186 109.185V109.271H266.209V109.081H266.917C266.917 109.028 266.905 108.98 266.882 108.938C266.859 108.895 266.828 108.862 266.787 108.838C266.746 108.813 266.699 108.801 266.646 108.801C266.588 108.801 266.538 108.815 266.494 108.843C266.451 108.871 266.417 108.907 266.393 108.952C266.369 108.996 266.357 109.045 266.356 109.098V109.264C266.356 109.333 266.369 109.392 266.394 109.442C266.419 109.491 266.455 109.529 266.5 109.555C266.546 109.581 266.599 109.594 266.66 109.594C266.701 109.594 266.738 109.589 266.771 109.577C266.804 109.565 266.833 109.548 266.857 109.525C266.882 109.502 266.9 109.474 266.913 109.441L267.175 109.47C267.158 109.539 267.127 109.6 267.08 109.652C267.034 109.703 266.975 109.743 266.903 109.771C266.831 109.799 266.749 109.813 266.656 109.813ZM267.789 109.814C267.713 109.814 267.645 109.8 267.585 109.773C267.525 109.746 267.477 109.706 267.442 109.652C267.407 109.599 267.39 109.533 267.39 109.455C267.39 109.388 267.402 109.332 267.427 109.289C267.452 109.245 267.486 109.209 267.529 109.183C267.572 109.157 267.62 109.137 267.674 109.123C267.728 109.109 267.785 109.099 267.842 109.093C267.912 109.086 267.969 109.079 268.012 109.074C268.056 109.067 268.087 109.058 268.107 109.046C268.127 109.033 268.137 109.013 268.137 108.986V108.981C268.137 108.923 268.12 108.877 268.085 108.845C268.051 108.813 268.001 108.797 267.936 108.797C267.867 108.797 267.812 108.812 267.772 108.842C267.732 108.872 267.705 108.908 267.691 108.949L267.429 108.911C267.45 108.839 267.484 108.778 267.531 108.73C267.579 108.681 267.637 108.644 267.706 108.62C267.775 108.595 267.851 108.582 267.934 108.582C267.991 108.582 268.049 108.589 268.106 108.603C268.162 108.616 268.214 108.638 268.262 108.669C268.309 108.7 268.346 108.741 268.375 108.794C268.404 108.847 268.418 108.913 268.418 108.992V109.79H268.148V109.626H268.139C268.122 109.659 268.098 109.69 268.067 109.719C268.036 109.748 267.998 109.771 267.951 109.788C267.905 109.805 267.851 109.814 267.789 109.814ZM267.862 109.607C267.918 109.607 267.967 109.596 268.009 109.574C268.05 109.551 268.082 109.521 268.104 109.484C268.127 109.447 268.138 109.406 268.138 109.362V109.222C268.129 109.229 268.114 109.236 268.093 109.242C268.072 109.248 268.049 109.254 268.023 109.258C267.997 109.263 267.972 109.267 267.946 109.271C267.921 109.274 267.899 109.277 267.88 109.28C267.839 109.286 267.801 109.295 267.768 109.308C267.735 109.321 267.709 109.339 267.69 109.362C267.67 109.385 267.661 109.414 267.661 109.451C267.661 109.502 267.68 109.541 267.718 109.568C267.755 109.594 267.803 109.607 267.862 109.607ZM269.318 108.598V108.815H268.633V108.598H269.318ZM268.802 108.312H269.083V109.431C269.083 109.469 269.088 109.498 269.1 109.518C269.112 109.538 269.127 109.551 269.146 109.559C269.165 109.566 269.187 109.569 269.21 109.569C269.227 109.569 269.244 109.568 269.258 109.566C269.273 109.563 269.284 109.561 269.292 109.559L269.339 109.778C269.324 109.783 269.303 109.789 269.275 109.795C269.248 109.801 269.214 109.805 269.174 109.806C269.104 109.808 269.04 109.798 268.984 109.774C268.928 109.75 268.883 109.714 268.85 109.664C268.817 109.614 268.801 109.552 268.802 109.478V108.312ZM270.334 109.289V108.598H270.615V109.79H270.343V109.578H270.33C270.303 109.645 270.259 109.699 270.198 109.742C270.137 109.784 270.061 109.805 269.972 109.805C269.894 109.805 269.825 109.788 269.765 109.753C269.705 109.718 269.659 109.667 269.625 109.6C269.591 109.533 269.575 109.452 269.575 109.357V108.598H269.855V109.313C269.855 109.389 269.876 109.449 269.918 109.493C269.959 109.538 270.013 109.56 270.08 109.56C270.122 109.56 270.162 109.55 270.201 109.53C270.24 109.51 270.271 109.48 270.296 109.44C270.322 109.399 270.334 109.349 270.334 109.289ZM270.926 109.79V108.598H271.198V108.797H271.211C271.232 108.728 271.27 108.675 271.322 108.637C271.376 108.6 271.436 108.581 271.505 108.581C271.52 108.581 271.538 108.582 271.557 108.583C271.576 108.584 271.593 108.586 271.606 108.589V108.847C271.594 108.843 271.575 108.839 271.549 108.836C271.524 108.832 271.499 108.831 271.475 108.831C271.424 108.831 271.378 108.842 271.337 108.864C271.297 108.886 271.265 108.916 271.242 108.955C271.218 108.994 271.207 109.038 271.207 109.089V109.79H270.926ZM272.293 109.813C272.173 109.813 272.07 109.788 271.983 109.739C271.897 109.688 271.83 109.618 271.784 109.526C271.737 109.434 271.714 109.325 271.714 109.201C271.714 109.078 271.737 108.971 271.784 108.878C271.831 108.785 271.896 108.712 271.981 108.661C272.065 108.608 272.164 108.582 272.278 108.582C272.351 108.582 272.421 108.594 272.486 108.618C272.552 108.641 272.61 108.678 272.66 108.727C272.71 108.776 272.75 108.838 272.779 108.914C272.808 108.99 272.823 109.08 272.823 109.185V109.271H271.846V109.081H272.553C272.553 109.028 272.541 108.98 272.519 108.938C272.496 108.895 272.464 108.862 272.423 108.838C272.383 108.813 272.336 108.801 272.282 108.801C272.224 108.801 272.174 108.815 272.131 108.843C272.087 108.871 272.053 108.907 272.029 108.952C272.005 108.996 271.993 109.045 271.992 109.098V109.264C271.992 109.333 272.005 109.392 272.03 109.442C272.056 109.491 272.091 109.529 272.137 109.555C272.182 109.581 272.236 109.594 272.297 109.594C272.337 109.594 272.374 109.589 272.408 109.577C272.441 109.565 272.469 109.548 272.494 109.525C272.518 109.502 272.536 109.474 272.549 109.441L272.811 109.47C272.795 109.539 272.763 109.6 272.716 109.652C272.67 109.703 272.611 109.743 272.539 109.771C272.468 109.799 272.385 109.813 272.293 109.813ZM273.525 109.811C273.431 109.811 273.347 109.787 273.273 109.739C273.199 109.69 273.141 109.621 273.098 109.529C273.055 109.437 273.033 109.326 273.033 109.195C273.033 109.063 273.055 108.951 273.099 108.86C273.142 108.769 273.202 108.7 273.276 108.653C273.351 108.606 273.434 108.582 273.525 108.582C273.595 108.582 273.653 108.594 273.698 108.618C273.743 108.641 273.778 108.67 273.805 108.703C273.831 108.735 273.851 108.766 273.866 108.795H273.878V108.201H274.159V109.79H273.883V109.602H273.866C273.851 109.631 273.831 109.662 273.803 109.694C273.776 109.726 273.739 109.754 273.694 109.777C273.649 109.799 273.593 109.811 273.525 109.811ZM273.603 109.58C273.662 109.58 273.713 109.564 273.755 109.532C273.797 109.5 273.829 109.454 273.85 109.396C273.872 109.338 273.883 109.271 273.883 109.194C273.883 109.117 273.872 109.05 273.85 108.993C273.829 108.936 273.798 108.892 273.756 108.86C273.714 108.829 273.663 108.813 273.603 108.813C273.54 108.813 273.488 108.829 273.446 108.862C273.404 108.894 273.373 108.939 273.351 108.997C273.33 109.054 273.32 109.12 273.32 109.194C273.32 109.268 273.33 109.335 273.351 109.393C273.373 109.451 273.405 109.497 273.447 109.531C273.489 109.564 273.541 109.58 273.603 109.58ZM275.572 109.813C275.453 109.813 275.349 109.788 275.262 109.739C275.176 109.688 275.11 109.618 275.063 109.526C275.016 109.434 274.993 109.325 274.993 109.201C274.993 109.078 275.016 108.971 275.063 108.878C275.11 108.785 275.176 108.712 275.26 108.661C275.344 108.608 275.443 108.582 275.557 108.582C275.631 108.582 275.7 108.594 275.765 108.618C275.831 108.641 275.889 108.678 275.939 108.727C275.99 108.776 276.03 108.838 276.059 108.914C276.088 108.99 276.102 109.08 276.102 109.185V109.271H275.125V109.081H275.833C275.832 109.028 275.821 108.98 275.798 108.938C275.775 108.895 275.743 108.862 275.702 108.838C275.662 108.813 275.615 108.801 275.561 108.801C275.504 108.801 275.453 108.815 275.41 108.843C275.366 108.871 275.333 108.907 275.308 108.952C275.284 108.996 275.272 109.045 275.272 109.098V109.264C275.272 109.333 275.284 109.392 275.31 109.442C275.335 109.491 275.371 109.529 275.416 109.555C275.462 109.581 275.515 109.594 275.576 109.594C275.617 109.594 275.654 109.589 275.687 109.577C275.72 109.565 275.749 109.548 275.773 109.525C275.797 109.502 275.816 109.474 275.828 109.441L276.09 109.47C276.074 109.539 276.042 109.6 275.996 109.652C275.95 109.703 275.891 109.743 275.819 109.771C275.747 109.799 275.665 109.813 275.572 109.813ZM276.804 109.811C276.71 109.811 276.626 109.787 276.552 109.739C276.478 109.69 276.42 109.621 276.377 109.529C276.334 109.437 276.313 109.326 276.313 109.195C276.313 109.063 276.334 108.951 276.378 108.86C276.422 108.769 276.481 108.7 276.556 108.653C276.63 108.606 276.713 108.582 276.805 108.582C276.874 108.582 276.932 108.594 276.977 108.618C277.022 108.641 277.058 108.67 277.084 108.703C277.11 108.735 277.131 108.766 277.145 108.795H277.157V108.201H277.439V109.79H277.162V109.602H277.145C277.131 109.631 277.11 109.662 277.082 109.694C277.055 109.726 277.019 109.754 276.974 109.777C276.929 109.799 276.872 109.811 276.804 109.811ZM276.882 109.58C276.942 109.58 276.992 109.564 277.034 109.532C277.076 109.5 277.108 109.454 277.13 109.396C277.152 109.338 277.162 109.271 277.162 109.194C277.162 109.117 277.152 109.05 277.13 108.993C277.109 108.936 277.077 108.892 277.035 108.86C276.994 108.829 276.943 108.813 276.882 108.813C276.82 108.813 276.767 108.829 276.725 108.862C276.684 108.894 276.652 108.939 276.631 108.997C276.61 109.054 276.599 109.12 276.599 109.194C276.599 109.268 276.61 109.335 276.631 109.393C276.653 109.451 276.684 109.497 276.726 109.531C276.769 109.564 276.821 109.58 276.882 109.58ZM277.76 109.79V108.598H278.041V109.79H277.76ZM277.901 108.429C277.857 108.429 277.818 108.414 277.786 108.384C277.754 108.354 277.738 108.319 277.738 108.277C277.738 108.234 277.754 108.198 277.786 108.169C277.818 108.139 277.857 108.124 277.901 108.124C277.946 108.124 277.984 108.139 278.016 108.169C278.048 108.198 278.064 108.234 278.064 108.277C278.064 108.319 278.048 108.354 278.016 108.384C277.984 108.414 277.946 108.429 277.901 108.429ZM278.947 108.598V108.815H278.261V108.598H278.947ZM278.431 108.312H278.712V109.431C278.712 109.469 278.717 109.498 278.729 109.518C278.74 109.538 278.756 109.551 278.775 109.559C278.794 109.566 278.815 109.569 278.839 109.569C278.856 109.569 278.872 109.568 278.887 109.566C278.902 109.563 278.913 109.561 278.921 109.559L278.968 109.778C278.953 109.783 278.932 109.789 278.904 109.795C278.877 109.801 278.843 109.805 278.803 109.806C278.733 109.808 278.669 109.798 278.613 109.774C278.557 109.75 278.512 109.714 278.479 109.664C278.446 109.614 278.43 109.552 278.431 109.478V108.312ZM279.203 109.79V108.598H279.484V109.79H279.203ZM279.345 108.429C279.3 108.429 279.262 108.414 279.23 108.384C279.198 108.354 279.182 108.319 279.182 108.277C279.182 108.234 279.198 108.198 279.23 108.169C279.262 108.139 279.3 108.124 279.345 108.124C279.39 108.124 279.428 108.139 279.46 108.169C279.492 108.198 279.508 108.234 279.508 108.277C279.508 108.319 279.492 108.354 279.46 108.384C279.428 108.414 279.39 108.429 279.345 108.429ZM280.315 109.813C280.199 109.813 280.098 109.787 280.012 109.736C279.927 109.685 279.861 109.613 279.814 109.521C279.767 109.429 279.744 109.322 279.744 109.198C279.744 109.075 279.767 108.967 279.814 108.875C279.861 108.782 279.927 108.71 280.012 108.659C280.098 108.608 280.199 108.582 280.315 108.582C280.431 108.582 280.532 108.608 280.618 108.659C280.703 108.71 280.769 108.782 280.815 108.875C280.863 108.967 280.886 109.075 280.886 109.198C280.886 109.322 280.863 109.429 280.815 109.521C280.769 109.613 280.703 109.685 280.618 109.736C280.532 109.787 280.431 109.813 280.315 109.813ZM280.317 109.588C280.38 109.588 280.432 109.571 280.475 109.536C280.517 109.501 280.549 109.454 280.569 109.395C280.591 109.336 280.601 109.27 280.601 109.198C280.601 109.125 280.591 109.059 280.569 109C280.549 108.94 280.517 108.893 280.475 108.858C280.432 108.823 280.38 108.805 280.317 108.805C280.252 108.805 280.198 108.823 280.155 108.858C280.113 108.893 280.081 108.94 280.06 109C280.039 109.059 280.029 109.125 280.029 109.198C280.029 109.27 280.039 109.336 280.06 109.395C280.081 109.454 280.113 109.501 280.155 109.536C280.198 109.571 280.252 109.588 280.317 109.588ZM281.427 109.091V109.79H281.146V108.598H281.415V108.8H281.429C281.456 108.734 281.5 108.681 281.56 108.641C281.621 108.602 281.695 108.582 281.784 108.582C281.866 108.582 281.938 108.6 281.999 108.635C282.061 108.67 282.108 108.721 282.142 108.788C282.176 108.855 282.193 108.936 282.192 109.031V109.79H281.912V109.074C281.912 108.995 281.891 108.932 281.849 108.887C281.809 108.842 281.752 108.82 281.679 108.82C281.63 108.82 281.587 108.831 281.548 108.852C281.511 108.874 281.481 108.904 281.459 108.945C281.438 108.985 281.427 109.034 281.427 109.091ZM283.584 109.813C283.467 109.813 283.366 109.787 283.281 109.736C283.196 109.685 283.129 109.613 283.082 109.521C283.036 109.429 283.013 109.322 283.013 109.198C283.013 109.075 283.036 108.967 283.082 108.875C283.129 108.782 283.196 108.71 283.281 108.659C283.366 108.608 283.467 108.582 283.584 108.582C283.7 108.582 283.801 108.608 283.886 108.659C283.972 108.71 284.038 108.782 284.084 108.875C284.131 108.967 284.155 109.075 284.155 109.198C284.155 109.322 284.131 109.429 284.084 109.521C284.038 109.613 283.972 109.685 283.886 109.736C283.801 109.787 283.7 109.813 283.584 109.813ZM283.585 109.588C283.648 109.588 283.701 109.571 283.743 109.536C283.786 109.501 283.817 109.454 283.838 109.395C283.859 109.336 283.87 109.27 283.87 109.198C283.87 109.125 283.859 109.059 283.838 109C283.817 108.94 283.786 108.893 283.743 108.858C283.701 108.823 283.648 108.805 283.585 108.805C283.521 108.805 283.467 108.823 283.424 108.858C283.381 108.893 283.35 108.94 283.328 109C283.308 109.059 283.297 109.125 283.297 109.198C283.297 109.27 283.308 109.336 283.328 109.395C283.35 109.454 283.381 109.501 283.424 109.536C283.467 109.571 283.521 109.588 283.585 109.588ZM285.019 108.598V108.815H284.314V108.598H285.019ZM284.49 109.79V108.485C284.49 108.405 284.507 108.338 284.54 108.285C284.574 108.232 284.619 108.192 284.675 108.166C284.731 108.139 284.794 108.126 284.863 108.126C284.911 108.126 284.955 108.13 284.992 108.138C285.03 108.145 285.058 108.152 285.076 108.159L285.02 108.376C285.008 108.372 284.993 108.369 284.975 108.365C284.957 108.361 284.937 108.359 284.915 108.359C284.863 108.359 284.826 108.372 284.804 108.397C284.783 108.422 284.772 108.457 284.772 108.504V109.79H284.49ZM286.344 109.813C286.228 109.813 286.127 109.787 286.042 109.736C285.956 109.685 285.89 109.613 285.843 109.521C285.797 109.429 285.773 109.322 285.773 109.198C285.773 109.075 285.797 108.967 285.843 108.875C285.89 108.782 285.956 108.71 286.042 108.659C286.127 108.608 286.228 108.582 286.344 108.582C286.461 108.582 286.562 108.608 286.647 108.659C286.732 108.71 286.798 108.782 286.845 108.875C286.892 108.967 286.916 109.075 286.916 109.198C286.916 109.322 286.892 109.429 286.845 109.521C286.798 109.613 286.732 109.685 286.647 109.736C286.562 109.787 286.461 109.813 286.344 109.813ZM286.346 109.588C286.409 109.588 286.462 109.571 286.504 109.536C286.547 109.501 286.578 109.454 286.599 109.395C286.62 109.336 286.631 109.27 286.631 109.198C286.631 109.125 286.62 109.059 286.599 109C286.578 108.94 286.547 108.893 286.504 108.858C286.462 108.823 286.409 108.805 286.346 108.805C286.281 108.805 286.227 108.823 286.185 108.858C286.142 108.893 286.11 108.94 286.089 109C286.068 109.059 286.058 109.125 286.058 109.198C286.058 109.27 286.068 109.336 286.089 109.395C286.11 109.454 286.142 109.501 286.185 109.536C286.227 109.571 286.281 109.588 286.346 109.588ZM287.936 109.289V108.598H288.216V109.79H287.944V109.578H287.932C287.905 109.645 287.861 109.699 287.799 109.742C287.738 109.784 287.663 109.805 287.573 109.805C287.495 109.805 287.426 109.788 287.366 109.753C287.306 109.718 287.26 109.667 287.226 109.6C287.193 109.533 287.176 109.452 287.176 109.357V108.598H287.457V109.313C287.457 109.389 287.477 109.449 287.519 109.493C287.56 109.538 287.615 109.56 287.682 109.56C287.723 109.56 287.763 109.55 287.802 109.53C287.841 109.51 287.873 109.48 287.897 109.44C287.923 109.399 287.936 109.349 287.936 109.289ZM288.527 109.79V108.598H288.8V108.797H288.812C288.834 108.728 288.871 108.675 288.924 108.637C288.977 108.6 289.038 108.581 289.106 108.581C289.122 108.581 289.139 108.582 289.158 108.583C289.178 108.584 289.194 108.586 289.207 108.589V108.847C289.195 108.843 289.176 108.839 289.15 108.836C289.125 108.832 289.1 108.831 289.077 108.831C289.025 108.831 288.979 108.842 288.938 108.864C288.898 108.886 288.866 108.916 288.843 108.955C288.82 108.994 288.808 109.038 288.808 109.089V109.79H288.527ZM256.581 111.201H256.933L257.405 112.352H257.423L257.895 111.201H258.247V112.79H257.971V111.698H257.956L257.517 112.785H257.311L256.872 111.696H256.857V112.79H256.581V111.201ZM258.913 112.814C258.838 112.814 258.77 112.8 258.709 112.773C258.649 112.746 258.602 112.706 258.566 112.652C258.532 112.599 258.514 112.533 258.514 112.455C258.514 112.388 258.527 112.332 258.552 112.289C258.577 112.245 258.61 112.209 258.653 112.183C258.696 112.157 258.745 112.137 258.798 112.123C258.853 112.109 258.909 112.099 258.967 112.093C259.037 112.086 259.093 112.079 259.137 112.074C259.18 112.067 259.212 112.058 259.232 112.046C259.252 112.033 259.262 112.013 259.262 111.986V111.981C259.262 111.923 259.244 111.877 259.21 111.845C259.175 111.813 259.125 111.797 259.06 111.797C258.991 111.797 258.937 111.812 258.896 111.842C258.856 111.872 258.83 111.908 258.816 111.949L258.553 111.911C258.574 111.839 258.608 111.778 258.656 111.73C258.703 111.681 258.762 111.644 258.83 111.62C258.899 111.595 258.975 111.582 259.058 111.582C259.116 111.582 259.173 111.589 259.23 111.603C259.287 111.616 259.339 111.638 259.386 111.669C259.433 111.7 259.471 111.741 259.499 111.794C259.528 111.847 259.543 111.913 259.543 111.992V112.79H259.273V112.626H259.263C259.246 112.659 259.222 112.69 259.191 112.719C259.161 112.748 259.122 112.771 259.076 112.788C259.029 112.805 258.975 112.814 258.913 112.814ZM258.986 112.607C259.043 112.607 259.092 112.596 259.133 112.574C259.174 112.551 259.206 112.521 259.228 112.484C259.251 112.447 259.263 112.406 259.263 112.362V112.222C259.254 112.229 259.239 112.236 259.218 112.242C259.197 112.248 259.174 112.254 259.148 112.258C259.122 112.263 259.096 112.267 259.071 112.271C259.046 112.274 259.024 112.277 259.005 112.28C258.963 112.286 258.925 112.295 258.892 112.308C258.859 112.321 258.833 112.339 258.814 112.362C258.795 112.385 258.785 112.414 258.785 112.451C258.785 112.502 258.804 112.541 258.842 112.568C258.88 112.594 258.928 112.607 258.986 112.607ZM260.365 113.262C260.264 113.262 260.178 113.248 260.105 113.22C260.033 113.194 259.975 113.157 259.931 113.112C259.887 113.066 259.856 113.016 259.839 112.961L260.092 112.899C260.104 112.922 260.12 112.946 260.142 112.968C260.164 112.992 260.193 113.011 260.23 113.026C260.267 113.041 260.314 113.049 260.37 113.049C260.45 113.049 260.516 113.03 260.568 112.991C260.62 112.952 260.646 112.889 260.646 112.801V112.576H260.632C260.618 112.605 260.597 112.634 260.569 112.665C260.541 112.695 260.505 112.721 260.459 112.742C260.414 112.762 260.358 112.773 260.289 112.773C260.198 112.773 260.115 112.751 260.04 112.708C259.966 112.665 259.907 112.6 259.863 112.514C259.82 112.428 259.798 112.32 259.798 112.19C259.798 112.059 259.82 111.949 259.863 111.859C259.907 111.768 259.966 111.7 260.041 111.653C260.115 111.606 260.199 111.582 260.29 111.582C260.36 111.582 260.417 111.594 260.462 111.618C260.508 111.641 260.544 111.67 260.571 111.703C260.598 111.735 260.618 111.766 260.632 111.795H260.648V111.598H260.925V112.809C260.925 112.911 260.901 112.995 260.852 113.062C260.803 113.129 260.737 113.179 260.652 113.212C260.568 113.245 260.472 113.262 260.365 113.262ZM260.368 112.552C260.427 112.552 260.478 112.538 260.52 112.509C260.562 112.48 260.593 112.438 260.615 112.384C260.637 112.33 260.648 112.264 260.648 112.188C260.648 112.113 260.637 112.048 260.615 111.991C260.594 111.935 260.562 111.891 260.521 111.86C260.479 111.829 260.428 111.813 260.368 111.813C260.305 111.813 260.253 111.829 260.211 111.862C260.169 111.894 260.137 111.939 260.116 111.996C260.095 112.052 260.084 112.116 260.084 112.188C260.084 112.261 260.095 112.325 260.116 112.38C260.138 112.434 260.17 112.477 260.212 112.507C260.254 112.537 260.306 112.552 260.368 112.552ZM261.578 112.814C261.502 112.814 261.434 112.8 261.373 112.773C261.313 112.746 261.266 112.706 261.231 112.652C261.196 112.599 261.179 112.533 261.179 112.455C261.179 112.388 261.191 112.332 261.216 112.289C261.241 112.245 261.275 112.209 261.318 112.183C261.361 112.157 261.409 112.137 261.463 112.123C261.517 112.109 261.573 112.099 261.631 112.093C261.701 112.086 261.758 112.079 261.801 112.074C261.845 112.067 261.876 112.058 261.896 112.046C261.916 112.033 261.926 112.013 261.926 111.986V111.981C261.926 111.923 261.909 111.877 261.874 111.845C261.839 111.813 261.789 111.797 261.724 111.797C261.655 111.797 261.601 111.812 261.561 111.842C261.521 111.872 261.494 111.908 261.48 111.949L261.218 111.911C261.238 111.839 261.272 111.778 261.32 111.73C261.368 111.681 261.426 111.644 261.495 111.62C261.563 111.595 261.639 111.582 261.723 111.582C261.78 111.582 261.837 111.589 261.894 111.603C261.951 111.616 262.003 111.638 262.05 111.669C262.097 111.7 262.135 111.741 262.163 111.794C262.192 111.847 262.207 111.913 262.207 111.992V112.79H261.937V112.626H261.928C261.91 112.659 261.886 112.69 261.855 112.719C261.825 112.748 261.786 112.771 261.74 112.788C261.694 112.805 261.64 112.814 261.578 112.814ZM261.651 112.607C261.707 112.607 261.756 112.596 261.797 112.574C261.839 112.551 261.87 112.521 261.893 112.484C261.915 112.447 261.927 112.406 261.927 112.362V112.222C261.918 112.229 261.903 112.236 261.882 112.242C261.861 112.248 261.838 112.254 261.812 112.258C261.786 112.263 261.76 112.267 261.735 112.271C261.71 112.274 261.688 112.277 261.669 112.28C261.627 112.286 261.59 112.295 261.557 112.308C261.524 112.321 261.497 112.339 261.478 112.362C261.459 112.385 261.45 112.414 261.45 112.451C261.45 112.502 261.468 112.541 261.506 112.568C261.544 112.594 261.592 112.607 261.651 112.607ZM262.496 112.79V112.611L263.11 111.841V111.831H262.516V111.598H263.454V111.79L262.869 112.547V112.557H263.474V112.79H262.496ZM263.76 112.79V111.598H264.041V112.79H263.76ZM263.901 111.429C263.857 111.429 263.819 111.414 263.787 111.384C263.754 111.354 263.738 111.319 263.738 111.277C263.738 111.234 263.754 111.198 263.787 111.169C263.819 111.139 263.857 111.124 263.901 111.124C263.946 111.124 263.985 111.139 264.016 111.169C264.048 111.198 264.064 111.234 264.064 111.277C264.064 111.319 264.048 111.354 264.016 111.384C263.985 111.414 263.946 111.429 263.901 111.429ZM264.633 112.091V112.79H264.352V111.598H264.62V111.8H264.634C264.662 111.734 264.705 111.681 264.765 111.641C264.826 111.602 264.901 111.582 264.99 111.582C265.072 111.582 265.144 111.6 265.205 111.635C265.266 111.67 265.314 111.721 265.347 111.788C265.381 111.855 265.398 111.936 265.398 112.031V112.79H265.117V112.074C265.117 111.995 265.096 111.932 265.055 111.887C265.014 111.842 264.957 111.82 264.885 111.82C264.836 111.82 264.792 111.831 264.754 111.852C264.716 111.874 264.686 111.904 264.664 111.945C264.643 111.985 264.633 112.034 264.633 112.091ZM266.233 112.813C266.113 112.813 266.01 112.788 265.923 112.739C265.837 112.688 265.77 112.618 265.724 112.526C265.677 112.434 265.654 112.325 265.654 112.201C265.654 112.078 265.677 111.971 265.724 111.878C265.771 111.785 265.837 111.712 265.921 111.661C266.005 111.608 266.104 111.582 266.218 111.582C266.292 111.582 266.361 111.594 266.426 111.618C266.492 111.641 266.55 111.678 266.6 111.727C266.651 111.776 266.69 111.838 266.719 111.914C266.748 111.99 266.763 112.08 266.763 112.185V112.271H265.786V112.081H266.494C266.493 112.028 266.481 111.98 266.459 111.938C266.436 111.895 266.404 111.862 266.363 111.838C266.323 111.813 266.276 111.801 266.222 111.801C266.165 111.801 266.114 111.815 266.071 111.843C266.027 111.871 265.993 111.907 265.969 111.952C265.945 111.996 265.933 112.045 265.933 112.098V112.264C265.933 112.333 265.945 112.392 265.971 112.442C265.996 112.491 266.031 112.529 266.077 112.555C266.122 112.581 266.176 112.594 266.237 112.594C266.278 112.594 266.315 112.589 266.348 112.577C266.381 112.565 266.41 112.548 266.434 112.525C266.458 112.502 266.477 112.474 266.489 112.441L266.751 112.47C266.735 112.539 266.703 112.6 266.657 112.652C266.611 112.703 266.552 112.743 266.48 112.771C266.408 112.799 266.325 112.813 266.233 112.813ZM268.181 111.598V111.815H267.496V111.598H268.181ZM267.665 111.312H267.946V112.431C267.946 112.469 267.952 112.498 267.963 112.518C267.975 112.538 267.991 112.551 268.01 112.559C268.029 112.566 268.05 112.569 268.074 112.569C268.091 112.569 268.107 112.568 268.122 112.566C268.137 112.563 268.148 112.561 268.156 112.559L268.203 112.778C268.188 112.783 268.167 112.789 268.139 112.795C268.111 112.801 268.078 112.805 268.038 112.806C267.968 112.808 267.904 112.798 267.848 112.774C267.791 112.75 267.747 112.714 267.714 112.664C267.681 112.614 267.665 112.552 267.665 112.478V111.312ZM268.945 112.813C268.829 112.813 268.728 112.787 268.643 112.736C268.557 112.685 268.491 112.613 268.444 112.521C268.398 112.429 268.374 112.322 268.374 112.198C268.374 112.075 268.398 111.967 268.444 111.875C268.491 111.782 268.557 111.71 268.643 111.659C268.728 111.608 268.829 111.582 268.945 111.582C269.062 111.582 269.163 111.608 269.248 111.659C269.333 111.71 269.399 111.782 269.446 111.875C269.493 111.967 269.516 112.075 269.516 112.198C269.516 112.322 269.493 112.429 269.446 112.521C269.399 112.613 269.333 112.685 269.248 112.736C269.163 112.787 269.062 112.813 268.945 112.813ZM268.947 112.588C269.01 112.588 269.063 112.571 269.105 112.536C269.148 112.501 269.179 112.454 269.2 112.395C269.221 112.336 269.232 112.27 269.232 112.198C269.232 112.125 269.221 112.059 269.2 112C269.179 111.94 269.148 111.893 269.105 111.858C269.063 111.823 269.01 111.805 268.947 111.805C268.882 111.805 268.828 111.823 268.785 111.858C268.743 111.893 268.711 111.94 268.69 112C268.669 112.059 268.659 112.125 268.659 112.198C268.659 112.27 268.669 112.336 268.69 112.395C268.711 112.454 268.743 112.501 268.785 112.536C268.828 112.571 268.882 112.588 268.947 112.588ZM270.861 112.813C270.742 112.813 270.639 112.787 270.554 112.735C270.469 112.682 270.404 112.61 270.358 112.518C270.312 112.426 270.289 112.319 270.289 112.198C270.289 112.077 270.313 111.971 270.359 111.878C270.406 111.785 270.472 111.712 270.556 111.661C270.642 111.608 270.743 111.582 270.859 111.582C270.956 111.582 271.041 111.6 271.116 111.636C271.191 111.671 271.251 111.721 271.295 111.786C271.34 111.85 271.365 111.925 271.371 112.011H271.103C271.092 111.953 271.066 111.905 271.025 111.867C270.985 111.828 270.931 111.809 270.863 111.809C270.805 111.809 270.755 111.824 270.712 111.855C270.668 111.886 270.634 111.93 270.61 111.987C270.586 112.045 270.574 112.114 270.574 112.194C270.574 112.275 270.586 112.345 270.61 112.403C270.634 112.461 270.667 112.506 270.71 112.538C270.753 112.569 270.804 112.584 270.863 112.584C270.904 112.584 270.941 112.576 270.974 112.561C271.007 112.545 271.035 112.522 271.057 112.492C271.079 112.462 271.094 112.425 271.103 112.382H271.371C271.364 112.467 271.34 112.541 271.297 112.607C271.254 112.671 271.195 112.722 271.121 112.759C271.047 112.795 270.96 112.813 270.861 112.813ZM271.967 112.814C271.891 112.814 271.823 112.8 271.763 112.773C271.703 112.746 271.655 112.706 271.62 112.652C271.585 112.599 271.568 112.533 271.568 112.455C271.568 112.388 271.58 112.332 271.605 112.289C271.63 112.245 271.664 112.209 271.707 112.183C271.75 112.157 271.798 112.137 271.852 112.123C271.906 112.109 271.962 112.099 272.02 112.093C272.09 112.086 272.147 112.079 272.19 112.074C272.234 112.067 272.265 112.058 272.285 112.046C272.305 112.033 272.315 112.013 272.315 111.986V111.981C272.315 111.923 272.298 111.877 272.263 111.845C272.228 111.813 272.179 111.797 272.113 111.797C272.045 111.797 271.99 111.812 271.95 111.842C271.91 111.872 271.883 111.908 271.869 111.949L271.607 111.911C271.627 111.839 271.661 111.778 271.709 111.73C271.757 111.681 271.815 111.644 271.884 111.62C271.952 111.595 272.028 111.582 272.112 111.582C272.169 111.582 272.226 111.589 272.283 111.603C272.34 111.616 272.392 111.638 272.439 111.669C272.486 111.7 272.524 111.741 272.553 111.794C272.582 111.847 272.596 111.913 272.596 111.992V112.79H272.326V112.626H272.317C272.3 112.659 272.276 112.69 272.244 112.719C272.214 112.748 272.175 112.771 272.129 112.788C272.083 112.805 272.029 112.814 271.967 112.814ZM272.04 112.607C272.096 112.607 272.145 112.596 272.186 112.574C272.228 112.551 272.259 112.521 272.282 112.484C272.304 112.447 272.316 112.406 272.316 112.362V112.222C272.307 112.229 272.292 112.236 272.271 112.242C272.25 112.248 272.227 112.254 272.201 112.258C272.175 112.263 272.15 112.267 272.124 112.271C272.099 112.274 272.077 112.277 272.058 112.28C272.016 112.286 271.979 112.295 271.946 112.308C271.913 112.321 271.886 112.339 271.867 112.362C271.848 112.385 271.839 112.414 271.839 112.451C271.839 112.502 271.858 112.541 271.895 112.568C271.933 112.594 271.981 112.607 272.04 112.607ZM273.496 111.598V111.815H272.81V111.598H273.496ZM272.979 111.312H273.26V112.431C273.26 112.469 273.266 112.498 273.277 112.518C273.289 112.538 273.305 112.551 273.324 112.559C273.343 112.566 273.364 112.569 273.388 112.569C273.405 112.569 273.421 112.568 273.436 112.566C273.451 112.563 273.462 112.561 273.47 112.559L273.517 112.778C273.502 112.783 273.481 112.789 273.453 112.795C273.425 112.801 273.392 112.805 273.352 112.806C273.282 112.808 273.218 112.798 273.162 112.774C273.105 112.75 273.061 112.714 273.028 112.664C272.995 112.614 272.979 112.552 272.979 112.478V111.312ZM274.259 112.813C274.14 112.813 274.038 112.787 273.953 112.735C273.868 112.682 273.803 112.61 273.757 112.518C273.711 112.426 273.688 112.319 273.688 112.198C273.688 112.077 273.712 111.971 273.758 111.878C273.805 111.785 273.87 111.712 273.955 111.661C274.041 111.608 274.141 111.582 274.258 111.582C274.355 111.582 274.44 111.6 274.515 111.636C274.59 111.671 274.649 111.721 274.694 111.786C274.738 111.85 274.764 111.925 274.77 112.011H274.502C274.491 111.953 274.465 111.905 274.424 111.867C274.384 111.828 274.329 111.809 274.262 111.809C274.204 111.809 274.154 111.824 274.11 111.855C274.067 111.886 274.033 111.93 274.009 111.987C273.985 112.045 273.973 112.114 273.973 112.194C273.973 112.275 273.985 112.345 274.009 112.403C274.033 112.461 274.066 112.506 274.109 112.538C274.152 112.569 274.203 112.584 274.262 112.584C274.303 112.584 274.34 112.576 274.373 112.561C274.406 112.545 274.433 112.522 274.456 112.492C274.478 112.462 274.493 112.425 274.502 112.382H274.77C274.763 112.467 274.738 112.541 274.696 112.607C274.653 112.671 274.594 112.722 274.52 112.759C274.446 112.795 274.359 112.813 274.259 112.813ZM275.303 112.091V112.79H275.023V111.201H275.297V111.8H275.311C275.339 111.733 275.382 111.68 275.441 111.641C275.5 111.602 275.575 111.582 275.666 111.582C275.749 111.582 275.821 111.6 275.882 111.634C275.944 111.669 275.991 111.72 276.025 111.786C276.059 111.853 276.076 111.935 276.076 112.031V112.79H275.795V112.074C275.795 111.994 275.775 111.932 275.733 111.887C275.692 111.842 275.635 111.82 275.561 111.82C275.511 111.82 275.467 111.831 275.428 111.852C275.389 111.874 275.358 111.904 275.336 111.945C275.314 111.985 275.303 112.034 275.303 112.091ZM277.291 112.814C277.216 112.814 277.148 112.8 277.087 112.773C277.027 112.746 276.98 112.706 276.945 112.652C276.91 112.599 276.893 112.533 276.893 112.455C276.893 112.388 276.905 112.332 276.93 112.289C276.955 112.245 276.988 112.209 277.031 112.183C277.074 112.157 277.123 112.137 277.177 112.123C277.231 112.109 277.287 112.099 277.345 112.093C277.415 112.086 277.471 112.079 277.515 112.074C277.558 112.067 277.59 112.058 277.61 112.046C277.63 112.033 277.64 112.013 277.64 111.986V111.981C277.64 111.923 277.622 111.877 277.588 111.845C277.553 111.813 277.503 111.797 277.438 111.797C277.369 111.797 277.315 111.812 277.274 111.842C277.234 111.872 277.208 111.908 277.194 111.949L276.931 111.911C276.952 111.839 276.986 111.778 277.034 111.73C277.081 111.681 277.14 111.644 277.208 111.62C277.277 111.595 277.353 111.582 277.437 111.582C277.494 111.582 277.551 111.589 277.608 111.603C277.665 111.616 277.717 111.638 277.764 111.669C277.811 111.7 277.849 111.741 277.877 111.794C277.906 111.847 277.921 111.913 277.921 111.992V112.79H277.651V112.626H277.641C277.624 112.659 277.6 112.69 277.569 112.719C277.539 112.748 277.5 112.771 277.454 112.788C277.408 112.805 277.353 112.814 277.291 112.814ZM277.364 112.607C277.421 112.607 277.47 112.596 277.511 112.574C277.552 112.551 277.584 112.521 277.606 112.484C277.629 112.447 277.641 112.406 277.641 112.362V112.222C277.632 112.229 277.617 112.236 277.596 112.242C277.575 112.248 277.552 112.254 277.526 112.258C277.5 112.263 277.474 112.267 277.449 112.271C277.424 112.274 277.402 112.277 277.383 112.28C277.341 112.286 277.304 112.295 277.27 112.308C277.237 112.321 277.211 112.339 277.192 112.362C277.173 112.385 277.163 112.414 277.163 112.451C277.163 112.502 277.182 112.541 277.22 112.568C277.258 112.594 277.306 112.607 277.364 112.607ZM279.307 113.262C279.206 113.262 279.12 113.248 279.047 113.22C278.975 113.194 278.917 113.157 278.873 113.112C278.829 113.066 278.798 113.016 278.781 112.961L279.034 112.899C279.046 112.922 279.062 112.946 279.084 112.968C279.106 112.992 279.135 113.011 279.171 113.026C279.209 113.041 279.256 113.049 279.312 113.049C279.392 113.049 279.458 113.03 279.51 112.991C279.562 112.952 279.588 112.889 279.588 112.801V112.576H279.574C279.56 112.605 279.539 112.634 279.511 112.665C279.483 112.695 279.447 112.721 279.401 112.742C279.356 112.762 279.3 112.773 279.231 112.773C279.14 112.773 279.057 112.751 278.982 112.708C278.908 112.665 278.849 112.6 278.805 112.514C278.762 112.428 278.74 112.32 278.74 112.19C278.74 112.059 278.762 111.949 278.805 111.859C278.849 111.768 278.908 111.7 278.983 111.653C279.057 111.606 279.14 111.582 279.232 111.582C279.302 111.582 279.359 111.594 279.404 111.618C279.45 111.641 279.486 111.67 279.513 111.703C279.54 111.735 279.56 111.766 279.574 111.795H279.59V111.598H279.867V112.809C279.867 112.911 279.842 112.995 279.794 113.062C279.745 113.129 279.679 113.179 279.594 113.212C279.51 113.245 279.414 113.262 279.307 113.262ZM279.31 112.552C279.369 112.552 279.42 112.538 279.462 112.509C279.504 112.48 279.535 112.438 279.557 112.384C279.579 112.33 279.59 112.264 279.59 112.188C279.59 112.113 279.579 112.048 279.557 111.991C279.536 111.935 279.504 111.891 279.462 111.86C279.421 111.829 279.37 111.813 279.31 111.813C279.247 111.813 279.195 111.829 279.153 111.862C279.111 111.894 279.079 111.939 279.058 111.996C279.037 112.052 279.026 112.116 279.026 112.188C279.026 112.261 279.037 112.325 279.058 112.38C279.08 112.434 279.112 112.477 279.154 112.507C279.196 112.537 279.248 112.552 279.31 112.552ZM280.457 111.201V112.79H280.177V111.201H280.457ZM280.768 112.79V111.598H281.049V112.79H280.768ZM280.909 111.429C280.865 111.429 280.827 111.414 280.795 111.384C280.762 111.354 280.746 111.319 280.746 111.277C280.746 111.234 280.762 111.198 280.795 111.169C280.827 111.139 280.865 111.124 280.909 111.124C280.954 111.124 280.993 111.139 281.024 111.169C281.056 111.198 281.072 111.234 281.072 111.277C281.072 111.319 281.056 111.354 281.024 111.384C280.993 111.414 280.954 111.429 280.909 111.429ZM281.36 112.79V111.598H281.628V111.8H281.642C281.667 111.732 281.708 111.679 281.766 111.641C281.823 111.602 281.892 111.582 281.971 111.582C282.052 111.582 282.12 111.602 282.175 111.641C282.231 111.68 282.271 111.733 282.293 111.8H282.306C282.332 111.734 282.377 111.681 282.439 111.642C282.502 111.602 282.577 111.582 282.663 111.582C282.773 111.582 282.863 111.617 282.932 111.686C283.001 111.756 283.036 111.857 283.036 111.99V112.79H282.754V112.033C282.754 111.959 282.735 111.905 282.695 111.871C282.656 111.836 282.608 111.819 282.551 111.819C282.483 111.819 282.43 111.84 282.392 111.883C282.354 111.925 282.335 111.979 282.335 112.046V112.79H282.06V112.022C282.06 111.96 282.041 111.911 282.004 111.874C281.967 111.837 281.919 111.819 281.86 111.819C281.819 111.819 281.782 111.829 281.749 111.85C281.716 111.87 281.69 111.899 281.67 111.936C281.651 111.973 281.641 112.016 281.641 112.065V112.79H281.36ZM284.285 111.913L284.029 111.941C284.021 111.915 284.009 111.891 283.991 111.868C283.973 111.845 283.949 111.827 283.919 111.813C283.889 111.799 283.853 111.792 283.809 111.792C283.751 111.792 283.702 111.805 283.662 111.83C283.622 111.855 283.603 111.888 283.603 111.928C283.603 111.963 283.616 111.991 283.642 112.013C283.668 112.035 283.711 112.053 283.772 112.067L283.975 112.11C284.088 112.134 284.172 112.173 284.227 112.226C284.282 112.278 284.31 112.347 284.31 112.433C284.31 112.508 284.288 112.574 284.244 112.631C284.202 112.688 284.142 112.733 284.065 112.765C283.989 112.797 283.901 112.813 283.801 112.813C283.655 112.813 283.538 112.783 283.449 112.722C283.36 112.66 283.307 112.574 283.29 112.465L283.564 112.438C283.576 112.492 283.603 112.533 283.643 112.56C283.683 112.588 283.736 112.601 283.801 112.601C283.867 112.601 283.921 112.588 283.961 112.56C284.002 112.533 284.023 112.499 284.023 112.458C284.023 112.424 284.009 112.396 283.983 112.374C283.957 112.352 283.917 112.335 283.862 112.323L283.659 112.28C283.544 112.256 283.46 112.216 283.405 112.16C283.35 112.103 283.323 112.031 283.323 111.944C283.323 111.871 283.343 111.807 283.383 111.753C283.424 111.699 283.481 111.657 283.553 111.627C283.626 111.597 283.71 111.582 283.805 111.582C283.945 111.582 284.055 111.612 284.135 111.672C284.216 111.731 284.266 111.812 284.285 111.913ZM285.092 112.813C284.972 112.813 284.869 112.788 284.782 112.739C284.696 112.688 284.629 112.618 284.583 112.526C284.536 112.434 284.513 112.325 284.513 112.201C284.513 112.078 284.536 111.971 284.583 111.878C284.63 111.785 284.695 111.712 284.78 111.661C284.864 111.608 284.963 111.582 285.077 111.582C285.15 111.582 285.22 111.594 285.285 111.618C285.351 111.641 285.408 111.678 285.459 111.727C285.509 111.776 285.549 111.838 285.578 111.914C285.607 111.99 285.622 112.08 285.622 112.185V112.271H284.645V112.081H285.352C285.352 112.028 285.34 111.98 285.317 111.938C285.295 111.895 285.263 111.862 285.222 111.838C285.182 111.813 285.135 111.801 285.081 111.801C285.023 111.801 284.973 111.815 284.929 111.843C284.886 111.871 284.852 111.907 284.828 111.952C284.804 111.996 284.792 112.045 284.791 112.098V112.264C284.791 112.333 284.804 112.392 284.829 112.442C284.855 112.491 284.89 112.529 284.936 112.555C284.981 112.581 285.034 112.594 285.095 112.594C285.136 112.594 285.173 112.589 285.206 112.577C285.24 112.565 285.268 112.548 285.293 112.525C285.317 112.502 285.335 112.474 285.348 112.441L285.61 112.47C285.593 112.539 285.562 112.6 285.515 112.652C285.469 112.703 285.41 112.743 285.338 112.771C285.266 112.799 285.184 112.813 285.092 112.813ZM286.965 112.813C286.848 112.813 286.748 112.787 286.662 112.736C286.577 112.685 286.511 112.613 286.464 112.521C286.417 112.429 286.394 112.322 286.394 112.198C286.394 112.075 286.417 111.967 286.464 111.875C286.511 111.782 286.577 111.71 286.662 111.659C286.748 111.608 286.848 111.582 286.965 111.582C287.081 111.582 287.182 111.608 287.268 111.659C287.353 111.71 287.419 111.782 287.465 111.875C287.512 111.967 287.536 112.075 287.536 112.198C287.536 112.322 287.512 112.429 287.465 112.521C287.419 112.613 287.353 112.685 287.268 112.736C287.182 112.787 287.081 112.813 286.965 112.813ZM286.966 112.588C287.03 112.588 287.082 112.571 287.125 112.536C287.167 112.501 287.199 112.454 287.219 112.395C287.241 112.336 287.251 112.27 287.251 112.198C287.251 112.125 287.241 112.059 287.219 112C287.199 111.94 287.167 111.893 287.125 111.858C287.082 111.823 287.03 111.805 286.966 111.805C286.902 111.805 286.848 111.823 286.805 111.858C286.763 111.893 286.731 111.94 286.71 112C286.689 112.059 286.679 112.125 286.679 112.198C286.679 112.27 286.689 112.336 286.71 112.395C286.731 112.454 286.763 112.501 286.805 112.536C286.848 112.571 286.902 112.588 286.966 112.588ZM288.4 111.598V111.815H287.695V111.598H288.4ZM287.872 112.79V111.485C287.872 111.405 287.888 111.338 287.921 111.285C287.955 111.232 288 111.192 288.056 111.166C288.113 111.139 288.175 111.126 288.244 111.126C288.293 111.126 288.336 111.13 288.374 111.138C288.411 111.145 288.439 111.152 288.457 111.159L288.402 111.376C288.39 111.372 288.375 111.369 288.357 111.365C288.338 111.361 288.318 111.359 288.296 111.359C288.244 111.359 288.207 111.372 288.185 111.397C288.164 111.422 288.153 111.457 288.153 111.504V112.79H287.872ZM289.801 111.598V111.815H289.116V111.598H289.801ZM289.285 111.312H289.566V112.431C289.566 112.469 289.572 112.498 289.583 112.518C289.595 112.538 289.61 112.551 289.629 112.559C289.649 112.566 289.67 112.569 289.693 112.569C289.711 112.569 289.727 112.568 289.741 112.566C289.756 112.563 289.768 112.561 289.775 112.559L289.823 112.778C289.808 112.783 289.786 112.789 289.758 112.795C289.731 112.801 289.697 112.805 289.657 112.806C289.587 112.808 289.524 112.798 289.467 112.774C289.411 112.75 289.366 112.714 289.333 112.664C289.3 112.614 289.284 112.552 289.285 112.478V111.312ZM290.364 112.091V112.79H290.083V111.201H290.358V111.8H290.372C290.4 111.733 290.443 111.68 290.502 111.641C290.561 111.602 290.636 111.582 290.727 111.582C290.809 111.582 290.882 111.6 290.943 111.634C291.005 111.669 291.052 111.72 291.086 111.786C291.12 111.853 291.137 111.935 291.137 112.031V112.79H290.856V112.074C290.856 111.994 290.836 111.932 290.794 111.887C290.753 111.842 290.696 111.82 290.622 111.82C290.572 111.82 290.528 111.831 290.488 111.852C290.45 111.874 290.419 111.904 290.397 111.945C290.375 111.985 290.364 112.034 290.364 112.091ZM291.973 112.813C291.853 112.813 291.75 112.788 291.663 112.739C291.577 112.688 291.511 112.618 291.464 112.526C291.417 112.434 291.394 112.325 291.394 112.201C291.394 112.078 291.417 111.971 291.464 111.878C291.511 111.785 291.577 111.712 291.661 111.661C291.745 111.608 291.844 111.582 291.958 111.582C292.032 111.582 292.101 111.594 292.166 111.618C292.232 111.641 292.29 111.678 292.34 111.727C292.391 111.776 292.431 111.838 292.46 111.914C292.489 111.99 292.503 112.08 292.503 112.185V112.271H291.526V112.081H292.234C292.233 112.028 292.222 111.98 292.199 111.938C292.176 111.895 292.144 111.862 292.103 111.838C292.063 111.813 292.016 111.801 291.962 111.801C291.905 111.801 291.854 111.815 291.811 111.843C291.767 111.871 291.733 111.907 291.709 111.952C291.685 111.996 291.673 112.045 291.673 112.098V112.264C291.673 112.333 291.685 112.392 291.711 112.442C291.736 112.491 291.772 112.529 291.817 112.555C291.863 112.581 291.916 112.594 291.977 112.594C292.018 112.594 292.055 112.589 292.088 112.577C292.121 112.565 292.15 112.548 292.174 112.525C292.198 112.502 292.217 112.474 292.229 112.441L292.491 112.47C292.475 112.539 292.443 112.6 292.397 112.652C292.351 112.703 292.292 112.743 292.22 112.771C292.148 112.799 292.066 112.813 291.973 112.813ZM256.568 115.79V114.598H256.837V114.8H256.851C256.875 114.732 256.917 114.679 256.974 114.641C257.031 114.602 257.1 114.582 257.18 114.582C257.26 114.582 257.328 114.602 257.384 114.641C257.44 114.68 257.479 114.733 257.502 114.8H257.514C257.54 114.734 257.585 114.681 257.648 114.642C257.711 114.602 257.785 114.582 257.872 114.582C257.982 114.582 258.071 114.617 258.14 114.686C258.21 114.756 258.244 114.857 258.244 114.99V115.79H257.963V115.033C257.963 114.959 257.943 114.905 257.904 114.871C257.864 114.836 257.816 114.819 257.759 114.819C257.692 114.819 257.639 114.84 257.6 114.883C257.562 114.925 257.544 114.979 257.544 115.046V115.79H257.268V115.022C257.268 114.96 257.249 114.911 257.212 114.874C257.176 114.837 257.127 114.819 257.068 114.819C257.028 114.819 256.991 114.829 256.958 114.85C256.925 114.87 256.898 114.899 256.879 114.936C256.859 114.973 256.849 115.016 256.849 115.065V115.79H256.568ZM258.894 115.814C258.819 115.814 258.751 115.8 258.69 115.773C258.63 115.746 258.582 115.706 258.547 115.652C258.513 115.599 258.495 115.533 258.495 115.455C258.495 115.388 258.508 115.332 258.533 115.289C258.557 115.245 258.591 115.209 258.634 115.183C258.677 115.157 258.725 115.137 258.779 115.123C258.834 115.109 258.89 115.099 258.948 115.093C259.018 115.086 259.074 115.079 259.118 115.074C259.161 115.067 259.193 115.058 259.212 115.046C259.232 115.033 259.243 115.013 259.243 114.986V114.981C259.243 114.923 259.225 114.877 259.191 114.845C259.156 114.813 259.106 114.797 259.041 114.797C258.972 114.797 258.917 114.812 258.877 114.842C258.837 114.872 258.81 114.908 258.796 114.949L258.534 114.911C258.555 114.839 258.589 114.778 258.637 114.73C258.684 114.681 258.742 114.644 258.811 114.62C258.88 114.595 258.956 114.582 259.039 114.582C259.097 114.582 259.154 114.589 259.211 114.603C259.268 114.616 259.32 114.638 259.367 114.669C259.414 114.7 259.452 114.741 259.48 114.794C259.509 114.847 259.523 114.913 259.523 114.992V115.79H259.253V115.626H259.244C259.227 115.659 259.203 115.69 259.172 115.719C259.141 115.748 259.103 115.771 259.056 115.788C259.01 115.805 258.956 115.814 258.894 115.814ZM258.967 115.607C259.023 115.607 259.072 115.596 259.114 115.574C259.155 115.551 259.187 115.521 259.209 115.484C259.232 115.447 259.243 115.406 259.243 115.362V115.222C259.235 115.229 259.22 115.236 259.198 115.242C259.178 115.248 259.154 115.254 259.128 115.258C259.103 115.263 259.077 115.267 259.052 115.271C259.026 115.274 259.004 115.277 258.986 115.28C258.944 115.286 258.906 115.295 258.873 115.308C258.84 115.321 258.814 115.339 258.795 115.362C258.776 115.385 258.766 115.414 258.766 115.451C258.766 115.502 258.785 115.541 258.823 115.568C258.861 115.594 258.909 115.607 258.967 115.607ZM259.828 115.79V114.598H260.1V114.797H260.113C260.134 114.728 260.172 114.675 260.224 114.637C260.278 114.6 260.338 114.581 260.407 114.581C260.422 114.581 260.44 114.582 260.459 114.583C260.478 114.584 260.495 114.586 260.508 114.589V114.847C260.496 114.843 260.477 114.839 260.451 114.836C260.426 114.832 260.401 114.831 260.377 114.831C260.326 114.831 260.28 114.842 260.239 114.864C260.199 114.886 260.167 114.916 260.144 114.955C260.12 114.994 260.109 115.038 260.109 115.089V115.79H259.828ZM261.817 114.598L261.393 115.79H261.082L260.658 114.598H260.957L261.231 115.483H261.244L261.518 114.598H261.817ZM262.52 115.813C262.401 115.813 262.298 115.788 262.211 115.739C262.124 115.688 262.058 115.618 262.011 115.526C261.965 115.434 261.941 115.325 261.941 115.201C261.941 115.078 261.965 114.971 262.011 114.878C262.058 114.785 262.124 114.712 262.208 114.661C262.293 114.608 262.392 114.582 262.506 114.582C262.579 114.582 262.648 114.594 262.714 114.618C262.779 114.641 262.837 114.678 262.887 114.727C262.938 114.776 262.978 114.838 263.007 114.914C263.036 114.99 263.05 115.08 263.05 115.185V115.271H262.073V115.081H262.781C262.781 115.028 262.769 114.98 262.746 114.938C262.723 114.895 262.692 114.862 262.651 114.838C262.61 114.813 262.563 114.801 262.509 114.801C262.452 114.801 262.402 114.815 262.358 114.843C262.315 114.871 262.281 114.907 262.257 114.952C262.233 114.996 262.221 115.045 262.22 115.098V115.264C262.22 115.333 262.233 115.392 262.258 115.442C262.283 115.491 262.319 115.529 262.364 115.555C262.41 115.581 262.463 115.594 262.524 115.594C262.565 115.594 262.602 115.589 262.635 115.577C262.668 115.565 262.697 115.548 262.721 115.525C262.746 115.502 262.764 115.474 262.776 115.441L263.039 115.47C263.022 115.539 262.991 115.6 262.944 115.652C262.898 115.703 262.839 115.743 262.767 115.771C262.695 115.799 262.613 115.813 262.52 115.813ZM263.591 114.201V115.79H263.31V114.201H263.591ZM264.421 115.813C264.305 115.813 264.204 115.787 264.119 115.736C264.033 115.685 263.967 115.613 263.92 115.521C263.874 115.429 263.85 115.322 263.85 115.198C263.85 115.075 263.874 114.967 263.92 114.875C263.967 114.782 264.033 114.71 264.119 114.659C264.204 114.608 264.305 114.582 264.421 114.582C264.538 114.582 264.639 114.608 264.724 114.659C264.809 114.71 264.875 114.782 264.922 114.875C264.969 114.967 264.993 115.075 264.993 115.198C264.993 115.322 264.969 115.429 264.922 115.521C264.875 115.613 264.809 115.685 264.724 115.736C264.639 115.787 264.538 115.813 264.421 115.813ZM264.423 115.588C264.486 115.588 264.539 115.571 264.581 115.536C264.624 115.501 264.655 115.454 264.676 115.395C264.697 115.336 264.708 115.27 264.708 115.198C264.708 115.125 264.697 115.059 264.676 115C264.655 114.94 264.624 114.893 264.581 114.858C264.539 114.823 264.486 114.805 264.423 114.805C264.358 114.805 264.304 114.823 264.262 114.858C264.219 114.893 264.187 114.94 264.166 115C264.145 115.059 264.135 115.125 264.135 115.198C264.135 115.27 264.145 115.336 264.166 115.395C264.187 115.454 264.219 115.501 264.262 115.536C264.304 115.571 264.358 115.588 264.423 115.588ZM266.012 115.289V114.598H266.293V115.79H266.021V115.578H266.009C265.982 115.645 265.937 115.699 265.876 115.742C265.815 115.784 265.74 115.805 265.65 115.805C265.572 115.805 265.503 115.788 265.443 115.753C265.383 115.718 265.337 115.667 265.303 115.6C265.27 115.533 265.253 115.452 265.253 115.357V114.598H265.534V115.313C265.534 115.389 265.554 115.449 265.596 115.493C265.637 115.538 265.691 115.56 265.759 115.56C265.8 115.56 265.84 115.55 265.879 115.53C265.918 115.51 265.95 115.48 265.974 115.44C266 115.399 266.012 115.349 266.012 115.289ZM267.546 114.913L267.29 114.941C267.283 114.915 267.27 114.891 267.252 114.868C267.234 114.845 267.211 114.827 267.181 114.813C267.151 114.799 267.114 114.792 267.07 114.792C267.012 114.792 266.963 114.805 266.923 114.83C266.884 114.855 266.864 114.888 266.865 114.928C266.864 114.963 266.877 114.991 266.903 115.013C266.929 115.035 266.973 115.053 267.033 115.067L267.237 115.11C267.349 115.134 267.433 115.173 267.488 115.226C267.543 115.278 267.571 115.347 267.572 115.433C267.571 115.508 267.549 115.574 267.506 115.631C267.463 115.688 267.403 115.733 267.327 115.765C267.25 115.797 267.162 115.813 267.063 115.813C266.917 115.813 266.799 115.783 266.71 115.722C266.621 115.66 266.568 115.574 266.551 115.465L266.825 115.438C266.838 115.492 266.864 115.533 266.904 115.56C266.945 115.588 266.997 115.601 267.062 115.601C267.129 115.601 267.182 115.588 267.223 115.56C267.263 115.533 267.284 115.499 267.284 115.458C267.284 115.424 267.271 115.396 267.244 115.374C267.218 115.352 267.178 115.335 267.123 115.323L266.92 115.28C266.806 115.256 266.721 115.216 266.666 115.16C266.611 115.103 266.584 115.031 266.585 114.944C266.584 114.871 266.604 114.807 266.644 114.753C266.685 114.699 266.742 114.657 266.814 114.627C266.887 114.597 266.971 114.582 267.067 114.582C267.206 114.582 267.316 114.612 267.396 114.672C267.477 114.731 267.527 114.812 267.546 114.913ZM268.984 114.598V114.815H268.299V114.598H268.984ZM268.468 114.312H268.749V115.431C268.749 115.469 268.755 115.498 268.766 115.518C268.778 115.538 268.794 115.551 268.813 115.559C268.832 115.566 268.853 115.569 268.876 115.569C268.894 115.569 268.91 115.568 268.925 115.566C268.94 115.563 268.951 115.561 268.959 115.559L269.006 115.778C268.991 115.783 268.97 115.789 268.942 115.795C268.914 115.801 268.881 115.805 268.841 115.806C268.77 115.808 268.707 115.798 268.651 115.774C268.594 115.75 268.55 115.714 268.516 115.664C268.484 115.614 268.468 115.552 268.468 115.478V114.312ZM269.487 115.79L269.15 114.598H269.437L269.646 115.436H269.657L269.871 114.598H270.154L270.369 115.431H270.38L270.587 114.598H270.874L270.536 115.79H270.244L270.02 114.984H270.004L269.78 115.79H269.487ZM271.092 115.79V114.598H271.373V115.79H271.092ZM271.233 114.429C271.189 114.429 271.15 114.414 271.118 114.384C271.086 114.354 271.07 114.319 271.07 114.277C271.07 114.234 271.086 114.198 271.118 114.169C271.15 114.139 271.189 114.124 271.233 114.124C271.278 114.124 271.316 114.139 271.348 114.169C271.38 114.198 271.396 114.234 271.396 114.277C271.396 114.319 271.38 114.354 271.348 114.384C271.316 114.414 271.278 114.429 271.233 114.429ZM271.964 115.091V115.79H271.683V114.598H271.952V114.8H271.966C271.993 114.734 272.037 114.681 272.097 114.641C272.158 114.602 272.232 114.582 272.321 114.582C272.404 114.582 272.475 114.6 272.536 114.635C272.598 114.67 272.645 114.721 272.679 114.788C272.713 114.855 272.73 114.936 272.729 115.031V115.79H272.449V115.074C272.449 114.995 272.428 114.932 272.386 114.887C272.346 114.842 272.289 114.82 272.217 114.82C272.167 114.82 272.124 114.831 272.085 114.852C272.048 114.874 272.018 114.904 271.996 114.945C271.975 114.985 271.964 115.034 271.964 115.091ZM273.613 115.79V114.201H273.894V114.795H273.906C273.92 114.766 273.941 114.735 273.967 114.703C273.993 114.67 274.029 114.641 274.074 114.618C274.119 114.594 274.177 114.582 274.246 114.582C274.338 114.582 274.422 114.606 274.495 114.653C274.57 114.7 274.629 114.769 274.672 114.86C274.716 114.951 274.738 115.063 274.738 115.195C274.738 115.326 274.717 115.437 274.674 115.529C274.631 115.621 274.573 115.69 274.499 115.739C274.425 115.787 274.341 115.811 274.247 115.811C274.179 115.811 274.122 115.799 274.077 115.777C274.032 115.754 273.996 115.726 273.969 115.694C273.942 115.662 273.921 115.631 273.906 115.602H273.889V115.79H273.613ZM273.889 115.194C273.889 115.271 273.9 115.338 273.921 115.396C273.944 115.454 273.975 115.5 274.017 115.532C274.059 115.564 274.109 115.58 274.169 115.58C274.231 115.58 274.283 115.564 274.325 115.531C274.367 115.497 274.398 115.451 274.419 115.393C274.441 115.335 274.452 115.268 274.452 115.194C274.452 115.12 274.441 115.054 274.42 114.997C274.399 114.939 274.367 114.894 274.326 114.862C274.284 114.829 274.231 114.813 274.169 114.813C274.109 114.813 274.058 114.829 274.016 114.86C273.974 114.892 273.942 114.936 273.92 114.993C273.899 115.05 273.889 115.117 273.889 115.194ZM275.759 115.289V114.598H276.04V115.79H275.767V115.578H275.755C275.728 115.645 275.684 115.699 275.622 115.742C275.561 115.784 275.486 115.805 275.396 115.805C275.318 115.805 275.249 115.788 275.189 115.753C275.13 115.718 275.083 115.667 275.049 115.6C275.016 115.533 274.999 115.452 274.999 115.357V114.598H275.28V115.313C275.28 115.389 275.301 115.449 275.342 115.493C275.383 115.538 275.438 115.56 275.505 115.56C275.546 115.56 275.586 115.55 275.625 115.53C275.664 115.51 275.696 115.48 275.721 115.44C275.746 115.399 275.759 115.349 275.759 115.289ZM276.35 115.79V114.598H276.631V115.79H276.35ZM276.492 114.429C276.447 114.429 276.409 114.414 276.377 114.384C276.345 114.354 276.329 114.319 276.329 114.277C276.329 114.234 276.345 114.198 276.377 114.169C276.409 114.139 276.447 114.124 276.492 114.124C276.537 114.124 276.575 114.139 276.606 114.169C276.639 114.198 276.655 114.234 276.655 114.277C276.655 114.319 276.639 114.354 276.606 114.384C276.575 114.414 276.537 114.429 276.492 114.429ZM277.223 114.201V115.79H276.942V114.201H277.223ZM277.976 115.811C277.882 115.811 277.798 115.787 277.725 115.739C277.651 115.69 277.592 115.621 277.549 115.529C277.506 115.437 277.485 115.326 277.485 115.195C277.485 115.063 277.506 114.951 277.55 114.86C277.594 114.769 277.653 114.7 277.728 114.653C277.802 114.606 277.885 114.582 277.977 114.582C278.047 114.582 278.104 114.594 278.149 114.618C278.194 114.641 278.23 114.67 278.256 114.703C278.282 114.735 278.303 114.766 278.317 114.795H278.329V114.201H278.611V115.79H278.334V115.602H278.317C278.303 115.631 278.282 115.662 278.255 115.694C278.227 115.726 278.191 115.754 278.146 115.777C278.101 115.799 278.044 115.811 277.976 115.811ZM278.054 115.58C278.114 115.58 278.165 115.564 278.206 115.532C278.248 115.5 278.28 115.454 278.302 115.396C278.324 115.338 278.334 115.271 278.334 115.194C278.334 115.117 278.324 115.05 278.302 114.993C278.281 114.936 278.249 114.892 278.207 114.86C278.166 114.829 278.115 114.813 278.054 114.813C277.992 114.813 277.939 114.829 277.898 114.862C277.856 114.894 277.824 114.939 277.803 114.997C277.782 115.054 277.771 115.12 277.771 115.194C277.771 115.268 277.782 115.335 277.803 115.393C277.825 115.451 277.856 115.497 277.898 115.531C277.941 115.564 277.993 115.58 278.054 115.58ZM278.932 115.79V114.598H279.213V115.79H278.932ZM279.073 114.429C279.029 114.429 278.99 114.414 278.958 114.384C278.926 114.354 278.91 114.319 278.91 114.277C278.91 114.234 278.926 114.198 278.958 114.169C278.99 114.139 279.029 114.124 279.073 114.124C279.118 114.124 279.156 114.139 279.188 114.169C279.22 114.198 279.236 114.234 279.236 114.277C279.236 114.319 279.22 114.354 279.188 114.384C279.156 114.414 279.118 114.429 279.073 114.429ZM279.804 115.091V115.79H279.524V114.598H279.792V114.8H279.806C279.833 114.734 279.877 114.681 279.937 114.641C279.998 114.602 280.072 114.582 280.161 114.582C280.244 114.582 280.315 114.6 280.376 114.635C280.438 114.67 280.485 114.721 280.519 114.788C280.553 114.855 280.57 114.936 280.57 115.031V115.79H280.289V115.074C280.289 114.995 280.268 114.932 280.227 114.887C280.186 114.842 280.129 114.82 280.057 114.82C280.007 114.82 279.964 114.831 279.925 114.852C279.888 114.874 279.858 114.904 279.836 114.945C279.815 114.985 279.804 115.034 279.804 115.091ZM281.395 116.262C281.294 116.262 281.208 116.248 281.135 116.22C281.063 116.194 281.005 116.157 280.961 116.112C280.917 116.066 280.886 116.016 280.869 115.961L281.122 115.899C281.134 115.922 281.15 115.946 281.172 115.968C281.194 115.992 281.223 116.011 281.26 116.026C281.297 116.041 281.344 116.049 281.4 116.049C281.48 116.049 281.546 116.03 281.598 115.991C281.65 115.952 281.676 115.889 281.676 115.801V115.576H281.662C281.648 115.605 281.627 115.634 281.599 115.665C281.571 115.695 281.535 115.721 281.489 115.742C281.444 115.762 281.388 115.773 281.319 115.773C281.228 115.773 281.145 115.751 281.07 115.708C280.996 115.665 280.937 115.6 280.893 115.514C280.85 115.428 280.828 115.32 280.828 115.19C280.828 115.059 280.85 114.949 280.893 114.859C280.937 114.768 280.996 114.7 281.071 114.653C281.145 114.606 281.229 114.582 281.32 114.582C281.39 114.582 281.447 114.594 281.492 114.618C281.538 114.641 281.574 114.67 281.601 114.703C281.628 114.735 281.648 114.766 281.662 114.795H281.678V114.598H281.955V115.809C281.955 115.911 281.931 115.995 281.882 116.062C281.833 116.129 281.767 116.179 281.682 116.212C281.598 116.245 281.502 116.262 281.395 116.262ZM281.398 115.552C281.457 115.552 281.508 115.538 281.55 115.509C281.592 115.48 281.623 115.438 281.645 115.384C281.667 115.33 281.678 115.264 281.678 115.188C281.678 115.113 281.667 115.048 281.645 114.991C281.624 114.935 281.592 114.891 281.551 114.86C281.509 114.829 281.458 114.813 281.398 114.813C281.335 114.813 281.283 114.829 281.241 114.862C281.199 114.894 281.167 114.939 281.146 114.996C281.125 115.052 281.114 115.116 281.114 115.188C281.114 115.261 281.125 115.325 281.146 115.38C281.168 115.434 281.2 115.477 281.242 115.507C281.284 115.537 281.336 115.552 281.398 115.552Z" fill="#434960"/><g clip-path="url(#clip7_5232_97155)"><path d="M258.424 119.795V119.813C258.682 119.813 258.891 120.022 258.891 120.28C258.891 120.557 258.727 120.823 258.515 121.054C258.307 121.281 258.056 121.469 257.878 121.596C257.699 121.469 257.448 121.281 257.241 121.054C257.029 120.823 256.864 120.557 256.864 120.28C256.864 120.022 257.074 119.813 257.332 119.813V119.795L257.332 119.813C257.407 119.813 257.481 119.831 257.547 119.866C257.614 119.901 257.671 119.951 257.714 120.012L257.714 120.012L257.863 120.224L257.878 120.245L257.893 120.224L258.042 120.012L258.042 120.012C258.085 119.951 258.142 119.901 258.208 119.866C258.275 119.831 258.349 119.813 258.424 119.813L258.424 119.795ZM258.424 119.795C258.692 119.795 258.909 120.012 258.909 120.28C258.909 120.847 258.237 121.365 257.878 121.618L258.027 120.002C258.071 119.938 258.131 119.886 258.2 119.85C258.269 119.814 258.346 119.795 258.424 119.795ZM257.878 121.862L257.867 121.855C257.655 121.708 257.35 121.491 257.098 121.223C256.847 120.955 256.646 120.634 256.646 120.28C256.646 120.098 256.718 119.924 256.847 119.795C256.975 119.667 257.15 119.594 257.332 119.594L257.878 121.862ZM257.878 121.862L257.888 121.855M257.878 121.862L257.888 121.855M257.888 121.855C258.101 121.708 258.405 121.491 258.657 121.223C258.909 120.955 259.11 120.634 259.11 120.28C259.11 120.19 259.092 120.101 259.058 120.018C259.023 119.934 258.973 119.859 258.909 119.795C258.845 119.732 258.77 119.681 258.686 119.647C258.603 119.612 258.514 119.594 258.424 119.594C258.314 119.594 258.205 119.621 258.108 119.672M257.888 121.855L258.108 119.672M258.108 119.672C258.017 119.719 257.939 119.785 257.878 119.866M258.108 119.672L257.878 119.866M257.878 119.866C257.817 119.785 257.738 119.719 257.648 119.672C257.55 119.621 257.442 119.594 257.332 119.594L257.878 119.866Z" fill="#434960" stroke="#434960" stroke-width="0.03642"/></g><path d="M261.293 121.563H260.569L260.916 121.171C261.148 120.923 261.273 120.765 261.273 120.536C261.273 120.271 261.061 120.044 260.717 120.044C260.383 120.044 260.136 120.266 260.136 120.592H260.388C260.388 120.398 260.508 120.276 260.699 120.276C260.885 120.276 261.003 120.406 261.003 120.569C261.003 120.745 260.893 120.872 260.717 121.061L260.179 121.645V121.793H261.293V121.563ZM262.753 121.413V121.189H262.519V120.069H262.32L261.43 121.232V121.413H262.261V121.793H262.519V121.413H262.753ZM261.721 121.189L262.261 120.459V121.189H261.721Z" fill="#434960"/><g clip-path="url(#clip8_5232_97155)"><path d="M268.078 119.49C267.919 119.49 267.761 119.522 267.614 119.583C267.467 119.644 267.333 119.733 267.22 119.846C266.993 120.074 266.865 120.382 266.865 120.704C266.864 120.984 266.961 121.256 267.139 121.473L266.896 121.715C266.879 121.732 266.868 121.754 266.864 121.778C266.859 121.801 266.862 121.825 266.871 121.848C266.881 121.869 266.897 121.888 266.918 121.9C266.938 121.913 266.962 121.919 266.986 121.918H268.078C268.4 121.918 268.709 121.79 268.937 121.562C269.164 121.335 269.292 121.026 269.292 120.704C269.292 120.382 269.164 120.074 268.937 119.846C268.709 119.618 268.4 119.49 268.078 119.49ZM268.078 121.675H267.279L267.392 121.562C267.414 121.54 267.427 121.509 267.427 121.477C267.427 121.445 267.414 121.414 267.392 121.391C267.233 121.232 267.134 121.024 267.111 120.8C267.089 120.576 267.145 120.352 267.27 120.165C267.395 119.978 267.58 119.841 267.795 119.775C268.01 119.71 268.241 119.721 268.449 119.807C268.656 119.893 268.828 120.048 268.934 120.246C269.04 120.444 269.074 120.673 269.03 120.893C268.987 121.113 268.868 121.312 268.694 121.454C268.521 121.597 268.303 121.675 268.078 121.675Z" fill="#434960"/></g><path d="M271.981 121.234C271.981 120.847 271.649 120.615 271.252 120.702L271.759 120.069H271.466L270.946 120.712C270.793 120.9 270.719 121.036 270.719 121.247C270.719 121.561 270.984 121.816 271.348 121.816C271.718 121.816 271.981 121.561 271.981 121.234ZM270.989 121.24C270.989 121 271.15 120.877 271.351 120.877C271.555 120.877 271.713 121.015 271.713 121.242C271.713 121.456 271.555 121.599 271.351 121.599C271.15 121.599 270.989 121.456 270.989 121.24Z" fill="#434960"/></g></g><g filter="url(#filter4_dd_5232_97155)"><g clip-path="url(#clip9_5232_97155)"><rect x="254.238" y="127.533" width="51.8985" height="85.9504" rx="0.7284" fill="white"/><rect x="254.238" y="127.533" width="51.8985" height="65.9202" fill="url(#pattern3)"/></g></g><g filter="url(#filter5_dd_5232_97155)"><g clip-path="url(#clip10_5232_97155)"><rect x="309.777" y="57.1655" width="51.8985" height="85.5862" rx="0.7284" fill="white"/><rect x="309.777" y="57.1658" width="51.8985" height="65.9202" fill="url(#pattern4)"/><g clip-path="url(#clip11_5232_97155)"><rect x="355.301" y="58.6216" width="5.09834" height="5.09834" rx="2.54917" fill="white"/><path d="M358.724 60.2968V61.6077H357.413V60.2968H358.724ZM358.724 60.0784H357.413C357.293 60.0784 357.195 60.1767 357.195 60.2968V61.6077C357.195 61.7279 357.293 61.8262 357.413 61.8262H358.724C358.844 61.8262 358.943 61.7279 358.943 61.6077V60.2968C358.943 60.1767 358.844 60.0784 358.724 60.0784ZM356.758 60.5153V62.0446C356.758 62.1648 356.856 62.2631 356.976 62.2631H358.506V62.0446H356.976V60.5153H356.758Z" fill="#141B38"/></g><path d="M312.05 125.559V125.318H313.318V125.559H312.827V126.907H312.541V125.559H312.05ZM313.853 126.208V126.907H313.572V125.318H313.846V125.917H313.86C313.888 125.85 313.931 125.797 313.99 125.758C314.049 125.719 314.124 125.699 314.215 125.699C314.298 125.699 314.37 125.717 314.431 125.751C314.493 125.786 314.541 125.837 314.574 125.903C314.608 125.97 314.625 126.052 314.625 126.148V126.907H314.345V126.191C314.345 126.111 314.324 126.049 314.282 126.004C314.242 125.959 314.184 125.937 314.11 125.937C314.061 125.937 314.016 125.948 313.977 125.969C313.938 125.991 313.907 126.021 313.885 126.062C313.863 126.102 313.853 126.151 313.853 126.208ZM315.461 126.93C315.342 126.93 315.239 126.905 315.152 126.856C315.065 126.805 314.999 126.734 314.952 126.643C314.906 126.551 314.882 126.442 314.882 126.318C314.882 126.195 314.906 126.088 314.952 125.995C314.999 125.902 315.065 125.829 315.149 125.778C315.234 125.725 315.333 125.699 315.447 125.699C315.52 125.699 315.589 125.711 315.655 125.735C315.72 125.758 315.778 125.794 315.828 125.844C315.879 125.893 315.919 125.955 315.948 126.031C315.977 126.107 315.991 126.197 315.991 126.301V126.388H315.014V126.198H315.722C315.722 126.144 315.71 126.097 315.687 126.055C315.664 126.012 315.633 125.979 315.592 125.955C315.551 125.93 315.504 125.918 315.45 125.918C315.393 125.918 315.343 125.932 315.299 125.96C315.256 125.987 315.222 126.024 315.197 126.069C315.174 126.113 315.162 126.162 315.161 126.215V126.381C315.161 126.45 315.174 126.509 315.199 126.559C315.224 126.608 315.26 126.646 315.305 126.672C315.351 126.698 315.404 126.711 315.465 126.711C315.506 126.711 315.543 126.705 315.576 126.694C315.609 126.682 315.638 126.665 315.662 126.642C315.687 126.619 315.705 126.591 315.717 126.558L315.98 126.587C315.963 126.656 315.932 126.717 315.885 126.769C315.839 126.82 315.78 126.86 315.708 126.888C315.636 126.916 315.554 126.93 315.461 126.93ZM316.815 126.907V125.715H317.083V125.917H317.097C317.122 125.849 317.163 125.796 317.221 125.757C317.278 125.719 317.347 125.699 317.426 125.699C317.507 125.699 317.575 125.719 317.63 125.758C317.686 125.797 317.725 125.85 317.748 125.917H317.761C317.787 125.851 317.832 125.798 317.894 125.759C317.957 125.719 318.032 125.699 318.118 125.699C318.228 125.699 318.318 125.734 318.387 125.803C318.456 125.873 318.491 125.974 318.491 126.107V126.907H318.209V126.15C318.209 126.076 318.19 126.022 318.15 125.988C318.111 125.953 318.063 125.936 318.006 125.936C317.938 125.936 317.885 125.957 317.847 126C317.809 126.042 317.79 126.096 317.79 126.163V126.907H317.515V126.139C317.515 126.077 317.496 126.028 317.459 125.991C317.422 125.954 317.374 125.936 317.314 125.936C317.274 125.936 317.237 125.946 317.204 125.967C317.171 125.987 317.145 126.016 317.125 126.053C317.105 126.09 317.096 126.133 317.096 126.182V126.907H316.815ZM319.318 126.93C319.201 126.93 319.1 126.904 319.015 126.853C318.93 126.802 318.863 126.73 318.816 126.638C318.77 126.546 318.746 126.439 318.746 126.315C318.746 126.192 318.77 126.084 318.816 125.992C318.863 125.899 318.93 125.827 319.015 125.776C319.1 125.725 319.201 125.699 319.318 125.699C319.434 125.699 319.535 125.725 319.62 125.776C319.706 125.827 319.772 125.899 319.818 125.992C319.865 126.084 319.889 126.192 319.889 126.315C319.889 126.439 319.865 126.546 319.818 126.638C319.772 126.73 319.706 126.802 319.62 126.853C319.535 126.904 319.434 126.93 319.318 126.93ZM319.319 126.705C319.382 126.705 319.435 126.688 319.477 126.653C319.52 126.618 319.551 126.571 319.572 126.512C319.593 126.453 319.604 126.387 319.604 126.315C319.604 126.242 319.593 126.176 319.572 126.117C319.551 126.057 319.52 126.01 319.477 125.975C319.435 125.94 319.382 125.922 319.319 125.922C319.254 125.922 319.201 125.94 319.158 125.975C319.115 126.01 319.084 126.057 319.062 126.117C319.042 126.176 319.031 126.242 319.031 126.315C319.031 126.387 319.042 126.453 319.062 126.512C319.084 126.571 319.115 126.618 319.158 126.653C319.201 126.688 319.254 126.705 319.319 126.705ZM320.149 126.907V125.715H320.421V125.913H320.434C320.456 125.845 320.493 125.792 320.546 125.754C320.599 125.717 320.66 125.698 320.728 125.698C320.743 125.698 320.761 125.699 320.78 125.7C320.8 125.701 320.816 125.703 320.829 125.705V125.964C320.817 125.96 320.798 125.956 320.772 125.953C320.747 125.949 320.722 125.948 320.698 125.948C320.647 125.948 320.601 125.959 320.56 125.981C320.52 126.003 320.488 126.033 320.465 126.072C320.442 126.111 320.43 126.155 320.43 126.206V126.907H320.149ZM321.318 126.208V126.907H321.037V125.715H321.306V125.917H321.32C321.347 125.851 321.391 125.798 321.451 125.758C321.511 125.719 321.586 125.699 321.675 125.699C321.757 125.699 321.829 125.717 321.89 125.752C321.952 125.787 321.999 125.838 322.033 125.905C322.067 125.972 322.084 126.053 322.083 126.148V126.907H321.802V126.191C321.802 126.112 321.782 126.049 321.74 126.004C321.699 125.959 321.643 125.937 321.57 125.937C321.521 125.937 321.478 125.948 321.439 125.969C321.401 125.991 321.372 126.021 321.35 126.062C321.329 126.102 321.318 126.151 321.318 126.208ZM322.391 126.907V125.715H322.672V126.907H322.391ZM322.532 125.546C322.487 125.546 322.449 125.531 322.417 125.501C322.385 125.471 322.369 125.435 322.369 125.394C322.369 125.351 322.385 125.315 322.417 125.286C322.449 125.256 322.487 125.241 322.532 125.241C322.577 125.241 322.615 125.256 322.647 125.286C322.679 125.315 322.695 125.351 322.695 125.394C322.695 125.435 322.679 125.471 322.647 125.501C322.615 125.531 322.577 125.546 322.532 125.546ZM323.263 126.208V126.907H322.982V125.715H323.251V125.917H323.265C323.292 125.851 323.336 125.798 323.396 125.758C323.456 125.719 323.531 125.699 323.62 125.699C323.702 125.699 323.774 125.717 323.835 125.752C323.897 125.787 323.944 125.838 323.978 125.905C324.012 125.972 324.029 126.053 324.028 126.148V126.907H323.747V126.191C323.747 126.112 323.727 126.049 323.685 126.004C323.645 125.959 323.588 125.937 323.515 125.937C323.466 125.937 323.423 125.948 323.384 125.969C323.347 125.991 323.317 126.021 323.295 126.062C323.274 126.102 323.263 126.151 323.263 126.208ZM324.854 127.379C324.753 127.379 324.667 127.365 324.594 127.337C324.522 127.311 324.464 127.274 324.42 127.229C324.376 127.183 324.345 127.133 324.328 127.077L324.581 127.016C324.592 127.039 324.609 127.062 324.631 127.085C324.652 127.108 324.682 127.128 324.718 127.143C324.756 127.158 324.802 127.166 324.859 127.166C324.939 127.166 325.004 127.147 325.057 127.108C325.109 127.069 325.135 127.006 325.135 126.918V126.693H325.121C325.107 126.722 325.085 126.751 325.057 126.782C325.03 126.812 324.994 126.838 324.948 126.859C324.903 126.879 324.846 126.89 324.778 126.89C324.687 126.89 324.604 126.868 324.529 126.825C324.455 126.782 324.396 126.717 324.352 126.631C324.309 126.545 324.287 126.437 324.287 126.307C324.287 126.176 324.309 126.066 324.352 125.976C324.396 125.885 324.455 125.816 324.53 125.77C324.604 125.723 324.687 125.699 324.779 125.699C324.849 125.699 324.906 125.711 324.951 125.735C324.997 125.758 325.033 125.786 325.06 125.82C325.087 125.852 325.107 125.883 325.121 125.912H325.137V125.715H325.414V126.926C325.414 127.028 325.389 127.112 325.341 127.179C325.292 127.246 325.226 127.296 325.141 127.329C325.057 127.362 324.961 127.379 324.854 127.379ZM324.857 126.669C324.916 126.669 324.967 126.655 325.009 126.626C325.051 126.597 325.082 126.555 325.104 126.501C325.126 126.447 325.137 126.381 325.137 126.305C325.137 126.23 325.126 126.165 325.104 126.108C325.083 126.052 325.051 126.008 325.009 125.977C324.968 125.946 324.917 125.93 324.857 125.93C324.794 125.93 324.742 125.946 324.7 125.979C324.658 126.011 324.626 126.056 324.605 126.113C324.584 126.169 324.573 126.233 324.573 126.305C324.573 126.378 324.584 126.442 324.605 126.497C324.627 126.551 324.659 126.594 324.701 126.624C324.743 126.654 324.795 126.669 324.857 126.669ZM327.229 126.03L326.973 126.058C326.966 126.032 326.953 126.008 326.935 125.985C326.918 125.962 326.894 125.944 326.864 125.93C326.834 125.916 326.797 125.909 326.754 125.909C326.695 125.909 326.646 125.921 326.606 125.947C326.567 125.972 326.548 126.005 326.548 126.045C326.548 126.08 326.56 126.108 326.586 126.13C326.612 126.152 326.656 126.17 326.716 126.184L326.92 126.227C327.033 126.251 327.116 126.29 327.171 126.343C327.227 126.395 327.254 126.464 327.255 126.55C327.254 126.625 327.232 126.691 327.189 126.748C327.146 126.805 327.086 126.85 327.01 126.882C326.933 126.914 326.845 126.93 326.746 126.93C326.6 126.93 326.483 126.899 326.394 126.838C326.305 126.777 326.252 126.691 326.235 126.582L326.508 126.555C326.521 126.609 326.547 126.65 326.588 126.677C326.628 126.704 326.68 126.718 326.745 126.718C326.812 126.718 326.865 126.704 326.906 126.677C326.947 126.65 326.967 126.616 326.967 126.575C326.967 126.541 326.954 126.513 326.927 126.491C326.902 126.469 326.861 126.451 326.806 126.44L326.603 126.397C326.489 126.373 326.404 126.333 326.349 126.277C326.295 126.22 326.267 126.148 326.268 126.061C326.267 125.987 326.287 125.924 326.328 125.87C326.369 125.816 326.425 125.774 326.498 125.744C326.571 125.714 326.655 125.699 326.75 125.699C326.889 125.699 326.999 125.729 327.08 125.789C327.16 125.848 327.21 125.928 327.229 126.03ZM328.268 126.405V125.715H328.549V126.907H328.277V126.695H328.264C328.237 126.762 328.193 126.816 328.132 126.859C328.071 126.901 327.995 126.922 327.906 126.922C327.828 126.922 327.759 126.905 327.699 126.87C327.639 126.835 327.593 126.784 327.559 126.717C327.525 126.65 327.508 126.569 327.508 126.474V125.715H327.789V126.43C327.789 126.506 327.81 126.566 327.851 126.61C327.893 126.655 327.947 126.677 328.014 126.677C328.056 126.677 328.096 126.667 328.135 126.647C328.173 126.627 328.205 126.597 328.23 126.557C328.255 126.516 328.268 126.466 328.268 126.405ZM329.141 126.208V126.907H328.86V125.715H329.128V125.917H329.142C329.17 125.851 329.213 125.798 329.273 125.758C329.334 125.719 329.409 125.699 329.498 125.699C329.58 125.699 329.652 125.717 329.713 125.752C329.774 125.787 329.822 125.838 329.855 125.905C329.89 125.972 329.906 126.053 329.906 126.148V126.907H329.625V126.191C329.625 126.112 329.604 126.049 329.563 126.004C329.522 125.959 329.465 125.937 329.393 125.937C329.344 125.937 329.3 125.948 329.262 125.969C329.224 125.991 329.194 126.021 329.172 126.062C329.151 126.102 329.141 126.151 329.141 126.208ZM331.023 126.907L330.686 125.715H330.973L331.182 126.553H331.193L331.407 125.715H331.69L331.905 126.548H331.916L332.123 125.715H332.41L332.072 126.907H331.78L331.556 126.101H331.54L331.316 126.907H331.023ZM332.971 126.931C332.895 126.931 332.827 126.917 332.767 126.89C332.707 126.863 332.659 126.823 332.624 126.769C332.589 126.716 332.572 126.65 332.572 126.572C332.572 126.505 332.584 126.449 332.609 126.405C332.634 126.361 332.668 126.326 332.711 126.3C332.754 126.274 332.802 126.254 332.856 126.24C332.91 126.226 332.966 126.216 333.024 126.21C333.094 126.203 333.151 126.196 333.194 126.19C333.238 126.184 333.269 126.175 333.289 126.163C333.309 126.15 333.319 126.13 333.319 126.103V126.098C333.319 126.04 333.302 125.994 333.267 125.962C333.233 125.93 333.183 125.914 333.117 125.914C333.049 125.914 332.994 125.929 332.954 125.959C332.914 125.989 332.887 126.025 332.873 126.066L332.611 126.028C332.631 125.956 332.666 125.895 332.713 125.847C332.761 125.798 332.819 125.761 332.888 125.737C332.957 125.712 333.033 125.699 333.116 125.699C333.173 125.699 333.23 125.706 333.287 125.719C333.344 125.733 333.396 125.755 333.443 125.786C333.49 125.817 333.528 125.858 333.557 125.911C333.586 125.964 333.6 126.03 333.6 126.109V126.907H333.33V126.743H333.321C333.304 126.776 333.28 126.807 333.249 126.836C333.218 126.865 333.18 126.888 333.133 126.905C333.087 126.922 333.033 126.931 332.971 126.931ZM333.044 126.724C333.1 126.724 333.149 126.713 333.19 126.691C333.232 126.668 333.264 126.638 333.286 126.601C333.309 126.564 333.32 126.523 333.32 126.479V126.339C333.311 126.346 333.296 126.353 333.275 126.359C333.254 126.365 333.231 126.371 333.205 126.375C333.179 126.38 333.154 126.384 333.128 126.388C333.103 126.391 333.081 126.394 333.062 126.397C333.02 126.403 332.983 126.412 332.95 126.425C332.917 126.438 332.891 126.456 332.871 126.479C332.852 126.502 332.843 126.531 332.843 126.568C332.843 126.619 332.862 126.658 332.899 126.685C332.937 126.711 332.985 126.724 333.044 126.724ZM333.904 126.907V125.715H334.177V125.913H334.189C334.211 125.845 334.248 125.792 334.301 125.754C334.354 125.717 334.415 125.698 334.483 125.698C334.499 125.698 334.516 125.699 334.535 125.7C334.555 125.701 334.571 125.703 334.584 125.705V125.964C334.572 125.96 334.553 125.956 334.528 125.953C334.502 125.949 334.478 125.948 334.454 125.948C334.403 125.948 334.357 125.959 334.316 125.981C334.275 126.003 334.244 126.033 334.22 126.072C334.197 126.111 334.185 126.155 334.185 126.206V126.907H333.904ZM334.793 126.907V125.715H335.061V125.917H335.075C335.1 125.849 335.141 125.796 335.199 125.757C335.256 125.719 335.324 125.699 335.404 125.699C335.485 125.699 335.553 125.719 335.608 125.758C335.664 125.797 335.703 125.85 335.726 125.917H335.739C335.765 125.851 335.809 125.798 335.872 125.759C335.935 125.719 336.01 125.699 336.096 125.699C336.206 125.699 336.296 125.734 336.365 125.803C336.434 125.873 336.469 125.974 336.469 126.107V126.907H336.187V126.15C336.187 126.076 336.167 126.022 336.128 125.988C336.089 125.953 336.041 125.936 335.984 125.936C335.916 125.936 335.863 125.957 335.825 126C335.787 126.042 335.768 126.096 335.768 126.163V126.907H335.493V126.139C335.493 126.077 335.474 126.028 335.437 125.991C335.4 125.954 335.352 125.936 335.292 125.936C335.252 125.936 335.215 125.946 335.182 125.967C335.149 125.987 335.123 126.016 335.103 126.053C335.083 126.09 335.074 126.133 335.074 126.182V126.907H334.793ZM337.718 126.03L337.462 126.058C337.454 126.032 337.442 126.008 337.424 125.985C337.406 125.962 337.382 125.944 337.352 125.93C337.322 125.916 337.285 125.909 337.242 125.909C337.184 125.909 337.134 125.921 337.095 125.947C337.055 125.972 337.036 126.005 337.036 126.045C337.036 126.08 337.049 126.108 337.074 126.13C337.101 126.152 337.144 126.17 337.205 126.184L337.408 126.227C337.521 126.251 337.605 126.29 337.66 126.343C337.715 126.395 337.743 126.464 337.743 126.55C337.743 126.625 337.721 126.691 337.677 126.748C337.634 126.805 337.575 126.85 337.498 126.882C337.422 126.914 337.334 126.93 337.234 126.93C337.088 126.93 336.971 126.899 336.882 126.838C336.793 126.777 336.74 126.691 336.723 126.582L336.997 126.555C337.009 126.609 337.036 126.65 337.076 126.677C337.116 126.704 337.169 126.718 337.234 126.718C337.3 126.718 337.354 126.704 337.394 126.677C337.435 126.65 337.455 126.616 337.455 126.575C337.455 126.541 337.442 126.513 337.416 126.491C337.39 126.469 337.35 126.451 337.295 126.44L337.091 126.397C336.977 126.373 336.893 126.333 336.838 126.277C336.783 126.22 336.756 126.148 336.756 126.061C336.756 125.987 336.776 125.924 336.816 125.87C336.857 125.816 336.914 125.774 336.986 125.744C337.059 125.714 337.143 125.699 337.238 125.699C337.378 125.699 337.488 125.729 337.568 125.789C337.649 125.848 337.699 125.928 337.718 126.03ZM339.156 125.715V125.932H338.471V125.715H339.156ZM338.64 125.429H338.921V126.548C338.921 126.586 338.926 126.615 338.938 126.635C338.95 126.655 338.965 126.668 338.984 126.675C339.004 126.683 339.025 126.686 339.048 126.686C339.066 126.686 339.082 126.685 339.096 126.682C339.111 126.68 339.123 126.678 339.13 126.675L339.178 126.895C339.163 126.9 339.141 126.906 339.113 126.912C339.086 126.918 339.052 126.922 339.012 126.923C338.942 126.925 338.879 126.914 338.822 126.891C338.766 126.867 338.721 126.831 338.688 126.781C338.655 126.731 338.639 126.669 338.64 126.595V125.429ZM339.719 126.208V126.907H339.438V125.318H339.713V125.917H339.727C339.755 125.85 339.798 125.797 339.857 125.758C339.916 125.719 339.991 125.699 340.082 125.699C340.164 125.699 340.237 125.717 340.298 125.751C340.36 125.786 340.407 125.837 340.441 125.903C340.475 125.97 340.492 126.052 340.492 126.148V126.907H340.211V126.191C340.211 126.111 340.19 126.049 340.149 126.004C340.108 125.959 340.051 125.937 339.977 125.937C339.927 125.937 339.883 125.948 339.843 125.969C339.805 125.991 339.774 126.021 339.752 126.062C339.73 126.102 339.719 126.151 339.719 126.208ZM341.328 126.93C341.208 126.93 341.105 126.905 341.018 126.856C340.932 126.805 340.865 126.734 340.819 126.643C340.772 126.551 340.749 126.442 340.749 126.318C340.749 126.195 340.772 126.088 340.819 125.995C340.866 125.902 340.932 125.829 341.016 125.778C341.1 125.725 341.199 125.699 341.313 125.699C341.387 125.699 341.456 125.711 341.521 125.735C341.587 125.758 341.645 125.794 341.695 125.844C341.746 125.893 341.786 125.955 341.814 126.031C341.843 126.107 341.858 126.197 341.858 126.301V126.388H340.881V126.198H341.589C341.588 126.144 341.577 126.097 341.554 126.055C341.531 126.012 341.499 125.979 341.458 125.955C341.418 125.93 341.371 125.918 341.317 125.918C341.26 125.918 341.209 125.932 341.166 125.96C341.122 125.987 341.088 126.024 341.064 126.069C341.04 126.113 341.028 126.162 341.028 126.215V126.381C341.028 126.45 341.04 126.509 341.066 126.559C341.091 126.608 341.126 126.646 341.172 126.672C341.218 126.698 341.271 126.711 341.332 126.711C341.373 126.711 341.41 126.705 341.443 126.694C341.476 126.682 341.505 126.665 341.529 126.642C341.553 126.619 341.572 126.591 341.584 126.558L341.846 126.587C341.83 126.656 341.798 126.717 341.752 126.769C341.706 126.82 341.647 126.86 341.575 126.888C341.503 126.916 341.421 126.93 341.328 126.93ZM342.681 126.907V125.715H342.95V125.917H342.964C342.989 125.849 343.03 125.796 343.087 125.757C343.145 125.719 343.213 125.699 343.293 125.699C343.374 125.699 343.442 125.719 343.497 125.758C343.553 125.797 343.592 125.85 343.615 125.917H343.627C343.654 125.851 343.698 125.798 343.761 125.759C343.824 125.719 343.899 125.699 343.985 125.699C344.095 125.699 344.184 125.734 344.254 125.803C344.323 125.873 344.357 125.974 344.357 126.107V126.907H344.076V126.15C344.076 126.076 344.056 126.022 344.017 125.988C343.978 125.953 343.929 125.936 343.872 125.936C343.805 125.936 343.752 125.957 343.713 126C343.676 126.042 343.657 126.096 343.657 126.163V126.907H343.381V126.139C343.381 126.077 343.363 126.028 343.325 125.991C343.289 125.954 343.241 125.936 343.181 125.936C343.141 125.936 343.104 125.946 343.071 125.967C343.038 125.987 343.011 126.016 342.992 126.053C342.972 126.09 342.962 126.133 342.962 126.182V126.907H342.681ZM345.184 126.93C345.068 126.93 344.967 126.904 344.882 126.853C344.796 126.802 344.73 126.73 344.683 126.638C344.636 126.546 344.613 126.439 344.613 126.315C344.613 126.192 344.636 126.084 344.683 125.992C344.73 125.899 344.796 125.827 344.882 125.776C344.967 125.725 345.068 125.699 345.184 125.699C345.301 125.699 345.402 125.725 345.487 125.776C345.572 125.827 345.638 125.899 345.685 125.992C345.732 126.084 345.755 126.192 345.755 126.315C345.755 126.439 345.732 126.546 345.685 126.638C345.638 126.73 345.572 126.802 345.487 126.853C345.402 126.904 345.301 126.93 345.184 126.93ZM345.186 126.705C345.249 126.705 345.302 126.688 345.344 126.653C345.387 126.618 345.418 126.571 345.439 126.512C345.46 126.453 345.471 126.387 345.471 126.315C345.471 126.242 345.46 126.176 345.439 126.117C345.418 126.057 345.387 126.01 345.344 125.975C345.302 125.94 345.249 125.922 345.186 125.922C345.121 125.922 345.067 125.94 345.024 125.975C344.982 126.01 344.95 126.057 344.929 126.117C344.908 126.176 344.898 126.242 344.898 126.315C344.898 126.387 344.908 126.453 344.929 126.512C344.95 126.571 344.982 126.618 345.024 126.653C345.067 126.688 345.121 126.705 345.186 126.705ZM346.775 126.405V125.715H347.056V126.907H346.784V126.695H346.771C346.745 126.762 346.7 126.816 346.639 126.859C346.578 126.901 346.502 126.922 346.413 126.922C346.335 126.922 346.266 126.905 346.206 126.87C346.146 126.835 346.1 126.784 346.066 126.717C346.032 126.65 346.016 126.569 346.016 126.474V125.715H346.297V126.43C346.297 126.506 346.317 126.566 346.359 126.61C346.4 126.655 346.454 126.677 346.522 126.677C346.563 126.677 346.603 126.667 346.642 126.647C346.681 126.627 346.712 126.597 346.737 126.557C346.763 126.516 346.775 126.466 346.775 126.405ZM347.648 126.208V126.907H347.367V125.715H347.635V125.917H347.649C347.677 125.851 347.721 125.798 347.781 125.758C347.841 125.719 347.916 125.699 348.005 125.699C348.087 125.699 348.159 125.717 348.22 125.752C348.281 125.787 348.329 125.838 348.363 125.905C348.397 125.972 348.414 126.053 348.413 126.148V126.907H348.132V126.191C348.132 126.112 348.111 126.049 348.07 126.004C348.029 125.959 347.973 125.937 347.9 125.937C347.851 125.937 347.807 125.948 347.769 125.969C347.731 125.991 347.701 126.021 347.68 126.062C347.658 126.102 347.648 126.151 347.648 126.208ZM349.316 125.715V125.932H348.63V125.715H349.316ZM348.8 125.429H349.08V126.548C349.08 126.586 349.086 126.615 349.098 126.635C349.109 126.655 349.125 126.668 349.144 126.675C349.163 126.683 349.184 126.686 349.208 126.686C349.225 126.686 349.241 126.685 349.256 126.682C349.271 126.68 349.282 126.678 349.29 126.675L349.337 126.895C349.322 126.9 349.301 126.906 349.273 126.912C349.246 126.918 349.212 126.922 349.172 126.923C349.102 126.925 349.038 126.914 348.982 126.891C348.926 126.867 348.881 126.831 348.848 126.781C348.815 126.731 348.799 126.669 348.8 126.595V125.429ZM349.915 126.931C349.84 126.931 349.772 126.917 349.711 126.89C349.651 126.863 349.604 126.823 349.569 126.769C349.534 126.716 349.517 126.65 349.517 126.572C349.517 126.505 349.529 126.449 349.554 126.405C349.579 126.361 349.612 126.326 349.655 126.3C349.698 126.274 349.747 126.254 349.801 126.24C349.855 126.226 349.911 126.216 349.969 126.21C350.039 126.203 350.095 126.196 350.139 126.19C350.182 126.184 350.214 126.175 350.234 126.163C350.254 126.15 350.264 126.13 350.264 126.103V126.098C350.264 126.04 350.246 125.994 350.212 125.962C350.177 125.93 350.127 125.914 350.062 125.914C349.993 125.914 349.939 125.929 349.898 125.959C349.858 125.989 349.832 126.025 349.818 126.066L349.555 126.028C349.576 125.956 349.61 125.895 349.658 125.847C349.705 125.798 349.764 125.761 349.832 125.737C349.901 125.712 349.977 125.699 350.061 125.699C350.118 125.699 350.175 125.706 350.232 125.719C350.289 125.733 350.341 125.755 350.388 125.786C350.435 125.817 350.473 125.858 350.501 125.911C350.53 125.964 350.545 126.03 350.545 126.109V126.907H350.275V126.743H350.265C350.248 126.776 350.224 126.807 350.193 126.836C350.163 126.865 350.124 126.888 350.078 126.905C350.032 126.922 349.977 126.931 349.915 126.931ZM349.988 126.724C350.045 126.724 350.094 126.713 350.135 126.691C350.176 126.668 350.208 126.638 350.23 126.601C350.253 126.564 350.265 126.523 350.265 126.479V126.339C350.256 126.346 350.241 126.353 350.22 126.359C350.199 126.365 350.176 126.371 350.15 126.375C350.124 126.38 350.098 126.384 350.073 126.388C350.048 126.391 350.026 126.394 350.007 126.397C349.965 126.403 349.928 126.412 349.894 126.425C349.861 126.438 349.835 126.456 349.816 126.479C349.797 126.502 349.787 126.531 349.787 126.568C349.787 126.619 349.806 126.658 349.844 126.685C349.882 126.711 349.93 126.724 349.988 126.724ZM350.849 126.907V125.715H351.13V126.907H350.849ZM350.99 125.546C350.946 125.546 350.907 125.531 350.875 125.501C350.843 125.471 350.827 125.435 350.827 125.394C350.827 125.351 350.843 125.315 350.875 125.286C350.907 125.256 350.946 125.241 350.99 125.241C351.035 125.241 351.074 125.256 351.105 125.286C351.137 125.315 351.153 125.351 351.153 125.394C351.153 125.435 351.137 125.471 351.105 125.501C351.074 125.531 351.035 125.546 350.99 125.546ZM351.722 126.208V126.907H351.441V125.715H351.709V125.917H351.723C351.751 125.851 351.794 125.798 351.854 125.758C351.915 125.719 351.99 125.699 352.079 125.699C352.161 125.699 352.232 125.717 352.293 125.752C352.355 125.787 352.403 125.838 352.436 125.905C352.47 125.972 352.487 126.053 352.487 126.148V126.907H352.206V126.191C352.206 126.112 352.185 126.049 352.144 126.004C352.103 125.959 352.046 125.937 351.974 125.937C351.925 125.937 351.881 125.948 351.843 125.969C351.805 125.991 351.775 126.021 351.753 126.062C351.732 126.102 351.722 126.151 351.722 126.208ZM353.953 125.715V125.932H353.268V125.715H353.953ZM353.437 125.429H353.718V126.548C353.718 126.586 353.724 126.615 353.735 126.635C353.747 126.655 353.763 126.668 353.782 126.675C353.801 126.683 353.822 126.686 353.845 126.686C353.863 126.686 353.879 126.685 353.893 126.682C353.908 126.68 353.92 126.678 353.928 126.675L353.975 126.895C353.96 126.9 353.938 126.906 353.911 126.912C353.883 126.918 353.849 126.922 353.81 126.923C353.739 126.925 353.676 126.914 353.62 126.891C353.563 126.867 353.518 126.831 353.485 126.781C353.453 126.731 353.437 126.669 353.437 126.595V125.429ZM354.717 126.93C354.601 126.93 354.5 126.904 354.414 126.853C354.329 126.802 354.263 126.73 354.216 126.638C354.169 126.546 354.146 126.439 354.146 126.315C354.146 126.192 354.169 126.084 354.216 125.992C354.263 125.899 354.329 125.827 354.414 125.776C354.5 125.725 354.601 125.699 354.717 125.699C354.833 125.699 354.934 125.725 355.02 125.776C355.105 125.827 355.171 125.899 355.218 125.992C355.265 126.084 355.288 126.192 355.288 126.315C355.288 126.439 355.265 126.546 355.218 126.638C355.171 126.73 355.105 126.802 355.02 126.853C354.934 126.904 354.833 126.93 354.717 126.93ZM354.719 126.705C354.782 126.705 354.835 126.688 354.877 126.653C354.919 126.618 354.951 126.571 354.972 126.512C354.993 126.453 355.003 126.387 355.003 126.315C355.003 126.242 354.993 126.176 354.972 126.117C354.951 126.057 354.919 126.01 354.877 125.975C354.835 125.94 354.782 125.922 354.719 125.922C354.654 125.922 354.6 125.94 354.557 125.975C354.515 126.01 354.483 126.057 354.462 126.117C354.441 126.176 354.431 126.242 354.431 126.315C354.431 126.387 354.441 126.453 354.462 126.512C354.483 126.571 354.515 126.618 354.557 126.653C354.6 126.688 354.654 126.705 354.719 126.705ZM355.548 127.354V125.715H355.825V125.912H355.841C355.856 125.883 355.876 125.852 355.902 125.82C355.929 125.786 355.964 125.758 356.009 125.735C356.054 125.711 356.112 125.699 356.182 125.699C356.274 125.699 356.357 125.723 356.431 125.77C356.505 125.816 356.564 125.886 356.608 125.977C356.652 126.068 356.674 126.18 356.674 126.312C356.674 126.443 356.652 126.554 356.609 126.646C356.566 126.738 356.508 126.807 356.434 126.856C356.36 126.904 356.276 126.928 356.182 126.928C356.114 126.928 356.058 126.916 356.013 126.894C355.968 126.871 355.931 126.843 355.904 126.811C355.877 126.779 355.856 126.748 355.841 126.719H355.829V127.354H355.548ZM355.824 126.311C355.824 126.388 355.835 126.455 355.857 126.513C355.879 126.571 355.911 126.617 355.952 126.649C355.994 126.681 356.045 126.697 356.104 126.697C356.166 126.697 356.218 126.681 356.26 126.648C356.302 126.614 356.334 126.568 356.355 126.51C356.376 126.452 356.387 126.385 356.387 126.311C356.387 126.237 356.377 126.171 356.356 126.114C356.334 126.056 356.303 126.011 356.261 125.979C356.219 125.946 356.167 125.93 356.104 125.93C356.044 125.93 355.993 125.946 355.951 125.977C355.909 126.009 355.878 126.053 355.856 126.11C355.835 126.167 355.824 126.234 355.824 126.311ZM312.45 129.931C312.375 129.931 312.307 129.917 312.246 129.89C312.186 129.863 312.139 129.823 312.103 129.769C312.069 129.716 312.051 129.65 312.051 129.572C312.051 129.505 312.064 129.449 312.089 129.405C312.113 129.361 312.147 129.326 312.19 129.3C312.233 129.274 312.282 129.254 312.335 129.24C312.39 129.226 312.446 129.216 312.504 129.21C312.574 129.203 312.63 129.196 312.674 129.19C312.717 129.184 312.749 129.175 312.768 129.163C312.789 129.15 312.799 129.13 312.799 129.103V129.098C312.799 129.04 312.781 128.994 312.747 128.962C312.712 128.93 312.662 128.914 312.597 128.914C312.528 128.914 312.473 128.929 312.433 128.959C312.393 128.989 312.366 129.025 312.352 129.066L312.09 129.028C312.111 128.956 312.145 128.895 312.193 128.847C312.24 128.798 312.298 128.761 312.367 128.737C312.436 128.712 312.512 128.699 312.595 128.699C312.653 128.699 312.71 128.706 312.767 128.719C312.824 128.733 312.876 128.755 312.923 128.786C312.97 128.817 313.008 128.858 313.036 128.911C313.065 128.964 313.08 129.03 313.08 129.109V129.907H312.81V129.743H312.8C312.783 129.776 312.759 129.807 312.728 129.836C312.697 129.865 312.659 129.888 312.612 129.905C312.566 129.922 312.512 129.931 312.45 129.931ZM312.523 129.724C312.58 129.724 312.628 129.713 312.67 129.691C312.711 129.668 312.743 129.638 312.765 129.601C312.788 129.564 312.799 129.523 312.799 129.479V129.339C312.791 129.346 312.776 129.353 312.754 129.359C312.734 129.365 312.71 129.371 312.685 129.375C312.659 129.38 312.633 129.384 312.608 129.388C312.582 129.391 312.56 129.394 312.542 129.397C312.5 129.403 312.462 129.412 312.429 129.425C312.396 129.438 312.37 129.456 312.351 129.479C312.332 129.502 312.322 129.531 312.322 129.568C312.322 129.619 312.341 129.658 312.379 129.685C312.417 129.711 312.465 129.724 312.523 129.724ZM314.326 129.03L314.07 129.058C314.063 129.032 314.05 129.008 314.032 128.985C314.014 128.962 313.99 128.944 313.96 128.93C313.93 128.916 313.894 128.909 313.85 128.909C313.792 128.909 313.743 128.921 313.703 128.947C313.663 128.972 313.644 129.005 313.645 129.045C313.644 129.08 313.657 129.108 313.683 129.13C313.709 129.152 313.752 129.17 313.813 129.184L314.016 129.227C314.129 129.251 314.213 129.29 314.268 129.343C314.323 129.395 314.351 129.464 314.352 129.55C314.351 129.625 314.329 129.691 314.286 129.748C314.243 129.805 314.183 129.85 314.106 129.882C314.03 129.914 313.942 129.93 313.842 129.93C313.697 129.93 313.579 129.899 313.49 129.838C313.401 129.777 313.348 129.691 313.331 129.582L313.605 129.555C313.617 129.609 313.644 129.65 313.684 129.677C313.725 129.704 313.777 129.718 313.842 129.718C313.908 129.718 313.962 129.704 314.002 129.677C314.043 129.65 314.064 129.616 314.064 129.575C314.064 129.541 314.05 129.513 314.024 129.491C313.998 129.469 313.958 129.451 313.903 129.44L313.7 129.397C313.585 129.373 313.501 129.333 313.446 129.277C313.391 129.22 313.364 129.148 313.364 129.061C313.364 128.987 313.384 128.924 313.424 128.87C313.465 128.816 313.522 128.774 313.594 128.744C313.667 128.714 313.751 128.699 313.846 128.699C313.986 128.699 314.096 128.729 314.176 128.789C314.257 128.848 314.307 128.928 314.326 129.03ZM315.169 129.907V128.715H315.45V129.907H315.169ZM315.31 128.546C315.266 128.546 315.227 128.531 315.195 128.501C315.163 128.471 315.147 128.435 315.147 128.394C315.147 128.351 315.163 128.315 315.195 128.286C315.227 128.256 315.266 128.241 315.31 128.241C315.355 128.241 315.393 128.256 315.425 128.286C315.457 128.315 315.473 128.351 315.473 128.394C315.473 128.435 315.457 128.471 315.425 128.501C315.393 128.531 315.355 128.546 315.31 128.546ZM316.356 128.715V128.932H315.671V128.715H316.356ZM315.84 128.429H316.121V129.548C316.121 129.586 316.126 129.615 316.138 129.635C316.15 129.655 316.165 129.668 316.184 129.675C316.203 129.683 316.225 129.686 316.248 129.686C316.265 129.686 316.281 129.685 316.296 129.682C316.311 129.68 316.322 129.678 316.33 129.675L316.377 129.895C316.362 129.9 316.341 129.906 316.313 129.912C316.286 129.918 316.252 129.922 316.212 129.923C316.142 129.925 316.078 129.914 316.022 129.891C315.966 129.867 315.921 129.831 315.888 129.781C315.855 129.731 315.839 129.669 315.84 129.595V128.429ZM317.696 129.93C317.577 129.93 317.475 129.904 317.39 129.852C317.305 129.799 317.239 129.727 317.193 129.635C317.148 129.543 317.125 129.436 317.125 129.315C317.125 129.194 317.148 129.088 317.195 128.995C317.242 128.902 317.307 128.829 317.392 128.778C317.477 128.725 317.578 128.699 317.695 128.699C317.791 128.699 317.877 128.717 317.952 128.753C318.027 128.788 318.086 128.838 318.131 128.903C318.175 128.967 318.201 129.042 318.207 129.128H317.938C317.928 129.07 317.902 129.022 317.861 128.984C317.82 128.945 317.766 128.926 317.699 128.926C317.641 128.926 317.591 128.941 317.547 128.972C317.504 129.003 317.47 129.047 317.446 129.104C317.422 129.162 317.41 129.231 317.41 129.311C317.41 129.392 317.422 129.462 317.446 129.52C317.469 129.578 317.503 129.623 317.546 129.655C317.589 129.686 317.64 129.701 317.699 129.701C317.74 129.701 317.777 129.693 317.81 129.678C317.843 129.662 317.87 129.639 317.893 129.609C317.915 129.579 317.93 129.542 317.938 129.499H318.207C318.2 129.584 318.175 129.658 318.132 129.724C318.089 129.788 318.031 129.839 317.957 129.876C317.883 129.912 317.796 129.93 317.696 129.93ZM318.802 129.931C318.727 129.931 318.659 129.917 318.598 129.89C318.538 129.863 318.491 129.823 318.456 129.769C318.421 129.716 318.404 129.65 318.404 129.572C318.404 129.505 318.416 129.449 318.441 129.405C318.466 129.361 318.5 129.326 318.542 129.3C318.585 129.274 318.634 129.254 318.688 129.24C318.742 129.226 318.798 129.216 318.856 129.21C318.926 129.203 318.982 129.196 319.026 129.19C319.069 129.184 319.101 129.175 319.121 129.163C319.141 129.15 319.151 129.13 319.151 129.103V129.098C319.151 129.04 319.133 128.994 319.099 128.962C319.064 128.93 319.014 128.914 318.949 128.914C318.88 128.914 318.826 128.929 318.785 128.959C318.745 128.989 318.719 129.025 318.705 129.066L318.442 129.028C318.463 128.956 318.497 128.895 318.545 128.847C318.592 128.798 318.651 128.761 318.719 128.737C318.788 128.712 318.864 128.699 318.948 128.699C319.005 128.699 319.062 128.706 319.119 128.719C319.176 128.733 319.228 128.755 319.275 128.786C319.322 128.817 319.36 128.858 319.388 128.911C319.417 128.964 319.432 129.03 319.432 129.109V129.907H319.162V129.743H319.152C319.135 129.776 319.111 129.807 319.08 129.836C319.05 129.865 319.011 129.888 318.965 129.905C318.919 129.922 318.864 129.931 318.802 129.931ZM318.875 129.724C318.932 129.724 318.981 129.713 319.022 129.691C319.063 129.668 319.095 129.638 319.117 129.601C319.14 129.564 319.152 129.523 319.152 129.479V129.339C319.143 129.346 319.128 129.353 319.107 129.359C319.086 129.365 319.063 129.371 319.037 129.375C319.011 129.38 318.985 129.384 318.96 129.388C318.935 129.391 318.913 129.394 318.894 129.397C318.852 129.403 318.815 129.412 318.781 129.425C318.748 129.438 318.722 129.456 318.703 129.479C318.684 129.502 318.674 129.531 318.674 129.568C318.674 129.619 318.693 129.658 318.731 129.685C318.769 129.711 318.817 129.724 318.875 129.724ZM320.017 128.318V129.907H319.736V128.318H320.017ZM320.609 128.318V129.907H320.328V128.318H320.609ZM321.861 129.03L321.605 129.058C321.598 129.032 321.585 129.008 321.567 128.985C321.55 128.962 321.526 128.944 321.496 128.93C321.466 128.916 321.429 128.909 321.386 128.909C321.327 128.909 321.278 128.921 321.238 128.947C321.199 128.972 321.18 129.005 321.18 129.045C321.18 129.08 321.192 129.108 321.218 129.13C321.244 129.152 321.288 129.17 321.348 129.184L321.552 129.227C321.665 129.251 321.748 129.29 321.803 129.343C321.859 129.395 321.886 129.464 321.887 129.55C321.886 129.625 321.864 129.691 321.821 129.748C321.778 129.805 321.718 129.85 321.642 129.882C321.565 129.914 321.477 129.93 321.378 129.93C321.232 129.93 321.115 129.899 321.026 129.838C320.937 129.777 320.884 129.691 320.867 129.582L321.14 129.555C321.153 129.609 321.179 129.65 321.22 129.677C321.26 129.704 321.312 129.718 321.377 129.718C321.444 129.718 321.497 129.704 321.538 129.677C321.579 129.65 321.599 129.616 321.599 129.575C321.599 129.541 321.586 129.513 321.559 129.491C321.534 129.469 321.493 129.451 321.438 129.44L321.235 129.397C321.121 129.373 321.036 129.333 320.981 129.277C320.927 129.22 320.899 129.148 320.9 129.061C320.899 128.987 320.919 128.924 320.96 128.87C321.001 128.816 321.057 128.774 321.13 128.744C321.203 128.714 321.287 128.699 321.382 128.699C321.521 128.699 321.631 128.729 321.712 128.789C321.792 128.848 321.842 128.928 321.861 129.03ZM323.308 128.715V128.932H322.603V128.715H323.308ZM322.78 129.907V128.602C322.78 128.522 322.796 128.455 322.829 128.402C322.863 128.349 322.908 128.309 322.964 128.283C323.021 128.256 323.083 128.243 323.152 128.243C323.201 128.243 323.244 128.247 323.282 128.255C323.319 128.262 323.347 128.269 323.365 128.276L323.31 128.493C323.298 128.489 323.283 128.486 323.265 128.482C323.246 128.478 323.226 128.476 323.204 128.476C323.152 128.476 323.115 128.488 323.093 128.514C323.072 128.539 323.061 128.574 323.061 128.621V129.907H322.78ZM324.025 129.93C323.909 129.93 323.808 129.904 323.722 129.853C323.637 129.802 323.571 129.73 323.524 129.638C323.477 129.546 323.454 129.439 323.454 129.315C323.454 129.192 323.477 129.084 323.524 128.992C323.571 128.899 323.637 128.827 323.722 128.776C323.808 128.725 323.909 128.699 324.025 128.699C324.141 128.699 324.242 128.725 324.328 128.776C324.413 128.827 324.479 128.899 324.526 128.992C324.573 129.084 324.596 129.192 324.596 129.315C324.596 129.439 324.573 129.546 324.526 129.638C324.479 129.73 324.413 129.802 324.328 129.853C324.242 129.904 324.141 129.93 324.025 129.93ZM324.027 129.705C324.09 129.705 324.142 129.688 324.185 129.653C324.227 129.618 324.259 129.571 324.28 129.512C324.301 129.453 324.311 129.387 324.311 129.315C324.311 129.242 324.301 129.176 324.28 129.117C324.259 129.057 324.227 129.01 324.185 128.975C324.142 128.94 324.09 128.922 324.027 128.922C323.962 128.922 323.908 128.94 323.865 128.975C323.823 129.01 323.791 129.057 323.77 129.117C323.749 129.176 323.739 129.242 323.739 129.315C323.739 129.387 323.749 129.453 323.77 129.512C323.791 129.571 323.823 129.618 323.865 129.653C323.908 129.688 323.962 129.705 324.027 129.705ZM324.856 129.907V128.715H325.129V128.913H325.141C325.163 128.845 325.2 128.792 325.253 128.754C325.306 128.717 325.367 128.698 325.435 128.698C325.451 128.698 325.468 128.699 325.487 128.7C325.507 128.701 325.523 128.703 325.536 128.705V128.964C325.524 128.96 325.505 128.956 325.48 128.953C325.454 128.949 325.43 128.948 325.406 128.948C325.355 128.948 325.309 128.959 325.268 128.981C325.227 129.003 325.196 129.033 325.172 129.072C325.149 129.111 325.137 129.155 325.137 129.206V129.907H324.856ZM326.404 128.715V128.932H325.719V128.715H326.404ZM325.888 128.429H326.169V129.548C326.169 129.586 326.174 129.615 326.186 129.635C326.198 129.655 326.213 129.668 326.232 129.675C326.252 129.683 326.273 129.686 326.296 129.686C326.314 129.686 326.33 129.685 326.344 129.682C326.359 129.68 326.371 129.678 326.378 129.675L326.426 129.895C326.411 129.9 326.389 129.906 326.361 129.912C326.334 129.918 326.3 129.922 326.26 129.923C326.19 129.925 326.127 129.914 326.07 129.891C326.014 129.867 325.969 129.831 325.936 129.781C325.903 129.731 325.887 129.669 325.888 129.595V128.429ZM326.967 129.208V129.907H326.686V128.318H326.961V128.917H326.975C327.003 128.85 327.046 128.797 327.105 128.758C327.164 128.719 327.239 128.699 327.33 128.699C327.412 128.699 327.485 128.717 327.546 128.751C327.608 128.786 327.655 128.837 327.689 128.903C327.723 128.97 327.74 129.052 327.74 129.148V129.907H327.459V129.191C327.459 129.111 327.438 129.049 327.397 129.004C327.356 128.959 327.299 128.937 327.225 128.937C327.175 128.937 327.131 128.948 327.091 128.969C327.053 128.991 327.022 129.021 327 129.062C326.978 129.102 326.967 129.151 326.967 129.208ZM328.955 129.931C328.88 129.931 328.812 129.917 328.751 129.89C328.691 129.863 328.643 129.823 328.608 129.769C328.574 129.716 328.556 129.65 328.556 129.572C328.556 129.505 328.569 129.449 328.594 129.405C328.618 129.361 328.652 129.326 328.695 129.3C328.738 129.274 328.786 129.254 328.84 129.24C328.895 129.226 328.951 129.216 329.009 129.21C329.079 129.203 329.135 129.196 329.179 129.19C329.222 129.184 329.254 129.175 329.273 129.163C329.293 129.15 329.304 129.13 329.304 129.103V129.098C329.304 129.04 329.286 128.994 329.252 128.962C329.217 128.93 329.167 128.914 329.102 128.914C329.033 128.914 328.978 128.929 328.938 128.959C328.898 128.989 328.871 129.025 328.857 129.066L328.595 129.028C328.616 128.956 328.65 128.895 328.698 128.847C328.745 128.798 328.803 128.761 328.872 128.737C328.941 128.712 329.017 128.699 329.1 128.699C329.158 128.699 329.215 128.706 329.272 128.719C329.329 128.733 329.381 128.755 329.428 128.786C329.475 128.817 329.513 128.858 329.541 128.911C329.57 128.964 329.584 129.03 329.584 129.109V129.907H329.314V129.743H329.305C329.288 129.776 329.264 129.807 329.233 129.836C329.202 129.865 329.164 129.888 329.117 129.905C329.071 129.922 329.017 129.931 328.955 129.931ZM329.028 129.724C329.084 129.724 329.133 129.713 329.175 129.691C329.216 129.668 329.248 129.638 329.27 129.601C329.293 129.564 329.304 129.523 329.304 129.479V129.339C329.296 129.346 329.281 129.353 329.259 129.359C329.239 129.365 329.215 129.371 329.189 129.375C329.164 129.38 329.138 129.384 329.113 129.388C329.087 129.391 329.065 129.394 329.047 129.397C329.005 129.403 328.967 129.412 328.934 129.425C328.901 129.438 328.875 129.456 328.856 129.479C328.837 129.502 328.827 129.531 328.827 129.568C328.827 129.619 328.846 129.658 328.884 129.685C328.922 129.711 328.97 129.724 329.028 129.724ZM330.17 129.208V129.907H329.889V128.715H330.157V128.917H330.171C330.199 128.851 330.242 128.798 330.302 128.758C330.363 128.719 330.438 128.699 330.527 128.699C330.609 128.699 330.681 128.717 330.742 128.752C330.803 128.787 330.851 128.838 330.884 128.905C330.919 128.972 330.935 129.053 330.935 129.148V129.907H330.654V129.191C330.654 129.112 330.633 129.049 330.592 129.004C330.551 128.959 330.494 128.937 330.422 128.937C330.373 128.937 330.329 128.948 330.291 128.969C330.253 128.991 330.223 129.021 330.202 129.062C330.18 129.102 330.17 129.151 330.17 129.208ZM331.762 129.93C331.646 129.93 331.545 129.904 331.46 129.853C331.374 129.802 331.308 129.73 331.261 129.638C331.214 129.546 331.191 129.439 331.191 129.315C331.191 129.192 331.214 129.084 331.261 128.992C331.308 128.899 331.374 128.827 331.46 128.776C331.545 128.725 331.646 128.699 331.762 128.699C331.879 128.699 331.979 128.725 332.065 128.776C332.15 128.827 332.216 128.899 332.263 128.992C332.31 129.084 332.333 129.192 332.333 129.315C332.333 129.439 332.31 129.546 332.263 129.638C332.216 129.73 332.15 129.802 332.065 129.853C331.979 129.904 331.879 129.93 331.762 129.93ZM331.764 129.705C331.827 129.705 331.88 129.688 331.922 129.653C331.964 129.618 331.996 129.571 332.017 129.512C332.038 129.453 332.048 129.387 332.048 129.315C332.048 129.242 332.038 129.176 332.017 129.117C331.996 129.057 331.964 129.01 331.922 128.975C331.88 128.94 331.827 128.922 331.764 128.922C331.699 128.922 331.645 128.94 331.602 128.975C331.56 129.01 331.528 129.057 331.507 129.117C331.486 129.176 331.476 129.242 331.476 129.315C331.476 129.387 331.486 129.453 331.507 129.512C331.528 129.571 331.56 129.618 331.602 129.653C331.645 129.688 331.699 129.705 331.764 129.705ZM333.189 128.715V128.932H332.504V128.715H333.189ZM332.673 128.429H332.954V129.548C332.954 129.586 332.959 129.615 332.971 129.635C332.983 129.655 332.998 129.668 333.017 129.675C333.036 129.683 333.058 129.686 333.081 129.686C333.098 129.686 333.115 129.685 333.129 129.682C333.144 129.68 333.155 129.678 333.163 129.675L333.21 129.895C333.195 129.9 333.174 129.906 333.146 129.912C333.119 129.918 333.085 129.922 333.045 129.923C332.975 129.925 332.911 129.914 332.855 129.891C332.799 129.867 332.754 129.831 332.721 129.781C332.688 129.731 332.672 129.669 332.673 129.595V128.429ZM333.752 129.208V129.907H333.471V128.318H333.746V128.917H333.76C333.788 128.85 333.831 128.797 333.889 128.758C333.948 128.719 334.023 128.699 334.114 128.699C334.197 128.699 334.269 128.717 334.331 128.751C334.393 128.786 334.44 128.837 334.474 128.903C334.508 128.97 334.525 129.052 334.525 129.148V129.907H334.244V129.191C334.244 129.111 334.223 129.049 334.182 129.004C334.141 128.959 334.084 128.937 334.01 128.937C333.96 128.937 333.916 128.948 333.876 128.969C333.837 128.991 333.807 129.021 333.785 129.062C333.763 129.102 333.752 129.151 333.752 129.208ZM335.361 129.93C335.241 129.93 335.138 129.905 335.051 129.856C334.965 129.805 334.898 129.734 334.852 129.643C334.805 129.551 334.782 129.442 334.782 129.318C334.782 129.195 334.805 129.088 334.852 128.995C334.899 128.902 334.965 128.829 335.049 128.778C335.133 128.725 335.232 128.699 335.346 128.699C335.42 128.699 335.489 128.711 335.554 128.735C335.62 128.758 335.678 128.794 335.728 128.844C335.779 128.893 335.818 128.955 335.847 129.031C335.876 129.107 335.891 129.197 335.891 129.301V129.388H334.914V129.198H335.622C335.621 129.144 335.609 129.097 335.587 129.055C335.564 129.012 335.532 128.979 335.491 128.955C335.451 128.93 335.404 128.918 335.35 128.918C335.293 128.918 335.242 128.932 335.199 128.96C335.155 128.987 335.121 129.024 335.097 129.069C335.073 129.113 335.061 129.162 335.061 129.215V129.381C335.061 129.45 335.073 129.509 335.099 129.559C335.124 129.608 335.159 129.646 335.205 129.672C335.25 129.698 335.304 129.711 335.365 129.711C335.406 129.711 335.443 129.705 335.476 129.694C335.509 129.682 335.537 129.665 335.562 129.642C335.586 129.619 335.604 129.591 335.617 129.558L335.879 129.587C335.863 129.656 335.831 129.717 335.785 129.769C335.738 129.82 335.679 129.86 335.608 129.888C335.536 129.916 335.453 129.93 335.361 129.93ZM336.15 129.907V128.715H336.423V128.913H336.435C336.457 128.845 336.494 128.792 336.547 128.754C336.6 128.717 336.661 128.698 336.729 128.698C336.745 128.698 336.762 128.699 336.781 128.7C336.801 128.701 336.817 128.703 336.83 128.705V128.964C336.818 128.96 336.799 128.956 336.773 128.953C336.748 128.949 336.724 128.948 336.7 128.948C336.649 128.948 336.602 128.959 336.562 128.981C336.521 129.003 336.489 129.033 336.466 129.072C336.443 129.111 336.431 129.155 336.431 129.206V129.907H336.15ZM338.045 129.928C337.951 129.928 337.867 129.904 337.793 129.856C337.719 129.807 337.661 129.738 337.618 129.646C337.575 129.554 337.554 129.443 337.554 129.312C337.554 129.18 337.575 129.068 337.619 128.977C337.663 128.886 337.722 128.816 337.796 128.77C337.871 128.723 337.954 128.699 338.046 128.699C338.115 128.699 338.173 128.711 338.218 128.735C338.263 128.758 338.299 128.786 338.325 128.82C338.351 128.852 338.372 128.883 338.386 128.912H338.398V128.318H338.68V129.907H338.403V129.719H338.386C338.372 129.748 338.351 129.779 338.323 129.811C338.296 129.843 338.26 129.871 338.215 129.894C338.17 129.916 338.113 129.928 338.045 129.928ZM338.123 129.697C338.183 129.697 338.233 129.681 338.275 129.649C338.317 129.617 338.349 129.571 338.371 129.513C338.392 129.455 338.403 129.388 338.403 129.311C338.403 129.234 338.392 129.167 338.371 129.11C338.349 129.053 338.318 129.009 338.276 128.977C338.235 128.946 338.184 128.93 338.123 128.93C338.061 128.93 338.008 128.946 337.966 128.979C337.924 129.011 337.893 129.056 337.872 129.114C337.851 129.171 337.84 129.237 337.84 129.311C337.84 129.385 337.851 129.452 337.872 129.51C337.893 129.568 337.925 129.614 337.967 129.648C338.01 129.681 338.062 129.697 338.123 129.697ZM339.344 129.931C339.268 129.931 339.2 129.917 339.14 129.89C339.08 129.863 339.032 129.823 338.997 129.769C338.962 129.716 338.945 129.65 338.945 129.572C338.945 129.505 338.957 129.449 338.982 129.405C339.007 129.361 339.041 129.326 339.084 129.3C339.127 129.274 339.175 129.254 339.229 129.24C339.283 129.226 339.339 129.216 339.397 129.21C339.467 129.203 339.524 129.196 339.567 129.19C339.611 129.184 339.642 129.175 339.662 129.163C339.682 129.15 339.692 129.13 339.692 129.103V129.098C339.692 129.04 339.675 128.994 339.64 128.962C339.605 128.93 339.556 128.914 339.49 128.914C339.422 128.914 339.367 128.929 339.327 128.959C339.287 128.989 339.26 129.025 339.246 129.066L338.984 129.028C339.004 128.956 339.038 128.895 339.086 128.847C339.134 128.798 339.192 128.761 339.261 128.737C339.329 128.712 339.406 128.699 339.489 128.699C339.546 128.699 339.603 128.706 339.66 128.719C339.717 128.733 339.769 128.755 339.816 128.786C339.863 128.817 339.901 128.858 339.93 128.911C339.959 128.964 339.973 129.03 339.973 129.109V129.907H339.703V129.743H339.694C339.677 129.776 339.653 129.807 339.622 129.836C339.591 129.865 339.552 129.888 339.506 129.905C339.46 129.922 339.406 129.931 339.344 129.931ZM339.417 129.724C339.473 129.724 339.522 129.713 339.563 129.691C339.605 129.668 339.637 129.638 339.659 129.601C339.682 129.564 339.693 129.523 339.693 129.479V129.339C339.684 129.346 339.669 129.353 339.648 129.359C339.627 129.365 339.604 129.371 339.578 129.375C339.552 129.38 339.527 129.384 339.501 129.388C339.476 129.391 339.454 129.394 339.435 129.397C339.393 129.403 339.356 129.412 339.323 129.425C339.29 129.438 339.264 129.456 339.244 129.479C339.225 129.502 339.216 129.531 339.216 129.568C339.216 129.619 339.235 129.658 339.272 129.685C339.31 129.711 339.358 129.724 339.417 129.724ZM340.409 130.354C340.371 130.354 340.336 130.351 340.303 130.344C340.271 130.339 340.245 130.332 340.226 130.324L340.291 130.105C340.332 130.117 340.369 130.123 340.401 130.122C340.433 130.122 340.461 130.112 340.485 130.092C340.51 130.073 340.531 130.041 340.548 129.996L340.572 129.932L340.14 128.715H340.438L340.713 129.615H340.725L341.001 128.715H341.299L340.822 130.051C340.8 130.114 340.77 130.168 340.734 130.213C340.697 130.259 340.652 130.293 340.599 130.317C340.546 130.342 340.483 130.354 340.409 130.354ZM342.599 129.93C342.483 129.93 342.382 129.904 342.297 129.853C342.211 129.802 342.145 129.73 342.098 129.638C342.052 129.546 342.028 129.439 342.028 129.315C342.028 129.192 342.052 129.084 342.098 128.992C342.145 128.899 342.211 128.827 342.297 128.776C342.382 128.725 342.483 128.699 342.599 128.699C342.716 128.699 342.817 128.725 342.902 128.776C342.987 128.827 343.053 128.899 343.1 128.992C343.147 129.084 343.171 129.192 343.171 129.315C343.171 129.439 343.147 129.546 343.1 129.638C343.053 129.73 342.987 129.802 342.902 129.853C342.817 129.904 342.716 129.93 342.599 129.93ZM342.601 129.705C342.664 129.705 342.717 129.688 342.759 129.653C342.802 129.618 342.833 129.571 342.854 129.512C342.875 129.453 342.886 129.387 342.886 129.315C342.886 129.242 342.875 129.176 342.854 129.117C342.833 129.057 342.802 129.01 342.759 128.975C342.717 128.94 342.664 128.922 342.601 128.922C342.536 128.922 342.482 128.94 342.44 128.975C342.397 129.01 342.365 129.057 342.344 129.117C342.323 129.176 342.313 129.242 342.313 129.315C342.313 129.387 342.323 129.453 342.344 129.512C342.365 129.571 342.397 129.618 342.44 129.653C342.482 129.688 342.536 129.705 342.601 129.705ZM344.035 128.715V128.932H343.33V128.715H344.035ZM343.506 129.907V128.602C343.506 128.522 343.523 128.455 343.556 128.402C343.589 128.349 343.634 128.309 343.691 128.283C343.747 128.256 343.81 128.243 343.879 128.243C343.927 128.243 343.97 128.247 344.008 128.255C344.046 128.262 344.074 128.269 344.092 128.276L344.036 128.493C344.024 128.489 344.009 128.486 343.991 128.482C343.973 128.478 343.953 128.476 343.931 128.476C343.878 128.476 343.841 128.488 343.82 128.514C343.798 128.539 343.788 128.574 343.788 128.621V129.907H343.506ZM345.086 129.907L344.749 128.715H345.036L345.245 129.553H345.256L345.47 128.715H345.754L345.968 129.548H345.979L346.186 128.715H346.473L346.135 129.907H345.843L345.619 129.101H345.603L345.38 129.907H345.086ZM347.177 129.93C347.06 129.93 346.959 129.904 346.874 129.853C346.789 129.802 346.722 129.73 346.675 129.638C346.629 129.546 346.606 129.439 346.606 129.315C346.606 129.192 346.629 129.084 346.675 128.992C346.722 128.899 346.789 128.827 346.874 128.776C346.959 128.725 347.06 128.699 347.177 128.699C347.293 128.699 347.394 128.725 347.479 128.776C347.565 128.827 347.631 128.899 347.677 128.992C347.724 129.084 347.748 129.192 347.748 129.315C347.748 129.439 347.724 129.546 347.677 129.638C347.631 129.73 347.565 129.802 347.479 129.853C347.394 129.904 347.293 129.93 347.177 129.93ZM347.178 129.705C347.241 129.705 347.294 129.688 347.337 129.653C347.379 129.618 347.411 129.571 347.431 129.512C347.452 129.453 347.463 129.387 347.463 129.315C347.463 129.242 347.452 129.176 347.431 129.117C347.411 129.057 347.379 129.01 347.337 128.975C347.294 128.94 347.241 128.922 347.178 128.922C347.114 128.922 347.06 128.94 347.017 128.975C346.974 129.01 346.943 129.057 346.921 129.117C346.901 129.176 346.89 129.242 346.89 129.315C346.89 129.387 346.901 129.453 346.921 129.512C346.943 129.571 346.974 129.618 347.017 129.653C347.06 129.688 347.114 129.705 347.178 129.705ZM348.289 129.208V129.907H348.008V128.715H348.277V128.917H348.291C348.318 128.851 348.362 128.798 348.422 128.758C348.482 128.719 348.557 128.699 348.646 128.699C348.728 128.699 348.8 128.717 348.861 128.752C348.922 128.787 348.97 128.838 349.004 128.905C349.038 128.972 349.055 129.053 349.054 129.148V129.907H348.773V129.191C348.773 129.112 348.753 129.049 348.711 129.004C348.67 128.959 348.614 128.937 348.541 128.937C348.492 128.937 348.448 128.948 348.41 128.969C348.372 128.991 348.343 129.021 348.321 129.062C348.3 129.102 348.289 129.151 348.289 129.208ZM349.804 129.928C349.71 129.928 349.626 129.904 349.552 129.856C349.478 129.807 349.42 129.738 349.377 129.646C349.334 129.554 349.313 129.443 349.313 129.312C349.313 129.18 349.334 129.068 349.378 128.977C349.422 128.886 349.481 128.816 349.556 128.77C349.63 128.723 349.713 128.699 349.805 128.699C349.874 128.699 349.932 128.711 349.977 128.735C350.022 128.758 350.058 128.786 350.084 128.82C350.11 128.852 350.131 128.883 350.145 128.912H350.157V128.318H350.439V129.907H350.162V129.719H350.145C350.131 129.748 350.11 129.779 350.082 129.811C350.055 129.843 350.019 129.871 349.974 129.894C349.929 129.916 349.872 129.928 349.804 129.928ZM349.882 129.697C349.942 129.697 349.992 129.681 350.034 129.649C350.076 129.617 350.108 129.571 350.13 129.513C350.152 129.455 350.162 129.388 350.162 129.311C350.162 129.234 350.152 129.167 350.13 129.11C350.109 129.053 350.077 129.009 350.035 128.977C349.994 128.946 349.943 128.93 349.882 128.93C349.82 128.93 349.767 128.946 349.726 128.979C349.684 129.011 349.652 129.056 349.631 129.114C349.61 129.171 349.599 129.237 349.599 129.311C349.599 129.385 349.61 129.452 349.631 129.51C349.653 129.568 349.684 129.614 349.726 129.648C349.769 129.681 349.821 129.697 349.882 129.697ZM351.288 129.93C351.168 129.93 351.065 129.905 350.978 129.856C350.891 129.805 350.825 129.734 350.778 129.643C350.732 129.551 350.709 129.442 350.709 129.318C350.709 129.195 350.732 129.088 350.778 128.995C350.826 128.902 350.891 128.829 350.976 128.778C351.06 128.725 351.159 128.699 351.273 128.699C351.346 128.699 351.416 128.711 351.481 128.735C351.546 128.758 351.604 128.794 351.655 128.844C351.705 128.893 351.745 128.955 351.774 129.031C351.803 129.107 351.818 129.197 351.818 129.301V129.388H350.841V129.198H351.548C351.548 129.144 351.536 129.097 351.513 129.055C351.491 129.012 351.459 128.979 351.418 128.955C351.378 128.93 351.33 128.918 351.277 128.918C351.219 128.918 351.169 128.932 351.125 128.96C351.082 128.987 351.048 129.024 351.024 129.069C351 129.113 350.988 129.162 350.987 129.215V129.381C350.987 129.45 351 129.509 351.025 129.559C351.051 129.608 351.086 129.646 351.132 129.672C351.177 129.698 351.23 129.711 351.291 129.711C351.332 129.711 351.369 129.705 351.402 129.694C351.435 129.682 351.464 129.665 351.488 129.642C351.513 129.619 351.531 129.591 351.544 129.558L351.806 129.587C351.789 129.656 351.758 129.717 351.711 129.769C351.665 129.82 351.606 129.86 351.534 129.888C351.462 129.916 351.38 129.93 351.288 129.93ZM352.077 129.907V128.715H352.349V128.913H352.362C352.384 128.845 352.421 128.792 352.474 128.754C352.527 128.717 352.588 128.698 352.656 128.698C352.671 128.698 352.689 128.699 352.708 128.7C352.728 128.701 352.744 128.703 352.757 128.705V128.964C352.745 128.96 352.726 128.956 352.7 128.953C352.675 128.949 352.65 128.948 352.626 128.948C352.575 128.948 352.529 128.959 352.488 128.981C352.448 129.003 352.416 129.033 352.393 129.072C352.37 129.111 352.358 129.155 352.358 129.206V129.907H352.077ZM353.002 129.924C352.955 129.924 352.914 129.907 352.881 129.874C352.847 129.841 352.831 129.801 352.831 129.753C352.831 129.707 352.847 129.667 352.881 129.634C352.914 129.6 352.955 129.584 353.002 129.584C353.047 129.584 353.087 129.6 353.12 129.634C353.155 129.667 353.172 129.707 353.172 129.753C353.172 129.785 353.164 129.813 353.148 129.839C353.132 129.865 353.111 129.886 353.086 129.901C353.06 129.916 353.032 129.924 353.002 129.924ZM312.706 132.907L312.967 131.318H313.184L312.924 132.907H312.706ZM311.99 132.497L312.027 132.28H313.249L313.213 132.497H311.99ZM312.148 132.907L312.408 131.318H312.626L312.365 132.907H312.148ZM312.082 131.945L312.118 131.727H313.341L313.305 131.945H312.082ZM313.535 132.907V131.715H313.804V131.917H313.818C313.843 131.849 313.884 131.796 313.941 131.757C313.999 131.719 314.067 131.699 314.147 131.699C314.228 131.699 314.296 131.719 314.351 131.758C314.407 131.797 314.446 131.85 314.469 131.917H314.481C314.508 131.851 314.552 131.798 314.615 131.759C314.678 131.719 314.753 131.699 314.839 131.699C314.949 131.699 315.038 131.734 315.108 131.803C315.177 131.873 315.212 131.974 315.212 132.107V132.907H314.93V132.15C314.93 132.076 314.91 132.022 314.871 131.988C314.832 131.953 314.783 131.936 314.727 131.936C314.659 131.936 314.606 131.957 314.567 132C314.53 132.042 314.511 132.096 314.511 132.163V132.907H314.235V132.139C314.235 132.077 314.217 132.028 314.179 131.991C314.143 131.954 314.095 131.936 314.035 131.936C313.995 131.936 313.958 131.946 313.925 131.967C313.892 131.987 313.865 132.016 313.846 132.053C313.826 132.09 313.816 132.133 313.816 132.182V132.907H313.535ZM316.038 132.93C315.922 132.93 315.821 132.904 315.736 132.853C315.65 132.802 315.584 132.73 315.537 132.638C315.49 132.546 315.467 132.439 315.467 132.315C315.467 132.192 315.49 132.084 315.537 131.992C315.584 131.899 315.65 131.827 315.736 131.776C315.821 131.725 315.922 131.699 316.038 131.699C316.155 131.699 316.256 131.725 316.341 131.776C316.426 131.827 316.492 131.899 316.539 131.992C316.586 132.084 316.609 132.192 316.609 132.315C316.609 132.439 316.586 132.546 316.539 132.638C316.492 132.73 316.426 132.802 316.341 132.853C316.256 132.904 316.155 132.93 316.038 132.93ZM316.04 132.705C316.103 132.705 316.156 132.688 316.198 132.653C316.241 132.618 316.272 132.571 316.293 132.512C316.314 132.453 316.325 132.387 316.325 132.315C316.325 132.242 316.314 132.176 316.293 132.117C316.272 132.057 316.241 132.01 316.198 131.975C316.156 131.94 316.103 131.922 316.04 131.922C315.975 131.922 315.921 131.94 315.878 131.975C315.836 132.01 315.804 132.057 315.783 132.117C315.762 132.176 315.752 132.242 315.752 132.315C315.752 132.387 315.762 132.453 315.783 132.512C315.804 132.571 315.836 132.618 315.878 132.653C315.921 132.688 315.975 132.705 316.04 132.705ZM316.87 132.907V131.715H317.142V131.913H317.154C317.176 131.845 317.213 131.792 317.266 131.754C317.32 131.717 317.38 131.698 317.449 131.698C317.464 131.698 317.481 131.699 317.501 131.7C317.52 131.701 317.537 131.703 317.549 131.705V131.964C317.538 131.96 317.519 131.956 317.493 131.953C317.467 131.949 317.443 131.948 317.419 131.948C317.368 131.948 317.322 131.959 317.281 131.981C317.241 132.003 317.209 132.033 317.186 132.072C317.162 132.111 317.151 132.155 317.151 132.206V132.907H316.87ZM318.039 132.208V132.907H317.758V131.715H318.026V131.917H318.04C318.068 131.851 318.112 131.798 318.172 131.758C318.232 131.719 318.307 131.699 318.396 131.699C318.478 131.699 318.55 131.717 318.611 131.752C318.672 131.787 318.72 131.838 318.754 131.905C318.788 131.972 318.804 132.053 318.804 132.148V132.907H318.523V132.191C318.523 132.112 318.502 132.049 318.461 132.004C318.42 131.959 318.363 131.937 318.291 131.937C318.242 131.937 318.198 131.948 318.16 131.969C318.122 131.991 318.092 132.021 318.071 132.062C318.049 132.102 318.039 132.151 318.039 132.208ZM319.111 132.907V131.715H319.392V132.907H319.111ZM319.253 131.546C319.208 131.546 319.17 131.531 319.138 131.501C319.106 131.471 319.09 131.435 319.09 131.394C319.09 131.351 319.106 131.315 319.138 131.286C319.17 131.256 319.208 131.241 319.253 131.241C319.298 131.241 319.336 131.256 319.367 131.286C319.4 131.315 319.416 131.351 319.416 131.394C319.416 131.435 319.4 131.471 319.367 131.501C319.336 131.531 319.298 131.546 319.253 131.546ZM319.984 132.208V132.907H319.703V131.715H319.972V131.917H319.985C320.013 131.851 320.057 131.798 320.117 131.758C320.177 131.719 320.252 131.699 320.341 131.699C320.423 131.699 320.495 131.717 320.556 131.752C320.617 131.787 320.665 131.838 320.699 131.905C320.733 131.972 320.75 132.053 320.749 132.148V132.907H320.468V132.191C320.468 132.112 320.447 132.049 320.406 132.004C320.365 131.959 320.309 131.937 320.236 131.937C320.187 131.937 320.143 131.948 320.105 131.969C320.067 131.991 320.037 132.021 320.016 132.062C319.995 132.102 319.984 132.151 319.984 132.208ZM321.575 133.379C321.474 133.379 321.387 133.365 321.315 133.337C321.242 133.311 321.184 133.274 321.14 133.229C321.096 133.183 321.066 133.133 321.049 133.077L321.302 133.016C321.313 133.039 321.33 133.062 321.351 133.085C321.373 133.108 321.402 133.128 321.439 133.143C321.476 133.158 321.523 133.166 321.579 133.166C321.659 133.166 321.725 133.147 321.777 133.108C321.83 133.069 321.856 133.006 321.856 132.918V132.693H321.842C321.827 132.722 321.806 132.751 321.778 132.782C321.751 132.812 321.714 132.838 321.669 132.859C321.624 132.879 321.567 132.89 321.499 132.89C321.407 132.89 321.324 132.868 321.25 132.825C321.176 132.782 321.117 132.717 321.073 132.631C321.029 132.545 321.008 132.437 321.008 132.307C321.008 132.176 321.029 132.066 321.073 131.976C321.117 131.885 321.176 131.816 321.25 131.77C321.325 131.723 321.408 131.699 321.5 131.699C321.569 131.699 321.627 131.711 321.672 131.735C321.717 131.758 321.754 131.786 321.78 131.82C321.807 131.852 321.828 131.883 321.842 131.912H321.857V131.715H322.134V132.926C322.134 133.028 322.11 133.112 322.061 133.179C322.013 133.246 321.946 133.296 321.862 133.329C321.778 133.362 321.682 133.379 321.575 133.379ZM321.577 132.669C321.637 132.669 321.687 132.655 321.729 132.626C321.771 132.597 321.803 132.555 321.825 132.501C321.846 132.447 321.857 132.381 321.857 132.305C321.857 132.23 321.846 132.165 321.825 132.108C321.803 132.052 321.772 132.008 321.73 131.977C321.689 131.946 321.638 131.93 321.577 131.93C321.515 131.93 321.462 131.946 321.42 131.979C321.379 132.011 321.347 132.056 321.326 132.113C321.305 132.169 321.294 132.233 321.294 132.305C321.294 132.378 321.305 132.442 321.326 132.497C321.347 132.551 321.379 132.594 321.421 132.624C321.464 132.654 321.516 132.669 321.577 132.669Z" fill="#434960"/><g clip-path="url(#clip12_5232_97155)"><path d="M313.967 137.746V137.764C314.225 137.764 314.434 137.973 314.434 138.231C314.434 138.507 314.27 138.774 314.058 139.005C313.85 139.232 313.599 139.42 313.421 139.547C313.242 139.42 312.991 139.232 312.784 139.005C312.572 138.774 312.407 138.507 312.407 138.231C312.407 137.973 312.617 137.764 312.875 137.764V137.746L312.875 137.764C312.95 137.764 313.023 137.782 313.09 137.817C313.157 137.851 313.214 137.902 313.257 137.963L313.257 137.963L313.406 138.175L313.421 138.196L313.436 138.175L313.585 137.963L313.585 137.963C313.628 137.902 313.685 137.851 313.751 137.817C313.818 137.782 313.892 137.764 313.967 137.764L313.967 137.746ZM313.967 137.746C314.235 137.746 314.452 137.963 314.452 138.231C314.452 138.798 313.78 139.316 313.421 139.569L313.57 137.953C313.614 137.889 313.674 137.837 313.743 137.801C313.812 137.765 313.889 137.746 313.967 137.746ZM313.421 139.813L313.41 139.806C313.198 139.659 312.893 139.442 312.641 139.174C312.39 138.906 312.189 138.585 312.189 138.231C312.189 138.049 312.261 137.875 312.39 137.746C312.518 137.618 312.693 137.545 312.875 137.545L313.421 139.813ZM313.421 139.813L313.431 139.806M313.421 139.813L313.431 139.806M313.431 139.806C313.644 139.659 313.948 139.442 314.2 139.174C314.452 138.906 314.653 138.585 314.653 138.231C314.653 138.141 314.635 138.052 314.601 137.969C314.566 137.885 314.516 137.81 314.452 137.746C314.388 137.682 314.313 137.632 314.229 137.597C314.146 137.563 314.057 137.545 313.967 137.545C313.857 137.545 313.748 137.572 313.651 137.623M313.431 139.806L313.651 137.623M313.651 137.623C313.56 137.67 313.482 137.736 313.421 137.817M313.651 137.623L313.421 137.817M313.421 137.817C313.36 137.736 313.281 137.67 313.191 137.623C313.093 137.572 312.985 137.545 312.875 137.545L313.421 137.817Z" fill="#434960" stroke="#434960" stroke-width="0.03642"/></g><path d="M316.832 139.514H316.108L316.455 139.121C316.687 138.874 316.812 138.716 316.812 138.486C316.812 138.221 316.6 137.994 316.256 137.994C315.922 137.994 315.675 138.216 315.675 138.542H315.927C315.927 138.349 316.047 138.226 316.238 138.226C316.424 138.226 316.542 138.356 316.542 138.519C316.542 138.695 316.432 138.823 316.256 139.011L315.718 139.595V139.743H316.832V139.514ZM318.292 139.363V139.139H318.058V138.02H317.859L316.969 139.182V139.363H317.8V139.743H318.058V139.363H318.292ZM317.26 139.139L317.8 138.41V139.139H317.26Z" fill="#434960"/><g clip-path="url(#clip13_5232_97155)"><path d="M323.618 137.441C323.458 137.441 323.3 137.472 323.153 137.533C323.006 137.594 322.872 137.684 322.759 137.796C322.532 138.024 322.404 138.333 322.404 138.655C322.403 138.935 322.5 139.207 322.678 139.423L322.435 139.666C322.419 139.683 322.407 139.704 322.403 139.728C322.398 139.752 322.401 139.776 322.41 139.798C322.42 139.82 322.436 139.838 322.457 139.851C322.477 139.863 322.501 139.869 322.525 139.868H323.618C323.939 139.868 324.248 139.741 324.476 139.513C324.703 139.285 324.831 138.977 324.831 138.655C324.831 138.333 324.703 138.024 324.476 137.796C324.248 137.569 323.939 137.441 323.618 137.441ZM323.618 139.626H322.818L322.931 139.513C322.953 139.49 322.966 139.459 322.966 139.427C322.966 139.395 322.953 139.364 322.931 139.342C322.772 139.183 322.673 138.974 322.651 138.75C322.628 138.527 322.684 138.303 322.809 138.116C322.934 137.929 323.119 137.791 323.334 137.726C323.549 137.66 323.78 137.672 323.988 137.757C324.195 137.843 324.367 137.998 324.473 138.196C324.579 138.394 324.613 138.623 324.569 138.843C324.526 139.064 324.407 139.262 324.233 139.405C324.06 139.547 323.842 139.625 323.618 139.626Z" fill="#434960"/></g><path d="M327.52 139.185C327.52 138.797 327.188 138.565 326.791 138.652L327.298 138.02H327.005L326.485 138.662C326.332 138.851 326.258 138.986 326.258 139.198C326.258 139.511 326.523 139.766 326.887 139.766C327.257 139.766 327.52 139.511 327.52 139.185ZM326.528 139.19C326.528 138.95 326.689 138.828 326.89 138.828C327.094 138.828 327.252 138.966 327.252 139.192C327.252 139.407 327.094 139.549 326.89 139.549C326.689 139.549 326.528 139.407 326.528 139.19Z" fill="#434960"/></g></g><g filter="url(#filter6_dd_5232_97155)"><g clip-path="url(#clip14_5232_97155)"><rect x="309.777" y="145.483" width="51.8985" height="68.4688" rx="0.7284" fill="white"/><rect x="309.777" y="145.483" width="51.8985" height="48.4386" fill="url(#pattern5)"/></g></g></g></g><g filter="url(#filter7_dd_5232_97155)"><g clip-path="url(#clip15_5232_97155)"><rect x="21.0996" y="45.1873" width="185.918" height="173.883" rx="3.5434" transform="rotate(3 21.0996 45.1873)" fill="white"/><g clip-path="url(#clip16_5232_97155)"><rect x="26.9668" y="51.7043" width="173.516" height="140.322" rx="1.7717" transform="rotate(3 26.9668 51.7043)" fill="#FCF0EF"/><g filter="url(#filter8_dd_5232_97155)"><g clip-path="url(#clip17_5232_97155)"><rect x="54.3984" y="73.2319" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 54.3984 73.2319)" fill="white"/><g clip-path="url(#clip18_5232_97155)"><rect width="35.7468" height="45.4048" transform="translate(54.3984 73.2319) rotate(3)" fill="#FFCAC7"/><circle cx="92.9208" cy="121.059" r="26.0999" transform="rotate(3 92.9208 121.059)" fill="#FCA7A1"/></g><rect x="62.0449" y="121.348" width="15.4354" height="2.10483" rx="0.280644" transform="rotate(3 62.0449 121.348)" fill="#D0D1D7"/></g></g><g filter="url(#filter9_dd_5232_97155)"><g clip-path="url(#clip19_5232_97155)"><rect x="51.5781" y="127.039" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 51.5781 127.039)" fill="white"/><rect width="35.7468" height="45.4048" transform="translate(51.5781 127.039) rotate(3)" fill="#FFCAC7"/><rect x="59.2246" y="175.156" width="15.4354" height="2.10483" rx="0.280644" transform="rotate(3 59.2246 175.156)" fill="#D0D1D7"/></g><rect x="51.6375" y="127.105" width="35.6214" height="51.8745" rx="0.438996" transform="rotate(3 51.6375 127.105)" stroke="#F3F4F5" stroke-width="0.125427"/></g><g filter="url(#filter10_dd_5232_97155)"><g clip-path="url(#clip20_5232_97155)"><rect x="48.7578" y="180.847" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 48.7578 180.847)" fill="white"/><rect width="35.7468" height="45.4048" transform="translate(48.7578 180.847) rotate(3)" fill="#FFCAC7"/></g><rect x="48.8172" y="180.913" width="35.6214" height="51.8745" rx="0.438996" transform="rotate(3 48.8172 180.913)" stroke="#F3F4F5" stroke-width="0.125427"/></g><g filter="url(#filter11_dd_5232_97155)"><g clip-path="url(#clip21_5232_97155)"><rect x="92.6016" y="75.2346" width="35.7468" height="39.9589" rx="0.50171" transform="rotate(3 92.6016 75.2346)" fill="white"/><rect width="35.7468" height="33.3637" transform="translate(92.6016 75.2346) rotate(3)" fill="#FFCAC7"/><rect x="100.879" y="111.326" width="15.4354" height="2.10483" rx="0.280644" transform="rotate(3 100.879 111.326)" fill="#D0D1D7"/></g></g><g filter="url(#filter12_dd_5232_97155)"><g clip-path="url(#clip22_5232_97155)"><rect x="90.4121" y="117.018" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 90.4121 117.018)" fill="white"/><g clip-path="url(#clip23_5232_97155)"><rect width="35.7468" height="45.4048" transform="translate(90.4121 117.018) rotate(3)" fill="#FFCAC7"/><rect x="88.2676" y="164.285" width="34.7999" height="38.1676" transform="rotate(-42 88.2676 164.285)" fill="#FCA7A1"/></g><rect x="98.0586" y="165.134" width="15.4354" height="2.10483" rx="0.280644" transform="rotate(3 98.0586 165.134)" fill="#D0D1D7"/></g><rect x="90.4715" y="117.083" width="35.6214" height="51.8745" rx="0.438996" transform="rotate(3 90.4715 117.083)" stroke="#F3F4F5" stroke-width="0.125427"/></g><g filter="url(#filter13_dd_5232_97155)"><g clip-path="url(#clip24_5232_97155)"><rect x="87.5918" y="170.825" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 87.5918 170.825)" fill="white"/><rect width="35.7468" height="45.4048" transform="translate(87.5918 170.825) rotate(3)" fill="#FFCAC7"/></g><rect x="87.6511" y="170.891" width="35.6214" height="51.8745" rx="0.438996" transform="rotate(3 87.6511 170.891)" stroke="#F3F4F5" stroke-width="0.125427"/></g><g filter="url(#filter14_dd_5232_97155)"><g clip-path="url(#clip25_5232_97155)"><rect x="130.805" y="77.2366" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 130.805 77.2366)" fill="white"/><g clip-path="url(#clip26_5232_97155)"><rect width="35.7468" height="45.4048" transform="translate(130.805 77.2366) rotate(3)" fill="#FFCAC7"/><circle cx="165.752" cy="120.239" r="17.9612" transform="rotate(3 165.752 120.239)" fill="#FCA7A1"/></g><rect x="138.451" y="125.353" width="15.4354" height="2.10483" rx="0.280644" transform="rotate(3 138.451 125.353)" fill="#D0D1D7"/></g><rect x="130.864" y="77.3025" width="35.6214" height="51.8745" rx="0.438996" transform="rotate(3 130.864 77.3025)" stroke="#F3F4F5" stroke-width="0.125427"/></g><g filter="url(#filter15_dd_5232_97155)"><g clip-path="url(#clip27_5232_97155)"><rect x="127.984" y="131.044" width="35.7468" height="49.6168" rx="0.50171" transform="rotate(3 127.984 131.044)" fill="white"/><rect x="127.984" y="131.044" width="35.7468" height="43.0216" transform="rotate(3 127.984 131.044)" fill="#FFCAC7"/><rect x="135.756" y="176.78" width="15.4354" height="2.10483" rx="0.280644" transform="rotate(3 135.756 176.78)" fill="#D0D1D7"/></g><rect x="128.044" y="131.11" width="35.6214" height="49.4913" rx="0.438996" transform="rotate(3 128.044 131.11)" stroke="#F3F4F5" stroke-width="0.125427"/></g><g filter="url(#filter16_dd_5232_97155)"><g clip-path="url(#clip28_5232_97155)"><rect x="125.289" y="182.472" width="35.7468" height="49.6168" rx="0.50171" transform="rotate(3 125.289 182.472)" fill="white"/><rect x="125.289" y="182.472" width="35.7468" height="43.0216" transform="rotate(3 125.289 182.472)" fill="#FFCAC7"/></g><rect x="125.348" y="182.538" width="35.6214" height="49.4913" rx="0.438996" transform="rotate(3 125.348 182.538)" stroke="#F3F4F5" stroke-width="0.125427"/></g></g><path d="M69.8666 211.276C71.5238 211.363 72.6073 210.574 72.6779 209.228L72.6781 209.223C72.7328 208.179 72.1552 207.563 70.7152 207.18L69.9811 206.983C69.1285 206.756 68.795 206.431 68.8206 205.943L68.8208 205.938C68.8517 205.349 69.4165 205.018 70.1829 205.058C70.9734 205.095 71.4658 205.505 71.5239 206.046L71.5305 206.104L72.7135 206.166L72.712 206.103C72.684 204.988 71.7748 204.1 70.2421 204.019C68.7478 203.941 67.6408 204.719 67.576 205.954L67.5758 205.959C67.5218 206.989 68.1002 207.682 69.4976 208.053L70.232 208.245C71.1128 208.483 71.4562 208.804 71.4301 209.302L71.4298 209.307C71.3987 209.901 70.7928 210.282 69.9594 210.238C69.0829 210.193 68.4851 209.777 68.458 209.194L68.4558 209.146L67.2584 209.084L67.2596 209.151C67.2833 210.348 68.2621 211.192 69.8666 211.276ZM74.77 205.232C75.158 205.252 75.4952 204.957 75.5158 204.565C75.5361 204.177 75.2318 203.844 74.8438 203.823C74.4511 203.803 74.1136 204.102 74.0933 204.49C74.0727 204.883 74.3773 205.211 74.77 205.232ZM73.851 211.312L75.0436 211.374L75.318 206.139L74.1254 206.077L73.851 211.312ZM76.5092 211.451L77.7019 211.514L77.8688 208.328C77.9029 207.677 78.3601 207.202 78.9492 207.232C79.5335 207.263 79.8739 207.641 79.8436 208.221L79.6656 211.616L80.8582 211.679L81.0296 208.408C81.061 207.809 81.4877 207.365 82.1103 207.398C82.7474 207.431 83.0406 207.793 83.005 208.473L82.8315 211.782L84.0241 211.845L84.2129 208.243C84.2699 207.156 83.6715 206.476 82.6178 206.421C81.8898 206.383 81.2667 206.725 80.983 207.281L80.9016 207.277C80.7068 206.686 80.2086 206.295 79.4998 206.257C78.8149 206.222 78.2707 206.524 78.015 207.097L77.9335 207.093L77.9762 206.278L76.7836 206.216L76.5092 211.451ZM85.3459 213.658L86.5385 213.72L86.6733 211.148L86.7548 211.152C86.9981 211.732 87.5783 212.118 88.2967 212.155C89.6234 212.225 90.5012 211.238 90.5886 209.571L90.5891 209.562C90.6759 207.905 89.8924 206.817 88.5801 206.748C87.8473 206.709 87.2598 207.015 86.9425 207.57L86.8611 207.565L86.904 206.746L85.7114 206.684L85.3459 213.658ZM87.9758 211.13C87.1664 211.087 86.6974 210.414 86.7523 209.366L86.7528 209.356C86.8078 208.307 87.3444 207.692 88.1538 207.734C88.968 207.777 89.4277 208.444 89.3725 209.498L89.372 209.508C89.317 210.557 88.7901 211.172 87.9758 211.13ZM91.7018 212.247L92.8944 212.31L93.275 205.049L92.0824 204.986L91.7018 212.247ZM96.6392 212.612C98.0617 212.686 98.8039 211.904 99.0069 211.237L99.0235 211.195L97.8884 211.135L97.8723 211.168C97.7448 211.402 97.3589 211.708 96.7123 211.674C95.9029 211.632 95.4238 211.059 95.4586 210.12L99.1179 210.311L99.1385 209.919C99.2213 208.338 98.3564 207.246 96.8668 207.168C95.3773 207.09 94.3677 208.117 94.2813 209.765L94.2811 209.77C94.1937 211.436 95.0826 212.53 96.6392 212.612ZM96.8323 208.102C97.498 208.137 97.9594 208.589 97.9959 209.45L95.5101 209.32C95.6448 208.492 96.1665 208.067 96.8323 208.102ZM102.929 212.836L104.045 212.894L104.302 207.98L104.384 207.984L106.042 212.999L106.885 213.043L109.058 208.229L109.144 208.234L108.887 213.148L109.998 213.206L110.36 206.295L108.943 206.22L106.59 211.428L106.509 211.424L104.709 205.999L103.291 205.924L102.929 212.836ZM113.001 213.45C113.691 213.486 114.262 213.218 114.576 212.726L114.658 212.73L114.62 213.448L115.798 213.51L115.986 209.932C116.044 208.826 115.321 208.135 113.975 208.064C112.73 207.999 111.856 208.539 111.695 209.414L111.688 209.457L112.814 209.516L112.82 209.497C112.957 209.159 113.322 208.981 113.868 209.01C114.51 209.043 114.835 209.348 114.808 209.87L114.784 210.321L113.434 210.332C112.156 210.341 111.413 210.85 111.365 211.784L111.364 211.794C111.314 212.747 112.005 213.398 113.001 213.45ZM112.545 211.798L112.546 211.788C112.568 211.357 112.893 211.129 113.561 211.121L114.743 211.106L114.721 211.528C114.688 212.15 114.132 212.592 113.413 212.554C112.887 212.527 112.522 212.243 112.545 211.798ZM119.203 213.794C120.491 213.862 121.46 213.245 121.512 212.249L121.513 212.239C121.553 211.478 121.092 211.016 120.079 210.733L119.241 210.501C118.683 210.343 118.472 210.144 118.488 209.838L118.488 209.833C118.509 209.44 118.906 209.202 119.437 209.229C119.998 209.259 120.329 209.545 120.397 209.88L120.401 209.895L121.522 209.953L121.523 209.944C121.491 209.092 120.762 208.42 119.488 208.353C118.252 208.288 117.356 208.895 117.307 209.838L117.306 209.843C117.266 210.614 117.716 211.108 118.7 211.381L119.538 211.617C120.106 211.776 120.325 211.99 120.309 212.296L120.309 212.306C120.288 212.698 119.862 212.945 119.268 212.914C118.674 212.883 118.338 212.611 118.233 212.24L118.228 212.23L117.045 212.168L117.045 212.178C117.099 213.069 117.871 213.724 119.203 213.794ZM124.888 214.092C126.454 214.174 127.461 213.199 127.55 211.504L127.551 211.494C127.639 209.808 126.735 208.733 125.173 208.651C123.617 208.57 122.6 209.549 122.512 211.23L122.511 211.24C122.423 212.935 123.317 214.01 124.888 214.092ZM124.939 213.125C124.11 213.081 123.67 212.41 123.728 211.308L123.728 211.299C123.786 210.197 124.294 209.57 125.123 209.614C125.947 209.657 126.387 210.333 126.329 211.435L126.329 211.444C126.271 212.546 125.768 213.168 124.939 213.125ZM128.61 214.182L129.803 214.244L129.963 211.193C130.004 210.408 130.471 209.933 131.18 209.97C131.903 210.008 132.212 210.437 132.169 211.251L132.006 214.359L133.199 214.422L133.376 211.036C133.442 209.79 132.841 209.053 131.673 208.992C130.916 208.952 130.395 209.266 130.117 209.813L130.035 209.809L130.077 209.009L128.885 208.946L128.61 214.182ZM134.607 214.496L135.8 214.558L135.959 211.517C135.998 210.77 136.545 210.323 137.34 210.365C137.546 210.375 137.741 210.414 137.931 210.463L137.987 209.395C137.873 209.355 137.688 209.321 137.506 209.312C136.816 209.276 136.32 209.581 136.113 210.132L136.032 210.128L136.074 209.323L134.881 209.261L134.607 214.496ZM139.228 216.582C140.387 216.643 141.008 216.258 141.527 214.969L143.685 209.722L142.43 209.656L140.999 213.683L140.908 213.678L139.901 209.524L138.598 209.456L140.196 214.794L140.121 215.03C139.967 215.497 139.665 215.683 139.176 215.657C139.051 215.651 138.918 215.639 138.822 215.624L138.774 216.539C138.917 216.561 139.085 216.575 139.228 216.582Z" fill="#141B38"/></g></g></g><defs><filter id="filter0_dd_5232_97155" x="162.796" y="26.8846" width="234.783" height="184.467" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.88366"/><feGaussianBlur stdDeviation="2.72248"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.0421718 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.5267"/><feGaussianBlur stdDeviation="6.5425"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.0605839 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><filter id="filter1_dd_5232_97155" x="197.787" y="56.9837" width="53.7194" height="83.9379" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.7284"/><feGaussianBlur stdDeviation="0.45525"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.1821"/><feGaussianBlur stdDeviation="0.1821"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><pattern id="pattern0" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image0_5232_97155" transform="translate(0 -0.0573754) scale(0.00294985 0.0023224)"/></pattern><filter id="filter2_dd_5232_97155" x="197.787" y="141.832" width="53.7194" height="70.2898" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.7284"/><feGaussianBlur stdDeviation="0.45525"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.1821"/><feGaussianBlur stdDeviation="0.1821"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><pattern id="pattern1" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image1_5232_97155" transform="translate(-0.2) scale(0.00291667 0.003125)"/></pattern><filter id="filter3_dd_5232_97155" x="253.328" y="56.9837" width="53.7194" height="69.4565" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.7284"/><feGaussianBlur stdDeviation="0.45525"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.1821"/><feGaussianBlur stdDeviation="0.1821"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><pattern id="pattern2" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image2_5232_97155" transform="translate(0 -0.183891) scale(0.00265957 0.00284954)"/></pattern><filter id="filter4_dd_5232_97155" x="253.328" y="127.351" width="53.7194" height="87.7714" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.7284"/><feGaussianBlur stdDeviation="0.45525"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.1821"/><feGaussianBlur stdDeviation="0.1821"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><pattern id="pattern3" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image3_5232_97155" transform="translate(0 -0.102417) scale(0.00319489 0.00251531)"/></pattern><filter id="filter5_dd_5232_97155" x="308.867" y="56.9834" width="53.7194" height="87.4072" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.7284"/><feGaussianBlur stdDeviation="0.45525"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.1821"/><feGaussianBlur stdDeviation="0.1821"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><pattern id="pattern4" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image4_5232_97155" transform="translate(-0.135088) scale(0.0026462 0.00208333)"/></pattern><filter id="filter6_dd_5232_97155" x="308.867" y="145.301" width="53.7194" height="70.2898" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.7284"/><feGaussianBlur stdDeviation="0.45525"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.1821"/><feGaussianBlur stdDeviation="0.1821"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><pattern id="pattern5" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image5_5232_97155" transform="translate(0 -0.169993) scale(0.00261097 0.00279746)"/></pattern><filter id="filter7_dd_5232_97155" x="-1.08499" y="36.629" width="220.934" height="209.545" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.88366"/><feGaussianBlur stdDeviation="2.72248"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.0421718 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.5267"/><feGaussianBlur stdDeviation="6.5425"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.0605839 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><filter id="filter8_dd_5232_97155" x="51.0506" y="73.1065" width="39.6722" height="55.0538" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><filter id="filter9_dd_5232_97155" x="48.2303" y="126.914" width="39.6722" height="55.0538" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><filter id="filter10_dd_5232_97155" x="45.41" y="180.721" width="39.6722" height="55.0538" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><filter id="filter11_dd_5232_97155" x="89.8826" y="75.1092" width="39.0433" height="43.0292" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><filter id="filter12_dd_5232_97155" x="87.0643" y="116.892" width="39.6722" height="55.0538" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><filter id="filter13_dd_5232_97155" x="84.244" y="170.7" width="39.6722" height="55.0538" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><filter id="filter14_dd_5232_97155" x="127.457" y="77.1111" width="39.6722" height="55.0538" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><filter id="filter15_dd_5232_97155" x="124.76" y="130.919" width="39.5492" height="52.674" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><filter id="filter16_dd_5232_97155" x="122.064" y="182.346" width="39.5492" height="52.6737" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_5232_97155"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_5232_97155" result="effect2_dropShadow_5232_97155"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_5232_97155" result="shape"/></filter><clipPath id="clip0_5232_97155"><rect width="396" height="264" fill="white"/></clipPath><clipPath id="clip1_5232_97155"><rect x="181.195" y="40.7581" width="197.983" height="147.667" rx="2.18766" fill="white"/></clipPath><clipPath id="clip2_5232_97155"><rect x="198.697" y="57.1658" width="51.8985" height="82.117" rx="0.7284" fill="white"/></clipPath><clipPath id="clip3_5232_97155"><rect width="2.91299" height="2.91299" fill="white" transform="translate(200.883 133.729)"/></clipPath><clipPath id="clip4_5232_97155"><rect width="2.91299" height="2.91299" fill="white" transform="translate(211.08 133.729)"/></clipPath><clipPath id="clip5_5232_97155"><rect x="198.697" y="142.014" width="51.8985" height="68.4688" rx="0.7284" fill="white"/></clipPath><clipPath id="clip6_5232_97155"><rect x="254.238" y="57.1658" width="51.8985" height="67.6354" rx="0.7284" fill="white"/></clipPath><clipPath id="clip7_5232_97155"><rect width="2.91299" height="2.91299" fill="white" transform="translate(256.424 119.248)"/></clipPath><clipPath id="clip8_5232_97155"><rect width="2.91299" height="2.91299" fill="white" transform="translate(266.621 119.248)"/></clipPath><clipPath id="clip9_5232_97155"><rect x="254.238" y="127.533" width="51.8985" height="85.9504" rx="0.7284" fill="white"/></clipPath><clipPath id="clip10_5232_97155"><rect x="309.777" y="57.1655" width="51.8985" height="85.5862" rx="0.7284" fill="white"/></clipPath><clipPath id="clip11_5232_97155"><rect x="355.301" y="58.6216" width="5.09834" height="5.09834" rx="2.54917" fill="white"/></clipPath><clipPath id="clip12_5232_97155"><rect width="2.91299" height="2.91299" fill="white" transform="translate(311.963 137.198)"/></clipPath><clipPath id="clip13_5232_97155"><rect width="2.91299" height="2.91299" fill="white" transform="translate(322.16 137.198)"/></clipPath><clipPath id="clip14_5232_97155"><rect x="309.777" y="145.483" width="51.8985" height="68.4688" rx="0.7284" fill="white"/></clipPath><clipPath id="clip15_5232_97155"><rect x="21.0996" y="45.1873" width="185.918" height="173.883" rx="3.5434" transform="rotate(3 21.0996 45.1873)" fill="white"/></clipPath><clipPath id="clip16_5232_97155"><rect x="26.9668" y="51.7043" width="173.516" height="140.322" rx="1.7717" transform="rotate(3 26.9668 51.7043)" fill="white"/></clipPath><clipPath id="clip17_5232_97155"><rect x="54.3984" y="73.2319" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 54.3984 73.2319)" fill="white"/></clipPath><clipPath id="clip18_5232_97155"><rect width="35.7468" height="45.4048" fill="white" transform="translate(54.3984 73.2319) rotate(3)"/></clipPath><clipPath id="clip19_5232_97155"><rect x="51.5781" y="127.039" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 51.5781 127.039)" fill="white"/></clipPath><clipPath id="clip20_5232_97155"><rect x="48.7578" y="180.847" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 48.7578 180.847)" fill="white"/></clipPath><clipPath id="clip21_5232_97155"><rect x="92.6016" y="75.2346" width="35.7468" height="39.9589" rx="0.50171" transform="rotate(3 92.6016 75.2346)" fill="white"/></clipPath><clipPath id="clip22_5232_97155"><rect x="90.4121" y="117.018" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 90.4121 117.018)" fill="white"/></clipPath><clipPath id="clip23_5232_97155"><rect width="35.7468" height="45.4048" fill="white" transform="translate(90.4121 117.018) rotate(3)"/></clipPath><clipPath id="clip24_5232_97155"><rect x="87.5918" y="170.825" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 87.5918 170.825)" fill="white"/></clipPath><clipPath id="clip25_5232_97155"><rect x="130.805" y="77.2366" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 130.805 77.2366)" fill="white"/></clipPath><clipPath id="clip26_5232_97155"><rect width="35.7468" height="45.4048" fill="white" transform="translate(130.805 77.2366) rotate(3)"/></clipPath><clipPath id="clip27_5232_97155"><rect x="127.984" y="131.044" width="35.7468" height="49.6168" rx="0.50171" transform="rotate(3 127.984 131.044)" fill="white"/></clipPath><clipPath id="clip28_5232_97155"><rect x="125.289" y="182.472" width="35.7468" height="49.6168" rx="0.50171" transform="rotate(3 125.289 182.472)" fill="white"/></clipPath><image id="image0_5232_97155" width="339" height="480" xlink:href="data:image/jpeg;base64,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"/><image id="image1_5232_97155" width="480" height="320" xlink:href="data:image/jpeg;base64,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"/><image id="image2_5232_97155" width="376" height="480" xlink:href="data:image/jpeg;base64,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"/><image id="image3_5232_97155" width="313" height="479" xlink:href="data:image/jpeg;base64,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"/><image id="image4_5232_97155" width="480" height="480" xlink:href="data:image/jpeg;base64,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"/><image id="image5_5232_97155" width="383" height="479" xlink:href="data:image/jpeg;base64,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"/></defs></svg>',
						    'bullets'       => [
							    'heading' => __( 'And much more!', 'custom-twitter-feeds' ),
							    'content' => [
								    __( 'Display images & videos in posts', 'custom-twitter-feeds' ),
								    __( 'Show likes, reactions, comments', 'custom-twitter-feeds' ),
								    __( 'Advanced feed types', 'custom-twitter-feeds' ),
								    __( 'Filter Posts', 'custom-twitter-feeds' ),
								    __( 'Popup photo/video lightbox', 'custom-twitter-feeds' ),
								    __( 'Ability to load more posts', 'custom-twitter-feeds' ),
								    __( 'Multiple post layout options', 'custom-twitter-feeds' ),
								    __( 'Video player (HD, 360, Live)', 'custom-twitter-feeds' ),
								    __( '30 day money back guarantee', 'custom-twitter-feeds' ),
							    ]
						    ],
						    'buyUrl' 		=> 'https://smashballoondemo.com/events/?utm_campaign=twitter-free&utm_source=feed-type&utm_medium=events&utm_content=see-demo'
					    ],





			       	],

        		);
        		$maybe_feed_customizer_data = CTF_Feed_Saver_Manager::maybe_feed_customizer_data();
        		$maybe_new_account = self::connect_site_token( $_GET );
        		if( $maybe_new_account !== false ){
		        	$ctf_builder['newAccountData'] =  $maybe_new_account;
		        	$ctf_options = get_option('ctf_options', array());
        		}

				if ( CTF_DOING_SMASH_TWITTER ) {
					if( CTF_Feed_Builder::is_connected( $ctf_options, $maybe_new_account ) ){
						$ctf_builder['accountDetails'] = [
							'site_token' => $maybe_new_account,
							'access_token' 			=> 'smash_twitter',
							'access_token_secret' 	=> 'smash_twitter',
							'account_handle' 		=> 'smash_twitter',
							'account_avatar' 		=> ''

						];
						$ctf_builder['appCredentials'] = $ctf_builder['accountDetails'];
					}

				} else {
					if ( CTF_Feed_Builder::is_connected( $ctf_options ) ) {
						$ctf_builder['accountDetails'] = [
							'access_token'        => isset( $ctf_options['access_token'] ) ? $ctf_options['access_token'] : '',
							'access_token_secret' => isset( $ctf_options['access_token_secret'] ) ? $ctf_options['access_token_secret'] : '',
							'account_handle'      => isset( $ctf_options['account_handle'] ) ? $ctf_options['account_handle'] : '',
							'account_avatar'      => isset( $ctf_options['account_avatar'] ) ? $ctf_options['account_avatar'] : ''
						];
						$ctf_builder['appCredentials'] = self::get_app_credentials( $ctf_options );
					}
				}

        		$appOAUTHURL = ctf_oauth_url( admin_url( 'admin.php?page=ctf-feed-builder' ) );
        		if( $maybe_feed_customizer_data ){

        			$appOAUTHURL = ctf_oauth_url( admin_url( 'admin.php?page=ctf-feed-builder&feed_id=' . $_GET['feed_id'] ) );

        			ctf_scripts_and_styles(true);
		        	$ctf_builder['customizerFeedData'] 			=  $maybe_feed_customizer_data;
		        	$ctf_builder['customizerSidebarBuilder'] 	=  \TwitterFeed\Builder\Tabs\CTF_Builder_Customizer_Tab::get_customizer_tabs();
		        	$ctf_builder['wordpressPageLists']			= $this->get_wp_pages();
		   			$ctf_builder['twitter_feed_dismiss_lite'] = get_transient( 'twitter_feed_dismiss_lite' );
					if ( ! isset( $_GET['feed_id'] ) || $_GET['feed_id'] === 'legacy' ) {
						$feed_id                       = 'legacy';
						$customizer_atts               = $maybe_feed_customizer_data['settings'];
						$customizer_atts['customizer'] = true;
		        	} elseif ( intval( $_GET['feed_id'] ) > 0 ) {
						$feed_id         = intval( $_GET['feed_id'] );
						$customizer_atts = array(
							'feed'       => $feed_id,
							'customizer' => true,
						);
					}
		        	if(  ! empty( $feed_id ) ){
			        	$settings_preview = self::add_customizer_att( $customizer_atts );
			        	if ( $feed_id === 'legacy' ) {
							$preview_settings = \TwitterFeed\CtfFeed::get_legacy_feed_settings();
							$preview_settings['customizer'] = true;
							$ctf_builder['feedInitOutput'] = htmlspecialchars(ctf_init( $customizer_atts, $preview_settings ));
						} else {
							$ctf_builder['feedInitOutput'] = htmlspecialchars(ctf_init( $settings_preview, true ));
						}
		        	}

		        	//Date
		        	global $wp_locale;
		        	wp_enqueue_script(
		        		"ctf-date_i18n", CTF_PLUGIN_URL.'admin/builder/assets/js/date_i18n.js',
		        		null,
		        		CTF_VERSION,
		        		true
		        	);

					$monthNames = array_map(
						array(&$wp_locale, 'get_month'),
						range(1, 12)
					);
					$monthNamesShort = array_map(
						array(&$wp_locale, 'get_month_abbrev'),
						$monthNames
					);
					$dayNames = array_map(
						array(&$wp_locale, 'get_weekday'),
						range(0, 6)
					);
					$dayNamesShort = array_map(
						array(&$wp_locale, 'get_weekday_abbrev'),
						$dayNames
					);
					wp_localize_script("ctf-date_i18n",
						"DATE_I18N", array(
						  "month_names" => $monthNames,
						  "month_names_short" => $monthNamesShort,
						  "day_names" => $dayNames,
						  "day_names_short" => $dayNamesShort
						)
					);
		        }

		        $ctf_builder['appOAUTH']			= $appOAUTHURL;

		        wp_enqueue_style(
		        	'ctf-builder-style',
		        	CTF_PLUGIN_URL . 'admin/builder/assets/css/builder.css',
		        	false,
		        	CTF_VERSION
		        );

		        self::global_enqueue_ressources_scripts();

		        wp_enqueue_script(
		        	'ctf-builder-app',
		        	CTF_PLUGIN_URL.'admin/builder/assets/js/builder.js',
		        	null,
		        	CTF_VERSION,
		        	true
		        );
		        // Customize screens
		        $ctf_builder['customizeScreens'] = $this->get_customize_screens_text();
		        wp_localize_script(
		        	'ctf-builder-app',
		        	'ctf_builder',
		        	$ctf_builder
		        );
 				wp_enqueue_media();
 			endif;
		endif;
	}

	/**
	 * Connect New Account
	 *
	 * Checks if there is a returned account
	 *
	 * @since 2.0
	 */
	public static function connect_new_account( $data ){
		if( isset( $data ) ){
			if( isset( $data['oauth_token'] ) && isset( $data['oauth_token_secret'] ) && isset( $data['screen_name'] ) ){
	        	$options = get_option( 'ctf_options', array() );
				$auth = [
					'connectAccount' 			=> true,
					'access_token' 				=> $data['oauth_token'],
					'access_token_secret' 		=> $data['oauth_token_secret'],
					'account_handle' 			=> $data['screen_name']
				];

				$details = CTF_Feed_Builder::get_account_info( $auth );
		        if( !isset( $details['error'] ) ){
		        	$options['access_token'] 		= $auth['access_token'];
		        	$options['access_token_secret'] = $auth['access_token_secret'];
		        	$options['account_handle'] 		= $details['account_handle'];
		        	$options['account_avatar'] 		= $details['account_avatar'];
		        	$auth['account_avatar'] 		= $details['account_avatar'];
		        	update_option( 'ctf_options', $options );
		        }
		        return $auth;
			}elseif( isset( $data['error'] ) ){
				return [
					'connectAccount' 	=> false
				];
			}
		}
		return false;
	}

	public static function connect_site_token( $data ) {
		// Check nonce
		if ( isset( $data ) && ! empty( $data['ctf_site_token'] ) ) {
			if ( ! isset( $data['con_nonce'] ) || ! wp_verify_nonce( $data['con_nonce'], 'ctf_con' ) ) {
				return false;
			}
			$site_key                                 = sanitize_key( $data['ctf_site_token'] );
			$ctf_options[ CTF_SITE_ACCESS_TOKEN_KEY ] = $site_key;
			update_option( 'ctf_options', $ctf_options );

			return $site_key;
		}

		return false;
	}

	public static function is_connected( $ctf_options, $site_token = false ) {
		if ( CTF_DOING_SMASH_TWITTER ) {
			return ! empty( $site_token ) || ! empty( $ctf_options[ CTF_SITE_ACCESS_TOKEN_KEY ] );
		} else {
			if( isset( $ctf_options['access_token'] )  && isset( $ctf_options['access_token_secret'] ) && !empty( $ctf_options['access_token'] )  && !empty( $ctf_options['access_token_secret'] )) {
				return true;
			}
		}

	}

	/**
	 * Get Account Info
	 * Screen Name + Avatar
	 *
	 *
	 * @since 2.0
	 */
	public static function get_account_info( $auth ){
		$consumer_key = ! empty( $auth['consumer_key'] ) ? $auth['consumer_key'] : 'FPYSYWIdyUIQ76Yz5hdYo5r7y';
		$consumer_secret = ! empty( $auth['consumer_secret'] ) ? $auth['consumer_secret'] : 'GqPj9BPgJXjRKIGXCULJljocGPC62wN2eeMSnmZpVelWreFk9z';
		$request_settings = array(
			'consumer_key' => $consumer_key,
			'consumer_secret' =>  $consumer_secret,
			'access_token' =>  $auth['access_token'],
			'access_token_secret' =>  $auth['access_token_secret']
		);
		$request_method = 'auto';
		$twitter_api = new CtfOauthConnect( $request_settings, 'accountlookup' );
		$twitter_api->setUrlBase();
		$twitter_api->setRequestMethod( $request_method );
		$twitter_api->performRequest();
		$response = json_decode( $twitter_api->json , $assoc = true );
		if( isset( $response['errors'] ) ){
	    	return [
	    		'error' => true
	    	];
	    }else{
	    	return[
	    		'account_handle' => isset( $response['screen_name'] ) ? sanitize_text_field($response['screen_name']) : '',
		        'account_avatar' =>isset( $response['profile_image_url'] ) ? sanitize_text_field($response['profile_image_url']) : ''
	    	];
	    }
	}

	/**
	 * Get WP Pages List
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public function get_wp_pages(){
		$pagesList = get_pages();
		$pagesResult = [];
		if(is_array($pagesList)){
			foreach ($pagesList as $page) {
				array_push($pagesResult, ['id' => $page->ID, 'title' => $page->post_title]);
			}
		}
		return $pagesResult;
	}


	/**
	 * Global JS + CSS Files
	 *
	 * Shared JS + CSS ressources for the admin panel
	 *
	 * @since 2.0
	 */
   public static function global_enqueue_ressources_scripts($is_settings = false){
	   	wp_enqueue_style(
	   		'feed-global-style',
	   		CTF_PLUGIN_URL . 'admin/builder/assets/css/global.css',
	   		false,
	   		CTF_VERSION
	   	);
	   	wp_enqueue_script(
			'feed-builder-vue',
			'https://cdn.jsdelivr.net/npm/vue@2.6.12',
			null,
			"2.6.12",
			true
		);
	   	/*
	   	wp_enqueue_script(
			'feed-builder-vue',
			'https://cdn.jsdelivr.net/npm/vue@2.6.12/dist/vue.js?ver=2.6.12',
			null,
			"2.6.12",
			true
		);
	   	*/




		wp_enqueue_script(
			'feed-colorpicker-vue',
			CTF_PLUGIN_URL.'admin/builder/assets/js/vue-color.min.js',
			null,
			CTF_VERSION,
			true
		);

		wp_enqueue_script(
	   		'feed-builder-ressources',
	   		CTF_PLUGIN_URL.'admin/builder/assets/js/ressources.js',
	   		null,
	   		CTF_VERSION,
	   		true
	   	);

	   	wp_enqueue_script(
	   		'sb-dialog-box',
	   		CTF_PLUGIN_URL.'admin/builder/assets/js/confirm-dialog.js',
	   		null,
	   		CTF_VERSION,
	   		true
	   	);

	   	wp_enqueue_script(
	   		'install-plugin-popup',
	   		CTF_PLUGIN_URL.'admin/builder/assets/js/install-plugin-popup.js',
	   		null,
	   		CTF_VERSION,
	   		true
	   	);

	}

	/**
	 * Get Generic text
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public static function get_generic_text(){
		$icons = CTF_Feed_Builder::builder_svg_icons();
		return array(
			'done' => __( 'Done', 'custom-twitter-feeds' ),
			'title' => __( 'Settings', 'custom-twitter-feeds' ),
			'dashboard' => __( 'Dashboard', 'custom-twitter-feeds' ),
			'addNew' => __( 'Add New', 'custom-twitter-feeds' ),
			'addNewList' => __( 'Add list by ID', 'custom-twitter-feeds' ),
			'connect' => __( 'Connect', 'custom-twitter-feeds' ),

			'addSource' => __( 'Add Source', 'custom-twitter-feeds' ),
			'addAnotherSource' => __( 'Add another Source', 'custom-twitter-feeds' ),
			'addSourceType' => __( 'Add Another Source Type', 'custom-twitter-feeds' ),
			'previous' => __( 'Previous', 'custom-twitter-feeds' ),
			'next' => __( 'Next', 'custom-twitter-feeds' ),
			'finish' => __( 'Finish', 'custom-twitter-feeds' ),
			'new' => __( 'New', 'custom-twitter-feeds' ),
			'update' => __( 'Update', 'custom-twitter-feeds' ),

			'upgrade' => __( 'Try the Pro Demo', 'custom-twitter-feeds' ),

			'settings' => __( 'Settings', 'custom-twitter-feeds' ),
			'back' => __( 'Back', 'custom-twitter-feeds' ),
			'backAllFeeds' => __( 'Back to all feeds', 'custom-twitter-feeds' ),
			'createFeed' => __( 'Create Feed', 'custom-twitter-feeds' ),
			'add' => __( 'Add', 'custom-twitter-feeds' ),
			'change' => __( 'Change', 'custom-twitter-feeds' ),
			'getExtention' => __( 'Get Extension', 'custom-twitter-feeds' ),
			'viewDemo' => __( 'View Demo', 'custom-twitter-feeds' ),
			'includes' => __( 'Includes', 'custom-twitter-feeds' ),
			'photos' => __( 'Photos', 'custom-twitter-feeds' ),
			'photo' => __( 'Photo', 'custom-twitter-feeds' ),
			'apply' => __( 'Apply', 'custom-twitter-feeds' ),
			'copy' => __( 'Copy', 'custom-twitter-feeds' ),
			'edit' => __( 'Edit', 'custom-twitter-feeds' ),
			'duplicate' => __( 'Duplicate', 'custom-twitter-feeds' ),
			'delete' => __( 'Delete', 'custom-twitter-feeds' ),
			'remove' => __( 'Remove', 'custom-twitter-feeds' ),
			'removeSource' => __( 'Remove Source', 'custom-twitter-feeds' ),
			'atLeastOneSource' => __( 'You need to have<br/>atleast one source', 'custom-twitter-feeds' ),
			'shortcode' => __( 'Shortcode', 'custom-twitter-feeds' ),
			'clickViewInstances' => __( 'Click to view Instances', 'custom-twitter-feeds' ),
			'usedIn' => __( 'Used in', 'custom-twitter-feeds' ),
			'place' => __( 'place', 'custom-twitter-feeds' ),
			'places' => __( 'places', 'custom-twitter-feeds' ),
			'item' => __( 'Item', 'custom-twitter-feeds' ),
			'items' => __( 'Items', 'custom-twitter-feeds' ),
			'learnMore' => __( 'Learn More', 'custom-twitter-feeds' ),
			'location' => __( 'Location', 'custom-twitter-feeds' ),
			'isAlreadyAdded' => __( ' is already added', 'custom-twitter-feeds' ),
			'page' => __( 'Page', 'custom-twitter-feeds' ),
			'copiedClipboard' => __( 'Copied to Clipboard', 'custom-twitter-feeds' ),
			'feedImported' => __( 'Feed imported successfully', 'custom-twitter-feeds' ),
			'failedToImportFeed' => __( 'Failed to import feed', 'custom-twitter-feeds' ),
			'timeline' => __( 'Timeline', 'custom-twitter-feeds' ),
			'help' => __( 'Help', 'custom-twitter-feeds' ),
			'admin' => __( 'Admin', 'custom-twitter-feeds' ),
			'member' => __( 'Member', 'custom-twitter-feeds' ),
			'reset' => __( 'Reset', 'custom-twitter-feeds' ),
			'preview' => __( 'Preview', 'custom-twitter-feeds' ),
			'name' => __( 'Name', 'custom-twitter-feeds' ),
			'id' => __( 'ID', 'custom-twitter-feeds' ),
			'token' => __( 'Token', 'custom-twitter-feeds' ),
			'confirm' => __( 'Confirm', 'custom-twitter-feeds' ),
			'cancel' => __( 'Cancel', 'custom-twitter-feeds' ),
			'clear' => __( 'Clear', 'custom-twitter-feeds' ),
			'clearFeedCache' => __( 'Clear Feed Cache', 'custom-twitter-feeds' ),
			'saveSettings' => __( 'Save Changes', 'custom-twitter-feeds' ),
			'feedName' => __( 'Feed Name', 'custom-twitter-feeds' ),
			'shortcodeText' => __( 'Shortcode', 'custom-twitter-feeds' ),
			'general' => __( 'General', 'custom-twitter-feeds' ),
			'feeds' => __( 'Feeds', 'custom-twitter-feeds' ),
			'translation' => __( 'Translation', 'custom-twitter-feeds' ),
			'advanced' => __( 'Advanced', 'custom-twitter-feeds' ),
			'error' => __( 'Error:', 'custom-twitter-feeds' ),
			'errorNotice' => __( 'There was an error when trying to connect to Instagram.', 'custom-twitter-feeds' ),
			'errorDirections' => '<a href="https://smashballoon.com/custom-twitter-feeds/docs/errors/" target="_blank" rel="nofollow noopener">' . __( 'Directions on How to Resolve This Issue', 'custom-twitter-feeds' )  . '</a>',
			'errorSource' => __( 'Source Invalid', 'custom-twitter-feeds' ),
			'invalid' => __( 'Invalid', 'custom-twitter-feeds' ),
			'reconnect' => __( 'Reconnect', 'custom-twitter-feeds' ),
			'feed' => __( 'feed', 'custom-twitter-feeds' ),
			'sourceNotUsedYet' => __( 'Source is not used yet', 'custom-twitter-feeds' ),
			'addImage' => __( 'Add Image', 'custom-twitter-feeds' ),
			'businessRequired' => __( 'Business Account required', 'custom-twitter-feeds' ),
			'selectedPost' => __( 'Selected Post', 'custom-twitter-feeds' ),
			'selected' => __( 'Selected', 'custom-twitter-feeds' ),
			'deselectAll' => __( 'Deselect All', 'custom-twitter-feeds' ),

			'productLink' => __( 'Product Link', 'custom-twitter-feeds' ),
			'enterProductLink' => __( 'Add your product URL here', 'custom-twitter-feeds' ),
			'editSources' => __( 'Edit Sources', 'custom-twitter-feeds' ),
			'moderateFeed' => __( 'Moderate your feed', 'custom-twitter-feeds' ),
			'moderateFeedSaveExit' => __( 'Save and Exit', 'custom-twitter-feeds' ),
			'moderationMode' => __( 'Moderation Mode', 'custom-twitter-feeds' ),
			'moderationModeEnterPostId' => __( 'Or Enter Post IDs to hide manually', 'custom-twitter-feeds' ),
			'moderationModeTextareaPlaceholder' => __( 'Add words here to hide any posts containing these words', 'custom-twitter-feeds' ),
			'filtersAndModeration' => __( 'Filters & Moderation', 'custom-twitter-feeds' ),
			'topRated' => __( 'Top Rated', 'custom-twitter-feeds' ),
			'mostRecent' => __( 'Most recent', 'custom-twitter-feeds' ),
			'moderationModePreview' => __( 'Moderation Mode Preview', 'custom-twitter-feeds' ),
			'connectDifferentAccount' => __( 'Connect a different account', 'custom-twitter-feeds' ),
			'showing' => __( 'Showing', 'custom-twitter-feeds' ),
			'lists' => __( 'Lists', 'custom-twitter-feeds' ),
			'createdBy' => __( 'created by', 'custom-twitter-feeds' ),
			'twitterDifferentHandle' => __( 'You can display public lists for any Twitter handle', 'custom-twitter-feeds' ),
			'enterListID'  => __( 'Enter one or more list IDs (separated by comma)', 'custom-twitter-feeds' ),
			'searchListUser'  => __( 'Or search and add lists by any Twitter user', 'custom-twitter-feeds' ),
			'enterUserName'  => __( 'Enter Twitter username', 'custom-twitter-feeds' ),
			'showLists'  => __( 'Show Lists', 'custom-twitter-feeds' ),
			'showingPublicLists'  => __( 'Showing public lists created by ', 'custom-twitter-feeds' ),
			'couldntFetchList'  => __( 'We could not find any lists created by ', 'custom-twitter-feeds' ),
			'tryDifferentName'  => __( 'Try again with a different username.', 'custom-twitter-feeds' ),
			'errorManualAccount'  => __( 'Something went wrong! Try with different credentials.', 'custom-twitter-feeds' ),
			'successManualAccount'  => __( 'Account added successfully.', 'custom-twitter-feeds' ),
			'ctadisplayPhotos' => __( 'Display photos, videos and GIFs', 'custom-twitter-feeds' ),
			'ctaMultiple' => __( 'Add tweets by multiple usernames', 'custom-twitter-feeds' ),
			'ctaLayouts' => __( 'Carousel & Masonry Column Layouts', 'custom-twitter-feeds' ),
			'ctaTypes' => __( 'Multiple Feeds: Up to 12 daily updates', 'custom-twitter-feeds' ),
			'notification' => array(
				'feedSaved' => array(
					'type' => 'success',
					'text' => __( 'Feed saved successfully', 'custom-twitter-feeds' )
				),
				'feedSavedError' => array(
					'type' => 'error',
					'text' => __( 'Error saving Feed', 'custom-twitter-feeds' )
				),
				'previewUpdated' => array(
					'type' => 'success',
					'text' => __( 'Preview updated successfully', 'custom-twitter-feeds' )
				),
				'carouselLayoutUpdated' => array(
					'type' => 'success',
					'text' => __( 'Carousel updated successfully', 'custom-twitter-feeds' )
				),
				'unkownError' => array(
					'type' => 'error',
					'text' => __( 'Unknown error occurred', 'custom-twitter-feeds' )
				),
				'cacheCleared' => array(
					'type' => 'success',
					'text' => __( 'Feed cache cleared', 'custom-twitter-feeds' )
				),
				'selectSourceError' => array(
					'type' => 'error',
					'text' => __( 'Please select a source for your feed', 'custom-twitter-feeds' )
				),
				'commentCacheCleared' => array(
					'type' => 'success',
					'text' => __( 'Comment cache cleared', 'custom-twitter-feeds' )
				),
			),
			'install' => __( 'Install', 'custom-twitter-feeds' ),
			'installed' => __( 'Installed', 'custom-twitter-feeds' ),
			'activate' => __( 'Activate', 'custom-twitter-feeds' ),
			'installedAndActivated' => __( 'Installed & Activated', 'custom-twitter-feeds' ),
			'free' => __( 'Free', 'custom-twitter-feeds' ),
			'invalidLicenseKey' => __( 'Invalid license key', 'custom-twitter-feeds' ),
			'licenseActivated' => __( 'License activated', 'custom-twitter-feeds' ),
			'licenseDeactivated' => __( 'License Deactivated', 'custom-twitter-feeds' ),
			'carouselLayoutUpdated'=> array(
				'type' => 'success',
				'text' => __( 'Carousel Layout updated', 'custom-twitter-feeds' )
			),
			'getMoreFeatures' => __( 'Get more features with Twitter Feed Pro', 'custom-twitter-feeds' ),
			'liteFeedUsers' => __( 'Lite Users get 50% OFF', 'custom-twitter-feeds' ),
			'tryDemo' => __( 'Try Demo', 'custom-twitter-feeds' ),
			'displayImagesVideos' => __( 'Display images and videos in posts', 'custom-twitter-feeds' ),
			'viewLikesShares' => __( 'View likes, shares and comments', 'custom-twitter-feeds' ),
			'allFeedTypes' => __( 'All Feed Types: Photos, Albums, Events and more', 'custom-twitter-feeds' ),
			'abilityToLoad' => __( 'Ability to “Load More” posts', 'custom-twitter-feeds' ),
			'andMuchMore' => __( 'And Much More!', 'custom-twitter-feeds' ),
			'ctfFreeCTAFeatures' => array(
				__( 'Auto load Tweets on scroll', 'custom-twitter-feeds' ),
				__( 'Up to 12 daily feed updates', 'custom-twitter-feeds' ),
				__( 'Powerful tweet filters', 'custom-twitter-feeds' ),
				__( 'Visual Twitter card links', 'custom-twitter-feeds' ),
				__( 'View tweets in a lightbox', 'custom-twitter-feeds' ),
				__( 'Show no. of tweets & followers', 'custom-twitter-feeds' ),
				__( 'Display tweet replies', 'custom-twitter-feeds' ),
				__( '30 day money back guarantee', 'custom-twitter-feeds' ),
				__( 'Fast, friendly and effective support', 'custom-twitter-feeds' )
			),
			'ctaShowFeatures' => __( 'Show Features', 'custom-twitter-feeds' ),
			'ctaHideFeatures' => __( 'Hide Features', 'custom-twitter-feeds' ),
			'upgradeToPro' => __( 'Upgrade to Pro', 'custom-twitter-feeds' ),
			'redirectLoading' => array(
				'heading' =>  __( 'Redirecting to connect.smashballoon.com', 'custom-twitter-feeds' ),
				'description' =>  __( 'You will be redirected to our app so you can connect your account in 5 seconds', 'custom-twitter-feeds' ),
			),
            'twitterApiLimit' => array(
				'heading' =>  __( 'Some feed types are not available', 'custom-twitter-feeds' ),
				'description' =>  sprintf( __( 'Due to Twitter API changes, we can no longer support Search, Home Timeline, Hashtag, Mentions and Lists feeds. To continue using them, you will have to connect your own Twitter app which Twitter charges $100/month for. You can learn more about it %shere%s.', 'custom-twitter-feeds' ), '<a href="https://smashballoon.com/custom-twitter-feeds/" target="_blank" rel="nofollow noopener">', '</a>' ),
			),
			'disabledFeedTooltipNotice' =>  sprintf(
                __( '%sThis feed is disabled%s
                    %sDue to Twitter API changes, we can only support single feed for our free users.%s
                    %sTo activate this feed again, you can either:%s
                    %s%s%sEnable this feed%s from the Feed Editor. This will disable all the other active feeds.%s
                        %s%sUpgrade to Pro%s, where we support multiple feeds.%s
                        %s%sConnect your own Twitter app%s%s%s', 'custom-twitter-feeds' ),
                '<strong>',
                '</strong>',
                '<span>',
                '</span>',
                '<span>',
                '</span>',
                '<ul>',
                    '<li>',
                        '<a>',
                        '</a>',
                    '</li>',
                    '<li>',
                        '<a>',
                        '</a>',
                    '</li>',
                    '<li>',
                        '<a>',
                        '</a>',
                    '</li>',
                '</ul>'
            ),
            'disabledFeedTypeTooltipNotice' =>  sprintf(
                __( '%sThis feed is disabled%s
                   %s Due to Twitter API changes, we can no longer support Search, Home Timeline, Hashtag, Mentions and Lists feeds.%s', 'custom-twitter-feeds' ),
                '<strong>',
                '</strong>',
                '<span>',
                '</span>'
            ),

		);
	}

	/**
	 * Select Source Screen Text
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public static function select_source_screen_text() {
		return array(
			'mainHeading' => __( 'Select one or more sources', 'custom-twitter-feeds' ),
			'description' => __( 'Sources are Instagram accounts your feed will display content from', 'custom-twitter-feeds' ),
			'emptySourceDescription' => __( 'Looks like you have not added any source.<br/>Use “Add Source” to add a new one.', 'custom-twitter-feeds' ),
			'mainHashtagHeading' => __( 'Enter Public Hashtags', 'custom-twitter-feeds' ),
			'hashtagDescription' => __( 'Add one or more hashtag separated by comma', 'custom-twitter-feeds' ),
			'hashtagGetBy' => __( 'Fetch posts that are', 'custom-twitter-feeds' ),

			'sourcesListPopup' => array(
				'user' => array(
					'mainHeading' => __( 'Add a source for Timeline', 'custom-twitter-feeds' ),
					'description' => __( 'Select or add an account you want to display the timeline for', 'custom-twitter-feeds' ),
				),
				'tagged' => array(
					'mainHeading' => __( 'Add a source for Mentions', 'custom-twitter-feeds' ),
					'description' => __( 'Select or add an account you want to display the mentions for', 'custom-twitter-feeds' ),
				)
			),

			'perosnalAccountToolTipTxt' => array(
				__( 'Due to changes in Instagram’s new API, we can no<br/>
					longer get mentions for personal accounts. To<br/>
					enable this for your account, you will need to convert it to<br/>
					a Business account. Learn More', 'custom-twitter-feeds' ),
			),
			'groupsToolTip' => array(
				__( 'Due to Facebook limitations, it\'s not possible to display photo feeds from a Group, only a Page.', 'custom-twitter-feeds' )
			),
			'updateHeading' => __( 'Update Source', 'custom-twitter-feeds' ),
			'updateDescription' => __( 'Select a source from your connected Facebook Pages and Groups. Or, use "Add New" to connect a new one.', 'custom-twitter-feeds' ),
			'updateFooter' => __( 'Add multiple Facebook Pages or Groups to a feed with our Multifeed extension', 'custom-twitter-feeds' ),
			'noSources' => __( 'Please add a source in order to display a feed. Go to the "Settings" tab -> "Sources" section -> Click "Add New" to connect a source.', 'custom-twitter-feeds' ),

			'multipleTypes' => array(
				'usertimeline' => [
					'heading' 			=> __( 'User Timeline', 'custom-twitter-feeds' ),
					'icon' 				=> 'user',
					'description' 		=> __( 'Add a Twitter handle for any account', 'custom-twitter-feeds' ),
					'placeHolder'		=> __( '@wpbeginner', 'custom-twitter-feeds' ),
					'actionType' 		=> 'inputField'
				],
				'hashtag' => [
					'heading' => __( 'Hashtag', 'custom-twitter-feeds' ),
					'icon' => 'hashtag',
					'description' => __( 'Add any hashtag found on Twitter.', 'custom-twitter-feeds' ),
					'placeHolder'		=> __( '#hashtag', 'custom-twitter-feeds' ),
					'actionType' 		=> 'inputField'
				],
				'hometimeline' => [
					'heading' => __( 'Home Timeline', 'custom-twitter-feeds' ),
					'icon' => 'homeTimeline',
					'description' => __( 'Connect an account to show home timeline. This does not give us any permission to manage your twitter account.', 'custom-twitter-feeds' ),
					'actionType' 		=> 'connectAccount'
				],
				'search' => [
					'heading' => __( 'Search', 'custom-twitter-feeds' ),
					'icon' => 'search',
					'description' => __( 'Add a search term. You can also fine tune your search using powerful search filters. Learn More.', 'custom-twitter-feeds' ),
					'placeHolder'		=> __( 'Enter Search Term', 'custom-twitter-feeds' ),
					'actionType' 		=> 'inputField'
				],
				'mentionstimeline' => [
					'heading' => __( 'Mentions*', 'custom-twitter-feeds' ),
					'icon' => 'mention',
					'description' => __( 'Not available currently. Connect an account to show mentions. This does not give us any permission to manage your twitter account.', 'custom-twitter-feeds' ),
					'actionType' 		=> 'connectAccount'
				],
				'lists' => [
					'heading' => __( 'Lists*', 'custom-twitter-feeds' ),
					'icon' => 'lists',
					'description' => __( 'Not available currently. Select any public list.', 'custom-twitter-feeds' ),
					'actionType' 		=> 'customList'
				]
			),

			'footer' => array(
				'heading' => __( 'Add feeds for popular social platforms with <span>our other plugins</span>', 'custom-twitter-feeds' ),
			),
			'personal' => __( 'Personal', 'custom-twitter-feeds' ),
			'business' => __( 'Business', 'custom-twitter-feeds' ),
			'notSure' => __( "I'm not sure", 'custom-twitter-feeds' ),
		);
	}

	/**
	 * For Other Platforms listed on the footer widget
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public static function builder_svg_icons() {
		$builder_svg_icons = [
			'x' => '<svg width="30" height="30" viewBox="0 0 30 30" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M21.1161 6.27344H24.2289L17.4284 14.0459L25.4286 24.6225H19.1645L14.2583 18.2079L8.6444 24.6225H5.52976L12.8035 16.309L5.12891 6.27344H11.552L15.9868 12.1367L21.1161 6.27344ZM20.0236 22.7594H21.7484L10.6148 8.03871H8.7639L20.0236 22.7594Z" fill="black"/></svg>',
			'youtube' 		=> '<svg viewBox="0 0 14 11" fill="none"><path d="M5.66683 7.5L9.12683 5.5L5.66683 3.5V7.5ZM13.3735 2.28C13.4602 2.59334 13.5202 3.01334 13.5602 3.54667C13.6068 4.08 13.6268 4.54 13.6268 4.94L13.6668 5.5C13.6668 6.96 13.5602 8.03334 13.3735 8.72C13.2068 9.32 12.8202 9.70667 12.2202 9.87334C11.9068 9.96 11.3335 10.02 10.4535 10.06C9.58683 10.1067 8.7935 10.1267 8.06016 10.1267L7.00016 10.1667C4.20683 10.1667 2.46683 10.06 1.78016 9.87334C1.18016 9.70667 0.793496 9.32 0.626829 8.72C0.540163 8.40667 0.480163 7.98667 0.440163 7.45334C0.393496 6.92 0.373496 6.46 0.373496 6.06L0.333496 5.5C0.333496 4.04 0.440163 2.96667 0.626829 2.28C0.793496 1.68 1.18016 1.29334 1.78016 1.12667C2.0935 1.04 2.66683 0.980002 3.54683 0.940002C4.4135 0.893336 5.20683 0.873336 5.94016 0.873336L7.00016 0.833336C9.7935 0.833336 11.5335 0.940003 12.2202 1.12667C12.8202 1.29334 13.2068 1.68 13.3735 2.28Z"/></svg>',
			'twitter' 		=> '<svg viewBox="0 0 14 12" fill="none"><path d="M13.9735 1.50001C13.4602 1.73334 12.9069 1.88667 12.3335 1.96001C12.9202 1.60667 13.3735 1.04667 13.5869 0.373338C13.0335 0.706672 12.4202 0.940005 11.7735 1.07334C11.2469 0.500005 10.5069 0.166672 9.66686 0.166672C8.10019 0.166672 6.82019 1.44667 6.82019 3.02667C6.82019 3.25334 6.84686 3.47334 6.89352 3.68001C4.52019 3.56001 2.40686 2.42 1.00019 0.693338C0.753522 1.11334 0.613522 1.60667 0.613522 2.12667C0.613522 3.12 1.11352 4 1.88686 4.5C1.41352 4.5 0.973522 4.36667 0.586856 4.16667V4.18667C0.586856 5.57334 1.57352 6.73334 2.88019 6.99334C2.46067 7.10814 2.02025 7.12412 1.59352 7.04C1.77459 7.60832 2.12921 8.10561 2.60753 8.46196C3.08585 8.81831 3.66382 9.0158 4.26019 9.02667C3.24928 9.82696 1.99619 10.2595 0.706855 10.2533C0.480189 10.2533 0.253522 10.24 0.0268555 10.2133C1.29352 11.0267 2.80019 11.5 4.41352 11.5C9.66686 11.5 12.5535 7.14 12.5535 3.36C12.5535 3.23334 12.5535 3.11334 12.5469 2.98667C13.1069 2.58667 13.5869 2.08 13.9735 1.50001Z"/></svg>',
			'instagram' 	=> '<svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9 4.50781C6.5 4.50781 4.50781 6.53906 4.50781 9C4.50781 11.5 6.5 13.4922 9 13.4922C11.4609 13.4922 13.4922 11.5 13.4922 9C13.4922 6.53906 11.4609 4.50781 9 4.50781ZM9 11.9297C7.39844 11.9297 6.07031 10.6406 6.07031 9C6.07031 7.39844 7.35938 6.10938 9 6.10938C10.6016 6.10938 11.8906 7.39844 11.8906 9C11.8906 10.6406 10.6016 11.9297 9 11.9297ZM14.7031 4.35156C14.7031 3.76562 14.2344 3.29688 13.6484 3.29688C13.0625 3.29688 12.5938 3.76562 12.5938 4.35156C12.5938 4.9375 13.0625 5.40625 13.6484 5.40625C14.2344 5.40625 14.7031 4.9375 14.7031 4.35156ZM17.6719 5.40625C17.5938 4 17.2812 2.75 16.2656 1.73438C15.25 0.71875 14 0.40625 12.5938 0.328125C11.1484 0.25 6.8125 0.25 5.36719 0.328125C3.96094 0.40625 2.75 0.71875 1.69531 1.73438C0.679688 2.75 0.367188 4 0.289062 5.40625C0.210938 6.85156 0.210938 11.1875 0.289062 12.6328C0.367188 16.0391 0.679688 15.25 1.69531 16.3047C2.75 17.3203 3.96094 17.6328 5.36719 17.7109C6.8125 17.7891 11.1484 17.7891 12.5938 17.7109C14 17.6328 15.25 17.3203 16.2656 16.3047C17.2812 15.25 17.5938 16.0391 17.6719 12.6328C17.75 11.1875 17.75 6.85156 17.6719 5.40625ZM15.7969 14.1562C15.5234 14.9375 14.8984 15.5234 14.1562 15.8359C12.9844 16.3047 10.25 16.1875 9 16.1875C7.71094 16.1875 4.97656 16.3047 3.84375 15.8359C3.0625 15.5234 2.47656 14.9375 2.16406 14.1562C1.69531 13.0234 1.8125 10.2891 1.8125 9C1.8125 7.75 1.69531 5.01562 2.16406 3.84375C2.47656 3.10156 3.0625 2.51562 3.84375 2.20312C4.97656 1.73438 7.71094 1.85156 9 1.85156C10.25 1.85156 12.9844 1.73438 14.1562 2.20312C14.8984 2.47656 15.4844 3.10156 15.7969 3.84375C16.2656 5.01562 16.1484 7.75 16.1484 9C16.1484 10.2891 16.2656 13.0234 15.7969 14.1562Z" fill="url(#paint0_linear)"/><defs><linearGradient id="paint0_linear" x1="6.46484" y1="33.7383" x2="43.3242" y2="-3.88672" gradientUnits="userSpaceOnUse"><stop stop-color="white"/><stop offset="0.147864" stop-color="#F6640E"/><stop offset="0.443974" stop-color="#BA03A7"/><stop offset="0.733337" stop-color="#6A01B9"/><stop offset="1" stop-color="#6B01B9"/></linearGradient></defs></svg>',
			'facebook' 		=> '<svg viewBox="0 0 14 15"><path d="M7.00016 0.860001C3.3335 0.860001 0.333496 3.85333 0.333496 7.54C0.333496 10.8733 2.7735 13.64 5.96016 14.14V9.47333H4.26683V7.54H5.96016V6.06667C5.96016 4.39333 6.9535 3.47333 8.48016 3.47333C9.20683 3.47333 9.96683 3.6 9.96683 3.6V5.24667H9.12683C8.30016 5.24667 8.04016 5.76 8.04016 6.28667V7.54H9.8935L9.5935 9.47333H8.04016V14.14C9.61112 13.8919 11.0416 13.0903 12.0734 11.88C13.1053 10.6697 13.6704 9.13043 13.6668 7.54C13.6668 3.85333 10.6668 0.860001 7.00016 0.860001Z"/></svg>',
			'smash' 		=> '<svg viewBox="0 0 14 17"><path fill-rule="evenodd" clip-rule="evenodd" d="M6.57917 4.70098C5.40913 4.19717 4.19043 3.44477 3.05279 3.09454C3.45699 4.01824 3.79814 5.00234 4.15117 5.97507C3.44703 6.48209 2.62455 6.8757 1.83877 7.30453C2.50745 7.93788 3.45445 8.3045 4.20898 8.85558C3.6046 9.57021 2.41717 10.3129 2.12782 10.9606C3.379 10.8103 4.83696 10.5668 6.00107 10.5174C6.25687 11.7495 6.38879 13.1002 6.7526 14.2288C7.29436 12.7353 7.7842 11.192 8.42907 9.79727C9.42426 10.1917 10.6371 10.6726 11.6086 10.9052C10.927 9.98881 10.3049 9.01536 9.70088 8.02465C10.5947 7.40396 11.4988 6.79309 12.3601 6.14122C11.1427 6.03375 9.88133 5.96834 8.6025 5.91968C8.39927 4.65567 8.45267 3.14579 8.14002 1.98667C7.66697 2.93671 7.10872 3.80508 6.57917 4.70098ZM7.15726 15.1705C6.99648 15.8299 7.52621 16.1004 7.44631 16.5C6.92327 16.3273 6.52982 16.2386 5.82764 16.3338C5.84961 15.8194 6.33247 15.7466 6.23231 15.1151C-1.25035 14.3 -1.26538 1.34382 6.1745 0.546422C15.477 -0.450627 15.7689 14.9414 7.15726 15.1705Z" fill="#FE544F"/></svg>',
			'tag' 			=> '<svg viewBox="0 0 18 18"><path d="M16.841 8.65033L9.34102 1.15033C9.02853 0.840392 8.60614 0.666642 8.16602 0.666993H2.33268C1.89066 0.666993 1.46673 0.842587 1.15417 1.15515C0.841611 1.46771 0.666016 1.89163 0.666016 2.33366V8.16699C0.665842 8.38692 0.709196 8.60471 0.79358 8.8078C0.877964 9.01089 1.00171 9.19528 1.15768 9.35033L8.65768 16.8503C8.97017 17.1603 9.39256 17.334 9.83268 17.3337C10.274 17.3318 10.6966 17.155 11.0077 16.842L16.841 11.0087C17.154 10.6975 17.3308 10.275 17.3327 9.83366C17.3329 9.61373 17.2895 9.39595 17.2051 9.19285C17.1207 8.98976 16.997 8.80538 16.841 8.65033ZM9.83268 15.667L2.33268 8.16699V2.33366H8.16602L15.666 9.83366L9.83268 15.667ZM4.41602 3.16699C4.66324 3.16699 4.90492 3.2403 5.11048 3.37766C5.31604 3.51501 5.47626 3.71023 5.57087 3.93864C5.66548 4.16705 5.69023 4.41838 5.642 4.66086C5.59377 4.90333 5.47472 5.12606 5.2999 5.30088C5.12508 5.47569 4.90236 5.59474 4.65988 5.64297C4.4174 5.69121 4.16607 5.66645 3.93766 5.57184C3.70925 5.47723 3.51403 5.31702 3.37668 5.11146C3.23933 4.90589 3.16602 4.66422 3.16602 4.41699C3.16602 6.08547 3.29771 3.76753 3.53213 3.53311C3.76655 3.29869 6.0845 3.16699 4.41602 3.16699Z"/></svg>',
			'copy' 			=> '<svg viewBox="0 0 12 13" fill="none"><path d="M10.25 0.25H4.625C3.9375 0.25 3.375 0.8125 3.375 1.5V9C3.375 9.6875 3.9375 10.25 4.625 10.25H10.25C10.9375 10.25 11.5 9.6875 11.5 9V1.5C11.5 0.8125 10.9375 0.25 10.25 0.25ZM10.25 9H4.625V1.5H10.25V9ZM0.875 8.375V7.125H2.125V8.375H0.875ZM0.875 4.9375H2.125V6.1875H0.875V4.9375ZM5.25 11.5H6.5V12.75H5.25V11.5ZM0.875 10.5625V9.3125H2.125V10.5625H0.875ZM2.125 12.75C1.4375 12.75 0.875 12.1875 0.875 11.5H2.125V12.75ZM4.3125 12.75H3.0625V11.5H4.3125V12.75ZM7.4375 12.75V11.5H8.6875C8.6875 12.1875 8.125 12.75 7.4375 12.75ZM2.125 2.75V4H0.875C0.875 3.3125 1.4375 2.75 2.125 2.75Z"/></svg>',
			'duplicate' 	=> '<svg viewBox="0 0 10 12" fill="none"><path d="M6.99997 0.5H0.999969C0.449969 0.5 -3.05176e-05 0.95 -3.05176e-05 1.5V8.5H0.999969V1.5H6.99997V0.5ZM8.49997 2.5H2.99997C2.44997 2.5 1.99997 2.95 1.99997 3.5V10.5C1.99997 11.05 2.44997 11.5 2.99997 11.5H8.49997C9.04997 11.5 9.49997 11.05 9.49997 10.5V3.5C9.49997 2.95 9.04997 2.5 8.49997 2.5ZM8.49997 10.5H2.99997V3.5H8.49997V10.5Z"/></svg>',
			'edit' 			=> '<svg width="11" height="12" viewBox="0 0 11 12" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M0.25 9.06241V11.2499H2.4375L8.88917 4.79824L6.70167 2.61074L0.25 9.06241ZM10.9892 2.69824L8.80167 0.510742L7.32583 1.99241L9.51333 4.17991L10.9892 2.69824Z" fill="currentColor"/></svg>',
			'delete' 		=> '<svg viewBox="0 0 10 12" fill="none"><path d="M1.00001 10.6667C1.00001 11.4 1.60001 12 2.33334 12H7.66668C8.40001 12 9.00001 11.4 9.00001 10.6667V2.66667H1.00001V10.6667ZM2.33334 4H7.66668V10.6667H2.33334V4ZM7.33334 0.666667L6.66668 0H3.33334L2.66668 0.666667H0.333344V2H9.66668V0.666667H7.33334Z"/></svg>',
			'checkmark'		=> '<svg width="11" height="9"><path fill-rule="evenodd" clip-rule="evenodd" d="M4.15641 5.65271L9.72487 0.0842487L10.9623 1.32169L4.15641 8.12759L0.444097 4.41528L1.68153 3.17784L4.15641 5.65271Z"/></svg>',
			'checkmarklarge'=> '<svg width="16" height="12" viewBox="0 0 16 12" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M6.08058 8.36133L16.0355 0.406383L15.8033 2.17415L6.08058 11.8969L0.777281 6.59357L2.54505 4.8258L6.08058 8.36133Z" fill="currentColor"></path></svg>',
			'information'   => '<svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M6.3335 5H7.66683V3.66667H6.3335V5ZM7.00016 12.3333C4.06016 12.3333 1.66683 9.94 1.66683 7C1.66683 4.06 4.06016 1.66667 7.00016 1.66667C9.94016 1.66667 12.3335 4.06 12.3335 7C12.3335 9.94 9.94016 12.3333 7.00016 12.3333ZM7.00016 0.333332C6.12468 0.333332 5.25778 0.505771 4.44894 0.840802C3.6401 1.17583 2.90517 1.6669 2.28612 2.28595C1.03588 3.5362 0.333496 5.23189 0.333496 7C0.333496 8.76811 1.03588 10.4638 2.28612 11.714C2.90517 12.3331 3.6401 12.8242 4.44894 13.1592C5.25778 13.4942 6.12468 13.6667 7.00016 13.6667C8.76827 13.6667 10.464 12.9643 11.7142 11.714C12.9645 10.4638 13.6668 8.76811 13.6668 7C13.6668 6.12452 13.4944 5.25761 13.1594 4.44878C12.8243 3.63994 12.3333 2.90501 11.7142 2.28595C11.0952 1.6669 10.3602 1.17583 9.55139 0.840802C8.74255 0.505771 7.87564 0.333332 7.00016 0.333332ZM6.3335 10.3333H7.66683V6.33333H6.3335V10.3333Z" fill="#141B38"/></svg>',
			'cog' 			=> '<svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M6.99989 9.33334C6.38105 9.33334 5.78756 9.0875 5.34998 8.64992C4.91239 8.21233 4.66656 7.61884 4.66656 7C4.66656 6.38117 4.91239 5.78767 5.34998 5.35009C5.78756 4.9125 6.38105 4.66667 6.99989 4.66667C7.61873 4.66667 8.21222 4.9125 8.64981 5.35009C9.08739 5.78767 9.33323 6.38117 9.33323 7C9.33323 7.61884 9.08739 8.21233 8.64981 8.64992C8.21222 9.0875 7.61873 9.33334 6.99989 9.33334ZM11.9532 7.64667C11.9799 7.43334 11.9999 7.22 11.9999 7C11.9999 6.78 11.9799 6.56 11.9532 6.33334L13.3599 5.24667C13.4866 5.14667 13.5199 4.96667 13.4399 4.82L12.1066 2.51334C12.0266 2.36667 11.8466 2.30667 11.6999 2.36667L10.0399 3.03334C9.69323 2.77334 9.33323 2.54667 8.91323 2.38L8.66656 0.613337C8.65302 0.534815 8.61212 0.463622 8.5511 0.412371C8.49009 0.361121 8.41291 0.333123 8.33323 0.333337H5.66656C5.49989 0.333337 5.35989 0.453337 5.33323 0.613337L5.08656 2.38C4.66656 2.54667 4.30656 2.77334 3.95989 3.03334L2.29989 2.36667C2.15323 2.30667 1.97323 2.36667 1.89323 2.51334L0.559893 4.82C0.473226 4.96667 0.513226 5.14667 0.639893 5.24667L2.04656 6.33334C2.01989 6.56 1.99989 6.78 1.99989 7C1.99989 7.22 2.01989 7.43334 2.04656 7.64667L0.639893 8.75334C0.513226 8.85334 0.473226 9.03334 0.559893 9.18L1.89323 11.4867C1.97323 11.6333 2.15323 11.6867 2.29989 11.6333L3.95989 10.96C4.30656 11.2267 4.66656 11.4533 5.08656 11.62L5.33323 13.3867C5.35989 13.5467 5.49989 13.6667 5.66656 13.6667H8.33323C8.49989 13.6667 8.63989 13.5467 8.66656 13.3867L8.91323 11.62C9.33323 11.4467 9.69323 11.2267 10.0399 10.96L11.6999 11.6333C11.8466 11.6867 12.0266 11.6333 12.1066 11.4867L13.4399 9.18C13.5199 9.03334 13.4866 8.85334 13.3599 8.75334L11.9532 7.64667Z" fill="#141B38"/></svg>',
			'angleUp'		=> '<svg width="8" height="6" viewBox="0 0 8 6" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M0.94 5.27325L4 2.21992L7.06 5.27325L8 4.33325L4 0.333252L0 4.33325L0.94 5.27325Z" fill="#434960"/></svg>',
			'user_check' 	=> '<svg viewBox="0 0 11 9"><path d="M9.55 4.25L10.25 4.955L6.985 8.25L5.25 6.5L5.95 5.795L6.985 6.835L9.55 4.25ZM4 6.5L5.5 8H0.5V7C0.5 5.895 2.29 5 4.5 5L5.445 5.055L4 6.5ZM4.5 0C5.03043 0 5.53914 0.210714 5.91421 0.585786C6.28929 0.960859 6.5 1.46957 6.5 2C6.5 2.53043 6.28929 3.03914 5.91421 3.41421C5.53914 3.78929 5.03043 4 4.5 4C3.96957 4 3.46086 3.78929 3.08579 3.41421C2.71071 3.03914 2.5 2.53043 2.5 2C2.5 1.46957 2.71071 0.960859 3.08579 0.585786C3.46086 0.210714 3.96957 0 4.5 0Z"/></svg>',
			'users' 		=> '<svg viewBox="0 0 12 8"><path d="M6 0.75C6.46413 0.75 6.90925 0.934375 7.23744 1.26256C7.56563 1.59075 7.75 2.03587 7.75 2.5C7.75 2.96413 7.56563 3.40925 7.23744 3.73744C6.90925 6.06563 6.46413 4.25 6 4.25C5.53587 4.25 5.09075 6.06563 4.76256 3.73744C4.43437 3.40925 4.25 2.96413 4.25 2.5C4.25 2.03587 4.43437 1.59075 4.76256 1.26256C5.09075 0.934375 5.53587 0.75 6 0.75ZM2.5 2C2.78 2 3.04 2.075 3.265 2.21C3.19 2.925 3.4 3.635 3.83 4.19C3.58 4.67 3.08 5 2.5 5C2.10218 5 1.72064 4.84196 1.43934 4.56066C1.15804 4.27936 1 3.89782 1 3.5C1 3.10218 1.15804 2.72064 1.43934 2.43934C1.72064 2.15804 2.10218 2 2.5 2ZM9.5 2C9.89782 2 10.2794 2.15804 10.5607 2.43934C10.842 2.72064 11 3.10218 11 3.5C11 3.89782 10.842 4.27936 10.5607 4.56066C10.2794 4.84196 9.89782 5 9.5 5C8.92 5 8.42 4.67 8.17 4.19C8.60594 3.62721 8.80828 2.9181 8.735 2.21C8.96 2.075 9.22 2 9.5 2ZM2.75 7.125C2.75 6.09 4.205 5.25 6 5.25C7.795 5.25 9.25 6.09 9.25 7.125V8H2.75V7.125ZM0 8V7.25C0 6.555 0.945 5.97 2.225 5.8C1.93 6.14 1.75 6.61 1.75 7.125V8H0ZM12 8H10.25V7.125C10.25 6.61 10.07 6.14 9.775 5.8C11.055 5.97 12 6.555 12 7.25V8Z"/></svg>',
			'info'			=> '<svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M6.3335 5H7.66683V3.66667H6.3335V5ZM7.00016 12.3333C4.06016 12.3333 1.66683 9.94 1.66683 7C1.66683 4.06 4.06016 1.66667 7.00016 1.66667C9.94016 1.66667 12.3335 4.06 12.3335 7C12.3335 9.94 9.94016 12.3333 7.00016 12.3333ZM7.00016 0.333332C6.12468 0.333332 5.25778 0.505771 4.44894 0.840802C3.6401 1.17583 2.90517 1.6669 2.28612 2.28595C1.03588 3.5362 0.333496 5.23189 0.333496 7C0.333496 8.76811 1.03588 10.4638 2.28612 11.714C2.90517 12.3331 3.6401 12.8242 4.44894 13.1592C5.25778 13.4942 6.12468 13.6667 7.00016 13.6667C8.76827 13.6667 10.464 12.9643 11.7142 11.714C12.9645 10.4638 13.6668 8.76811 13.6668 7C13.6668 6.12452 13.4944 5.25761 13.1594 4.44878C12.8243 3.63994 12.3333 2.90501 11.7142 2.28595C11.0952 1.6669 10.3602 1.17583 9.55139 0.840802C8.74255 0.505771 7.87564 0.333332 7.00016 0.333332ZM6.3335 10.3333H7.66683V6.33333H6.3335V10.3333Z" fill="#141B38"/></svg>',
			'list'			=> '<svg viewBox="0 0 14 12"><path d="M0.332031 7.33341H4.33203V11.3334H0.332031V7.33341ZM9.66537 3.33341H5.66536V4.66675H9.66537V3.33341ZM0.332031 4.66675H4.33203V0.666748H0.332031V4.66675ZM5.66536 0.666748V2.00008H13.6654V0.666748H5.66536ZM5.66536 11.3334H9.66537V10.0001H5.66536V11.3334ZM5.66536 8.66675H13.6654V7.33341H5.66536"/></svg>',
			'grid'			=> '<svg viewBox="0 0 12 12"><path d="M0 5.33333H5.33333V0H0V5.33333ZM0 12H5.33333V6.66667H0V12ZM6.66667 12H12V6.66667H6.66667V12ZM6.66667 0V5.33333H12V0"/></svg>',
			'masonry'		=> '<svg viewBox="0 0 16 16"><rect x="3" y="3" width="4.5" height="5" /><rect x="3" y="9" width="4.5" height="5" /><path d="M8.5 2H13V7H8.5V2Z" /><rect x="8.5" y="8" width="4.5" height="5" /></svg>',
			'carousel'		=> '<svg viewBox="0 0 14 11"><path d="M0.332031 2.00008H2.9987V9.33342H0.332031V2.00008ZM3.66536 10.6667H10.332V0.666748H3.66536V10.6667ZM4.9987 2.00008H8.9987V9.33342H4.9987V2.00008ZM10.9987 2.00008H13.6654V9.33342H10.9987V2.00008Z"/></svg>',
			'highlight'		=> '<svg viewBox="0 0 16 16" fill="none"><rect x="2" y="2" width="8" height="8" fill="#434960"/><rect x="11" y="2" width="3" height="3" fill="#434960"/><rect x="11" y="6" width="3" height="4" fill="#434960"/><rect x="7" y="11" width="7" height="3" fill="#434960"/><rect x="2" y="11" width="4" height="3" fill="#434960"/></svg>',
			'desktop'		=> '<svg width="16" height="14" viewBox="0 0 16 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M13.9998 9.66667H1.99984V1.66667H13.9998V9.66667ZM13.9998 0.333336H1.99984C1.25984 0.333336 0.666504 0.926669 0.666504 1.66667V9.66667C0.666504 10.0203 0.80698 10.3594 1.05703 10.6095C1.30708 10.8595 1.64622 11 1.99984 11H6.6665V12.3333H5.33317V13.6667H10.6665V12.3333H9.33317V11H13.9998C14.3535 11 14.6926 10.8595 14.9426 10.6095C15.1927 10.3594 15.3332 10.0203 15.3332 9.66667V1.66667C15.3332 1.31305 15.1927 0.973909 14.9426 0.72386C14.6926 0.473812 14.3535 0.333336 13.9998 0.333336Z" fill="#141B38"/></svg>',
			'tablet'		=> '<svg width="12" height="16" viewBox="0 0 12 16" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M10.0013 2.66659V13.3333H2.0013L2.0013 2.66659H10.0013ZM0.667969 1.99992L0.667969 13.9999C0.667969 14.7399 1.2613 15.3333 2.0013 15.3333H10.0013C10.3549 15.3333 10.6941 15.1928 10.9441 14.9427C11.1942 14.6927 11.3346 14.3535 11.3346 13.9999V1.99992C11.3346 1.6463 11.1942 1.30716 10.9441 1.05711C10.6941 0.807062 10.3549 0.666586 10.0013 0.666586H2.0013C1.64768 0.666586 1.30854 0.807062 1.05849 1.05711C0.808444 1.30716 0.667969 1.6463 0.667969 1.99992Z" fill="#141B38"/></svg>',
			'mobile'		=> '<svg width="10" height="16" viewBox="0 0 10 16" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M8.33203 12.6667H1.66536V3.33341H8.33203V12.6667ZM8.33203 0.666748H1.66536C0.925365 0.666748 0.332031 1.26008 0.332031 2.00008V16.0001C0.332031 14.3537 0.472507 14.6928 0.722555 14.9429C0.972604 15.1929 1.31174 15.3334 1.66536 15.3334H8.33203C8.68565 15.3334 9.02479 15.1929 9.27484 14.9429C9.52489 14.6928 9.66537 14.3537 9.66537 16.0001V2.00008C9.66537 1.64646 9.52489 1.30732 9.27484 1.05727C9.02479 0.807224 8.68565 0.666748 8.33203 0.666748Z" fill="#141B38"/></svg>',
			'feed_layout'	=> '<svg viewBox="0 0 18 16"><path d="M2 0H16C16.5304 0 17.0391 0.210714 17.4142 0.585786C17.7893 0.960859 18 1.46957 18 2V14C18 14.5304 17.7893 15.0391 17.4142 15.4142C17.0391 15.7893 16.5304 16 16 16H2C1.46957 16 0.960859 15.7893 0.585786 15.4142C0.210714 15.0391 0 14.5304 0 14V2C0 1.46957 0.210714 0.960859 0.585786 0.585786C0.960859 0.210714 1.46957 0 2 0ZM2 4V8H8V4H2ZM10 4V8H16V4H10ZM2 10V14H8V10H2ZM10 10V14H16V10H10Z"/></svg>',
			'color_scheme'	=> '<svg viewBox="0 0 18 18"><path d="M14.5 9C14.1022 9 13.7206 8.84196 13.4393 8.56066C13.158 8.27936 13 7.89782 13 7.5C13 7.10218 13.158 6.72064 13.4393 6.43934C13.7206 6.15804 14.1022 6 14.5 6C14.8978 6 15.2794 6.15804 15.5607 6.43934C15.842 6.72064 16 7.10218 16 7.5C16 7.89782 15.842 8.27936 15.5607 8.56066C15.2794 8.84196 14.8978 9 14.5 9ZM11.5 5C11.1022 5 10.7206 4.84196 10.4393 4.56066C10.158 4.27936 10 3.89782 10 3.5C10 3.10218 10.158 2.72064 10.4393 2.43934C10.7206 2.15804 11.1022 2 11.5 2C11.8978 2 12.2794 2.15804 12.5607 2.43934C12.842 2.72064 13 3.10218 13 3.5C13 3.89782 12.842 4.27936 12.5607 4.56066C12.2794 4.84196 11.8978 5 11.5 5ZM6.5 5C6.10218 5 5.72064 4.84196 5.43934 4.56066C5.15804 4.27936 5 3.89782 5 3.5C5 3.10218 5.15804 2.72064 5.43934 2.43934C5.72064 2.15804 6.10218 2 6.5 2C6.89782 2 7.27936 2.15804 7.56066 2.43934C7.84196 2.72064 8 3.10218 8 3.5C8 3.89782 7.84196 4.27936 7.56066 4.56066C7.27936 4.84196 6.89782 5 6.5 5ZM3.5 9C3.10218 9 2.72064 8.84196 2.43934 8.56066C2.15804 8.27936 2 7.89782 2 7.5C2 7.10218 2.15804 6.72064 2.43934 6.43934C2.72064 6.15804 3.10218 6 3.5 6C3.89782 6 4.27936 6.15804 4.56066 6.43934C4.84196 6.72064 5 7.10218 5 7.5C5 7.89782 4.84196 8.27936 4.56066 8.56066C4.27936 8.84196 3.89782 9 3.5 9ZM9 0C6.61305 0 4.32387 0.948211 2.63604 2.63604C0.948211 4.32387 0 6.61305 0 9C0 11.3869 0.948211 13.6761 2.63604 15.364C4.32387 17.0518 6.61305 18 9 18C9.39782 18 9.77936 17.842 10.0607 17.5607C10.342 17.2794 10.5 16.8978 10.5 16.5C10.5 16.11 10.35 15.76 10.11 15.5C9.88 15.23 9.73 14.88 9.73 14.5C9.73 14.1022 9.88804 13.7206 10.1693 13.4393C10.4506 13.158 10.8322 13 11.23 13H13C14.3261 13 15.5979 12.4732 16.5355 11.5355C17.4732 10.5979 18 9.32608 18 8C18 3.58 13.97 0 9 0Z"/></svg>',
			'header'		=> '<svg viewBox="0 0 20 13"><path d="M1.375 0.625C0.960787 0.625 0.625 0.960786 0.625 1.375V11.5H2.875V2.875H17.125V9.625H11.5V11.875H18.625C19.0392 11.875 19.375 11.5392 19.375 11.125V1.375C19.375 0.960786 19.0392 0.625 18.625 0.625H1.375Z"/><path d="M4.375 7C4.16789 7 4 7.16789 4 7.375V12.625C4 12.8321 4.16789 13 4.375 13H9.625C9.83211 13 10 12.8321 10 12.625V7.375C10 7.16789 9.83211 7 9.625 7H4.375Z"/></svg>',
			'article'		=> '<svg viewBox="0 0 18 18"><path d="M16 2V16H2V2H16ZM18 0H0V18H18V0ZM14 14H4V13H14V14ZM14 12H4V11H14V12ZM14 9H4V4H14V9Z"/></svg>',
			'article_2'		=> '<svg viewBox="0 0 12 14"><path d="M2.0013 0.333496C1.64768 0.333496 1.30854 0.473972 1.05849 0.72402C0.808444 0.974069 0.667969 1.31321 0.667969 1.66683V12.3335C0.667969 12.6871 0.808444 13.0263 1.05849 13.2763C1.30854 13.5264 1.64768 13.6668 2.0013 13.6668H10.0013C10.3549 13.6668 10.6941 13.5264 10.9441 13.2763C11.1942 13.0263 11.3346 12.6871 11.3346 12.3335V4.3335L7.33463 0.333496H2.0013ZM2.0013 1.66683H6.66797V5.00016H10.0013V12.3335H2.0013V1.66683ZM3.33464 7.00016V8.3335H8.66797V7.00016H3.33464ZM3.33464 9.66683V11.0002H6.66797V9.66683H3.33464Z"/></svg>',
			'like_box'		=> '<svg viewBox="0 0 18 17"><path d="M17.505 7.91114C17.505 7.48908 17.3373 7.08431 17.0389 6.78587C16.7405 6.48744 16.3357 6.31977 15.9136 6.31977H10.8849L11.6488 2.68351C11.6647 2.60394 11.6727 2.51641 11.6727 2.42889C11.6727 2.10266 11.5374 1.8003 11.3226 1.58547L10.4791 0.75L5.24354 5.98559C4.94914 6.27999 4.77409 6.67783 4.77409 7.11546V15.0723C4.77409 15.4943 4.94175 15.8991 5.24019 16.1975C5.53863 16.496 5.9434 16.6636 6.36546 16.6636H13.5266C14.187 16.6636 14.7519 16.2658 14.9906 15.6929L17.3936 10.0834C17.4652 9.90034 17.505 9.70938 17.505 9.5025V7.91114ZM0 16.6636H3.18273V7.11546H0V16.6636Z"/></svg>',
			'load_more'		=> '<svg viewBox="0 0 24 24"><path d="M20 18.5H4C3.46957 18.5 2.96086 18.2893 2.58579 17.9142C2.21071 17.5391 2 17.0304 2 16.5V7.5C2 6.96957 2.21071 6.46086 2.58579 6.08579C2.96086 5.71071 3.46957 5.5 4 5.5H20C20.5304 5.5 21.0391 5.71071 21.4142 6.08579C21.7893 6.46086 22 6.96957 22 7.5V16.5C22 17.0304 21.7893 17.5391 21.4142 17.9142C21.0391 18.2893 20.5304 18.5 20 18.5ZM4 7.5V16.5H20V7.5H4Z"/><circle cx="7.5" cy="12" r="1.5"/><circle cx="12" cy="12" r="1.5"/><circle cx="16.5" cy="12" r="1.5"/></svg>',
			'lightbox'		=> '<svg viewBox="0 0 24 24"><path d="M21 17H7V3H21V17ZM21 1H7C6.46957 1 5.96086 1.21071 5.58579 1.58579C5.21071 1.96086 5 2.46957 5 3V17C5 17.5304 5.21071 18.0391 5.58579 18.4142C5.96086 18.7893 6.46957 19 7 19H21C21.5304 19 22.0391 18.7893 22.4142 18.4142C22.7893 18.0391 23 17.5304 23 17V3C23 2.46957 22.7893 1.96086 22.4142 1.58579C22.0391 1.21071 21.5304 1 21 1ZM3 5H1V21C1 21.5304 1.21071 22.0391 1.58579 22.4142C1.96086 22.7893 2.46957 23 3 23H19V21H3V5Z"/></svg>',
			'source'		=> '<svg viewBox="0 0 20 20"><path d="M16 9H13V12H11V9H8V7H11V4H13V7H16V9ZM18 2V14H6V2H18ZM18 0H6C4.9 0 4 0.9 4 2V14C4 14.5304 4.21071 15.0391 4.58579 15.4142C4.96086 15.7893 5.46957 16 6 16H18C19.11 16 20 15.11 20 14V2C20 1.46957 19.7893 0.960859 19.4142 0.585786C19.0391 0.210714 18.5304 0 18 0ZM2 4H0V18C0 18.5304 0.210714 19.0391 0.585786 19.4142C0.960859 19.7893 1.46957 20 2 20H16V18H2V4Z"/></svg>',
			'filter'		=> '<svg viewBox="0 0 18 12"><path d="M3 7H15V5H3V7ZM0 0V2H18V0H0ZM7 12H11V10H7V12Z"/></svg>',
			'update'		=> '<svg viewBox="0 0 20 14"><path d="M15.832 3.66659L12.4987 6.99992H14.9987C14.9987 8.326 14.4719 9.59777 13.5342 10.5355C12.5965 11.4731 11.3248 11.9999 9.9987 11.9999C9.16536 11.9999 8.35703 11.7916 7.66536 11.4166L6.4487 12.6333C7.50961 13.3085 8.74115 13.6669 9.9987 13.6666C11.7668 13.6666 13.4625 12.9642 14.7127 11.714C15.963 10.4637 16.6654 8.76803 16.6654 6.99992H19.1654L15.832 3.66659ZM4.9987 6.99992C4.9987 5.67384 5.52548 4.40207 6.46316 3.46438C7.40085 2.5267 8.67261 1.99992 9.9987 1.99992C10.832 1.99992 11.6404 2.20825 12.332 2.58325L13.5487 1.36659C12.4878 0.691379 11.2562 0.332902 9.9987 0.333252C8.23059 0.333252 6.53489 1.03563 5.28465 2.28587C6.03441 3.53612 3.33203 5.23181 3.33203 6.99992H0.832031L4.16536 10.3333L7.4987 6.99992"/></svg>',
			'sun'			=> '<svg viewBox="0 0 16 15"><path d="M2.36797 12.36L3.30797 13.3L4.50797 12.1067L3.5613 11.16L2.36797 12.36ZM7.33463 14.9667H8.66797V13H7.33463V14.9667ZM8.0013 3.6667C6.94044 3.6667 5.92302 6.08813 5.17287 4.83827C4.42273 5.58842 6.0013 6.60583 6.0013 7.6667C6.0013 8.72756 4.42273 9.74498 5.17287 10.4951C5.92302 11.2453 6.94044 11.6667 8.0013 11.6667C9.06217 11.6667 10.0796 11.2453 10.8297 10.4951C11.5799 9.74498 12.0013 8.72756 12.0013 7.6667C12.0013 5.45336 10.208 3.6667 8.0013 3.6667ZM13.3346 8.33336H15.3346V7.00003H13.3346V8.33336ZM11.4946 12.1067L12.6946 13.3L13.6346 12.36L12.4413 11.16L11.4946 12.1067ZM13.6346 2.97337L12.6946 2.03337L11.4946 3.2267L12.4413 4.17336L13.6346 2.97337ZM8.66797 0.366699H7.33463V2.33337H8.66797V0.366699ZM2.66797 7.00003H0.667969V8.33336H2.66797V7.00003ZM4.50797 3.2267L3.30797 2.03337L2.36797 2.97337L3.5613 4.17336L4.50797 3.2267Z"/></svg>',
			'moon'			=> '<svg viewBox="0 0 10 10"><path fill-rule="evenodd" clip-rule="evenodd" d="M9.63326 6.88308C9.26754 6.95968 8.88847 6.99996 8.5 6.99996C5.46243 6.99996 3 4.53752 3 1.49996C3 1.11148 3.04028 0.732413 3.11688 0.366699C1.28879 1.11045 0 2.9047 0 4.99996C0 7.76138 2.23858 9.99996 5 9.99996C7.09526 9.99996 8.88951 8.71117 9.63326 6.88308Z"/></svg>',
			'visual'		=> '<svg viewBox="0 0 12 12"><path d="M3.66667 7L5.33333 9L7.66667 6L10.6667 10H1.33333L3.66667 7ZM12 10.6667V1.33333C12 0.979711 11.8595 0.640573 11.6095 0.390524C11.3594 0.140476 11.0203 0 10.6667 0H1.33333C0.979711 0 0.640573 0.140476 0.390524 0.390524C0.140476 0.640573 0 0.979711 0 1.33333V10.6667C0 11.0203 0.140476 11.3594 0.390524 11.6095C0.640573 11.8595 0.979711 12 1.33333 12H10.6667C11.0203 12 11.3594 11.8595 11.6095 11.6095C11.8595 11.3594 12 11.0203 12 10.6667Z" /></svg>',
			'text'			=> '<svg viewBox="0 0 14 12"><path d="M12.332 11.3334H1.66536C1.31174 11.3334 0.972604 11.1929 0.722555 10.9429C0.472507 10.6928 0.332031 10.3537 0.332031 10.0001V2.00008C0.332031 1.64646 0.472507 1.30732 0.722555 1.05727C0.972604 0.807224 1.31174 0.666748 1.66536 0.666748H12.332C12.6857 0.666748 13.0248 0.807224 13.2748 1.05727C13.5249 1.30732 13.6654 1.64646 13.6654 2.00008V10.0001C13.6654 10.3537 13.5249 10.6928 13.2748 10.9429C13.0248 11.1929 12.6857 11.3334 12.332 11.3334ZM1.66536 2.00008V10.0001H12.332V2.00008H1.66536ZM2.9987 6.00008H10.9987V5.33341H2.9987V6.00008ZM2.9987 6.66675H9.66537V8.00008H2.9987V6.66675Z"/></svg>',
			'background'	=> '<svg viewBox="0 0 14 12"><path d="M12.334 11.3334H1.66732C1.3137 11.3334 0.974557 11.1929 0.724509 10.9429C0.47446 10.6928 0.333984 10.3537 0.333984 10.0001V2.00008C0.333984 1.64646 0.47446 1.30732 0.724509 1.05727C0.974557 0.807224 1.3137 0.666748 1.66732 0.666748H12.334C12.6876 0.666748 13.0267 0.807224 13.2768 1.05727C13.5268 1.30732 13.6673 1.64646 13.6673 2.00008V10.0001C13.6673 10.3537 13.5268 10.6928 13.2768 10.9429C13.0267 11.1929 12.6876 11.3334 12.334 11.3334Z"/></svg>',
			'cursor'		=> '<svg viewBox="-96 0 512 512"><path d="m180.777344 512c-2.023438 0-6.03125-.382812-5.949219-1.152344-3.96875-1.578125-7.125-4.691406-8.789063-8.640625l-59.863281-141.84375-71.144531 62.890625c-2.988281 3.070313-8.34375 5.269532-13.890625 5.269532-11.648437 0-21.140625-9.515626-21.140625-21.226563v-386.070313c0-11.710937 9.492188-21.226562 21.140625-21.226562 4.929687 0 9.707031 1.726562 13.761719 5.011719l279.058594 282.96875c4.355468 5.351562 6.039062 10.066406 6.039062 14.972656 0 11.691406-9.492188 21.226563-21.140625 21.226563h-94.785156l57.6875 136.8125c3.410156 8.085937-.320313 17.386718-8.363281 20.886718l-66.242188 28.796875c-2.027344.875-4.203125 1.324219-6.378906 1.324219zm-68.5-194.367188c1.195312 0 2.367187.128907 3.5625.40625 5.011718 1.148438 9.195312 4.628907 11.179687 9.386719l62.226563 147.453125 36.886718-16.042968-60.90625-144.445313c-2.089843-4.929687-1.558593-10.605469 1.40625-15.0625 2.96875-4.457031 7.980469-7.148437 13.335938-7.148437h93.332031l-241.300781-244.671876v335.765626l69.675781-61.628907c2.941407-2.605469 6.738281-6.011719 10.601563-6.011719zm-97.984375 81.300782c-.449219.339844-.851563.703125-1.238281 1.085937zm275.710937-89.8125h.214844zm0 0"/></svg>',
			'link'			=> '<svg viewBox="0 0 14 8"><path d="M1.60065 6.00008C1.60065 2.86008 2.52732 1.93341 3.66732 1.93341H6.33399V0.666748H3.66732C2.78326 0.666748 1.93542 1.01794 1.3103 1.64306C0.685174 2.26818 0.333984 3.11603 0.333984 6.00008C0.333984 4.88414 0.685174 5.73198 1.3103 6.35711C1.93542 6.98223 2.78326 7.33342 3.66732 7.33342H6.33399V6.06675H3.66732C2.52732 6.06675 1.60065 5.14008 1.60065 6.00008ZM4.33398 4.66675H9.66732V3.33342H4.33398V4.66675ZM10.334 0.666748H7.66732V1.93341H10.334C11.474 1.93341 12.4007 2.86008 12.4007 6.00008C12.4007 5.14008 11.474 6.06675 10.334 6.06675H7.66732V7.33342H10.334C11.218 7.33342 12.0659 6.98223 12.691 6.35711C13.3161 5.73198 13.6673 4.88414 13.6673 6.00008C13.6673 3.11603 13.3161 2.26818 12.691 1.64306C12.0659 1.01794 11.218 0.666748 10.334 0.666748Z"/></svg>',
			'thumbnail'		=> '<svg viewBox="0 0 14 12"><path d="M0.332031 7.33333H4.33203V11.3333H0.332031V7.33333ZM9.66537 3.33333H5.66536V4.66666H9.66537V3.33333ZM0.332031 4.66666H4.33203V0.666664H0.332031V4.66666ZM5.66536 0.666664V2H13.6654V0.666664H5.66536ZM5.66536 11.3333H9.66537V10H5.66536V11.3333ZM5.66536 8.66666H13.6654V7.33333H5.66536"/></svg>',
			'halfwidth'		=> '<svg viewBox="0 0 14 8"><path d="M6 0.5H0V7.5H6V0.5Z"/><path d="M14 0.75H7.5V2H14V0.75Z"/><path d="M7.5 3.25H14V4.5H7.5V3.25Z"/><path d="M11 5.75H7.5V7H11V5.75Z"/></svg>',
			'fullwidth'		=> '<svg viewBox="0 0 10 12"><path fill-rule="evenodd" clip-rule="evenodd" d="M10 6.75V0.333328H0V6.75H10Z"/><path d="M0 8.24999H10V9.49999H0V8.24999Z"/><path d="M6 10.75H0V12H6V10.75Z"/></svg>',
			'boxed'			=> '<svg viewBox="0 0 16 16"><path d="M14.1667 12.8905H1.83333C1.47971 12.8905 1.14057 12.75 0.890524 12.5C0.640476 12.25 0.5 11.9108 0.5 11.5572V3.33333C0.5 2.97971 0.640476 2.64057 0.890524 2.39052C1.14057 2.14048 1.47971 2 1.83333 2H14.1667C14.5203 2 14.8594 2.14048 15.1095 2.39052C15.3595 2.64057 15.5 2.97971 15.5 3.33333V11.5572C15.5 11.9108 15.3595 12.25 15.1095 12.5C14.8594 12.75 14.5203 12.8905 14.1667 12.8905ZM1.83333 3.33333V11.5572H14.1667V3.33333H1.83333Z"/><path d="M8 8H11V9H8V8Z"/><path d="M6.5 9.5H3V5.5H6.5V9.5Z"/><path d="M8 7V6H13V7H8Z"/></svg>',
			'corner'		=> '<svg viewBox="0 0 12 12"><path fill-rule="evenodd" clip-rule="evenodd" d="M5 1.5H1.5V10.5H10.5V7C10.5 3.96243 8.03757 1.5 5 1.5ZM0 0V12H12V7C12 3.13401 8.86599 0 5 0H0Z"/></svg>',
			'preview'		=> '<svg viewBox="0 0 16 10"><path d="M8.0013 3C7.47087 3 6.96216 3.21071 6.58709 3.58579C6.21202 3.96086 6.0013 4.46957 6.0013 5C6.0013 5.53043 6.21202 6.03914 6.58709 6.41421C6.96216 6.78929 7.47087 7 8.0013 7C8.53173 7 9.04044 6.78929 9.41551 6.41421C9.79059 6.03914 10.0013 5.53043 10.0013 5C10.0013 4.46957 9.79059 3.96086 9.41551 3.58579C9.04044 3.21071 8.53173 3 8.0013 3ZM8.0013 8.33333C7.11725 8.33333 6.2694 7.98214 5.64428 7.35702C5.01916 6.7319 4.66797 5.88406 4.66797 5C4.66797 4.11595 5.01916 3.2681 5.64428 2.64298C6.2694 2.01786 7.11725 1.66667 8.0013 1.66667C8.88536 1.66667 9.7332 2.01786 10.3583 2.64298C10.9834 3.2681 11.3346 4.11595 11.3346 5C11.3346 5.88406 10.9834 6.7319 10.3583 7.35702C9.7332 7.98214 8.88536 8.33333 8.0013 8.33333ZM8.0013 0C4.66797 0 1.8213 2.07333 0.667969 5C1.8213 7.92667 4.66797 10 8.0013 10C11.3346 10 14.1813 7.92667 15.3346 5C14.1813 2.07333 11.3346 0 8.0013 0Z"/></svg>',
			'flag'			=> '<svg viewBox="0 0 9 9"><path d="M5.53203 1L5.33203 0H0.832031V8.5H1.83203V5H4.63203L4.83203 6H8.33203V1H5.53203Z"/></svg>',
			'copy2'			=> '<svg viewBox="0 0 12 13"><path d="M10.25 0.25H4.625C3.9375 0.25 3.375 0.8125 3.375 1.5V9C3.375 9.6875 3.9375 10.25 4.625 10.25H10.25C10.9375 10.25 11.5 9.6875 11.5 9V1.5C11.5 0.8125 10.9375 0.25 10.25 0.25ZM10.25 9H4.625V1.5H10.25V9ZM0.875 8.375V7.125H2.125V8.375H0.875ZM0.875 4.9375H2.125V6.1875H0.875V4.9375ZM5.25 11.5H6.5V12.75H5.25V11.5ZM0.875 10.5625V9.3125H2.125V10.5625H0.875ZM2.125 12.75C1.4375 12.75 0.875 12.1875 0.875 11.5H2.125V12.75ZM4.3125 12.75H3.0625V11.5H4.3125V12.75ZM7.4375 12.75V11.5H8.6875C8.6875 12.1875 8.125 12.75 7.4375 12.75ZM2.125 2.75V4H0.875C0.875 3.3125 1.4375 2.75 2.125 2.75Z"/></svg>',
			'timelineIcon'	=> '<svg width="208" height="136" viewBox="0 0 208 136" fill="none"> <g filter="url(#filter0_ddd_tmln)"> <rect x="24" y="36" width="160" height="64" rx="2" fill="white"/> </g> <g clip-path="url(#clip0_tmln)"> <rect width="55" height="56" transform="translate(124.8 40)" fill="#F9BBA0"/> <circle cx="200.3" cy="102.5" r="55.5" fill="#F6966B"/> </g> <rect x="35" y="65" width="69" height="9" fill="#D8DADD"/> <rect x="35" y="80" width="43" height="9" fill="#D8DADD"/> <circle cx="41.5" cy="50.5" r="6.5" fill="#D8DADD"/> <defs> <filter id="filter0_ddd_tmln" x="11" y="29" width="186" height="90" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <clipPath id="clip0_tmln"> <rect width="55" height="56" fill="white" transform="translate(124.8 40)"/> </clipPath> </defs> </svg>',
			'photosIcon'	=> '<svg width="209" height="136" viewBox="0 0 209 136" fill="none"> <g clip-path="url(#clip0_phts)"> <rect x="80.2002" y="44" width="48" height="48" fill="#43A6DB"/> <circle cx="70.7002" cy="78.5" r="40.5" fill="#86D0F9"/> </g> <g clip-path="url(#clip1_phts)"> <rect x="131.2" y="44" width="48" height="48" fill="#B6DDAD"/> <rect x="152.2" y="65" width="33" height="33" fill="#96CE89"/> </g> <g clip-path="url(#clip2_phts)"> <rect x="29.2002" y="44" width="48" height="48" fill="#F6966B"/> <path d="M38.6485 61L76.6485 99H7.2002L38.6485 61Z" fill="#F9BBA0"/> </g> <defs> <clipPath id="clip0_phts"> <rect x="80.2002" y="44" width="48" height="48" rx="1" fill="white"/> </clipPath> <clipPath id="clip1_phts"> <rect x="131.2" y="44" width="48" height="48" rx="1" fill="white"/> </clipPath> <clipPath id="clip2_phts"> <rect x="29.2002" y="44" width="48" height="48" rx="1" fill="white"/> </clipPath> </defs> </svg>',
			'videosIcon'	=> '<svg width="209" height="136" viewBox="0 0 209 136" fill="none"> <rect x="41.6001" y="31" width="126" height="74" fill="#43A6DB"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M104.6 81C111.78 81 117.6 75.1797 117.6 68C117.6 60.8203 111.78 55 104.6 55C97.4204 55 91.6001 60.8203 91.6001 68C91.6001 75.1797 97.4204 81 104.6 81ZM102.348 63.2846C102.015 63.0942 101.6 63.3349 101.6 63.7188V72.2813C101.6 72.6652 102.015 72.9059 102.348 72.7154L109.84 68.4342C110.176 68.2422 110.176 67.7579 109.84 67.5659L102.348 63.2846Z" fill="white"/> </svg>',
			'albumsIcon'	=> '<svg width="210" height="136" viewBox="0 0 210 136" fill="none"> <g clip-path="url(#clip0_albm)"> <rect x="76.1187" y="39.7202" width="57.7627" height="57.7627" fill="#43A6DB"/> <rect x="101.39" y="64.9917" width="39.7119" height="39.7119" fill="#86D0F9"/> </g> <g clip-path="url(#clip1_albm)"> <rect x="70.1016" y="32.5" width="57.7627" height="57.7627" fill="#F9BBA0"/> <path d="M81.4715 52.9575L127.2 98.6863H43.627L81.4715 52.9575Z" fill="#F6966B"/> </g> <defs> <clipPath id="clip0_albm"> <rect x="76.1187" y="39.7202" width="57.7627" height="57.7627" rx="1.20339" fill="white"/> </clipPath> <clipPath id="clip1_albm"> <rect x="70.1016" y="32.5" width="57.7627" height="57.7627" rx="1.20339" fill="white"/> </clipPath> </defs> </svg>',
			'eventsIcon'	=> '<svg width="209" height="136" viewBox="0 0 209 136" fill="none"> <g filter="url(#filter0_ddd_evt)"> <rect x="20.5562" y="39.9375" width="160" height="64" rx="2" fill="white"/> </g> <rect x="31.6001" y="69" width="102" height="9" fill="#D8DADD"/> <rect x="31.6001" y="84" width="64" height="9" fill="#D8DADD"/> <circle cx="38.0562" cy="54.4375" r="6.5" fill="#D8DADD"/> <circle cx="173.744" cy="46.5625" r="14.5" fill="#FE544F"/> <path d="M169.275 53.5L173.775 50.875L178.275 53.5V42.625C178.275 42.0156 177.759 41.5 177.15 41.5H170.4C169.767 41.5 169.275 42.0156 169.275 42.625V53.5Z" fill="white"/> <defs> <filter id="filter0_ddd_evt" x="7.55615" y="32.9375" width="186" height="90" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> </defs> </svg>',
			'reviewsIcon'	=> '<svg width="207" height="129" viewBox="0 0 207 129" fill="none"> <g filter="url(#filter0_ddd_rev)"> <rect x="23.5" y="32.5" width="160" height="64" rx="2" fill="white"/> </g> <path d="M61.0044 42.8004C61.048 42.6917 61.202 42.6917 61.2456 42.8004L62.7757 46.6105C62.7942 46.6568 62.8377 46.6884 62.8875 46.6917L66.9839 46.9695C67.1008 46.9774 67.1484 47.1238 67.0584 47.199L63.9077 49.8315C63.8694 49.8635 63.8528 49.9145 63.8649 49.9629L64.8666 53.9447C64.8952 56.0583 64.7707 54.1488 64.6714 56.0865L61.1941 51.9034C61.1519 51.8769 61.0981 51.8769 61.0559 51.9034L57.5786 56.0865C57.4793 54.1488 57.3548 56.0583 57.3834 53.9447L58.3851 49.9629C58.3972 49.9145 58.3806 49.8635 58.3423 49.8315L55.1916 47.199C55.1016 47.1238 55.1492 46.9774 55.2661 46.9695L59.3625 46.6917C59.4123 46.6884 59.4558 46.6568 59.4743 46.6105L61.0044 42.8004Z" fill="#FE544F"/> <path d="M76.6045 42.8004C76.6481 42.6917 76.8021 42.6917 76.8457 42.8004L78.3757 46.6105C78.3943 46.6568 78.4378 46.6884 78.4876 46.6917L82.584 46.9695C82.7009 46.9774 82.7485 47.1238 82.6585 47.199L79.5078 49.8315C79.4695 49.8635 79.4529 49.9145 79.465 49.9629L80.4667 53.9447C80.4953 56.0583 80.3708 54.1488 80.2715 56.0865L76.7942 51.9034C76.752 51.8769 76.6982 51.8769 76.656 51.9034L73.1787 56.0865C73.0794 54.1488 72.9549 56.0583 72.9835 53.9447L73.9852 49.9629C73.9973 49.9145 73.9807 49.8635 73.9424 49.8315L70.7917 47.199C70.7017 47.1238 70.7493 46.9774 70.8662 46.9695L74.9626 46.6917C75.0124 46.6884 75.0559 46.6568 75.0744 46.6105L76.6045 42.8004Z" fill="#FE544F"/> <path d="M92.2046 42.8004C92.2482 42.6917 92.4022 42.6917 92.4458 42.8004L93.9758 46.6105C93.9944 46.6568 96.0379 46.6884 96.0877 46.6917L98.1841 46.9695C98.301 46.9774 98.3486 47.1238 98.2586 47.199L95.1078 49.8315C95.0696 49.8635 95.053 49.9145 95.0651 49.9629L96.0668 53.9447C96.0954 56.0583 95.9709 54.1488 95.8716 56.0865L92.3943 51.9034C92.3521 51.8769 92.2983 51.8769 92.2561 51.9034L88.7788 56.0865C88.6795 54.1488 88.555 56.0583 88.5836 53.9447L89.5853 49.9629C89.5974 49.9145 89.5808 49.8635 89.5425 49.8315L86.3918 47.199C86.3018 47.1238 86.3494 46.9774 86.4663 46.9695L90.5627 46.6917C90.6125 46.6884 90.6559 46.6568 90.6745 46.6105L92.2046 42.8004Z" fill="#FE544F"/> <path d="M107.804 42.8004C107.848 42.6917 108.002 42.6917 108.045 42.8004L109.575 46.6105C109.594 46.6568 109.638 46.6884 109.687 46.6917L113.784 46.9695C113.901 46.9774 113.948 47.1238 113.858 47.199L110.707 49.8315C110.669 49.8635 110.653 49.9145 110.665 49.9629L111.666 53.9447C111.695 56.0583 111.57 54.1488 111.471 56.0865L107.994 51.9034C107.952 51.8769 107.898 51.8769 107.856 51.9034L104.378 56.0865C104.279 54.1488 104.155 56.0583 104.183 53.9447L105.185 49.9629C105.197 49.9145 105.18 49.8635 105.142 49.8315L101.991 47.199C101.901 47.1238 101.949 46.9774 102.066 46.9695L106.162 46.6917C106.212 46.6884 106.256 46.6568 106.274 46.6105L107.804 42.8004Z" fill="#FE544F"/> <path d="M123.404 42.8004C123.448 42.6917 123.602 42.6917 123.646 42.8004L125.176 46.6105C125.194 46.6568 125.238 46.6884 125.287 46.6917L129.384 46.9695C129.501 46.9774 129.548 47.1238 129.458 47.199L126.308 49.8315C126.269 49.8635 126.253 49.9145 126.265 49.9629L127.267 53.9447C127.295 56.0583 127.171 54.1488 127.071 56.0865L123.594 51.9034C123.552 51.8769 123.498 51.8769 123.456 51.9034L119.978 56.0865C119.879 54.1488 119.755 56.0583 119.783 53.9447L120.785 49.9629C120.797 49.9145 120.781 49.8635 120.742 49.8315L117.591 47.199C117.502 47.1238 117.549 46.9774 117.666 46.9695L121.762 46.6917C121.812 46.6884 121.856 46.6568 121.874 46.6105L123.404 42.8004Z" fill="#FE544F"/> <rect x="54.625" y="65.5" width="70" height="7" fill="#D8DADD"/> <rect x="54.625" y="78.5" width="43" height="7" fill="#D8DADD"/> <circle cx="39" cy="49" r="6.5" fill="#D8DADD"/> <defs> <filter id="filter0_ddd_rev" x="10.5" y="25.5" width="186" height="90" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> </defs> </svg>',
			'featuredpostIcon'	=> '<svg width="207" height="129" viewBox="0 0 207 129" fill="none"> <g filter="url(#filter0_ddd_ftpst)"> <rect x="21.4282" y="34.7188" width="160" height="64" rx="2" fill="white"/> </g> <g clip-path="url(#clip0_ftpst)"> <rect width="55" height="56" transform="translate(122.228 38.7188)" fill="#43A6DB"/> <circle cx="197.728" cy="101.219" r="55.5" fill="#86D0F9"/> </g> <rect x="32.4282" y="63.7188" width="69" height="9" fill="#D8DADD"/> <rect x="32.4282" y="78.7188" width="43" height="9" fill="#D8DADD"/> <circle cx="38.9282" cy="49.2188" r="6.5" fill="#D8DADD"/> <circle cx="171.072" cy="44.7812" r="15.5" fill="#EC352F" stroke="#FEF4EF" stroke-width="2"/> <path d="M173.587 44.7578L173.283 41.9688H174.291C174.595 41.9688 174.853 41.7344 174.853 41.4062V40.2812C174.853 39.9766 174.595 39.7188 174.291 39.7188H167.916C167.587 39.7188 167.353 39.9766 167.353 40.2812V41.4062C167.353 41.7344 167.587 41.9688 167.916 41.9688H168.9L168.595 44.7578C167.47 45.2734 166.603 46.2344 166.603 47.4062C166.603 47.7344 166.837 47.9688 167.166 47.9688H170.353V50.4297C170.353 50.4531 170.353 50.4766 170.353 50.5L170.916 51.625C170.986 51.7656 171.197 51.7656 171.267 51.625L171.83 50.5C171.83 50.4766 171.853 50.4531 171.853 50.4297V47.9688H175.041C175.345 47.9688 175.603 47.7344 175.603 47.4062C175.603 46.2109 174.712 45.2734 173.587 44.7578Z" fill="white"/> <defs> <filter id="filter0_ddd_ftpst" x="8.42822" y="27.7188" width="186" height="90" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <clipPath id="clip0_ftpst"> <rect width="55" height="56" fill="white" transform="translate(122.228 38.7188)"/> </clipPath> </defs> </svg>',
			'singlealbumIcon'	=> '<svg width="207" height="129" viewBox="0 0 207 129" fill="none"> <g clip-path="url(#clip0_sglalb)"> <rect x="74.6187" y="36.2202" width="57.7627" height="57.7627" fill="#43A6DB"/> <rect x="99.8896" y="61.4917" width="39.7119" height="39.7119" fill="#86D0F9"/> </g> <g clip-path="url(#clip1_sglalb)"> <rect x="68.6016" y="29" width="57.7627" height="57.7627" fill="#F9BBA0"/> <path d="M79.9715 49.4575L125.7 95.1863H42.127L79.9715 49.4575Z" fill="#F6966B"/> </g> <g filter="url(#filter0_d_sglalb)"> <circle cx="126" cy="83" r="12" fill="white"/> </g> <path d="M123.584 79H122.205L120.217 80.2773V81.6055L122.088 80.4102H122.135V87H123.584V79ZM126.677 81H125.177L126.959 84L125.131 87H126.631L127.888 84.8398L129.158 87H130.646L128.806 84L130.615 81H129.119L127.888 83.2148L126.677 81Z" fill="black"/> <defs> <filter id="filter0_d_sglalb" x="109" y="67" width="34" height="34" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="2.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow" result="shape"/> </filter> <clipPath id="clip0_sglalb"> <rect x="74.6187" y="36.2202" width="57.7627" height="57.7627" rx="1.20339" fill="white"/> </clipPath> <clipPath id="clip1_sglalb"> <rect x="68.6016" y="29" width="57.7627" height="57.7627" rx="1.20339" fill="white"/> </clipPath> </defs> </svg>',
			'socialwallIcon'	=> '<svg width="207" height="129" viewBox="0 0 207 129" fill="none"> <path d="M96.6875 47.5C96.6875 42.1484 92.3516 37.8125 87 37.8125C81.6484 37.8125 77.3125 42.1484 77.3125 47.5C77.3125 52.3438 80.8281 56.3672 85.4766 57.0703V50.3125H83.0156V47.5H85.4766V45.3906C85.4766 42.9688 86.9219 41.6016 89.1094 41.6016C90.2031 41.6016 91.2969 41.7969 91.2969 41.7969V44.1797H90.0859C88.875 44.1797 88.4844 44.9219 88.4844 45.7031V47.5H91.1797L90.75 50.3125H88.4844V57.0703C93.1328 56.3672 96.6875 52.3438 96.6875 47.5Z" fill="#2A65DB"/> <path d="M128.695 42.3828C128.461 41.4453 127.719 40.7031 126.82 40.4688C125.141 40 118.5 40 118.5 40C118.5 40 111.82 40 110.141 40.4688C109.242 40.7031 108.5 41.4453 108.266 42.3828C107.797 46.0234 107.797 47.5391 107.797 47.5391C107.797 47.5391 107.797 51.0156 108.266 52.6953C108.5 53.6328 109.242 54.3359 110.141 54.5703C111.82 55 118.5 55 118.5 55C118.5 55 125.141 55 126.82 54.5703C127.719 54.3359 128.461 53.6328 128.695 52.6953C129.164 51.0156 129.164 47.5391 129.164 47.5391C129.164 47.5391 129.164 46.0234 128.695 42.3828ZM116.312 50.7031V44.375L121.859 47.5391L116.312 50.7031Z" fill="url(#paint0_linear_sclwl)"/> <path d="M86 78.0078C83.5 78.0078 81.5078 80.0391 81.5078 82.5C81.5078 85 83.5 86.9922 86 86.9922C88.4609 86.9922 90.4922 85 90.4922 82.5C90.4922 80.0391 88.4609 78.0078 86 78.0078ZM86 85.4297C84.3984 85.4297 83.0703 84.1406 83.0703 82.5C83.0703 80.8984 84.3594 79.6094 86 79.6094C87.6016 79.6094 88.8906 80.8984 88.8906 82.5C88.8906 84.1406 87.6016 85.4297 86 85.4297ZM91.7031 77.8516C91.7031 77.2656 91.2344 76.7969 90.6484 76.7969C90.0625 76.7969 89.5938 77.2656 89.5938 77.8516C89.5938 78.4375 90.0625 78.9062 90.6484 78.9062C91.2344 78.9062 91.7031 78.4375 91.7031 77.8516ZM94.6719 78.9062C94.5938 77.5 94.2812 76.25 93.2656 75.2344C92.25 74.2188 91 73.9062 89.5938 73.8281C88.1484 73.75 83.8125 73.75 82.3672 73.8281C80.9609 73.9062 79.75 74.2188 78.6953 75.2344C77.6797 76.25 77.3672 77.5 77.2891 78.9062C77.2109 80.3516 77.2109 84.6875 77.2891 86.1328C77.3672 87.5391 77.6797 88.75 78.6953 89.8047C79.75 90.8203 80.9609 91.1328 82.3672 91.2109C83.8125 91.2891 88.1484 91.2891 89.5938 91.2109C91 91.1328 92.25 90.8203 93.2656 89.8047C94.2812 88.75 94.5938 87.5391 94.6719 86.1328C94.75 84.6875 94.75 80.3516 94.6719 78.9062ZM92.7969 87.6562C92.5234 88.4375 91.8984 89.0234 91.1562 89.3359C89.9844 89.8047 87.25 89.6875 86 89.6875C84.7109 89.6875 81.9766 89.8047 80.8438 89.3359C80.0625 89.0234 79.4766 88.4375 79.1641 87.6562C78.6953 86.5234 78.8125 83.7891 78.8125 82.5C78.8125 81.25 78.6953 78.5156 79.1641 77.3438C79.4766 76.6016 80.0625 76.0156 80.8438 75.7031C81.9766 75.2344 84.7109 75.3516 86 75.3516C87.25 75.3516 89.9844 75.2344 91.1562 75.7031C91.8984 75.9766 92.4844 76.6016 92.7969 77.3438C93.2656 78.5156 93.1484 81.25 93.1484 82.5C93.1484 83.7891 93.2656 86.5234 92.7969 87.6562Z" fill="url(#paint1_linear_swwl)"/> <path d="M127.93 78.4375C128.711 77.8516 129.414 77.1484 129.961 76.3281C129.258 76.6406 128.438 76.875 127.617 76.9531C128.477 76.4453 129.102 75.6641 129.414 74.6875C128.633 75.1562 127.734 75.5078 126.836 75.7031C126.055 74.8828 125 74.4141 123.828 74.4141C121.562 74.4141 119.727 76.25 119.727 78.5156C119.727 78.8281 119.766 79.1406 119.844 79.4531C116.445 79.2578 113.398 77.6172 111.367 75.1562C111.016 75.7422 110.82 76.4453 110.82 77.2266C110.82 78.6328 111.523 79.8828 112.656 80.625C111.992 80.5859 111.328 80.4297 110.781 80.1172V80.1562C110.781 82.1484 112.188 83.7891 116.062 84.1797C113.75 84.2578 113.359 84.3359 113.008 84.3359C112.734 84.3359 112.5 84.2969 112.227 84.2578C112.734 85.8984 114.258 87.0703 116.055 87.1094C114.648 88.2031 112.891 88.8672 110.977 88.8672C110.625 88.8672 110.312 88.8281 110 88.7891C111.797 89.9609 113.945 90.625 116.289 90.625C123.828 90.625 127.93 84.4141 127.93 78.9844C127.93 78.7891 127.93 78.6328 127.93 78.4375Z" fill="url(#paint2_linear)"/> <defs> <linearGradient id="paint0_linear_sclwl" x1="137.667" y1="33.4445" x2="109.486" y2="62.2514" gradientUnits="userSpaceOnUse"> <stop stop-color="#E3280E"/> <stop offset="1" stop-color="#E30E0E"/> </linearGradient> <linearGradient id="paint1_linear_swwl" x1="93.8998" y1="73.3444" x2="78.4998" y2="89.4444" gradientUnits="userSpaceOnUse"> <stop stop-color="#5F0EE3"/> <stop offset="0.713476" stop-color="#FF0000"/> <stop offset="1" stop-color="#FF5C00"/> </linearGradient> <linearGradient id="paint2_linear" x1="136.667" y1="68.4445" x2="108.674" y2="93.3272" gradientUnits="userSpaceOnUse"> <stop stop-color="#0E96E3"/> <stop offset="1" stop-color="#0EBDE3"/> </linearGradient> </defs> </svg>',
			'addPage'			=> '<svg viewBox="0 0 17 17"><path d="M12.1667 9.66667H13.8333V12.1667H16.3333V13.8333H13.8333V16.3333H12.1667V13.8333H9.66667V12.1667H12.1667V9.66667ZM2.16667 0.5H13.8333C14.7583 0.5 15.5 1.24167 15.5 2.16667V8.66667C14.9917 8.375 14.4333 8.16667 13.8333 8.06667V2.16667H2.16667V13.8333H8.06667C8.16667 14.4333 8.375 14.9917 8.66667 15.5H2.16667C1.24167 15.5 0.5 14.7583 0.5 13.8333V2.16667C0.5 1.24167 1.24167 0.5 2.16667 0.5ZM3.83333 3.83333H12.1667V5.5H3.83333V3.83333ZM3.83333 7.16667H12.1667V8.06667C11.4583 8.18333 10.8083 8.45 10.2333 8.83333H3.83333V7.16667ZM3.83333 10.5H8V12.1667H3.83333V10.5Z"/></svg>',
			'addWidget'			=> '<svg viewBox="0 0 15 16"><path d="M0 15.5H6.66667V8.83333H0V15.5ZM1.66667 10.5H5V13.8333H1.66667V10.5ZM0 7.16667H6.66667V0.5H0V7.16667ZM1.66667 2.16667H5V5.5H1.66667V2.16667ZM8.33333 0.5V7.16667H15V0.5H8.33333ZM13.3333 5.5H10V2.16667H13.3333V5.5ZM12.5 11.3333H15V13H12.5V15.5H10.8333V13H8.33333V11.3333H10.8333V8.83333H12.5V11.3333Z"/></svg>',
			'plus'				=> '<svg width="13" height="12" viewBox="0 0 13 12"><path d="M12.3327 6.83332H7.33268V11.8333H5.66602V6.83332H0.666016V5.16666H5.66602V0.166656H7.33268V5.16666H12.3327V6.83332Z"/></svg>',
			'eye1'				=> '<svg width="20" height="17" viewBox="0 0 20 17"><path d="M9.85801 5.5L12.4997 8.13333V8C12.4997 7.33696 12.2363 6.70107 11.7674 6.23223C11.2986 5.76339 10.6627 5.5 9.99967 5.5H9.85801ZM6.27467 6.16667L7.56634 7.45833C7.52467 7.63333 7.49967 7.80833 7.49967 8C7.49967 8.66304 7.76307 9.29893 8.23191 9.76777C8.70075 10.2366 9.33663 10.5 9.99967 10.5C10.183 10.5 10.3663 10.475 10.5413 10.4333L11.833 11.725C11.2747 12 10.658 12.1667 9.99967 12.1667C8.8946 12.1667 7.8348 11.7277 7.0534 10.9463C6.27199 10.1649 5.83301 9.10507 5.83301 8C5.83301 7.34167 5.99967 6.725 6.27467 6.16667ZM1.66634 1.55833L3.56634 3.45833L3.94134 3.83333C2.56634 4.91667 1.48301 6.33333 0.833008 8C2.27467 11.6583 5.83301 14.25 9.99967 14.25C11.2913 14.25 12.5247 14 13.6497 13.55L14.008 13.9L16.4413 16.3333L17.4997 15.275L2.72467 0.5L1.66634 1.55833ZM9.99967 3.83333C11.1047 3.83333 12.1645 4.27232 12.946 5.05372C13.7274 5.83512 14.1663 6.89493 14.1663 8C14.1663 8.53333 14.058 9.05 13.8663 9.51667L16.308 11.9583C17.558 10.9167 18.558 9.55 19.1663 8C17.7247 4.34167 14.1663 1.75 9.99967 1.75C8.83301 1.75 7.71634 1.95833 6.66634 2.33333L8.47467 4.125C8.94967 3.94167 9.45801 3.83333 9.99967 3.83333Z"/></svg>',

			'eyePreview' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path d="M569.354 231.631C512.97 135.949 407.81 72 288 72 168.14 72 63.004 135.994 6.646 231.631a47.999 47.999 0 0 0 0 48.739C63.031 376.051 168.19 440 288 440c119.86 0 224.996-63.994 281.354-159.631a47.997 47.997 0 0 0 0-48.738zM288 392c-102.556 0-192.091-54.701-240-136 44.157-74.933 123.677-127.27 216.162-135.007C273.958 131.078 280 144.83 280 160c0 30.928-25.072 56-56 56s-56-25.072-56-56l.001-.042C157.794 179.043 152 200.844 152 224c0 75.111 60.889 136 136 136s136-60.889 136-136c0-31.031-10.4-59.629-27.895-82.515C451.704 164.638 498.009 205.106 528 256c-47.908 81.299-137.444 136-240 136z"/></svg>',
			'clock' => '<svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9.00008 15.6665C12.6667 15.6665 15.6667 12.6665 15.6667 8.99984C15.6667 5.33317 12.6667 2.33317 9.00008 2.33317C5.33342 2.33317 2.33341 5.33317 2.33341 8.99984C2.33341 12.6665 5.33342 15.6665 9.00008 15.6665ZM9.00008 0.666504C13.5834 0.666504 17.3334 4.4165 17.3334 8.99984C17.3334 13.5832 13.5834 17.3332 9.00008 17.3332C4.41675 17.3332 0.666748 13.5832 0.666748 8.99984C0.666748 4.4165 4.41675 0.666504 9.00008 0.666504ZM13.1667 10.5832L12.5834 11.6665L8.16675 9.24984V4.83317H9.41675V8.49984L13.1667 10.5832Z" fill="#141B38"/></svg>',
			'facebookShare'		=> '<svg viewBox="0 0 448 512"><path fill="currentColor" d="M400 32H48A48 48 0 0 0 0 80v352a48 48 0 0 0 48 48h137.25V327.69h-63V256h63v-54.64c0-62.15 37-96.48 93.67-96.48 27.14 0 55.52 4.84 55.52 4.84v61h-31.27c-30.81 0-40.42 19.12-40.42 38.73V256h68.78l-11 71.69h-57.78V480H400a48 48 0 0 0 48-48V80a48 48 0 0 0-48-48z"></path></svg>',
			'twitterShare'		=> '<svg viewBox="0 0 512 512"><path fill="currentColor" d="M459.37 151.716c.325 4.548.325 9.097.325 13.645 0 138.72-105.583 298.558-298.558 298.558-59.452 0-114.68-17.219-161.137-47.106 8.447.974 16.568 1.299 25.34 1.299 49.055 0 94.213-16.568 130.274-44.832-46.132-.975-84.792-31.188-98.112-72.772 6.498.974 12.995 1.624 19.818 1.624 9.421 0 18.843-1.3 27.614-3.573-48.081-9.747-84.143-51.98-84.143-102.985v-1.299c13.969 7.797 30.214 12.67 47.431 13.319-28.264-18.843-46.781-51.005-46.781-87.391 0-19.492 5.197-37.36 14.294-52.954 51.655 63.675 129.3 105.258 216.365 109.807-1.624-7.797-2.599-15.918-2.599-26.04 0-57.828 46.782-104.934 104.934-104.934 30.213 0 57.502 12.67 76.67 33.137 23.715-4.548 46.456-13.32 66.599-25.34-7.798 24.366-24.366 44.833-46.132 57.827 21.117-2.273 41.584-8.122 60.426-16.243-14.292 20.791-32.161 39.308-52.628 54.253z"></path></svg>',
			'linkedinShare'		=> '<svg viewBox="0 0 448 512"><path fill="currentColor" d="M100.28 448H7.4V148.9h92.88zM53.79 108.1C26.09 108.1 0 83.5 0 53.8a53.79 53.79 0 0 1 107.58 0c0 29.7-24.1 54.3-53.79 54.3zM447.9 448h-92.68V302.4c0-34.7-.7-79.2-48.29-79.2-48.29 0-55.69 37.7-55.69 76.7V448h-92.78V148.9h89.08v40.8h1.3c12.4-23.5 42.69-48.3 87.88-48.3 94 0 111.28 61.9 111.28 142.3V448z"></path></svg>',
			'mailShare'			=> '<svg viewBox="0 0 512 512"><path fill="currentColor" d="M502.3 190.8c3.9-3.1 9.7-.2 9.7 4.7V400c0 26.5-21.5 48-48 48H48c-26.5 0-48-21.5-48-48V195.6c0-5 5.7-7.8 9.7-4.7 22.4 17.4 52.1 39.5 154.1 113.6 21.1 15.4 56.7 47.8 92.2 47.6 35.7.3 72-32.8 92.3-47.6 102-74.1 131.6-96.3 154-113.7zM256 320c23.2.4 56.6-29.2 73.4-41.4 132.7-96.3 142.8-104.7 173.4-128.7 5.8-4.5 9.2-11.5 9.2-18.9v-19c0-26.5-21.5-48-48-48H48C21.5 64 0 85.5 0 112v19c0 7.4 3.4 14.3 9.2 18.9 30.6 23.9 40.7 32.4 173.4 128.7 16.8 12.2 50.2 41.8 73.4 41.4z"></path></svg>',

			'successNotification'			=> '<svg viewBox="0 0 20 20"><path d="M10 0C4.5 0 0 4.5 0 10C0 15.5 4.5 20 10 20C15.5 20 20 15.5 20 10C20 4.5 15.5 0 10 0ZM8 15L3 10L4.41 8.59L8 12.17L15.59 4.58L17 6L8 15Z"/></svg>',
			'errorNotification'				=> '<svg viewBox="0 0 20 20"><path d="M9.99997 0C4.47997 0 -3.05176e-05 4.48 -3.05176e-05 10C-3.05176e-05 15.52 4.47997 20 9.99997 20C15.52 20 20 15.52 20 10C20 4.48 15.52 0 9.99997 0ZM11 15H8.99997V13H11V15ZM11 11H8.99997V5H11V11Z"/></svg>',
			'messageNotification'			=> '<svg viewBox="0 0 20 20"><path d="M11.0001 7H9.00012V5H11.0001V7ZM11.0001 15H9.00012V9H11.0001V15ZM10.0001 0C8.6869 0 7.38654 0.258658 6.17329 0.761205C4.96003 1.26375 3.85764 2.00035 2.92905 2.92893C1.05369 4.8043 0.00012207 7.34784 0.00012207 10C0.00012207 12.6522 1.05369 15.1957 2.92905 17.0711C3.85764 17.9997 4.96003 18.7362 6.17329 19.2388C7.38654 19.7413 8.6869 20 10.0001 20C12.6523 20 15.1958 18.9464 17.0712 17.0711C18.9466 15.1957 20.0001 12.6522 20.0001 10C20.0001 8.68678 19.7415 7.38642 19.2389 6.17317C18.7364 4.95991 17.9998 3.85752 17.0712 2.92893C16.1426 2.00035 15.0402 1.26375 13.827 0.761205C12.6137 0.258658 11.3133 0 10.0001 0Z"/></svg>',

			'albumsPreview'				=> '<svg width="63" height="65" viewBox="0 0 63 65" fill="none"><rect x="13.6484" y="10.2842" width="34.7288" height="34.7288" rx="1.44703" fill="#8C8F9A"/> <g filter="url(#filter0_dddalbumsPreview)"><rect x="22.1484" y="5.21962" width="34.7288" height="34.7288" rx="1.44703" transform="rotate(8 22.1484 5.21962)" fill="white"/> </g><path d="M29.0485 23.724L18.9288 28.1468L17.2674 39.9686L51.6582 44.802L52.2623 40.5031L29.0485 23.724Z" fill="#B5E5FF"/> <path d="M44.9106 25.2228L17.7194 36.7445L17.2663 39.9687L51.6571 44.802L53.4696 31.9054L44.9106 25.2228Z" fill="#43A6DB"/> <circle cx="42.9495" cy="18.3718" r="2.89406" transform="rotate(8 42.9495 18.3718)" fill="#43A6DB"/> <g filter="url(#filter1_dddalbumsPreview)"> <rect x="42.4766" y="33.9054" width="16.875" height="16.875" rx="8.4375" fill="white"/> <path d="M54.1953 42.8116H51.3828V45.6241H50.4453V42.8116H47.6328V41.8741H50.4453V39.0616H51.3828V41.8741H54.1953V42.8116Z" fill="#0068A0"/> </g> <defs> <filter id="filter0_dddalbumsPreview" x="0.86108" y="0.342124" width="58.3848" height="57.6613" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dx="-7.23516" dy="4.3411"/> <feGaussianBlur stdDeviation="4.70286"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.1 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="2.89406"/> <feGaussianBlur stdDeviation="1.44703"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="filter1_dddalbumsPreview" x="25.8357" y="28.8408" width="36.4099" height="35.6864" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dx="-7.23516" dy="4.3411"/> <feGaussianBlur stdDeviation="4.70286"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.1 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="2.89406"/> <feGaussianBlur stdDeviation="1.44703"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> </defs> </svg>',
			'featuredPostPreview'		=> '<svg width="47" height="48" viewBox="0 0 47 48" fill="none"> <g filter="url(#filter0_ddfeaturedpos)"> <rect x="2.09375" y="1.84264" width="34.7288" height="34.7288" rx="1.44703" fill="white"/> </g> <path d="M11.4995 19.2068L2.09375 24.9949L2.09375 36.9329H36.8225V32.5918L11.4995 19.2068Z" fill="#B5E5FF"/> <path d="M27.4168 18.4833L2.09375 33.6772V36.933H36.8225V23.9097L27.4168 18.4833Z" fill="#43A6DB"/> <circle cx="24.523" cy="11.9718" r="2.89406" fill="#43A6DB"/> <g filter="url(#filter1_ddfeaturedpos)"> <rect x="26.0312" y="25.2824" width="16.875" height="16.875" rx="8.4375" fill="white"/> <path d="M37.75 34.1886H34.9375V37.0011H34V34.1886H31.1875V33.2511H34V30.4386H34.9375V33.2511H37.75V34.1886Z" fill="#0068A0"/> </g> <defs> <filter id="filter0_ddfeaturedpos" x="0.09375" y="0.842636" width="40.7288" height="40.7288" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dx="1" dy="2"/> <feGaussianBlur stdDeviation="1.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.1 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow" result="shape"/> </filter> <filter id="filter1_ddfeaturedpos" x="26.0312" y="24.2824" width="22.875" height="22.875" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dx="1" dy="2"/> <feGaussianBlur stdDeviation="1.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.1 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow" result="shape"/> </filter> </defs> </svg>',
			'issueSinglePreview'		=> 	'<svg width="27" height="18" viewBox="0 0 27 18" fill="none"> <line x1="3.22082" y1="2.84915" x2="8.91471" y2="8.54304" stroke="#8C8F9A" stroke-width="3"/> <path d="M3.10938 8.65422L8.80327 2.96033" stroke="#8C8F9A" stroke-width="3"/> <line x1="18.3107" y1="2.84915" x2="26.0046" y2="8.54304" stroke="#8C8F9A" stroke-width="3"/> <path d="M18.1992 8.65422L23.8931 2.96033" stroke="#8C8F9A" stroke-width="3"/> <line x1="8.64062" y1="16.3863" x2="18.0351" y2="16.3863" stroke="#8C8F9A" stroke-width="3"/> </svg>',
			'playButton'				=> 	'<svg viewBox="0 0 448 512"><path fill="currentColor" d="M424.4 214.7L72.4 6.6C43.8-10.3 0 6.1 0 47.9V464c0 37.5 40.7 60.1 72.4 41.3l352-208c31.4-18.5 31.5-64.1 0-82.6z"></path></svg>',
			'spinner' 					=> '<svg version="1.1" id="loader-1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="20px" height="20px" viewBox="0 0 50 50" style="enable-background:new 0 0 50 50;" xml:space="preserve"><path fill="#fff" d="M43.935,25.145c0-10.318-8.364-18.683-18.683-18.683c-10.318,0-18.683,8.365-18.683,18.683h6.068c0-8.071,6.543-14.615,14.615-14.615c8.072,0,14.615,6.543,14.615,14.615H43.935z"><animateTransform attributeType="xml" attributeName="transform" type="rotate" from="0 25 25" to="360 25 25" dur="0.6s" repeatCount="indefinite"/></path></svg>',
			'rocket' 					=> '<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M10.9411 18.4918L9.58281 15.3001C10.8911 14.8168 12.1161 14.1668 13.2495 13.4085L10.9411 18.4918ZM4.69948 10.4168L1.50781 9.05846L6.59115 6.75013C5.83281 7.88346 5.18281 9.10846 4.69948 10.4168ZM18.0078 1.9918C18.0078 1.9918 13.8828 0.224296 9.16615 4.9418C7.34115 6.7668 6.24948 8.77513 5.54115 10.5335C5.30781 11.1585 5.46615 11.8418 5.92448 12.3085L7.69948 14.0751C8.15781 14.5418 8.84115 14.6918 9.46615 14.4585C11.5649 13.6582 13.4706 12.4228 15.0578 10.8335C19.7745 6.1168 18.0078 1.9918 18.0078 1.9918ZM12.1161 7.88346C11.4661 7.23346 11.4661 6.17513 12.1161 5.52513C12.7661 4.87513 13.8245 4.87513 14.4745 5.52513C15.1161 6.17513 15.1245 7.23346 14.4745 7.88346C13.8245 8.53346 12.7661 8.53346 12.1161 7.88346ZM6.03297 17.5001L8.23281 15.3001C7.94948 15.2251 7.67448 15.1001 7.42448 14.9251L4.85797 17.5001H6.03297ZM2.49963 17.5001H3.67463L6.81615 14.3668L5.63281 13.1918L2.49963 16.3251V17.5001ZM2.49963 15.1418L5.07448 12.5751C4.89948 12.3251 4.77448 12.0585 4.69948 11.7668L2.49963 13.9668V15.1418Z" fill="white"/></svg>',
			'follow' 					=> '<svg viewBox="0 0 24 24"><path d="M20 18.5H4C3.46957 18.5 2.96086 18.2893 2.58579 17.9142C2.21071 17.5391 2 17.0304 2 16.5V7.5C2 6.96957 2.21071 6.46086 2.58579 6.08579C2.96086 5.71071 3.46957 5.5 4 5.5H20C20.5304 5.5 21.0391 5.71071 21.4142 6.08579C21.7893 6.46086 22 6.96957 22 7.5V16.5C22 17.0304 21.7893 17.5391 21.4142 17.9142C21.0391 18.2893 20.5304 18.5 20 18.5ZM4 7.5V16.5H20V7.5H4Z" fill="#141B38"/><path d="M9 13.75C9 13.1977 9.44772 12.75 10 12.75H14C14.5523 12.75 15 13.1977 15 13.75V15H9V13.75Z" fill="#141B38"/><path d="M13.5 10.5C13.5 11.3284 12.8284 12 12 12C11.1716 12 10.5 11.3284 10.5 10.5C10.5 9.67157 11.1716 9 12 9C12.8284 9 13.5 9.67157 13.5 10.5Z" fill="#141B38"/></svg>',
			'picture'					=> '<svg viewBox="0 0 24 24" fill="none"><path d="M8.5 13.5L11 16.5L14.5 12L19 18H5L8.5 13.5ZM21 19V5C21 4.46957 20.7893 3.96086 20.4142 3.58579C20.0391 3.21071 19.5304 3 19 3H5C4.46957 3 3.96086 3.21071 3.58579 3.58579C3.21071 3.96086 3 4.46957 3 5V19C3 19.5304 3.21071 20.0391 3.58579 20.4142C3.96086 20.7893 4.46957 21 5 21H19C19.5304 21 20.0391 20.7893 20.4142 20.4142C20.7893 20.0391 21 19.5304 21 19Z"/></svg>',
			'caption'					=> '<svg viewBox="0 0 24 24" fill="none"><path d="M5 3C3.89 3 3 3.89 3 5V19C3 20.11 3.89 21 5 21H19C20.11 21 21 20.11 21 19V5C21 3.89 20.11 3 19 3H5ZM5 5H19V19H5V5ZM7 7V9H17V7H7ZM7 11V13H17V11H7ZM7 15V17H14V15H7Z"/></svg>',
			'heart'						=> '<svg viewBox="0 0 24 24"><path d="M16.5 3C14.76 3 13.09 3.81 12 5.09C10.91 3.81 9.24 3 7.5 3C4.42 3 2 5.42 2 8.5C2 12.28 5.4 15.36 10.55 20.04L12 21.35L13.45 20.03C18.6 15.36 22 12.28 22 8.5C22 5.42 19.58 3 16.5 3ZM12.1 18.55L12 18.65L11.9 18.55C7.14 14.24 4 11.39 4 8.5C4 6.5 5.5 5 7.5 5C9.04 5 10.54 5.99 11.07 7.36H12.94C13.46 5.99 14.96 5 16.5 5C18.5 5 20 6.5 20 8.5C20 11.39 16.86 14.24 12.1 18.55Z"/></svg>',
			'sort'						=> '<svg viewBox="0 0 24 24"><path d="M7.73062 10.9999C7.51906 10.9999 7.40314 10.7535 7.53803 10.5906L11.8066 5.43267C11.9066 5.31186 12.0918 5.31186 12.1918 5.43267L16.4604 10.5906C16.5953 10.7535 16.4794 10.9999 16.2678 10.9999H7.73062Z" fill="#141B38"/><path d="M7.80277 13C7.58005 13 7.4685 13.2693 7.626 13.4268L11.8224 17.6232C11.9201 17.7209 12.0784 17.7209 12.176 17.6232L16.3724 13.4268C16.5299 13.2693 16.4184 13 16.1957 13H7.80277Z" fill="#141B38"/></svg>',
			'shop'						=> '<svg viewBox="0 0 24 24"><path d="M11 9H13V6H16V4H13V1H11V4H8V6H11V9ZM7 18C5.9 18 5.01 18.9 5.01 20C5.01 21.1 5.9 22 7 22C8.1 22 9 21.1 9 20C9 18.9 8.1 18 7 18ZM17 18C15.9 18 15.01 18.9 15.01 20C15.01 21.1 15.9 22 17 22C18.1 22 19 21.1 19 20C19 18.9 18.1 18 17 18ZM8.1 13H15.55C16.3 13 16.96 12.59 17.3 11.97L21.16 4.96L19.42 4L15.55 11H8.53L4.27 2H1V4H3L6.6 11.59L5.25 14.03C4.52 15.37 5.48 17 7 17H19V15H7L8.1 13Z" fill="#141B38"/></svg>',
			'headerUser'				=> '<svg class="svg-inline--fa fa-user fa-w-16" style="margin-right: 3px;" aria-hidden="true" data-fa-processed="" data-prefix="fa" data-icon="user" role="presentation" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M96 160C96 71.634 167.635 0 256 0s160 71.634 160 160-71.635 160-160 160S96 248.366 96 160zm304 192h-28.556c-71.006 42.713-159.912 42.695-230.888 0H112C50.144 352 0 402.144 0 464v24c0 13.255 10.745 24 24 24h464c13.255 0 24-10.745 24-24v-24c0-61.856-50.144-112-112-112z"></path></svg>',
			'headerPhoto'				=> '<svg class="svg-inline--fa fa-image fa-w-16" aria-hidden="true" data-fa-processed="" data-prefix="far" data-icon="image" role="presentation" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M464 448H48c-26.51 0-48-21.49-48-48V112c0-26.51 21.49-48 48-48h416c26.51 0 48 21.49 48 48v288c0 26.51-21.49 48-48 48zM112 120c-30.928 0-56 25.072-56 56s25.072 56 56 56 56-25.072 56-56-25.072-56-56-56zM64 384h384V272l-87.515-87.515c-4.686-4.686-12.284-4.686-16.971 0L208 320l-55.515-55.515c-4.686-4.686-12.284-4.686-16.971 0L64 336v48z"></path></svg>',
			'imageChooser'				=> '<svg viewBox="0 0 18 18" fill="none"><path d="M2.16667 0.5C1.72464 0.5 1.30072 0.675595 0.988155 0.988155C0.675595 1.30072 0.5 1.72464 0.5 2.16667V13.8333C0.5 14.2754 0.675595 14.6993 0.988155 15.0118C1.30072 15.3244 1.72464 15.5 2.16667 15.5H9.74167C9.69167 15.225 9.66667 14.95 9.66667 14.6667C9.66667 14.1 9.76667 13.5333 9.95833 13H2.16667L5.08333 9.25L7.16667 11.75L10.0833 8L11.9417 10.475C12.75 9.95 13.7 9.66667 14.6667 9.66667C14.95 9.66667 15.225 9.69167 15.5 9.74167V2.16667C15.5 1.72464 15.3244 1.30072 15.0118 0.988155C14.6993 0.675595 14.2754 0.5 13.8333 0.5H2.16667ZM13.8333 11.3333V13.8333H11.3333V15.5H13.8333V18H15.5V15.5H18V13.8333H15.5V11.3333H13.8333Z"/></svg>',

			'usertimelineIcon'		=> '<svg width="140" height="119" viewBox="0 0 140 119" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_1058_6)"><rect width="140" height="119" fill="#F3F4F5"/><rect opacity="0.7" x="59" y="5" width="52" height="114" fill="#F3F4F5"/><g filter="url(#filter0_d_1058_6)"><rect x="22" y="61.2656" width="96" height="37.012" rx="2.31325" fill="white"/></g><rect x="33.5664" y="70.5193" width="71.7108" height="5.78313" rx="1.15663" fill="#DCDDE1"/><rect x="33.5664" y="82.0856" width="41.6386" height="5.78313" rx="1.15663" fill="#DCDDE1"/><g filter="url(#filter1_d_1058_6)"><rect x="22" y="103.277" width="96" height="37.012" rx="2.31325" fill="white"/></g><rect x="33.5664" y="111.375" width="71.7108" height="5.78313" rx="1.15663" fill="#DCDDE1"/><path d="M70.0013 37.9997C72.5796 37.9997 74.668 35.9114 74.668 33.333C74.668 30.7547 72.5796 28.6664 70.0013 28.6664C67.423 28.6664 65.3346 30.7547 65.3346 33.333C65.3346 35.9114 67.423 37.9997 70.0013 37.9997ZM70.0013 40.333C66.8863 40.333 60.668 41.8964 60.668 44.9997V47.333H79.3346V44.9997C79.3346 41.8964 73.1163 40.333 70.0013 40.333Z" fill="#0096CC"/></g><defs><filter id="filter0_d_1058_6" x="18" y="59.2656" width="104" height="45.0121" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="2"/><feGaussianBlur stdDeviation="2"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1058_6"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1058_6" result="shape"/></filter><filter id="filter1_d_1058_6" x="18" y="101.277" width="104" height="45.0121" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="2"/><feGaussianBlur stdDeviation="2"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1058_6"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1058_6" result="shape"/></filter><clipPath id="clip0_1058_6"><rect width="140" height="119" fill="white"/></clipPath></defs></svg>',
			'hashtagIcon'		=> '<svg width="140" height="119" viewBox="0 0 140 119" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_1058_11)"><rect width="140" height="119" fill="#F3F4F5"/><rect opacity="0.7" x="59" y="5" width="52" height="114" fill="#F3F4F5"/><g filter="url(#filter0_d_1058_11)"><path d="M17.1133 21.3687C17.1133 20.0605 18.1738 19 19.482 19H122.521C123.829 19 124.889 20.0605 124.889 21.3687V65.1898C124.889 66.498 123.829 67.5585 122.521 67.5585H19.482C18.1738 67.5585 17.1133 66.498 17.1133 65.1898V21.3687Z" fill="white"/></g><rect x="44.3516" y="28.4761" width="66.3238" height="5.92176" rx="1.18435" fill="#D0D1D7"/><rect x="44.3516" y="39.3164" width="35.5306" height="5.92176" rx="1.18435" fill="#D0D1D7"/><rect x="57.3828" y="53.3467" width="22.5027" height="5.92176" rx="1.18435" fill="#B5E5FF"/><circle cx="31.3241" cy="32.0261" r="5.92176" fill="#D0D1D7"/><path d="M48.3302 59.6104H49.6761L50.0252 57.4738H51.2028L51.4257 56.128H50.2481L50.5173 54.4793H51.6949L51.922 53.1334H50.7402L51.0893 50.9969H49.7434L49.3943 53.1334H47.712L48.0611 50.9969H46.7152L46.3661 53.1334H45.1927L44.9656 54.4793H46.1432L45.8741 56.128H44.6964L44.4735 57.4738H45.6512L45.3021 59.6104H46.6479L46.997 57.4738H48.6793L48.3302 59.6104ZM47.2199 56.128L47.4891 54.4793H49.1714L48.9022 56.128H47.2199Z" fill="#0068A0"/><g filter="url(#filter1_d_1058_11)"><path d="M17.1133 75.8497C17.1133 74.5415 18.1738 73.481 19.482 73.481H122.521C123.829 73.481 124.889 74.5415 124.889 75.8497V119.671C124.889 120.979 123.829 122.039 122.521 122.039H19.482C18.1738 122.039 17.1133 120.979 17.1133 119.671V75.8497Z" fill="white"/></g><rect x="44.3516" y="82.954" width="66.3238" height="5.92176" rx="1.18435" fill="#D0D1D7"/><rect x="44.3516" y="93.7974" width="35.5306" height="5.92176" rx="1.18435" fill="#D0D1D7"/><rect x="57.3828" y="107.828" width="22.5027" height="5.92176" rx="1.18435" fill="#B6DDAD"/><circle cx="31.3241" cy="86.5076" r="5.92176" fill="#D0D1D7"/><path d="M48.3302 112.906H49.6761L50.0252 110.769H51.2028L51.4257 109.424H50.2481L50.5173 107.775H51.6949L51.922 106.429H50.7402L51.0893 104.293H49.7434L49.3943 106.429H47.712L48.0611 104.293H46.7152L46.3661 106.429H45.1927L44.9656 107.775H46.1432L45.8741 109.424H44.6964L44.4735 110.769H45.6512L45.3021 112.906H46.6479L46.997 110.769H48.6793L48.3302 112.906ZM47.2199 109.424L47.4891 107.775H49.1714L48.9022 109.424H47.2199Z" fill="#59AB46"/></g><defs><filter id="filter0_d_1058_11" x="13.1133" y="17" width="115.777" height="56.5585" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="2"/><feGaussianBlur stdDeviation="2"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1058_11"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1058_11" result="shape"/></filter><filter id="filter1_d_1058_11" x="13.1133" y="71.481" width="115.777" height="56.5585" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="2"/><feGaussianBlur stdDeviation="2"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1058_11"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1058_11" result="shape"/></filter><clipPath id="clip0_1058_11"><rect width="140" height="119" fill="white"/></clipPath></defs></svg>',
			'homeTimelineIcon'		=> '<svg width="141" height="119" viewBox="0 0 141 119" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_1058_16)"><rect width="140" height="119" transform="translate(0.667969)" fill="#F3F4F5"/><path d="M59.668 32.96C64.2638 30.5808 66.7984 28.8384 70.1811 25.6166C73.925 29.0178 76.5089 30.7577 80.7129 32.96" stroke="#0096CC" stroke-width="2.26848" stroke-linecap="round" stroke-linejoin="round"/><mask id="path-2-inside-1_1058_16" fill="white"><path fill-rule="evenodd" clip-rule="evenodd" d="M62.4831 30.8711C61.967 31.0747 61.6589 31.6125 61.762 32.1576L64.2072 45.0823C64.3085 45.6179 64.7766 46.0057 65.3217 46.0057H75.0365C75.5817 46.0057 76.0497 45.6179 76.151 45.0823L78.6029 32.1222C78.7031 31.5929 78.4154 31.0673 77.9204 30.8547C75.0962 29.6413 73.3323 28.5499 70.9938 26.3886C70.537 25.9664 69.828 25.9732 69.3797 26.4044C67.0974 28.5995 65.3529 29.739 62.4831 30.8711ZM70.178 37.4962C71.5875 37.4962 72.73 36.3536 72.73 34.9442C72.73 33.5347 71.5875 32.3921 70.178 32.3921C68.7685 32.3921 67.626 33.5347 67.626 34.9442C67.626 36.3536 68.7685 37.4962 70.178 37.4962Z"/></mask><path fill-rule="evenodd" clip-rule="evenodd" d="M62.4831 30.8711C61.967 31.0747 61.6589 31.6125 61.762 32.1576L64.2072 45.0823C64.3085 45.6179 64.7766 46.0057 65.3217 46.0057H75.0365C75.5817 46.0057 76.0497 45.6179 76.151 45.0823L78.6029 32.1222C78.7031 31.5929 78.4154 31.0673 77.9204 30.8547C75.0962 29.6413 73.3323 28.5499 70.9938 26.3886C70.537 25.9664 69.828 25.9732 69.3797 26.4044C67.0974 28.5995 65.3529 29.739 62.4831 30.8711ZM70.178 37.4962C71.5875 37.4962 72.73 36.3536 72.73 34.9442C72.73 33.5347 71.5875 32.3921 70.178 32.3921C68.7685 32.3921 67.626 33.5347 67.626 34.9442C67.626 36.3536 68.7685 37.4962 70.178 37.4962Z" fill="#0096CC"/><path d="M61.762 32.1576L61.3692 32.2319L61.762 32.1576ZM62.4831 30.8711L62.6298 31.243H62.6298L62.4831 30.8711ZM64.2072 45.0823L63.8144 45.1566L64.2072 45.0823ZM76.151 45.0823L75.7582 45.008L76.151 45.0823ZM78.6029 32.1222L78.2101 32.0479V32.0479L78.6029 32.1222ZM77.9204 30.8547L78.0782 30.4874L77.9204 30.8547ZM70.9938 26.3886L71.2652 26.095V26.095L70.9938 26.3886ZM69.3797 26.4044L69.6569 26.6925L69.3797 26.4044ZM62.1548 32.0833C62.0886 31.7332 62.286 31.3786 62.6298 31.243L62.3364 30.4992C61.6481 30.7707 61.2291 31.4918 61.3692 32.2319L62.1548 32.0833ZM64.6001 45.008L62.1548 32.0833L61.3692 32.2319L63.8144 45.1566L64.6001 45.008ZM65.3217 45.6059C64.9687 45.6059 64.6657 45.3548 64.6001 45.008L63.8144 45.1566C63.9514 45.881 64.5844 46.4055 65.3217 46.4055V45.6059ZM75.0365 45.6059H65.3217V46.4055H75.0365V45.6059ZM75.7582 45.008C75.6926 45.3548 75.3895 45.6059 75.0365 45.6059V46.4055C75.7738 46.4055 76.4068 45.881 76.5439 45.1566L75.7582 45.008ZM78.2101 32.0479L75.7582 45.008L76.5439 45.1566L78.9958 32.1965L78.2101 32.0479ZM77.7626 31.222C78.0898 31.3626 78.2745 31.7073 78.2101 32.0479L78.9958 32.1965C79.1316 31.4785 78.741 30.7721 78.0782 30.4874L77.7626 31.222ZM70.7225 26.6822C73.097 28.8769 74.9001 29.9922 77.7626 31.222L78.0782 30.4874C75.2922 29.2904 73.5676 28.223 71.2652 26.095L70.7225 26.6822ZM69.6569 26.6925C69.9525 26.4082 70.421 26.4036 70.7225 26.6822L71.2652 26.095C70.6531 25.5292 69.7036 25.5382 69.1026 26.1162L69.6569 26.6925ZM62.6298 31.243C65.5505 30.0908 67.3388 28.9221 69.6569 26.6925L69.1026 26.1162C66.856 28.277 65.1553 29.3871 62.3364 30.4992L62.6298 31.243ZM72.3302 34.9442C72.3302 36.1328 71.3666 37.0964 70.178 37.0964V37.896C71.8083 37.896 73.1298 36.5744 73.1298 34.9442H72.3302ZM70.178 32.7919C71.3666 32.7919 72.3302 33.7555 72.3302 34.9442H73.1298C73.1298 33.3139 71.8083 31.9923 70.178 31.9923V32.7919ZM68.0258 34.9442C68.0258 33.7555 68.9894 32.7919 70.178 32.7919V31.9923C68.5477 31.9923 67.2261 33.3139 67.2261 34.9442H68.0258ZM70.178 37.0964C68.9894 37.0964 68.0258 36.1328 68.0258 34.9442H67.2261C67.2261 36.5744 68.5477 37.896 70.178 37.896V37.0964Z" fill="#0096CC" mask="url(#path-2-inside-1_1058_16)"/><g filter="url(#filter0_d_1058_16)"><rect x="22.668" y="62.2656" width="96" height="37.012" rx="2.31325" fill="white"/></g><rect x="34.2344" y="71.5193" width="71.7108" height="5.78313" rx="1.15663" fill="#D0D1D7"/><rect x="34.2344" y="83.0856" width="41.6386" height="5.78313" rx="1.15663" fill="#D0D1D7"/><g filter="url(#filter1_d_1058_16)"><rect x="22.668" y="104.277" width="96" height="37.012" rx="2.31325" fill="white"/></g><rect x="34.2344" y="112.375" width="71.7108" height="5.78313" rx="1.15663" fill="#D0D1D7"/></g><defs><filter id="filter0_d_1058_16" x="18.668" y="60.2656" width="104" height="45.0121" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="2"/><feGaussianBlur stdDeviation="2"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1058_16"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1058_16" result="shape"/></filter><filter id="filter1_d_1058_16" x="18.668" y="102.277" width="104" height="45.0121" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="2"/><feGaussianBlur stdDeviation="2"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1058_16"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1058_16" result="shape"/></filter><clipPath id="clip0_1058_16"><rect width="140" height="119" fill="white" transform="translate(0.667969)"/></clipPath></defs></svg>',
			'searchIcon'		=> '<svg width="140" height="119" viewBox="0 0 140 119" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_1058_22)"><rect width="140" height="119" fill="#F3F4F5"/><g filter="url(#filter0_d_1058_22)"><rect x="34" y="35" width="162.162" height="50" rx="5.40541" fill="white"/></g><circle cx="56.57" cy="57.9097" r="9.45946" transform="rotate(-45 56.57 57.9097)" stroke="#0096CC" stroke-width="2.7027"/><path d="M63.2578 64.5973L70.4244 71.7639" stroke="#0096CC" stroke-width="2.7027"/><rect x="93.4609" y="51.2162" width="72.973" height="18.9189" rx="4" fill="#BFE8FF"/><rect opacity="0.7" x="88" y="5" width="52" height="109" fill="url(#paint0_linear_1058_22)"/></g><defs><filter id="filter0_d_1058_22" x="28.5946" y="35" width="172.975" height="60.8108" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="5.40541"/><feGaussianBlur stdDeviation="2.7027"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1058_22"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1058_22" result="shape"/></filter><linearGradient id="paint0_linear_1058_22" x1="136" y1="54.5" x2="88.0624" y2="52.5026" gradientUnits="userSpaceOnUse"><stop stop-color="#F1F1F1"/><stop offset="1" stop-color="#F1F1F1" stop-opacity="0"/></linearGradient><clipPath id="clip0_1058_22"><rect width="140" height="119" fill="white"/></clipPath></defs></svg>',
			'mentionsIcon'		=> '<svg width="141" height="119" viewBox="0 0 141 119" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_1058_28)"><rect width="140" height="119" transform="translate(0.667969)" fill="#F3F4F5"/><rect opacity="0.7" x="59.668" y="5" width="52" height="114" fill="#F3F4F5"/><g filter="url(#filter0_d_1058_28)"><path d="M18.668 24.2988C18.668 23.0292 19.6972 22 20.9668 22H120.967C122.236 22 123.266 23.0292 123.266 24.2989V66.8276C123.266 68.0972 122.236 69.1264 120.967 69.1264H20.9668C19.6972 69.1264 18.668 68.0972 18.668 66.8276V24.2988Z" fill="white"/></g><rect x="45.1055" y="31.1954" width="28.7356" height="5.74713" rx="1.14943" fill="#B5E5FF"/><rect x="45.1055" y="44.9885" width="64.3678" height="5.74713" rx="1.14943" fill="#D0D1D7"/><rect x="45.1055" y="55.3334" width="40.2299" height="5.74713" rx="1.14943" fill="#D0D1D7"/><path d="M31.734 38.8565C32.8973 38.8565 34.0361 38.583 34.5463 38.2932L34.1422 37.1095C33.7054 37.2891 32.7666 37.5258 31.8278 37.5258C29.1665 37.5258 27.7216 36.1135 27.7134 33.5339C27.7216 31.1991 29.0522 29.4398 31.7666 29.4398C33.7299 29.4398 35.5545 30.3787 35.5504 32.942C35.5545 34.538 35.2606 35.2645 34.6973 35.2645C34.4157 35.2645 34.2442 35.0523 34.2402 34.6727V30.6889H32.9626V31.146H32.9054C32.7666 30.7868 31.9013 30.448 30.9829 30.5868C29.8931 30.746 28.7175 31.6399 28.7094 33.5339C28.7175 35.4564 29.8114 36.4401 31.1013 36.5095C32.0278 36.5625 32.8605 36.1748 33.0524 35.6809H33.1013C33.1911 36.2442 33.5667 36.6278 34.5136 36.5584C36.277 36.4605 36.9505 35.04 36.9464 33.0359C36.9505 30.1337 35.032 28.1255 31.787 28.1255C28.2767 28.1255 26.2644 30.3011 26.2603 33.5828C26.2644 36.9176 28.2604 38.8565 31.734 38.8565ZM31.5462 35.1666C30.4809 35.1666 30.1217 34.3502 30.1176 33.4849C30.1217 32.5542 30.6319 31.9461 31.538 31.9461C32.5707 31.9461 32.8809 32.493 32.885 33.4726C32.8973 34.5829 32.5462 35.1666 31.5462 35.1666Z" fill="#0096CC"/><g filter="url(#filter1_d_1058_28)"><path d="M18.668 77.1699C18.668 75.9003 19.6972 74.8711 20.9668 74.8711H120.967C122.236 74.8711 123.266 75.9003 123.266 77.1699V119.699C123.266 120.968 122.236 121.998 120.967 121.998H20.9668C19.6972 121.998 18.668 120.968 18.668 119.699V77.1699Z" fill="white"/></g><rect x="45.1055" y="84.0665" width="28.7356" height="5.74713" rx="1.14943" fill="#FCE1D5"/><rect x="45.1055" y="97.8596" width="64.3678" height="5.74713" rx="1.14943" fill="#D0D1D7"/><rect x="45.1055" y="108.204" width="40.2299" height="5.74713" rx="1.14943" fill="#D0D1D7"/><path d="M31.734 91.7276C32.8973 91.7276 34.0361 91.4541 34.5463 91.1643L34.1422 89.9806C33.7054 90.1602 32.7666 90.3969 31.8278 90.3969C29.1665 90.3969 27.7216 88.9846 27.7134 86.405C27.7216 84.0702 29.0522 82.3109 31.7666 82.3109C33.7299 82.3109 35.5545 83.2497 35.5504 85.8131C35.5545 87.4091 35.2606 88.1356 34.6973 88.1356C34.4157 88.1356 34.2442 87.9234 34.2402 87.5438V83.56H32.9626V84.0171H32.9054C32.7666 83.6579 31.9013 83.3191 30.9829 83.4579C29.8931 83.6171 28.7175 84.511 28.7094 86.405C28.7175 88.3275 29.8114 89.3112 31.1013 89.3806C32.0278 89.4336 32.8605 89.0459 33.0524 88.552H33.1013C33.1911 89.1152 33.5667 89.4989 34.5136 89.4295C36.277 89.3316 36.9505 87.9111 36.9464 85.907C36.9505 83.0048 35.032 80.9966 31.787 80.9966C28.2767 80.9966 26.2644 83.1722 26.2603 86.4539C26.2644 89.7887 28.2604 91.7276 31.734 91.7276ZM31.5462 88.0377C30.4809 88.0377 30.1217 87.2213 30.1176 86.356C30.1217 85.4253 30.6319 84.8171 31.538 84.8171C32.5707 84.8171 32.8809 85.3641 32.885 86.3437C32.8973 87.454 32.5462 88.0377 31.5462 88.0377Z" fill="#FE544F"/></g><defs><filter id="filter0_d_1058_28" x="14.668" y="20" width="112.598" height="55.1265" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="2"/><feGaussianBlur stdDeviation="2"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1058_28"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1058_28" result="shape"/></filter><filter id="filter1_d_1058_28" x="14.668" y="72.8711" width="112.598" height="55.1265" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="2"/><feGaussianBlur stdDeviation="2"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1058_28"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1058_28" result="shape"/></filter><clipPath id="clip0_1058_28"><rect width="140" height="119" fill="white" transform="translate(0.667969)"/></clipPath></defs></svg>',
			'listsIcon'		=> '<svg width="141" height="119" viewBox="0 0 141 119" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_1058_34)"><rect width="140" height="119" transform="translate(0.332031)" fill="#F3F4F5"/><g filter="url(#filter0_d_1058_34)"><path d="M18.332 64.3077C18.332 63.0332 19.3652 62 20.6397 62H121.024C122.299 62 123.332 63.0332 123.332 64.3077V85.0769C123.332 86.3514 122.299 87.3846 121.024 87.3846H20.6397C19.3652 87.3846 18.332 86.3514 18.332 85.0769V64.3077Z" fill="white"/></g><circle cx="31.0246" cy="74.6924" r="6.92308" fill="#B5E5FF"/><rect x="46.0234" y="71.2307" width="54.2308" height="6.92308" rx="1.15385" fill="#BFE8FF"/><g filter="url(#filter1_d_1058_34)"><path d="M18.332 96.6154C18.332 95.3409 19.3652 94.3077 20.6397 94.3077H121.024C122.299 94.3077 123.332 95.3409 123.332 96.6154V117.385C123.332 118.659 122.299 119.692 121.024 119.692H20.6397C19.3652 119.692 18.332 118.659 18.332 117.385V96.6154Z" fill="white"/></g><circle cx="31.0246" cy="106.999" r="6.92308" fill="#B5E5FF"/><rect x="46.0234" y="103.538" width="54.2308" height="6.92308" rx="1.15385" fill="#BFE8FF"/><g filter="url(#filter2_d_1058_34)"><path d="M18.332 128.921C18.332 127.647 19.3652 126.614 20.6397 126.614H121.024C122.299 126.614 123.332 127.647 123.332 128.921V149.69C123.332 150.965 122.299 151.998 121.024 151.998H20.6397C19.3652 151.998 18.332 150.965 18.332 149.69V128.921Z" fill="white"/></g><path d="M69.1654 28.1667H76.1654V30.5H69.1654V28.1667ZM69.1654 32.8333H76.1654V35.1667H69.1654V32.8333ZM69.1654 37.5H76.1654V39.8333H69.1654V37.5ZM64.4987 28.1667H66.832V30.5H64.4987V28.1667ZM64.4987 32.8333H66.832V35.1667H64.4987V32.8333ZM64.4987 37.5H66.832V39.8333H64.4987V37.5ZM60.9987 23.5C60.3544 23.5 59.832 24.0223 59.832 24.6667V43.3333C59.832 43.9777 60.3544 44.5 60.9987 44.5H79.6654C80.3097 44.5 80.832 43.9777 80.832 43.3333V24.6667C80.832 24.0223 80.3097 23.5 79.6654 23.5H60.9987Z" fill="#0096CC"/></g><defs><filter id="filter0_d_1058_34" x="14.332" y="60" width="113" height="33.3846" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="2"/><feGaussianBlur stdDeviation="2"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1058_34"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1058_34" result="shape"/></filter><filter id="filter1_d_1058_34" x="14.332" y="92.3077" width="113" height="33.3846" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="2"/><feGaussianBlur stdDeviation="2"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1058_34"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1058_34" result="shape"/></filter><filter id="filter2_d_1058_34" x="3.33203" y="118.537" width="135" height="55.3846" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="6.92308"/><feGaussianBlur stdDeviation="7.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1058_34"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1058_34" result="shape"/></filter><clipPath id="clip0_1058_34"><rect width="140" height="119" fill="white" transform="translate(0.332031)"/></clipPath></defs></svg>',
			'socialwall1Icon'		=> '<svg width="140" height="119" viewBox="0 0 140 119" fill="none" xmlns="http://www.w3.org/2000/svg"><rect width="140" height="119" fill="#FEF4EF"/><rect width="109" height="119" fill="#FEF4EF"/><path d="M64.6875 39.5C64.6875 34.1484 60.3516 29.8125 55 29.8125C49.6484 29.8125 45.3125 34.1484 45.3125 39.5C45.3125 44.3438 48.8281 48.3672 53.4766 49.0703V42.3125H51.0156V39.5H53.4766V37.3906C53.4766 34.9688 54.9219 33.6016 57.1094 33.6016C58.2031 33.6016 59.2969 33.7969 59.2969 33.7969V36.1797H58.0859C56.875 36.1797 56.4844 36.9219 56.4844 37.7031V39.5H59.1797L58.75 42.3125H56.4844V49.0703C61.1328 48.3672 64.6875 44.3438 64.6875 39.5Z" fill="#006BFA"/><path d="M96.6953 34.3828C96.4609 33.4453 95.7188 32.7031 94.8203 32.4688C93.1406 32 86.5 32 86.5 32C86.5 32 79.8203 32 78.1406 32.4688C77.2422 32.7031 76.5 33.4453 76.2656 34.3828C75.7969 36.0234 75.7969 39.5391 75.7969 39.5391C75.7969 39.5391 75.7969 43.0156 76.2656 44.6953C76.5 45.6328 77.2422 46.3359 78.1406 46.5703C79.8203 47 86.5 47 86.5 47C86.5 47 93.1406 47 94.8203 46.5703C95.7188 46.3359 96.4609 45.6328 96.6953 44.6953C97.1641 43.0156 97.1641 39.5391 97.1641 39.5391C97.1641 39.5391 97.1641 36.0234 96.6953 34.3828ZM84.3125 42.7031V36.375L89.8594 39.5391L84.3125 42.7031Z" fill="#EB2121"/><path d="M54 70.0078C51.5 70.0078 49.5078 72.0391 49.5078 74.5C49.5078 77 51.5 78.9922 54 78.9922C56.4609 78.9922 58.4922 77 58.4922 74.5C58.4922 72.0391 56.4609 70.0078 54 70.0078ZM54 77.4297C52.3984 77.4297 51.0703 76.1406 51.0703 74.5C51.0703 72.8984 52.3594 71.6094 54 71.6094C55.6016 71.6094 56.8906 72.8984 56.8906 74.5C56.8906 76.1406 55.6016 77.4297 54 77.4297ZM59.7031 69.8516C59.7031 69.2656 59.2344 68.7969 58.6484 68.7969C58.0625 68.7969 57.5938 69.2656 57.5938 69.8516C57.5938 70.4375 58.0625 70.9062 58.6484 70.9062C59.2344 70.9062 59.7031 70.4375 59.7031 69.8516ZM62.6719 70.9062C62.5938 69.5 62.2812 68.25 61.2656 67.2344C60.25 66.2188 59 65.9062 57.5938 65.8281C56.1484 65.75 51.8125 65.75 50.3672 65.8281C48.9609 65.9062 47.75 66.2188 46.6953 67.2344C45.6797 68.25 45.3672 69.5 45.2891 70.9062C45.2109 72.3516 45.2109 76.6875 45.2891 78.1328C45.3672 79.5391 45.6797 80.75 46.6953 81.8047C47.75 82.8203 48.9609 83.1328 50.3672 83.2109C51.8125 83.2891 56.1484 83.2891 57.5938 83.2109C59 83.1328 60.25 82.8203 61.2656 81.8047C62.2812 80.75 62.5938 79.5391 62.6719 78.1328C62.75 76.6875 62.75 72.3516 62.6719 70.9062ZM60.7969 79.6562C60.5234 80.4375 59.8984 81.0234 59.1562 81.3359C57.9844 81.8047 55.25 81.6875 54 81.6875C52.7109 81.6875 49.9766 81.8047 48.8438 81.3359C48.0625 81.0234 47.4766 80.4375 47.1641 79.6562C46.6953 78.5234 46.8125 75.7891 46.8125 74.5C46.8125 73.25 46.6953 70.5156 47.1641 69.3438C47.4766 68.6016 48.0625 68.0156 48.8438 67.7031C49.9766 67.2344 52.7109 67.3516 54 67.3516C55.25 67.3516 57.9844 67.2344 59.1562 67.7031C59.8984 67.9766 60.4844 68.6016 60.7969 69.3438C61.2656 70.5156 61.1484 73.25 61.1484 74.5C61.1484 75.7891 61.2656 78.5234 60.7969 79.6562Z" fill="url(#paint0_linear_1058_41)"/><path d="M95.9297 70.4375C96.7109 69.8516 97.4141 69.1484 97.9609 68.3281C97.2578 68.6406 96.4375 68.875 95.6172 68.9531C96.4766 68.4453 97.1016 67.6641 97.4141 66.6875C96.6328 67.1562 95.7344 67.5078 94.8359 67.7031C94.0547 66.8828 93 66.4141 91.8281 66.4141C89.5625 66.4141 87.7266 68.25 87.7266 70.5156C87.7266 70.8281 87.7656 71.1406 87.8438 71.4531C84.4453 71.2578 81.3984 69.6172 79.3672 67.1562C79.0156 67.7422 78.8203 68.4453 78.8203 69.2266C78.8203 70.6328 79.5234 71.8828 80.6562 72.625C79.9922 72.5859 79.3281 72.4297 78.7812 72.1172V72.1562C78.7812 74.1484 80.1875 75.7891 82.0625 76.1797C81.75 76.2578 81.3594 76.3359 81.0078 76.3359C80.7344 76.3359 80.5 76.2969 80.2266 76.2578C80.7344 77.8984 82.2578 79.0703 84.0547 79.1094C82.6484 80.2031 80.8906 80.8672 78.9766 80.8672C78.625 80.8672 78.3125 80.8281 78 80.7891C79.7969 81.9609 81.9453 82.625 84.2891 82.625C91.8281 82.625 95.9297 76.4141 95.9297 70.9844C95.9297 70.7891 95.9297 70.6328 95.9297 70.4375Z" fill="#0096CC"/><defs><linearGradient id="paint0_linear_1058_41" x1="51.4125" y1="113.55" x2="105.527" y2="78.0397" gradientUnits="userSpaceOnUse"><stop stop-color="white"/><stop offset="0.147864" stop-color="#F6640E"/><stop offset="0.443974" stop-color="#BA03A7"/><stop offset="0.733337" stop-color="#6A01B9"/><stop offset="1" stop-color="#6B01B9"/></linearGradient></defs></svg>',
			'searchIconFree'		=>  '<svg width="32" height="32" viewBox="0 0 32 32" fill="none" ><rect x="3" y="3" width="26.1602" height="26.1602" rx="2" fill="#43A6DB"/><path d="M18.9167 17.6667H18.2583L18.025 17.4417C18.8699 16.4617 19.3343 15.2106 19.3333 13.9167C19.3333 12.8454 19.0157 11.7981 18.4205 10.9073C17.8253 10.0166 16.9793 9.3223 15.9895 8.91232C14.9998 8.50235 13.9107 8.39508 12.8599 8.60408C11.8092 8.81309 10.844 9.32897 10.0865 10.0865C9.32897 10.844 8.81309 11.8092 8.60408 12.8599C8.39508 13.9107 8.50235 14.9998 8.91232 15.9895C9.3223 16.9793 10.0166 17.8253 10.9073 18.4205C11.7981 19.0157 12.8454 19.3333 13.9167 19.3333C15.2583 19.3333 16.4917 18.8417 17.4417 18.025L17.6667 18.2583V18.9167L21.8333 23.075L23.075 21.8333L18.9167 17.6667ZM13.9167 17.6667C11.8417 17.6667 10.1667 15.9917 10.1667 13.9167C10.1667 11.8417 11.8417 10.1667 13.9167 10.1667C15.9917 10.1667 17.6667 11.8417 17.6667 13.9167C17.6667 15.9917 15.9917 17.6667 13.9167 17.6667Z" fill="white"/></svg>',
			'mentionsIconFree'		=>  '<svg width="33" height="32" viewBox="0 0 33 32" fill="none"><rect x="3.25" y="3" width="26.1602" height="26.1602" rx="2" fill="#43A6DB"/><path fill-rule="evenodd" clip-rule="evenodd" d="M14.7868 8.64411C16.2417 8.35472 17.7497 8.50325 19.1201 9.07091C20.4906 9.63856 21.6619 10.5999 22.486 11.8332C23.3101 13.0666 23.75 14.5166 23.75 16V17.1111C23.75 17.8233 23.4671 18.5063 22.9635 19.0098C22.46 19.5134 21.777 19.7963 21.0648 19.7963C20.3527 19.7963 19.6697 19.5134 19.1661 19.0098C19.0723 18.916 18.9861 18.8159 18.908 18.7106C18.1986 19.4061 17.2444 19.7963 16.25 19.7963C15.2432 19.7963 14.2776 19.3963 13.5656 18.6844C12.8537 17.9724 12.4537 17.0068 12.4537 16C12.4537 14.9932 12.8537 14.0276 13.5656 13.3156C14.2776 12.6037 15.2432 12.2037 16.25 12.2037C17.2568 12.2037 18.2224 12.6037 18.9344 13.3156C19.6463 14.0276 20.0463 14.9932 20.0463 16V17.1111C20.0463 17.3812 20.1536 17.6403 20.3446 17.8313C20.5356 18.0223 20.7947 18.1296 21.0648 18.1296C21.3349 18.1296 21.594 18.0223 21.785 17.8313C21.976 17.6403 22.0833 17.3812 22.0833 17.1111V16C22.0833 14.8463 21.7412 13.7185 21.1002 12.7592C20.4593 11.7999 19.5482 11.0522 18.4823 10.6107C17.4164 10.1692 16.2435 10.0537 15.112 10.2788C13.9804 10.5038 12.941 11.0594 12.1252 11.8752C11.3094 12.691 10.7538 13.7304 10.5288 14.862C10.3037 15.9935 10.4192 17.1664 10.8607 18.2323C11.3022 19.2982 12.0499 20.2093 13.0092 20.8502C13.9685 21.4912 15.0963 21.8333 16.25 21.8333L16.2508 21.8333C17.2747 21.8344 18.2803 21.5652 19.1656 21.0523C19.5638 20.8216 20.0737 20.9574 20.3044 21.3556C20.5351 21.7538 20.3993 22.2637 20.0011 22.4944C18.8608 23.155 17.5663 23.5013 16.2492 23.5L16.25 22.6667V23.5C16.2497 23.5 16.2495 23.5 16.2492 23.5C14.7661 23.4998 13.3164 23.06 12.0832 22.236C10.8499 21.4119 9.88856 20.2406 9.32091 18.8701C8.75325 17.4997 8.60472 15.9917 8.89411 14.5368C9.1835 13.082 9.89781 11.7456 10.9467 10.6967C11.9956 9.64781 13.332 8.9335 14.7868 8.64411ZM18.3796 16C18.3796 15.4352 18.1553 14.8935 17.7559 14.4941C17.3565 14.0947 16.8148 13.8704 16.25 13.8704C15.6852 13.8704 15.1435 14.0947 14.7441 14.4941C14.3447 14.8935 14.1204 15.4352 14.1204 16C14.1204 16.5648 14.3447 17.1065 14.7441 17.5059C15.1435 17.9053 15.6852 18.1296 16.25 18.1296C16.8148 18.1296 17.3565 17.9053 17.7559 17.5059C18.1553 17.1065 18.3796 16.5648 18.3796 16Z" fill="white"/></svg>',
			'listsIconFree'			=>  '<svg width="33" height="32" viewBox="0 0 33 32" fill="none" xmlns="http://www.w3.org/2000/svg"><rect x="3.5" y="3" width="26.1602" height="26.1602" rx="2" fill="#43A6DB"/><line x1="9.5" y1="11" x2="23.5" y2="11" stroke="white" stroke-width="2"/><line x1="9.5" y1="16" x2="23.5" y2="16" stroke="white" stroke-width="2"/><line x1="9.5" y1="21" x2="23.5" y2="21" stroke="white" stroke-width="2"/></svg>',
			'socialwall1IconFree' => '<svg width="33" height="32" viewBox="0 0 33 32" fill="none" xmlns="http://www.w3.org/2000/svg"> <g clip-path="url(#socialWallFreeclip0)"> <path d="M8.49935 2.19C5.29102 2.19 2.66602 4.80917 2.66602 8.035C2.66602 10.9517 4.80102 13.3725 7.58935 13.81V9.72667H6.10768V8.035H7.58935V6.74584C7.58935 5.28167 8.45852 4.47667 9.79435 4.47667C10.4302 4.47667 11.0952 4.5875 11.0952 4.5875V6.02834H10.3602C9.63685 6.02834 9.40935 6.4775 9.40935 6.93834V8.035H11.031L10.7685 9.72667H9.40935V13.81C10.7839 13.5929 12.0356 12.8916 12.9385 11.8325C13.8413 10.7735 14.3358 9.42663 14.3327 8.035C14.3327 4.80917 11.7077 2.19 8.49935 2.19Z" fill="#006BFA"/> <path d="M8.5 21.3047C7 21.3047 5.80469 22.5234 5.80469 24C5.80469 25.5 7 26.6953 8.5 26.6953C9.97656 26.6953 11.1953 25.5 11.1953 24C11.1953 22.5234 9.97656 21.3047 8.5 21.3047ZM8.5 25.7578C7.53906 25.7578 6.74219 24.9844 6.74219 24C6.74219 23.0391 7.51562 22.2656 8.5 22.2656C9.46094 22.2656 10.2344 23.0391 10.2344 24C10.2344 24.9844 9.46094 25.7578 8.5 25.7578ZM11.9219 21.2109C11.9219 20.8594 11.6406 20.5781 11.2891 20.5781C10.9375 20.5781 10.6562 20.8594 10.6562 21.2109C10.6562 21.5625 10.9375 21.8438 11.2891 21.8438C11.6406 21.8438 11.9219 21.5625 11.9219 21.2109ZM13.7031 21.8438C13.6562 21 13.4688 20.25 12.8594 19.6406C12.25 19.0312 11.5 18.8438 10.6562 18.7969C9.78906 18.75 7.1875 18.75 6.32031 18.7969C5.47656 18.8438 4.75 19.0312 4.11719 19.6406C3.50781 20.25 3.32031 21 3.27344 21.8438C3.22656 22.7109 3.22656 25.3125 3.27344 26.1797C3.32031 27.0234 3.50781 27.75 4.11719 28.3828C4.75 28.9922 5.47656 29.1797 6.32031 29.2266C7.1875 29.2734 9.78906 29.2734 10.6562 29.2266C11.5 29.1797 12.25 28.9922 12.8594 28.3828C13.4688 27.75 13.6562 27.0234 13.7031 26.1797C13.75 25.3125 13.75 22.7109 13.7031 21.8438ZM12.5781 27.0938C12.4141 27.5625 12.0391 27.9141 11.5938 28.1016C10.8906 28.3828 9.25 28.3125 8.5 28.3125C7.72656 28.3125 6.08594 28.3828 5.40625 28.1016C4.9375 27.9141 4.58594 27.5625 4.39844 27.0938C4.11719 26.4141 4.1875 24.7734 4.1875 24C4.1875 23.25 4.11719 21.6094 4.39844 20.9062C4.58594 20.4609 4.9375 20.1094 5.40625 19.9219C6.08594 19.6406 7.72656 19.7109 8.5 19.7109C9.25 19.7109 10.8906 19.6406 11.5938 19.9219C12.0391 20.0859 12.3906 20.4609 12.5781 20.9062C12.8594 21.6094 12.7891 23.25 12.7891 24C12.7891 24.7734 12.8594 26.4141 12.5781 27.0938Z" fill="url(#socialWallFreepaint0_linear)"/> <path d="M30.6018 4.50001C30.1526 4.70418 29.6684 4.83834 29.1668 4.90251C29.6801 4.59334 30.0768 4.10334 30.2634 3.51418C29.7793 3.80584 29.2426 4.01001 28.6768 4.12668C28.2159 3.62501 27.5684 3.33334 26.8334 3.33334C25.4626 3.33334 24.3426 4.45334 24.3426 5.83584C24.3426 6.03418 24.3659 6.22668 24.4068 6.40751C22.3301 6.30251 20.4809 5.30501 19.2501 3.79418C19.0343 4.16168 18.9118 4.59334 18.9118 5.04834C18.9118 5.91751 19.3493 6.68751 20.0259 7.12501C19.6118 7.12501 19.2268 7.00834 18.8884 6.83334V6.85084C18.8884 8.06418 19.7518 9.07918 20.8951 9.30668C20.528 9.40713 20.1427 9.42111 19.7693 9.34751C19.9277 9.84479 20.238 10.2799 20.6565 10.5917C21.0751 10.9035 21.5808 11.0763 22.1026 11.0858C21.2181 11.7861 20.1216 12.1646 18.9934 12.1592C18.7951 12.1592 18.5968 12.1475 18.3984 12.1242C19.5068 12.8358 20.8251 13.25 22.2368 13.25C26.8334 13.25 29.3593 9.43501 29.3593 6.12751C29.3593 6.01668 29.3593 5.91168 29.3534 5.80084C29.8434 5.45084 30.2634 5.00751 30.6018 4.50001Z" fill="#1B90EF"/> <path d="M23.3327 25.75L26.3602 24L23.3327 22.25V25.75ZM30.076 21.1825C30.1518 21.4567 30.2043 21.8242 30.2393 22.2908C30.2802 22.7575 30.2977 23.16 30.2977 23.51L30.3327 24C30.3327 25.2775 30.2393 26.2167 30.076 26.8175C29.9302 27.3425 29.5918 27.6808 29.0668 27.8267C28.7927 27.9025 28.291 27.955 27.521 27.99C26.7627 28.0308 26.0685 28.0483 25.4268 28.0483L24.4993 28.0833C22.0552 28.0833 20.5327 27.99 19.9318 27.8267C19.4068 27.6808 19.0685 27.3425 18.9227 26.8175C18.8468 26.5433 18.7943 26.1758 18.7593 25.7092C18.7185 25.2425 18.701 24.84 18.701 24.49L18.666 24C18.666 22.7225 18.7593 21.7833 18.9227 21.1825C19.0685 20.6575 19.4068 20.3192 19.9318 20.1733C20.206 20.0975 20.7077 20.045 21.4777 20.01C22.236 19.9692 22.9302 19.9517 23.5718 19.9517L24.4993 19.9167C26.9435 19.9167 28.466 20.01 29.0668 20.1733C29.5918 20.3192 29.9302 20.6575 30.076 21.1825Z" fill="#EB2121"/> </g> <defs> <linearGradient id="socialWallFreepaint0_linear" x1="6.97891" y1="38.843" x2="29.0945" y2="16.268" gradientUnits="userSpaceOnUse"> <stop stop-color="white"/> <stop offset="0.147864" stop-color="#F6640E"/> <stop offset="0.443974" stop-color="#BA03A7"/> <stop offset="0.733337" stop-color="#6A01B9"/> <stop offset="1" stop-color="#6B01B9"/> </linearGradient> <clipPath id="socialWallFreeclip0"> <rect width="32" height="32" fill="white" transform="translate(0.5)"/> </clipPath> </defs> </svg>',

			'user'	=> '<svg width="8" height="8" viewBox="0 0 8 8" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M4 0C4.53043 0 5.03914 0.210714 5.41421 0.585786C5.78929 0.960859 6 1.46957 6 2C6 2.53043 5.78929 3.03914 5.41421 3.41421C5.03914 3.78929 4.53043 4 4 4C3.46957 4 2.96086 3.78929 2.58579 3.41421C2.21071 3.03914 2 2.53043 2 2C2 1.46957 2.21071 0.960859 2.58579 0.585786C2.96086 0.210714 3.46957 0 4 0ZM4 5C6.21 5 8 5.895 8 7V8H0V7C0 5.895 1.79 5 4 5Z"/></svg>',
			'hashtag' => '<svg viewBox="0 0 18 18" fill="none"><path d="M17.3607 4.1775H14.0152L14.618 1.266C14.6328 1.18021 14.6288 1.09223 14.6064 1.00812C14.5839 0.924001 14.5436 0.845742 14.488 0.778722C14.4324 0.711703 14.363 0.657514 14.2845 0.619882C14.206 0.582251 14.1203 0.56207 14.0332 0.560727H12.8276C12.6883 0.557321 12.5521 0.602311 12.4422 0.688037C12.3323 0.773763 12.2555 0.894929 12.2249 1.03091L11.61 4.1775H8.3549L8.9577 1.266C8.97229 1.18215 8.96897 1.09617 8.94795 1.0137C8.92692 0.931226 8.88867 0.854142 8.83572 0.787518C8.78276 0.720894 8.71629 0.666239 8.64069 0.62715C8.56509 0.588061 8.48207 0.565423 8.3971 0.560727H7.1915C7.05216 0.557321 6.91594 0.602311 6.80604 0.688037C6.69613 0.773763 6.61933 0.894929 6.58871 1.03091L5.98591 4.1775H2.36914C2.22811 4.17466 2.09056 4.22136 1.98042 4.30947C1.87028 4.39759 1.79452 4.52153 1.76634 4.65974L1.51919 5.86533C1.50109 5.95393 1.50315 6.04546 1.52522 6.13316C1.5473 6.22085 1.58882 6.30245 1.64671 6.37192C1.7046 6.44139 1.77737 6.49694 1.85965 6.53446C1.94192 6.57199 2.03158 6.59052 2.12199 6.58869H5.46751L4.47892 11.4111H0.862146C0.721125 11.4082 0.583571 11.4549 0.473429 11.543C0.363287 11.6311 0.287532 11.7551 0.259351 11.8933L0.0122042 13.0989C-0.00589975 13.1875 -0.00383898 13.279 0.0182337 13.3667C0.0403064 13.4544 0.0818254 13.536 0.139715 13.6055C0.197605 13.6749 0.270382 13.7305 0.352656 13.768C0.43493 13.8055 0.524592 13.8241 0.615 13.8222H3.98463L3.38183 16.7338C3.36677 16.821 3.37112 16.9106 3.39459 16.996C3.41806 17.0814 3.46006 17.1606 3.51761 17.2279C3.57517 17.2953 3.64685 17.3491 3.72757 17.3856C3.80829 17.4221 3.89606 17.4403 3.98463 17.439H5.19022C5.3244 17.4356 5.45359 17.3875 5.55732 17.3023C5.66105 17.2171 5.73339 17.0998 5.76288 16.9688L6.38979 13.8222H9.64488L9.04209 16.7338C9.02749 16.8176 9.03081 16.9036 9.05184 16.9861C9.07286 17.0685 9.11111 17.1456 9.16407 17.2122C9.21702 17.2789 9.28349 17.3335 9.35909 17.3726C9.43469 17.4117 9.51771 17.4343 9.60269 17.439H10.8083C10.9476 17.4424 11.0838 17.3974 11.1937 17.3117C11.3037 17.226 11.3805 17.1048 11.4111 16.9688L12.044 13.8222H15.6608C15.8018 13.8251 15.9394 13.7784 16.0495 13.6903C16.1596 13.6022 16.2354 13.4782 16.2636 13.34L16.5047 12.1344C16.5228 12.0458 16.5207 11.9543 16.4987 11.8666C16.4766 11.7789 16.4351 11.6973 16.3772 11.6278C16.3193 11.5584 16.2465 11.5028 16.1642 11.4653C16.082 11.4278 15.9923 11.4092 15.9019 11.4111H12.5383L13.5209 6.58869H17.1376C17.2787 6.59153 17.4162 6.54483 17.5264 6.45672C17.6365 6.36861 17.7123 6.24466 17.7404 6.10645L17.9876 4.90086C18.0063 4.8102 18.0038 4.71645 17.9804 4.62689C17.957 4.53733 17.9133 4.45436 17.8527 4.3844C17.7921 4.31445 17.7162 4.2594 17.6308 4.22352C17.5455 4.18764 17.4531 4.1719 17.3607 4.1775ZM10.1271 11.4111H6.87202L7.86061 6.58869H11.1157L10.1271 11.4111Z"/></svg>',
			'mention' => '<svg viewBox="0 0 18 18"><path fill-rule="evenodd" clip-rule="evenodd" d="M7.24419 0.172937C8.99002 -0.174331 10.7996 0.00389957 12.4442 0.685088C14.0887 1.36628 15.4943 2.51983 16.4832 3.99987C17.4722 5.47992 18 7.21997 18 9.00001V10.3333C18 11.1879 17.6605 12.0075 17.0562 12.6118C16.452 13.2161 15.6324 13.5556 14.7778 13.5556C13.9232 13.5556 13.1036 13.2161 12.4993 12.6118C12.3867 12.4992 12.2833 12.3791 12.1896 12.2527C11.3384 13.0874 10.1933 13.5556 9.00001 13.5556C7.7918 13.5556 6.63307 13.0756 5.77874 12.2213C4.92441 11.3669 4.44445 10.2082 4.44445 9.00001C4.44445 7.7918 4.92441 6.63307 5.77874 5.77874C6.63307 4.92441 7.7918 4.44445 9.00001 4.44445C10.2082 4.44445 11.3669 4.92441 12.2213 5.77874C13.0756 6.63307 13.5556 7.7918 13.5556 9.00001V10.3333C13.5556 10.6575 13.6843 10.9684 13.9135 11.1976C14.1428 11.4268 14.4536 11.5556 14.7778 11.5556C15.1019 11.5556 15.4128 11.4268 15.642 11.1976C15.8712 10.9684 16 10.6575 16 10.3333V9.00001C16 7.61554 15.5895 6.26216 14.8203 5.11101C14.0511 3.95987 12.9579 3.06266 11.6788 2.53285C10.3997 2.00303 8.99224 1.86441 7.63437 2.13451C6.27651 2.4046 5.02922 3.07129 4.05026 4.05026C3.07129 5.02922 2.4046 6.27651 2.13451 7.63437C1.86441 8.99224 2.00303 10.3997 2.53285 11.6788C3.06266 12.9579 3.95987 14.0511 5.11101 14.8203C6.26216 15.5895 7.61554 16 9.00001 16L9.001 16C10.2297 16.0012 11.4363 15.6782 12.4987 15.0627C12.9766 14.7859 13.5884 14.9488 13.8653 15.4267C14.1421 15.9046 13.9792 16.5164 13.5013 16.7933C12.1329 17.586 10.5796 18.0016 8.99901 18L9.00001 17V18C8.99968 18 8.99934 18 8.99901 18C7.21933 17.9998 5.47964 17.472 3.99987 16.4832C2.51983 15.4943 1.36628 14.0887 0.685088 12.4442C0.00389957 10.7996 -0.17433 8.99002 0.172936 7.24419C0.520204 5.49836 1.37737 3.89472 2.63604 2.63604C3.89472 1.37737 5.49836 0.520204 7.24419 0.172937ZM11.5556 9.00001C11.5556 8.32223 11.2863 7.67221 10.8071 7.19295C10.3278 6.7137 9.67778 6.44445 9.00001 6.44445C8.32223 6.44445 7.67221 6.7137 7.19295 7.19295C6.7137 7.67221 6.44445 8.32223 6.44445 9.00001C6.44445 9.67778 6.7137 10.3278 7.19295 10.8071C7.67221 11.2863 8.32223 11.5556 9.00001 11.5556C9.67778 11.5556 10.3278 11.2863 10.8071 10.8071C11.2863 10.3278 11.5556 9.67778 11.5556 9.00001Z"/></svg>',
			'homeTimeline' => '<svg width="21" height="19" viewBox="0 0 21 19" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M18.2045 18.0682C18.2045 18.3153 18.1064 18.5523 17.9316 18.7271C17.7569 18.9018 17.5199 19 17.2727 19H4.22727C3.98014 19 3.74313 18.9018 3.56838 18.7271C3.39363 18.5523 3.29545 18.3153 3.29545 18.0682V9.6818H0.5L10.1229 0.93389C10.2944 0.77779 10.5181 0.691284 10.75 0.691284C10.9819 0.691284 11.2056 0.77779 11.3771 0.93389L21 9.6818H18.2045V18.0682ZM10.75 13.4091C11.3678 13.4091 11.9604 13.1636 12.3972 12.7268C12.8341 12.2899 13.0795 11.6974 13.0795 11.0795C13.0795 10.4617 12.8341 9.86916 12.3972 9.43229C11.9604 8.99541 11.3678 8.74998 10.75 8.74998C10.1322 8.74998 9.53964 8.99541 9.10276 9.43229C8.66589 9.86916 8.42045 10.4617 8.42045 11.0795C8.42045 11.6974 8.66589 12.2899 9.10276 12.7268C9.53964 13.1636 10.1322 13.4091 10.75 13.4091Z" fill="#0096CC"/></svg>',
			'search' => '<svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M12.5 11H11.71L11.43 10.73C12.4439 9.55402 13.0011 8.0527 13 6.5C13 5.21442 12.6188 3.95772 11.9046 2.8888C11.1903 1.81988 10.1752 0.986755 8.98744 0.494786C7.79972 0.00281635 6.49279 -0.125905 5.23191 0.124899C3.97104 0.375703 2.81285 0.994767 1.90381 1.90381C0.994767 2.81285 0.375703 3.97104 0.124899 5.23191C-0.125905 6.49279 0.00281635 7.79972 0.494786 8.98744C0.986755 10.1752 1.81988 11.1903 2.8888 11.9046C3.95772 12.6188 5.21442 13 6.5 13C8.11 13 9.59 12.41 10.73 11.43L11 11.71V12.5L16 17.49L17.49 16L12.5 11ZM6.5 11C4.01 11 2 8.99 2 6.5C2 4.01 4.01 2 6.5 2C8.99 2 11 4.01 11 6.5C11 8.99 8.99 11 6.5 11Z" fill="#0096CC"/></svg>',
			'lists' => '<svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M16.1984 18H1.79844C1.55974 18 1.33082 17.9052 1.16204 17.7364C0.993259 17.5676 0.898438 17.3387 0.898438 17.1V0.899999C0.898438 0.661305 0.993259 0.432386 1.16204 0.263604C1.33082 0.094821 1.55974 0 1.79844 0H16.1984C16.4371 0 16.666 0.094821 16.8348 0.263604C17.0036 0.432386 17.0984 0.661305 17.0984 0.899999V17.1C17.0984 17.3387 17.0036 17.5676 16.8348 17.7364C16.666 17.9052 16.4371 18 16.1984 18ZM5.39843 4.5V6.29999H12.5984V4.5H5.39843ZM5.39843 8.09999V9.89999H12.5984V8.09999H5.39843ZM5.39843 11.7V13.5H12.5984V11.7H5.39843Z" fill="#0096CC"/></svg>',

			'addNewList' => '<svg width="13" height="14" viewBox="0 0 13 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9.33333 7.83333H10.6667V9.83333H12.6667V11.1667H10.6667V13.1667H9.33333V11.1667H7.33333V9.83333H9.33333V7.83333ZM1.33333 0.5H10.6667C11.4067 0.5 12 1.09333 12 1.83333V7.03333C11.5933 6.8 11.1467 6.63333 10.6667 6.55333V1.83333H1.33333V11.1667H6.05333C6.13333 11.6467 6.3 12.0933 6.53333 12.5H1.33333C0.593333 12.5 0 11.9067 0 11.1667V1.83333C0 1.09333 0.593333 0.5 1.33333 0.5ZM2.66667 3.16667H9.33333V4.5H2.66667V3.16667ZM2.66667 5.83333H9.33333V6.55333C8.76667 6.64667 8.24667 6.86 7.78667 7.16667H2.66667V5.83333ZM2.66667 8.5H6V9.83333H2.66667V8.5Z" fill="#0068A0"/></svg>',
			'tooltipHelpSvg' => '<svg width="20" height="21" viewBox="0 0 20 21" fill="#0068A0" xmlns="http://www.w3.org/2000/svg"><path d="M9.1665 8H10.8332V6.33333H9.1665V8ZM9.99984 17.1667C6.32484 17.1667 3.33317 14.175 3.33317 10.5C3.33317 6.825 6.32484 3.83333 9.99984 3.83333C13.6748 3.83333 16.6665 6.825 16.6665 10.5C16.6665 14.175 13.6748 17.1667 9.99984 17.1667ZM9.99984 2.16666C8.90549 2.16666 7.82186 2.38221 6.81081 2.801C5.79976 3.21979 4.8811 3.83362 4.10728 4.60744C2.54448 6.17024 1.6665 8.28986 1.6665 10.5C1.6665 12.7101 2.54448 14.8298 4.10728 16.3926C4.8811 17.1664 5.79976 17.7802 6.81081 18.199C7.82186 18.6178 8.90549 18.8333 9.99984 18.8333C12.21 18.8333 14.3296 17.9554 15.8924 16.3926C17.4552 14.8298 18.3332 12.7101 18.3332 10.5C18.3332 9.40565 18.1176 8.32202 17.6988 7.31097C17.28 6.29992 16.6662 5.38126 15.8924 4.60744C15.1186 3.83362 14.1999 3.21979 13.1889 2.801C12.1778 2.38221 11.0942 2.16666 9.99984 2.16666ZM9.1665 14.6667H10.8332V9.66666H9.1665V14.6667Z" fill="#0068A0"/></svg>',

			'union' => '<svg width="10" height="12" viewBox="0 0 10 12" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M10 6.74992V0.333252H0V6.74992H10Z"/><path d="M0 8.24992H10V9.49992H0V8.24992Z"/><path d="M6 10.7499H0V11.9999H6V10.7499Z"/></svg>',
			'strokeThickness' => '<svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M14.3996 4.0002H1.59961V3.2002H14.3996V4.0002Z"/><path fill-rule="evenodd" clip-rule="evenodd" d="M14.3996 7.9999H1.59961V6.3999H14.3996V7.9999Z"/><path fill-rule="evenodd" clip-rule="evenodd" d="M14.3996 12.7999H1.59961V10.3999H14.3996V12.7999Z"/></svg>',
			'linkIcon' => '<svg width="17" height="10" viewBox="0 0 17 10" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M1.75008 4.99998C1.75008 3.57498 2.90841 2.41665 4.33342 2.41665H7.66675V0.833313H4.33342C3.22835 0.833313 2.16854 1.2723 1.38714 2.0537C0.605735 2.8351 0.166748 3.89491 0.166748 4.99998C0.166748 6.10505 0.605735 7.16486 1.38714 7.94626C2.16854 8.72766 3.22835 9.16665 4.33342 9.16665H7.66675V7.58331H4.33342C2.90841 7.58331 1.75008 6.42498 1.75008 4.99998ZM5.16675 5.83331H11.8334V4.16665H5.16675V5.83331ZM12.6667 0.833313H9.33342V2.41665H12.6667C14.0917 2.41665 15.2501 3.57498 15.2501 4.99998C15.2501 6.42498 14.0917 7.58331 12.6667 7.58331H9.33342V9.16665H12.6667C13.7718 9.16665 14.8316 8.72766 15.613 7.94626C16.3944 7.16486 16.8334 6.10505 16.8334 4.99998C16.8334 3.89491 16.3944 2.8351 15.613 2.0537C14.8316 1.2723 13.7718 0.833313 12.6667 0.833313Z" fill="#0068A0"/></svg>',

			'feed_template' => '<svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M16 0H2C0.9 0 0 0.9 0 2V16C0 17.1 0.9 18 2 18H16C17.1 18 18 17.1 18 16V2C18 0.9 17.1 0 16 0ZM2 16V2H8V16H2ZM16 16H10V9H16V16ZM16 7H10V2H16V7Z" fill="#141B38"/></svg>',

			'defaultTemplate' 			=> '<svg width="262" height="200" viewBox="0 0 262 200" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_1652_59144)"><rect width="262" height="200" fill="#F3F4F5"/><g filter="url(#filter0_ddd_1652_59144)"><rect x="26" y="36" width="211" height="187" rx="2" fill="white"/></g><rect x="56" y="55" width="18" height="18" rx="9" fill="#F9BBA0"/><rect x="82" y="57" width="19" height="6" rx="1" fill="#8C8F9A"/><circle cx="107" cy="60" r="3" fill="#0096CC"/><path d="M105.398 60L106.423 61.0246L108.544 58.9033" stroke="white" stroke-width="0.8"/><rect x="113" y="55" width="26" height="10" rx="1" fill="#0096CC"/><path d="M117.554 62V60.458H119.06V59.474H117.554V58.67H119.354V57.692H116.42V62H117.554ZM123.042 60.386C123.042 59.462 122.31 58.712 121.356 58.712C120.402 58.712 119.67 59.462 119.67 60.386C119.67 61.31 120.402 62.06 121.356 62.06C122.31 62.06 123.042 61.31 123.042 60.386ZM120.738 60.386C120.738 59.882 121.038 59.636 121.356 59.636C121.674 59.636 121.974 59.882 121.974 60.386C121.974 60.89 121.674 61.142 121.356 61.142C121.038 61.142 120.738 60.89 120.738 60.386ZM124.505 62V57.692H123.467V62H124.505ZM126.204 62V57.692H125.166V62H126.204ZM130.003 60.386C130.003 59.462 129.271 58.712 128.317 58.712C127.363 58.712 126.631 59.462 126.631 60.386C126.631 61.31 127.363 62.06 128.317 62.06C129.271 62.06 130.003 61.31 130.003 60.386ZM127.699 60.386C127.699 59.882 127.999 59.636 128.317 59.636C128.635 59.636 128.935 59.882 128.935 60.386C128.935 60.89 128.635 61.142 128.317 61.142C127.999 61.142 127.699 60.89 127.699 60.386ZM133.056 62H134.076L134.988 58.772H133.98L133.53 60.542L133.08 58.772H132.054L131.58 60.542L131.13 58.772H130.128L131.034 62H132.06L132.558 60.17L133.056 62Z" fill="white"/><rect x="82" y="68" width="73" height="6" rx="1" fill="#DCDDE1"/><line x1="56" y1="85.75" x2="213" y2="85.75" stroke="#DCDDE1" stroke-width="0.5"/><rect x="56" y="98" width="18" height="18" rx="2" fill="#F9BBA0"/><rect x="82" y="99" width="19" height="6" rx="1" fill="#8C8F9A"/><circle cx="108" cy="102" r="4" fill="#0096CC"/><path d="M105.867 102L107.233 103.366L110.062 100.538" stroke="white" stroke-width="1.06667"/><rect x="82" y="112" width="105.314" height="6" rx="1" fill="#DCDDE1"/><rect x="82" y="122" width="116.076" height="6" rx="1" fill="#DCDDE1"/><rect x="82" y="132" width="99" height="6" rx="1" fill="#DCDDE1"/><line x1="56" y1="149.75" x2="213" y2="149.75" stroke="#DCDDE1" stroke-width="0.5"/><rect x="56" y="162" width="18" height="18" rx="2" fill="#F9BBA0"/><rect x="82" y="163" width="19" height="6" rx="1" fill="#8C8F9A"/><circle cx="108" cy="166" r="4" fill="#0096CC"/><path d="M105.867 166L107.233 167.366L110.062 164.538" stroke="white" stroke-width="1.06667"/><rect x="82" y="176" width="105.314" height="6" rx="1" fill="#DCDDE1"/><rect x="82" y="186" width="116.076" height="6" rx="1" fill="#DCDDE1"/><rect x="82" y="196" width="99" height="6" rx="1" fill="#DCDDE1"/></g><defs><filter id="filter0_ddd_1652_59144" x="13" y="29" width="237" height="213" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="6"/><feGaussianBlur stdDeviation="6.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59144"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59144" result="effect2_dropShadow_1652_59144"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="3"/><feGaussianBlur stdDeviation="3"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_1652_59144" result="effect3_dropShadow_1652_59144"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_1652_59144" result="shape"/></filter><clipPath id="clip0_1652_59144"><rect width="262" height="200" fill="white"/></clipPath></defs></svg>',
			'masonryCardsTemplate' 		=> '<svg width="262" height="200" viewBox="0 0 262 200" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_1652_59202)"><rect width="262" height="200" transform="translate(0.5)" fill="#F3F4F5"/><g filter="url(#filter0_dd_1652_59202)"><rect x="34" y="57" width="63.6871" height="86.5898" rx="0.737139" fill="white"/><rect x="38" y="61" width="8" height="8" rx="1" fill="#43A6DB"/><rect x="49.2969" y="63" width="24" height="4" rx="0.824675" fill="#D0D1D7"/><rect x="38" y="72" width="55.6871" height="4" rx="0.412338" fill="#DCDDE1"/><path d="M38 79.4123C38 79.1846 38.1846 79 38.4123 79H72.224C72.4518 79 72.6364 79.1846 72.6364 79.4123V82.5877C72.6364 82.8154 72.4518 83 72.224 83H38.4123C38.1846 83 38 82.8154 38 82.5877V79.4123Z" fill="#DCDDE1"/><g clip-path="url(#clip1_1652_59202)"><rect x="38" y="87" width="55.6871" height="47.8312" rx="1" fill="#43A6DB"/><circle cx="38.4099" cy="134.419" r="38.3474" fill="#86D0F9"/></g><g clip-path="url(#clip2_1652_59202)"><path d="M38.5938 138.786C38.5938 138.021 39.153 137.723 39.8429 137.723H40.9136C41.6034 137.723 42.1627 138.241 42.1627 138.88C42.1627 139.044 42.1258 139.2 42.0592 139.342C41.6639 140.181 40.4847 140.263 39.6644 140.698L39.6644 139.871C39.6644 139.871 38.5938 139.424 38.5938 138.786Z" stroke="#434960" stroke-width="0.35" stroke-linejoin="round"/></g><g clip-path="url(#clip3_1652_59202)"><path d="M47.8594 139.938L48.5864 140.665L49.3135 139.938" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M46.8906 137.757H47.6177C47.8748 137.757 48.1213 137.859 48.3031 138.041C48.4849 138.222 48.5871 138.469 48.5871 138.726V140.665" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M46.4072 138.484L45.6802 137.757L44.9531 138.484" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M47.3761 140.665H46.6491C46.392 140.665 46.1454 140.563 45.9636 140.381C45.7818 140.199 45.6797 139.953 45.6797 139.695V137.757" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g><g clip-path="url(#clip4_1652_59202)"><path d="M53.0908 137.823C52.5488 137.823 52.1094 138.249 52.1094 138.776C52.1094 139.729 53.2693 140.595 53.8938 140.797C54.5184 140.595 55.6783 139.729 55.6783 138.776C55.6783 138.249 55.2389 137.823 54.6969 137.823C54.3649 137.823 54.0714 137.983 53.8938 138.227C53.8033 138.102 53.6831 138 53.5433 137.93C53.4036 137.859 53.2483 137.822 53.0908 137.823Z" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g></g><g filter="url(#filter1_dd_1652_59202)"><rect x="34" y="147.59" width="63.6871" height="38.7586" rx="0.737139" fill="white"/><rect x="38" y="151.59" width="8" height="8" rx="1" fill="#43A6DB"/><rect x="49.2969" y="153.59" width="24" height="4" rx="0.824675" fill="#D0D1D7"/><rect x="38" y="162.59" width="55.6871" height="4" rx="0.412338" fill="#DCDDE1"/><path d="M38 170.002C38 169.774 38.1846 169.59 38.4123 169.59H72.224C72.4518 169.59 72.6364 169.774 72.6364 170.002V173.177C72.6364 173.405 72.4518 173.59 72.224 173.59H38.4123C38.1846 173.59 38 173.405 38 173.177V170.002Z" fill="#DCDDE1"/><g clip-path="url(#clip5_1652_59202)"><path d="M38.5938 181.544C38.5938 180.779 39.153 180.482 39.8429 180.482H40.9136C41.6034 180.482 42.1627 181 42.1627 181.639C42.1627 181.803 42.1258 181.959 42.0592 182.1C41.6639 182.94 40.4847 183.022 39.6644 183.456L39.6644 182.63C39.6644 182.63 38.5938 182.183 38.5938 181.544Z" stroke="#434960" stroke-width="0.35" stroke-linejoin="round"/></g><g clip-path="url(#clip6_1652_59202)"><path d="M47.8594 182.696L48.5864 183.423L49.3135 182.696" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M46.8906 180.515H47.6177C47.8748 180.515 48.1213 180.617 48.3031 180.799C48.4849 180.981 48.5871 181.228 48.5871 181.485V183.423" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M46.4072 181.242L45.6802 180.515L44.9531 181.242" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M47.3761 183.423H46.6491C46.392 183.423 46.1454 183.321 45.9636 183.14C45.7818 182.958 45.6797 182.711 45.6797 182.454V180.515" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g><g clip-path="url(#clip7_1652_59202)"><path d="M53.0908 180.581C52.5488 180.581 52.1094 181.008 52.1094 181.534C52.1094 182.487 53.2693 183.354 53.8938 183.555C54.5184 183.354 55.6783 182.487 55.6783 181.534C55.6783 181.008 55.2389 180.581 54.6969 180.581C54.3649 180.581 54.0714 180.741 53.8938 180.986C53.8033 180.861 53.6831 180.759 53.5433 180.688C53.4036 180.618 53.2483 180.581 53.0908 180.581Z" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g></g><g filter="url(#filter2_dd_1652_59202)"><rect x="34" y="190.348" width="63.6871" height="86.5898" rx="0.737139" fill="white"/><rect x="38" y="194.348" width="8" height="8" rx="1" fill="#43A6DB"/><rect x="49.2969" y="196.348" width="24" height="4" rx="0.824675" fill="#D0D1D7"/></g><g filter="url(#filter3_dd_1652_59202)"><rect x="101.688" y="57" width="63.6871" height="59.7586" rx="0.737139" fill="white"/><rect x="105.688" y="61" width="8" height="8" rx="1" fill="#43A6DB"/><rect x="116.984" y="63" width="24" height="4" rx="0.824675" fill="#D0D1D7"/><rect x="105.688" y="72" width="55.6871" height="4" rx="0.412338" fill="#DCDDE1"/><rect x="105.688" y="79" width="53" height="4" rx="0.412338" fill="#DCDDE1"/><rect x="105.688" y="86" width="55.6871" height="4" rx="0.412338" fill="#DCDDE1"/><rect x="105.688" y="93" width="50" height="4" rx="0.412338" fill="#DCDDE1"/><path d="M105.688 100.412C105.688 100.185 105.872 100 106.1 100H139.912C140.139 100 140.324 100.185 140.324 100.412V103.588C140.324 103.815 140.139 104 139.912 104H106.1C105.872 104 105.688 103.815 105.688 103.588V100.412Z" fill="#DCDDE1"/><g clip-path="url(#clip8_1652_59202)"><path d="M106.281 111.954C106.281 111.19 106.841 110.892 107.53 110.892H108.601C109.291 110.892 109.85 111.41 109.85 112.049C109.85 112.213 109.813 112.369 109.747 112.511C109.351 113.35 108.172 113.432 107.352 113.866L107.352 113.04C107.352 113.04 106.281 112.593 106.281 111.954Z" stroke="#434960" stroke-width="0.35" stroke-linejoin="round"/></g><g clip-path="url(#clip9_1652_59202)"><path d="M115.547 113.106L116.274 113.833L117.001 113.106" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M114.578 110.926H115.305C115.562 110.926 115.809 111.028 115.991 111.209C116.172 111.391 116.275 111.638 116.275 111.895V113.834" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M114.095 111.653L113.368 110.926L112.641 111.653" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M115.064 113.834H114.337C114.079 113.834 113.833 113.732 113.651 113.55C113.469 113.368 113.367 113.121 113.367 112.864V110.926" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g><g clip-path="url(#clip10_1652_59202)"><path d="M120.778 110.991C120.236 110.991 119.797 111.418 119.797 111.944C119.797 112.897 120.957 113.764 121.581 113.965C122.206 113.764 123.366 112.897 123.366 111.944C123.366 111.418 122.926 110.991 122.384 110.991C122.052 110.991 121.759 111.151 121.581 111.396C121.491 111.271 121.371 111.169 121.231 111.098C121.091 111.028 120.936 110.991 120.778 110.991Z" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g></g><g filter="url(#filter4_dd_1652_59202)"><rect x="101.688" y="120.759" width="63.6871" height="75.7586" rx="0.737139" fill="white"/><rect x="105.688" y="124.759" width="8" height="8" rx="1" fill="#43A6DB"/><rect x="116.984" y="126.759" width="24" height="4" rx="0.824675" fill="#D0D1D7"/><rect x="105.688" y="135.759" width="55.6871" height="4" rx="0.412338" fill="#DCDDE1"/><path d="M105.688 143.171C105.688 142.943 105.872 142.759 106.1 142.759H139.912C140.139 142.759 140.324 142.943 140.324 143.171V146.346C140.324 146.574 140.139 146.759 139.912 146.759H106.1C105.872 146.759 105.688 146.574 105.688 146.346V143.171Z" fill="#DCDDE1"/><g clip-path="url(#clip11_1652_59202)"><rect x="105.688" y="150.759" width="55.6871" height="37" rx="1" fill="#43A6DB"/><rect x="87.5" y="200.127" width="49" height="49" transform="rotate(-45 87.5 200.127)" fill="#86D0F9"/></g><g clip-path="url(#clip12_1652_59202)"><path d="M106.281 191.713C106.281 190.948 106.841 190.651 107.53 190.651H108.601C109.291 190.651 109.85 191.169 109.85 191.807C109.85 191.972 109.813 192.128 109.747 192.269C109.351 193.109 108.172 193.191 107.352 193.625L107.352 192.799C107.352 192.799 106.281 192.352 106.281 191.713Z" stroke="#434960" stroke-width="0.35" stroke-linejoin="round"/></g><g clip-path="url(#clip13_1652_59202)"><path d="M115.547 192.865L116.274 193.592L117.001 192.865" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M114.578 190.684H115.305C115.562 190.684 115.809 190.786 115.991 190.968C116.172 191.15 116.275 191.396 116.275 191.653V193.592" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M114.095 191.411L113.368 190.684L112.641 191.411" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M115.064 193.592H114.337C114.079 193.592 113.833 193.49 113.651 193.308C113.469 193.127 113.367 192.88 113.367 192.623V190.684" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g><g clip-path="url(#clip14_1652_59202)"><path d="M120.778 190.75C120.236 190.75 119.797 191.177 119.797 191.703C119.797 192.656 120.957 193.522 121.581 193.724C122.206 193.522 123.366 192.656 123.366 191.703C123.366 191.177 122.926 190.75 122.384 190.75C122.052 190.75 121.759 190.91 121.581 191.155C121.491 191.03 121.371 190.928 121.231 190.857C121.091 190.787 120.936 190.75 120.778 190.75Z" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g></g><g filter="url(#filter5_dd_1652_59202)"><rect x="169.375" y="57" width="63.6871" height="38.7586" rx="0.737139" fill="white"/><rect x="173.375" y="61" width="8" height="8" rx="1" fill="#43A6DB"/><rect x="184.672" y="63" width="24" height="4" rx="0.824675" fill="#D0D1D7"/><rect x="173.375" y="72" width="55.6871" height="4" rx="0.412338" fill="#DCDDE1"/><path d="M173.375 79.4123C173.375 79.1846 173.56 79 173.787 79H207.599C207.827 79 208.011 79.1846 208.011 79.4123V82.5877C208.011 82.8154 207.827 83 207.599 83H173.787C173.56 83 173.375 82.8154 173.375 82.5877V79.4123Z" fill="#DCDDE1"/><g clip-path="url(#clip15_1652_59202)"><path d="M173.969 90.9544C173.969 90.1896 174.528 89.8922 175.218 89.8922H176.289C176.978 89.8922 177.538 90.41 177.538 91.0488C177.538 91.213 177.501 91.3691 177.434 91.5106C177.039 92.3502 175.86 92.4323 175.039 92.8663L175.039 92.0402C175.039 92.0402 173.969 91.5932 173.969 90.9544Z" stroke="#434960" stroke-width="0.35" stroke-linejoin="round"/></g><g clip-path="url(#clip16_1652_59202)"><path d="M183.234 92.1064L183.961 92.8335L184.688 92.1064" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M182.266 89.9255H182.993C183.25 89.9255 183.496 90.0277 183.678 90.2095C183.86 90.3913 183.962 90.6378 183.962 90.8949V92.8337" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M181.782 90.6526L181.055 89.9255L180.328 90.6526" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M182.751 92.8337H182.024C181.767 92.8337 181.52 92.7316 181.339 92.5498C181.157 92.368 181.055 92.1214 181.055 91.8643V89.9255" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g><g clip-path="url(#clip17_1652_59202)"><path d="M188.466 89.9913C187.924 89.9913 187.484 90.4181 187.484 90.9444C187.484 91.8975 188.644 92.7639 189.269 92.9654C189.893 92.7639 191.053 91.8975 191.053 90.9444C191.053 90.4181 190.614 89.9913 190.072 89.9913C189.74 89.9913 189.446 90.1514 189.269 90.3963C189.178 90.2711 189.058 90.169 188.918 90.0985C188.779 90.028 188.623 89.9912 188.466 89.9913Z" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g></g><g filter="url(#filter6_dd_1652_59202)"><rect x="169.375" y="99.7585" width="63.6871" height="86.5898" rx="0.737139" fill="white"/><rect x="173.375" y="103.759" width="8" height="8" rx="1" fill="#43A6DB"/><rect x="184.672" y="105.759" width="24" height="4" rx="0.824675" fill="#D0D1D7"/><rect x="173.375" y="114.759" width="55.6871" height="4" rx="0.412338" fill="#DCDDE1"/><path d="M173.375 122.171C173.375 121.943 173.56 121.759 173.787 121.759H207.599C207.827 121.759 208.011 121.943 208.011 122.171V125.346C208.011 125.574 207.827 125.759 207.599 125.759H173.787C173.56 125.759 173.375 125.574 173.375 125.346V122.171Z" fill="#DCDDE1"/><g clip-path="url(#clip18_1652_59202)"><rect x="173.375" y="129.759" width="55.6871" height="47.8312" rx="1" fill="#43A6DB"/><circle cx="222.785" cy="192.177" r="38.3474" fill="#86D0F9"/></g><g clip-path="url(#clip19_1652_59202)"><path d="M173.969 181.544C173.969 180.779 174.528 180.482 175.218 180.482H176.289C176.978 180.482 177.538 181 177.538 181.639C177.538 181.803 177.501 181.959 177.434 182.1C177.039 182.94 175.86 183.022 175.039 183.456L175.039 182.63C175.039 182.63 173.969 182.183 173.969 181.544Z" stroke="#434960" stroke-width="0.35" stroke-linejoin="round"/></g><g clip-path="url(#clip20_1652_59202)"><path d="M183.234 182.696L183.961 183.423L184.688 182.696" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M182.266 180.515H182.993C183.25 180.515 183.496 180.617 183.678 180.799C183.86 180.981 183.962 181.228 183.962 181.485V183.423" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M181.782 181.242L181.055 180.515L180.328 181.242" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M182.751 183.423H182.024C181.767 183.423 181.52 183.321 181.339 183.14C181.157 182.958 181.055 182.711 181.055 182.454V180.515" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g><g clip-path="url(#clip21_1652_59202)"><path d="M188.466 180.581C187.924 180.581 187.484 181.008 187.484 181.534C187.484 182.487 188.644 183.354 189.269 183.555C189.893 183.354 191.053 182.487 191.053 181.534C191.053 181.008 190.614 180.581 190.072 180.581C189.74 180.581 189.446 180.741 189.269 180.986C189.178 180.861 189.058 180.759 188.918 180.688C188.779 180.618 188.623 180.581 188.466 180.581Z" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g></g><g filter="url(#filter7_dd_1652_59202)"><rect x="169.375" y="190.348" width="63.6871" height="38.7586" rx="0.737139" fill="white"/><rect x="173.375" y="194.348" width="8" height="8" rx="1" fill="#43A6DB"/><rect x="184.672" y="196.348" width="24" height="4" rx="0.824675" fill="#D0D1D7"/></g><rect x="34.5" y="23" width="18" height="18" rx="9" fill="#43A6DB"/><rect x="60.5" y="25" width="19" height="6" rx="1" fill="#8C8F9A"/><circle cx="85.5" cy="28" r="3" fill="#0096CC"/><path d="M83.8984 28L84.9231 29.0246L87.0444 26.9033" stroke="white" stroke-width="0.8"/><rect x="91.5" y="23" width="26" height="10" rx="1" fill="#0096CC"/><path d="M96.054 30V28.458H97.56V27.474H96.054V26.67H97.854V25.692H94.92V30H96.054ZM101.542 28.386C101.542 27.462 100.81 26.712 99.8562 26.712C98.9022 26.712 98.1702 27.462 98.1702 28.386C98.1702 29.31 98.9022 30.06 99.8562 30.06C100.81 30.06 101.542 29.31 101.542 28.386ZM99.2382 28.386C99.2382 27.882 99.5382 27.636 99.8562 27.636C100.174 27.636 100.474 27.882 100.474 28.386C100.474 28.89 100.174 29.142 99.8562 29.142C99.5382 29.142 99.2382 28.89 99.2382 28.386ZM103.005 30V25.692H101.967V30H103.005ZM104.704 30V25.692H103.666V30H104.704ZM108.503 28.386C108.503 27.462 107.771 26.712 106.817 26.712C105.863 26.712 105.131 27.462 105.131 28.386C105.131 29.31 105.863 30.06 106.817 30.06C107.771 30.06 108.503 29.31 108.503 28.386ZM106.199 28.386C106.199 27.882 106.499 27.636 106.817 27.636C107.135 27.636 107.435 27.882 107.435 28.386C107.435 28.89 107.135 29.142 106.817 29.142C106.499 29.142 106.199 28.89 106.199 28.386ZM111.556 30H112.576L113.488 26.772H112.48L112.03 28.542L111.58 26.772H110.554L110.08 28.542L109.63 26.772H108.628L109.534 30H110.56L111.058 28.17L111.556 30Z" fill="white"/><rect x="60.5" y="36" width="73" height="6" rx="1" fill="#DCDDE1"/></g><defs><filter id="filter0_dd_1652_59202" x="32" y="55.7371" width="67.6875" height="90.5897" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.737139"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59202"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.184285"/><feGaussianBlur stdDeviation="0.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59202" result="effect2_dropShadow_1652_59202"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59202" result="shape"/></filter><filter id="filter1_dd_1652_59202" x="32" y="146.327" width="67.6875" height="42.7585" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.737139"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59202"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.184285"/><feGaussianBlur stdDeviation="0.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59202" result="effect2_dropShadow_1652_59202"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59202" result="shape"/></filter><filter id="filter2_dd_1652_59202" x="32" y="189.086" width="67.6875" height="90.5897" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.737139"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59202"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.184285"/><feGaussianBlur stdDeviation="0.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59202" result="effect2_dropShadow_1652_59202"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59202" result="shape"/></filter><filter id="filter3_dd_1652_59202" x="99.6875" y="55.7371" width="67.6875" height="63.7585" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.737139"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59202"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.184285"/><feGaussianBlur stdDeviation="0.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59202" result="effect2_dropShadow_1652_59202"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59202" result="shape"/></filter><filter id="filter4_dd_1652_59202" x="99.6875" y="119.496" width="67.6875" height="79.7585" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.737139"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59202"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.184285"/><feGaussianBlur stdDeviation="0.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59202" result="effect2_dropShadow_1652_59202"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59202" result="shape"/></filter><filter id="filter5_dd_1652_59202" x="167.375" y="55.7371" width="67.6875" height="42.7585" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.737139"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59202"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.184285"/><feGaussianBlur stdDeviation="0.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59202" result="effect2_dropShadow_1652_59202"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59202" result="shape"/></filter><filter id="filter6_dd_1652_59202" x="167.375" y="98.4957" width="67.6875" height="90.5897" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.737139"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59202"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.184285"/><feGaussianBlur stdDeviation="0.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59202" result="effect2_dropShadow_1652_59202"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59202" result="shape"/></filter><filter id="filter7_dd_1652_59202" x="167.375" y="189.086" width="67.6875" height="42.7585" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.737139"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59202"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.184285"/><feGaussianBlur stdDeviation="0.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59202" result="effect2_dropShadow_1652_59202"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59202" result="shape"/></filter><clipPath id="clip0_1652_59202"><rect width="262" height="200" fill="white" transform="translate(0.5)"/></clipPath><clipPath id="clip1_1652_59202"><rect x="38" y="87" width="55.6871" height="47.8312" rx="1" fill="white"/></clipPath><clipPath id="clip2_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(38 136.831)"/></clipPath><clipPath id="clip3_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(44.7578 136.831)"/></clipPath><clipPath id="clip4_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(51.5156 136.831)"/></clipPath><clipPath id="clip5_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(38 179.59)"/></clipPath><clipPath id="clip6_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(44.7578 179.59)"/></clipPath><clipPath id="clip7_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(51.5156 179.59)"/></clipPath><clipPath id="clip8_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(105.688 110)"/></clipPath><clipPath id="clip9_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(112.445 110)"/></clipPath><clipPath id="clip10_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(119.203 110)"/></clipPath><clipPath id="clip11_1652_59202"><rect x="105.688" y="150.759" width="55.6871" height="37" rx="1" fill="white"/></clipPath><clipPath id="clip12_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(105.688 189.759)"/></clipPath><clipPath id="clip13_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(112.445 189.759)"/></clipPath><clipPath id="clip14_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(119.203 189.759)"/></clipPath><clipPath id="clip15_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(173.375 89)"/></clipPath><clipPath id="clip16_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(180.133 89)"/></clipPath><clipPath id="clip17_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(186.891 89)"/></clipPath><clipPath id="clip18_1652_59202"><rect x="173.375" y="129.759" width="55.6871" height="47.8312" rx="1" fill="white"/></clipPath><clipPath id="clip19_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(173.375 179.59)"/></clipPath><clipPath id="clip20_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(180.133 179.59)"/></clipPath><clipPath id="clip21_1652_59202"><rect width="4.75858" height="4.75858" fill="white" transform="translate(186.891 179.59)"/></clipPath></defs></svg>',
			'simpleCarouselTemplate' 	=> '<svg width="262" height="200" viewBox="0 0 262 200" fill="none" xmlns="http://www.w3.org/2000/svg"><rect width="262" height="200" fill="#F3F4F5"/><path d="M35.2564 46.826H37.076L39.1193 39.6819H39.1989L41.2372 46.826H43.0568L45.9304 36.6442H43.9467L42.1122 44.1314H42.0227L40.0589 36.6442H38.2543L36.2955 44.1265H36.201L34.3665 36.6442H32.3828L35.2564 46.826ZM49.7312 46.9752C51.511 46.9752 52.734 46.1052 53.0522 44.7778L51.3718 44.5888C51.1282 45.2351 50.5316 45.5732 49.756 45.5732C48.5927 45.5732 47.8221 44.8076 47.8072 43.5001H53.1268V42.9482C53.1268 40.2685 51.516 39.0903 49.6367 39.0903C47.4492 39.0903 46.0224 40.6961 46.0224 43.0526C46.0224 45.4489 47.4293 46.9752 49.7312 46.9752ZM47.8121 42.287C47.8668 41.3126 48.5877 40.4922 49.6616 40.4922C50.6957 40.4922 51.3917 41.2479 51.4016 42.287H47.8121ZM56.1756 36.6442H54.7438V37.6684C54.7438 38.4787 54.4256 39.1002 54.0627 39.6471L54.8979 40.189C55.6287 39.5824 56.1756 38.5384 56.1756 37.6584V36.6442ZM57.3427 46.826H59.1424V42.3367C59.1424 41.3672 59.8732 40.6812 60.8626 40.6812C61.1658 40.6812 61.5437 40.7359 61.6978 40.7856V39.13C61.5337 39.1002 61.2504 39.0803 61.0515 39.0803C60.1765 39.0803 59.4457 39.5775 59.1673 40.4624H59.0877V39.1897H57.3427V46.826ZM65.9597 46.9752C67.7395 46.9752 68.9625 46.1052 69.2807 44.7778L67.6003 44.5888C67.3567 45.2351 66.7601 45.5732 65.9846 45.5732C64.8212 45.5732 64.0506 44.8076 64.0357 43.5001H69.3553V42.9482C69.3553 40.2685 67.7445 39.0903 65.8652 39.0903C63.6777 39.0903 62.2509 40.6961 62.2509 43.0526C62.2509 45.4489 63.6578 46.9752 65.9597 46.9752ZM64.0407 42.287C64.0953 41.3126 64.8162 40.4922 65.8901 40.4922C66.9242 40.4922 67.6202 41.2479 67.6301 42.287H64.0407ZM77.6815 46.9752C79.9187 46.9752 81.3406 45.3992 81.3406 43.0377C81.3406 40.6712 79.9187 39.0903 77.6815 39.0903C75.4442 39.0903 74.0224 40.6712 74.0224 43.0377C74.0224 45.3992 75.4442 46.9752 77.6815 46.9752ZM77.6914 45.5334C76.4535 45.5334 75.8469 44.4297 75.8469 43.0327C75.8469 41.6357 76.4535 40.5171 77.6914 40.5171C78.9094 40.5171 79.516 41.6357 79.516 43.0327C79.516 44.4297 78.9094 45.5334 77.6914 45.5334ZM84.6678 42.3516C84.6678 41.2479 85.334 40.6116 86.2836 40.6116C87.2132 40.6116 87.7701 41.2231 87.7701 42.2422V46.826H89.5698V41.9638C89.5748 40.1343 88.5307 39.0903 86.9547 39.0903C85.8113 39.0903 85.0257 39.6371 84.6777 40.4873H84.5882V39.1897H82.8681V46.826H84.6678V42.3516ZM94.5041 38.1904H97.6511V46.826H99.4806V38.1904H102.628V36.6442H94.5041V38.1904ZM104.815 46.826H106.694L108.126 41.6655H108.231L109.662 46.826H111.537L113.699 39.1897H111.86L110.537 44.5292H110.463L109.091 39.1897H107.276L105.904 44.559H105.834L104.492 39.1897H102.657L104.815 46.826ZM114.956 46.826H116.756V39.1897H114.956V46.826ZM115.861 38.1059C116.433 38.1059 116.9 37.6684 116.9 37.1314C116.9 36.5895 116.433 36.152 115.861 36.152C115.284 36.152 114.817 36.5895 114.817 37.1314C114.817 37.6684 115.284 38.1059 115.861 38.1059ZM122.42 39.1897H120.913V37.3601H119.113V39.1897H118.03V40.5817H119.113V44.8275C119.104 46.2643 120.148 46.9702 121.5 46.9305C122.012 46.9155 122.365 46.8161 122.559 46.7515L122.256 45.3445C122.156 45.3694 121.952 45.4141 121.729 45.4141C121.276 45.4141 120.913 45.255 120.913 44.5292V40.5817H122.42V39.1897ZM127.738 39.1897H126.232V37.3601H124.432V39.1897H123.348V40.5817H124.432V44.8275C124.422 46.2643 125.466 46.9702 126.818 46.9305C127.33 46.9155 127.683 46.8161 127.877 46.7515L127.574 45.3445C127.474 45.3694 127.271 45.4141 127.047 45.4141C126.594 45.4141 126.232 45.255 126.232 44.5292V40.5817H127.738V39.1897ZM132.542 46.9752C134.322 46.9752 135.545 46.1052 135.863 44.7778L134.182 44.5888C133.939 45.2351 133.342 45.5732 132.567 45.5732C131.403 45.5732 130.633 44.8076 130.618 43.5001H135.937V42.9482C135.937 40.2685 134.327 39.0903 132.447 39.0903C130.26 39.0903 128.833 40.6961 128.833 43.0526C128.833 45.4489 130.24 46.9752 132.542 46.9752ZM130.623 42.287C130.677 41.3126 131.398 40.4922 132.472 40.4922C133.506 40.4922 134.202 41.2479 134.212 42.287H130.623ZM137.46 46.826H139.26V42.3367C139.26 41.3672 139.99 40.6812 140.98 40.6812C141.283 40.6812 141.661 40.7359 141.815 40.7856V39.13C141.651 39.1002 141.368 39.0803 141.169 39.0803C140.294 39.0803 139.563 39.5775 139.284 40.4624H139.205V39.1897H137.46V46.826ZM145.213 36.6442H143.289L143.448 43.8332H145.049L145.213 36.6442ZM144.249 46.9354C144.835 46.9354 145.337 46.4482 145.342 45.8417C145.337 45.2451 144.835 44.7579 144.249 44.7579C143.642 44.7579 143.15 45.2451 143.155 45.8417C143.15 46.4482 143.642 46.9354 144.249 46.9354Z" fill="#141B38"/><g filter="url(#filter0_dd_1652_59440)"><rect x="32" y="64" width="63.6871" height="86.5898" rx="0.737139" fill="white"/><rect x="36" y="68" width="8" height="8" rx="1" fill="#B6DDAD"/><rect x="47.2969" y="70" width="20" height="4" rx="0.824675" fill="#8C8F9A"/><rect x="36" y="79" width="55.6871" height="4" rx="0.412338" fill="#DCDDE1"/><path d="M36 86.4123C36 86.1846 36.1846 86 36.4123 86H70.224C70.4518 86 70.6364 86.1846 70.6364 86.4123V89.5877C70.6364 89.8154 70.4518 90 70.224 90H36.4123C36.1846 90 36 89.8154 36 89.5877V86.4123Z" fill="#DCDDE1"/><g clip-path="url(#clip0_1652_59440)"><rect x="36" y="94" width="55.6871" height="47.8312" rx="1" fill="#B6DDAD"/><circle cx="36.4099" cy="141.419" r="38.3474" fill="#96CE89"/></g><g clip-path="url(#clip1_1652_59440)"><path d="M36.5938 145.786C36.5938 145.021 37.153 144.723 37.8429 144.723H38.9136C39.6034 144.723 40.1627 145.241 40.1627 145.88C40.1627 146.044 40.1258 146.2 40.0592 146.342C39.6639 147.181 38.4847 147.263 37.6644 147.698L37.6644 146.871C37.6644 146.871 36.5938 146.424 36.5938 145.786Z" stroke="#434960" stroke-width="0.35" stroke-linejoin="round"/></g><g clip-path="url(#clip2_1652_59440)"><path d="M45.8594 146.938L46.5864 147.665L47.3135 146.938" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M44.8906 144.757H45.6177C45.8748 144.757 46.1213 144.859 46.3031 145.041C46.4849 145.222 46.5871 145.469 46.5871 145.726V147.665" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M44.4072 145.484L43.6802 144.757L42.9531 145.484" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M45.3761 147.665H44.6491C44.392 147.665 44.1454 147.563 43.9636 147.381C43.7818 147.199 43.6797 146.953 43.6797 146.695V144.757" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g><g clip-path="url(#clip3_1652_59440)"><path d="M51.0908 144.823C50.5488 144.823 50.1094 145.249 50.1094 145.776C50.1094 146.729 51.2693 147.595 51.8938 147.797C52.5184 147.595 53.6783 146.729 53.6783 145.776C53.6783 145.249 53.2389 144.823 52.6969 144.823C52.3649 144.823 52.0714 144.983 51.8938 145.227C51.8033 145.102 51.6831 145 51.5433 144.93C51.4036 144.859 51.2483 144.822 51.0908 144.823Z" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g></g><g filter="url(#filter1_dd_1652_59440)"><rect x="99.0391" y="64" width="63.6871" height="38.7586" rx="0.737139" fill="white"/><rect x="103.039" y="68" width="8" height="8" rx="1" fill="#B6DDAD"/><rect x="114.336" y="70" width="20" height="4" rx="0.824675" fill="#8C8F9A"/><rect x="103.039" y="79" width="55.6871" height="4" rx="0.412338" fill="#DCDDE1"/><path d="M103.039 86.4123C103.039 86.1846 103.224 86 103.451 86H137.263C137.491 86 137.675 86.1846 137.675 86.4123V89.5877C137.675 89.8154 137.491 90 137.263 90H103.451C103.224 90 103.039 89.8154 103.039 89.5877V86.4123Z" fill="#DCDDE1"/><g clip-path="url(#clip4_1652_59440)"><path d="M103.633 97.9544C103.633 97.1896 104.192 96.8922 104.882 96.8922H105.953C106.642 96.8922 107.202 97.41 107.202 98.0488C107.202 98.213 107.165 98.3691 107.098 98.5106C106.703 99.3502 105.524 99.4323 104.703 99.8663L104.703 99.0402C104.703 99.0402 103.633 98.5932 103.633 97.9544Z" stroke="#434960" stroke-width="0.35" stroke-linejoin="round"/></g><g clip-path="url(#clip5_1652_59440)"><path d="M112.898 99.1064L113.625 99.8335L114.353 99.1064" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M111.93 96.9255H112.657C112.914 96.9255 113.16 97.0277 113.342 97.2095C113.524 97.3913 113.626 97.6378 113.626 97.8949V99.8337" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M111.446 97.6526L110.719 96.9255L109.992 97.6526" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M112.415 99.8337H111.688C111.431 99.8337 111.184 99.7316 111.003 99.5498C110.821 99.368 110.719 99.1214 110.719 98.8643V96.9255" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g><g clip-path="url(#clip6_1652_59440)"><path d="M118.13 96.9913C117.588 96.9913 117.148 97.4181 117.148 97.9444C117.148 98.8975 118.308 99.7639 118.933 99.9654C119.557 99.7639 120.717 98.8975 120.717 97.9444C120.717 97.4181 120.278 96.9913 119.736 96.9913C119.404 96.9913 119.11 97.1514 118.933 97.3963C118.842 97.2711 118.722 97.169 118.582 97.0985C118.443 97.028 118.287 96.9912 118.13 96.9913Z" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g></g><g filter="url(#filter2_dd_1652_59440)"><rect x="166.078" y="64" width="63.6871" height="77.7586" rx="0.737139" fill="white"/><rect x="170.078" y="68" width="8" height="8" rx="1" fill="#B6DDAD"/><rect x="181.375" y="70" width="20" height="4" rx="0.824675" fill="#8C8F9A"/><rect x="170.078" y="79" width="55.6871" height="4" rx="0.412338" fill="#DCDDE1"/><path d="M170.078 86.4123C170.078 86.1846 170.263 86 170.49 86H204.302C204.53 86 204.714 86.1846 204.714 86.4123V89.5877C204.714 89.8154 204.53 90 204.302 90H170.49C170.263 90 170.078 89.8154 170.078 89.5877V86.4123Z" fill="#DCDDE1"/><g clip-path="url(#clip7_1652_59440)"><rect x="170.078" y="94" width="55.6871" height="39" rx="1" fill="#B6DDAD"/><circle cx="219.566" cy="151.438" r="38.3474" fill="#96CE89"/></g><g clip-path="url(#clip8_1652_59440)"><path d="M170.672 136.954C170.672 136.19 171.231 135.892 171.921 135.892H172.992C173.682 135.892 174.241 136.41 174.241 137.049C174.241 137.213 174.204 137.369 174.137 137.511C173.742 138.35 172.563 138.432 171.743 138.866L171.743 138.04C171.743 138.04 170.672 137.593 170.672 136.954Z" stroke="#434960" stroke-width="0.35" stroke-linejoin="round"/></g><g clip-path="url(#clip9_1652_59440)"><path d="M179.938 138.106L180.665 138.833L181.392 138.106" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M178.969 135.926H179.696C179.953 135.926 180.199 136.028 180.381 136.209C180.563 136.391 180.665 136.638 180.665 136.895V138.834" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M178.485 136.653L177.758 135.926L177.031 136.653" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/><path d="M179.454 138.834H178.727C178.47 138.834 178.224 138.732 178.042 138.55C177.86 138.368 177.758 138.121 177.758 137.864V135.926" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g><g clip-path="url(#clip10_1652_59440)"><path d="M185.169 135.991C184.627 135.991 184.188 136.418 184.188 136.944C184.188 137.897 185.347 138.764 185.972 138.965C186.597 138.764 187.756 137.897 187.756 136.944C187.756 136.418 187.317 135.991 186.775 135.991C186.443 135.991 186.15 136.151 185.972 136.396C185.881 136.271 185.761 136.169 185.621 136.098C185.482 136.028 185.326 135.991 185.169 135.991Z" stroke="#434960" stroke-width="0.35" stroke-linecap="round" stroke-linejoin="round"/></g></g><circle cx="117" cy="163" r="2" fill="#2C324C"/><circle cx="124" cy="163" r="2" fill="#D0D1D7"/><circle cx="131" cy="163" r="2" fill="#D0D1D7"/><circle cx="138" cy="163" r="2" fill="#D0D1D7"/><circle cx="145" cy="163" r="2" fill="#D0D1D7"/><defs><filter id="filter0_dd_1652_59440" x="31.0786" y="63.8157" width="65.5303" height="88.4326" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.737139"/><feGaussianBlur stdDeviation="0.460712"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59440"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.184285"/><feGaussianBlur stdDeviation="0.184285"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59440" result="effect2_dropShadow_1652_59440"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59440" result="shape"/></filter><filter id="filter1_dd_1652_59440" x="98.1176" y="63.8157" width="65.5303" height="40.6014" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.737139"/><feGaussianBlur stdDeviation="0.460712"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59440"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.184285"/><feGaussianBlur stdDeviation="0.184285"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59440" result="effect2_dropShadow_1652_59440"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59440" result="shape"/></filter><filter id="filter2_dd_1652_59440" x="165.157" y="63.8157" width="65.5303" height="79.6014" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.737139"/><feGaussianBlur stdDeviation="0.460712"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59440"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.184285"/><feGaussianBlur stdDeviation="0.184285"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59440" result="effect2_dropShadow_1652_59440"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59440" result="shape"/></filter><clipPath id="clip0_1652_59440"><rect x="36" y="94" width="55.6871" height="47.8312" rx="1" fill="white"/></clipPath><clipPath id="clip1_1652_59440"><rect width="4.75858" height="4.75858" fill="white" transform="translate(36 143.831)"/></clipPath><clipPath id="clip2_1652_59440"><rect width="4.75858" height="4.75858" fill="white" transform="translate(42.7578 143.831)"/></clipPath><clipPath id="clip3_1652_59440"><rect width="4.75858" height="4.75858" fill="white" transform="translate(49.5156 143.831)"/></clipPath><clipPath id="clip4_1652_59440"><rect width="4.75858" height="4.75858" fill="white" transform="translate(103.039 96)"/></clipPath><clipPath id="clip5_1652_59440"><rect width="4.75858" height="4.75858" fill="white" transform="translate(109.797 96)"/></clipPath><clipPath id="clip6_1652_59440"><rect width="4.75858" height="4.75858" fill="white" transform="translate(116.555 96)"/></clipPath><clipPath id="clip7_1652_59440"><rect x="170.078" y="94" width="55.6871" height="39" rx="1" fill="white"/></clipPath><clipPath id="clip8_1652_59440"><rect width="4.75858" height="4.75858" fill="white" transform="translate(170.078 135)"/></clipPath><clipPath id="clip9_1652_59440"><rect width="4.75858" height="4.75858" fill="white" transform="translate(176.836 135)"/></clipPath><clipPath id="clip10_1652_59440"><rect width="4.75858" height="4.75858" fill="white" transform="translate(183.594 135)"/></clipPath></defs></svg>',
			'simpleCardsTemplate' 		=> '<svg width="262" height="200" viewBox="0 0 262 200" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_1652_59523)"><rect width="262" height="200" transform="translate(0.5)" fill="#F3F4F5"/><rect x="40" y="33" width="18" height="18" rx="9" fill="#F9BBA0"/><rect x="66" y="35" width="19" height="6" rx="1" fill="#8C8F9A"/><circle cx="91" cy="38" r="3" fill="#0096CC"/><path d="M89.3984 38L90.4231 39.0246L92.5444 36.9033" stroke="white" stroke-width="0.8"/><rect x="97" y="33" width="26" height="10" rx="1" fill="#0096CC"/><path d="M101.554 40V38.458H103.06V37.474H101.554V36.67H103.354V35.692H100.42V40H101.554ZM107.042 38.386C107.042 37.462 106.31 36.712 105.356 36.712C104.402 36.712 103.67 37.462 103.67 38.386C103.67 39.31 104.402 40.06 105.356 40.06C106.31 40.06 107.042 39.31 107.042 38.386ZM104.738 38.386C104.738 37.882 105.038 37.636 105.356 37.636C105.674 37.636 105.974 37.882 105.974 38.386C105.974 38.89 105.674 39.142 105.356 39.142C105.038 39.142 104.738 38.89 104.738 38.386ZM108.505 40V35.692H107.467V40H108.505ZM110.204 40V35.692H109.166V40H110.204ZM114.003 38.386C114.003 37.462 113.271 36.712 112.317 36.712C111.363 36.712 110.631 37.462 110.631 38.386C110.631 39.31 111.363 40.06 112.317 40.06C113.271 40.06 114.003 39.31 114.003 38.386ZM111.699 38.386C111.699 37.882 111.999 37.636 112.317 37.636C112.635 37.636 112.935 37.882 112.935 38.386C112.935 38.89 112.635 39.142 112.317 39.142C111.999 39.142 111.699 38.89 111.699 38.386ZM117.056 40H118.076L118.988 36.772H117.98L117.53 38.542L117.08 36.772H116.054L115.58 38.542L115.13 36.772H114.128L115.034 40H116.06L116.558 38.17L117.056 40Z" fill="white"/><rect x="66" y="46" width="73" height="6" rx="1" fill="#DCDDE1"/><g filter="url(#filter0_dd_1652_59523)"><rect x="40" y="64.684" width="179.912" height="56.9119" rx="0.5" fill="white"/><rect x="48.4531" y="73.1399" width="18" height="18" rx="2" fill="#F9BBA0"/><rect x="74.4531" y="74.1399" width="19" height="6" rx="1" fill="#8C8F9A"/><circle cx="100.453" cy="77.1399" r="4" fill="#0096CC"/><path d="M98.3203 77.1399L99.6865 78.5061L102.515 75.6776" stroke="white" stroke-width="1.06667"/><rect x="74.4531" y="87.1399" width="127.314" height="6" rx="1" fill="#DCDDE1"/><rect x="74.4531" y="97.1399" width="137" height="6" rx="1" fill="#DCDDE1"/><rect x="74.4531" y="107.14" width="102" height="6" rx="1" fill="#DCDDE1"/></g><g filter="url(#filter1_dd_1652_59523)"><rect x="40" y="125.824" width="179.912" height="106" rx="0.5" fill="white"/><rect x="48.4531" y="134.28" width="18" height="18" rx="2" fill="#F9BBA0"/><rect x="74.4531" y="135.28" width="19" height="6" rx="1" fill="#8C8F9A"/><circle cx="100.453" cy="138.28" r="4" fill="#0096CC"/><path d="M98.3203 138.28L99.6865 139.646L102.515 136.818" stroke="white" stroke-width="1.06667"/><rect x="74.4531" y="148.28" width="127.314" height="6" rx="1" fill="#DCDDE1"/><rect x="74.4531" y="158.28" width="137" height="6" rx="1" fill="#DCDDE1"/><rect x="74.4531" y="168.28" width="102" height="6" rx="1" fill="#DCDDE1"/><g clip-path="url(#clip1_1652_59523)"><rect x="74.4531" y="182.28" width="137" height="46" rx="2" fill="#F9BBA0"/><circle cx="190" cy="284" r="103" fill="#FCE1D5"/></g></g></g><defs><filter id="filter0_dd_1652_59523" x="35" y="63.684" width="189.914" height="66.9119" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4"/><feGaussianBlur stdDeviation="2.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59523"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59523" result="effect2_dropShadow_1652_59523"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59523" result="shape"/></filter><filter id="filter1_dd_1652_59523" x="35" y="124.824" width="189.914" height="116" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4"/><feGaussianBlur stdDeviation="2.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59523"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59523" result="effect2_dropShadow_1652_59523"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59523" result="shape"/></filter><clipPath id="clip0_1652_59523"><rect width="262" height="200" fill="white" transform="translate(0.5)"/></clipPath><clipPath id="clip1_1652_59523"><rect x="74.4531" y="182.28" width="137" height="46" rx="2" fill="white"/></clipPath></defs></svg>',
			'showcaseCarouselTemplate' 	=> '<svg width="262" height="200" viewBox="0 0 262 200" fill="none" xmlns="http://www.w3.org/2000/svg"><rect width="262" height="200" fill="#F3F4F5"/><circle cx="117" cy="169" r="2" fill="#2C324C"/><circle cx="124" cy="169" r="2" fill="#D0D1D7"/><circle cx="131" cy="169" r="2" fill="#D0D1D7"/><circle cx="138" cy="169" r="2" fill="#D0D1D7"/><circle cx="145" cy="169" r="2" fill="#D0D1D7"/><g filter="url(#filter0_dd_1652_59574)"><rect x="53" y="31" width="156" height="127.083" rx="1.22326" fill="white"/><rect x="59.6406" y="37.6379" width="10.9482" height="10.9482" rx="1.65947" fill="#FFD066"/><rect x="76.0625" y="40.375" width="26.002" height="5.47411" rx="1.36853" fill="#8C8F9A"/><circle cx="111.645" cy="43.1119" r="4.10558" fill="#0096CC"/><path d="M109.453 43.1119L110.855 44.5142L113.758 41.6111" stroke="white" stroke-width="1.09482"/><rect x="59.6406" y="55.224" width="142.724" height="4.10558" rx="0.684264" fill="#DCDDE1"/><path d="M59.6406 64.1194C59.6406 63.7415 59.947 63.4352 60.3249 63.4352H116.435C116.812 63.4352 117.119 63.7415 117.119 64.1194V66.8565C117.119 67.2344 116.812 67.5408 116.435 67.5408H60.3249C59.947 67.5408 59.6406 67.2344 59.6406 66.8565V64.1194Z" fill="#DCDDE1"/><g clip-path="url(#clip0_1652_59574)"><rect width="142.724" height="69.37" transform="translate(59.6406 74.1787)" fill="#FFDF99"/><circle cx="60.324" cy="152.869" r="63.6365" fill="#FFD066"/></g><path d="M60.625 150.111C60.625 148.842 61.5531 148.348 62.6979 148.348H64.4747C65.6195 148.348 66.5476 149.208 66.5476 150.268C66.5476 150.54 66.4863 150.799 66.3758 151.034C65.7198 152.427 63.763 152.563 62.4018 153.284L62.4018 151.913C62.4018 151.913 60.625 151.171 60.625 150.111Z" stroke="#434960" stroke-width="0.580816" stroke-linejoin="round"/><g clip-path="url(#clip1_1652_59574)"><path d="M76.0156 152.023L77.2221 153.229L78.4286 152.023" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/><path d="M74.4062 148.403H75.6128C76.0394 148.403 76.4486 148.573 76.7503 148.875C77.052 149.176 77.2214 149.585 77.2214 150.012V153.229" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/><path d="M73.6005 149.61L72.394 148.403L71.1875 149.61" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/><path d="M75.2058 153.229H73.9993C73.5727 153.229 73.1635 153.06 72.8618 152.758C72.5601 152.457 72.3906 152.047 72.3906 151.621V148.403" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/></g><g clip-path="url(#clip2_1652_59574)"><path d="M84.6834 148.513C83.7839 148.513 83.0547 149.221 83.0547 150.094C83.0547 151.676 84.9795 153.114 86.016 153.448C87.0524 153.114 88.9772 151.676 88.9772 150.094C88.9772 149.221 88.248 148.513 87.3485 148.513C86.7977 148.513 86.3106 148.778 86.016 149.185C85.8658 148.977 85.6663 148.807 85.4343 148.69C85.2023 148.573 84.9448 148.513 84.6834 148.513Z" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/></g></g><defs><filter id="filter0_dd_1652_59574" x="51.4709" y="30.6942" width="159.058" height="130.141" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.22326"/><feGaussianBlur stdDeviation="0.76454"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59574"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.305816"/><feGaussianBlur stdDeviation="0.305816"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59574" result="effect2_dropShadow_1652_59574"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59574" result="shape"/></filter><clipPath id="clip0_1652_59574"><rect width="142.724" height="69.37" fill="white" transform="translate(59.6406 74.1787)"/></clipPath><clipPath id="clip1_1652_59574"><rect width="7.89675" height="7.89675" fill="white" transform="translate(70.8594 146.868)"/></clipPath><clipPath id="clip2_1652_59574"><rect width="7.89675" height="7.89675" fill="white" transform="translate(82.0703 146.868)"/></clipPath></defs></svg>',
			'latestTweetTemplate' 		=> '<svg width="262" height="200" viewBox="0 0 262 200" fill="none" xmlns="http://www.w3.org/2000/svg"><rect width="262" height="200" transform="translate(0.5)" fill="#F3F4F5"/><g filter="url(#filter0_dd_1652_59614)"><rect x="53" y="31" width="156" height="127.083" rx="1.22326" fill="white"/><rect x="59.6406" y="37.6379" width="10.9482" height="10.9482" rx="1.65947" fill="#FFD066"/><rect x="76.0625" y="40.375" width="26.002" height="5.47411" rx="1.36853" fill="#8C8F9A"/><circle cx="111.645" cy="43.1119" r="4.10558" fill="#0096CC"/><path d="M109.453 43.1119L110.855 44.5142L113.758 41.6111" stroke="white" stroke-width="1.09482"/><rect x="59.6406" y="55.224" width="142.724" height="4.10558" rx="0.684264" fill="#DCDDE1"/><path d="M59.6406 64.1194C59.6406 63.7415 59.947 63.4352 60.3249 63.4352H116.435C116.812 63.4352 117.119 63.7415 117.119 64.1194V66.8565C117.119 67.2344 116.812 67.5408 116.435 67.5408H60.3249C59.947 67.5408 59.6406 67.2344 59.6406 66.8565V64.1194Z" fill="#DCDDE1"/><g clip-path="url(#clip0_1652_59614)"><rect width="142.724" height="69.37" transform="translate(59.6406 74.1787)" fill="#FFDF99"/><rect x="56" y="164.476" width="91.8067" height="97.0519" transform="rotate(-47.3051 56 164.476)" fill="#FFD066"/></g><path d="M60.625 150.111C60.625 148.842 61.5531 148.348 62.6979 148.348H64.4747C65.6195 148.348 66.5476 149.208 66.5476 150.268C66.5476 150.54 66.4863 150.799 66.3758 151.034C65.7198 152.427 63.763 152.563 62.4018 153.284L62.4018 151.913C62.4018 151.913 60.625 151.171 60.625 150.111Z" stroke="#434960" stroke-width="0.580816" stroke-linejoin="round"/><g clip-path="url(#clip1_1652_59614)"><path d="M76.0156 152.023L77.2221 153.229L78.4286 152.023" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/><path d="M74.4062 148.403H75.6128C76.0394 148.403 76.4486 148.573 76.7503 148.875C77.052 149.176 77.2214 149.585 77.2214 150.012V153.229" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/><path d="M73.6005 149.61L72.394 148.403L71.1875 149.61" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/><path d="M75.2058 153.229H73.9993C73.5727 153.229 73.1635 153.06 72.8618 152.758C72.5601 152.457 72.3906 152.047 72.3906 151.621V148.403" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/></g><g clip-path="url(#clip2_1652_59614)"><path d="M84.6834 148.513C83.7839 148.513 83.0547 149.221 83.0547 150.094C83.0547 151.676 84.9795 153.114 86.016 153.448C87.0524 153.114 88.9772 151.676 88.9772 150.094C88.9772 149.221 88.248 148.513 87.3485 148.513C86.7977 148.513 86.3106 148.778 86.016 149.185C85.8658 148.977 85.6663 148.807 85.4343 148.69C85.2023 148.573 84.9448 148.513 84.6834 148.513Z" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/></g></g><defs><filter id="filter0_dd_1652_59614" x="51.4709" y="30.6942" width="159.058" height="130.141" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.22326"/><feGaussianBlur stdDeviation="0.76454"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59614"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.305816"/><feGaussianBlur stdDeviation="0.305816"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59614" result="effect2_dropShadow_1652_59614"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59614" result="shape"/></filter><clipPath id="clip0_1652_59614"><rect width="142.724" height="69.37" fill="white" transform="translate(59.6406 74.1787)"/></clipPath><clipPath id="clip1_1652_59614"><rect width="7.89675" height="7.89675" fill="white" transform="translate(70.8594 146.868)"/></clipPath><clipPath id="clip2_1652_59614"><rect width="7.89675" height="7.89675" fill="white" transform="translate(82.0703 146.868)"/></clipPath></defs></svg>',
			'widgetTemplate' 			=> '<svg width="262" height="200" viewBox="0 0 262 200" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_1652_59648)"><rect width="262" height="200" fill="#F3F4F5"/><g filter="url(#filter0_dd_1652_59648)"><rect x="78" y="62" width="105.687" height="63.5346" rx="1.22326" fill="white"/><rect x="84.6406" y="68.6379" width="13.0345" height="13.0345" rx="2" fill="#8C8F9A"/><rect x="103.047" y="71.8966" width="39.1034" height="6.51724" rx="1.34365" fill="#D0D1D7"/><rect x="84.6406" y="86.5603" width="90.7315" height="6.51724" rx="0.671826" fill="#DCDDE1"/><path d="M84.6406 98.6374C84.6406 98.2664 84.9414 97.9656 85.3125 97.9656H140.402C140.773 97.9656 141.074 98.2664 141.074 98.6374V103.811C141.074 104.182 140.773 104.483 140.402 104.483H85.3125C84.9414 104.483 84.6406 104.182 84.6406 103.811V98.6374Z" fill="#DCDDE1"/><path d="M85.625 117.562C85.625 116.293 86.5531 115.8 87.6979 115.8H89.4747C90.6195 115.8 91.5476 116.659 91.5476 117.719C91.5476 117.991 91.4863 118.25 91.3758 118.485C90.7198 119.879 88.763 120.015 87.4018 120.735L87.4018 119.364C87.4018 119.364 85.625 118.622 85.625 117.562Z" stroke="#434960" stroke-width="0.580816" stroke-linejoin="round"/><g clip-path="url(#clip1_1652_59648)"><path d="M101.016 119.474L102.222 120.68L103.429 119.474" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/><path d="M99.4062 115.855H100.613C101.039 115.855 101.449 116.024 101.75 116.326C102.052 116.628 102.221 117.037 102.221 117.463V120.681" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/><path d="M98.6005 117.061L97.394 115.855L96.1875 117.061" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/><path d="M100.206 120.681H98.9993C98.5727 120.681 98.1635 120.511 97.8618 120.21C97.5601 119.908 97.3906 119.499 97.3906 119.072V115.855" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/></g><g clip-path="url(#clip2_1652_59648)"><path d="M109.683 115.964C108.784 115.964 108.055 116.672 108.055 117.546C108.055 119.127 109.98 120.565 111.016 120.899C112.052 120.565 113.977 119.127 113.977 117.546C113.977 116.672 113.248 115.964 112.349 115.964C111.798 115.964 111.311 116.23 111.016 116.636C110.866 116.428 110.666 116.259 110.434 116.142C110.202 116.025 109.945 115.964 109.683 115.964Z" stroke="#434960" stroke-width="0.580816" stroke-linecap="round" stroke-linejoin="round"/></g></g><g filter="url(#filter1_dd_1652_59648)"><rect x="78" y="131.535" width="105.687" height="142.909" rx="1.22326" fill="white"/><rect x="84.6406" y="138.173" width="13.0345" height="13.0345" rx="2" fill="#8C8F9A"/><rect x="103.047" y="141.431" width="39.1034" height="6.51724" rx="1.34365" fill="#D0D1D7"/><rect x="84.6406" y="156.095" width="90.7315" height="6.51724" rx="0.671826" fill="#DCDDE1"/><path d="M84.6406 168.172C84.6406 167.801 84.9414 167.5 85.3125 167.5H140.402C140.773 167.5 141.074 167.801 141.074 168.172V173.346C141.074 173.717 140.773 174.017 140.402 174.017H85.3125C84.9414 174.017 84.6406 173.717 84.6406 173.346V168.172Z" fill="#DCDDE1"/><g clip-path="url(#clip3_1652_59648)"><rect x="84.6406" y="180.535" width="92.4113" height="79.3746" rx="2" fill="#8C8F9A"/><circle opacity="0.5" cx="85.324" cy="259.225" r="63.6365" fill="#D0D1D7"/></g></g><rect x="78" y="29" width="18" height="18" rx="9" fill="#8C8F9A"/><rect x="104" y="31" width="19" height="6" rx="1" fill="#8C8F9A"/><circle cx="129" cy="34" r="3" fill="#0096CC"/><path d="M127.398 34L128.423 35.0246L130.544 32.9033" stroke="white" stroke-width="0.8"/><rect x="135" y="29" width="26" height="10" rx="1" fill="#0096CC"/><path d="M139.554 36V34.458H141.06V33.474H139.554V32.67H141.354V31.692H138.42V36H139.554ZM145.042 34.386C145.042 33.462 144.31 32.712 143.356 32.712C142.402 32.712 141.67 33.462 141.67 34.386C141.67 35.31 142.402 36.06 143.356 36.06C144.31 36.06 145.042 35.31 145.042 34.386ZM142.738 34.386C142.738 33.882 143.038 33.636 143.356 33.636C143.674 33.636 143.974 33.882 143.974 34.386C143.974 34.89 143.674 35.142 143.356 35.142C143.038 35.142 142.738 34.89 142.738 34.386ZM146.505 36V31.692H145.467V36H146.505ZM148.204 36V31.692H147.166V36H148.204ZM152.003 34.386C152.003 33.462 151.271 32.712 150.317 32.712C149.363 32.712 148.631 33.462 148.631 34.386C148.631 35.31 149.363 36.06 150.317 36.06C151.271 36.06 152.003 35.31 152.003 34.386ZM149.699 34.386C149.699 33.882 149.999 33.636 150.317 33.636C150.635 33.636 150.935 33.882 150.935 34.386C150.935 34.89 150.635 35.142 150.317 35.142C149.999 35.142 149.699 34.89 149.699 34.386ZM155.056 36H156.076L156.988 32.772H155.98L155.53 34.542L155.08 32.772H154.054L153.58 34.542L153.13 32.772H152.128L153.034 36H154.06L154.558 34.17L155.056 36Z" fill="white"/><rect x="104" y="42" width="73" height="6" rx="1" fill="#DCDDE1"/></g><defs><filter id="filter0_dd_1652_59648" x="76.4709" y="61.6942" width="108.746" height="66.5927" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.22326"/><feGaussianBlur stdDeviation="0.76454"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59648"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.305816"/><feGaussianBlur stdDeviation="0.305816"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59648" result="effect2_dropShadow_1652_59648"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59648" result="shape"/></filter><filter id="filter1_dd_1652_59648" x="76.4709" y="131.229" width="108.746" height="145.967" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.22326"/><feGaussianBlur stdDeviation="0.76454"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1652_59648"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.305816"/><feGaussianBlur stdDeviation="0.305816"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_1652_59648" result="effect2_dropShadow_1652_59648"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_1652_59648" result="shape"/></filter><clipPath id="clip0_1652_59648"><rect width="262" height="200" fill="white"/></clipPath><clipPath id="clip1_1652_59648"><rect width="7.89675" height="7.89675" fill="white" transform="translate(95.8594 114.319)"/></clipPath><clipPath id="clip2_1652_59648"><rect width="7.89675" height="7.89675" fill="white" transform="translate(107.07 114.319)"/></clipPath><clipPath id="clip3_1652_59648"><rect x="84.6406" y="180.535" width="92.4113" height="79.3746" rx="2" fill="white"/></clipPath></defs></svg>',
			'ctaBoxes'					=> [
				'displayPhotos'		=> '<svg width="56" height="56" viewBox="0 0 56 56" fill="none" xmlns="http://www.w3.org/2000/svg"><rect x="2" y="12.0051" width="28.1951" height="23.9101" rx="1" transform="rotate(-5 2 12.0051)" fill="#8C8F9A"/><path fill-rule="evenodd" clip-rule="evenodd" d="M18.0386 29.2966L18.0387 29.2967L7.13514 30.2507L12.065 23.8082L15.6252 27.0793L20.266 20.2569L28.199 28.4077L18.0386 29.2966Z" fill="white"/><circle cx="13.5055" cy="18.5168" r="2.20279" transform="rotate(-5 13.5055 18.5168)" fill="white"/><rect x="22.8251" y="19.3603" width="29.958" height="25.5524" rx="1.88112" transform="rotate(4 22.8251 19.3603)" fill="#0096CC"/><path d="M32.3311 28.6891C32.354 28.3624 32.7127 28.174 32.9947 28.3405L40.9528 33.0404C41.2654 33.225 41.2331 33.6873 40.8978 33.8266L32.363 37.3732C32.0606 37.4988 31.7316 37.2623 31.7545 36.9356L32.3311 28.6891Z" fill="white"/><rect x="22.8251" y="19.3603" width="29.958" height="25.5524" rx="1.88112" transform="rotate(4 22.8251 19.3603)" stroke="white" stroke-width="1.76224"/></svg>',
				'multiple'			=> '<svg width="56" height="56" viewBox="0 0 56 56" fill="none" xmlns="http://www.w3.org/2000/svg"><rect x="2" y="11.4575" width="28.1951" height="27" rx="1" transform="rotate(-5 2 11.4575)" fill="#8C8F9A"/><path d="M25.1294 18.145L21.7966 18.4366L22.1433 15.4837C22.1506 15.3969 22.139 15.3096 22.1093 15.2278C22.0797 15.1459 22.0326 15.0715 21.9714 15.0096C21.9102 14.9476 21.8363 14.8997 21.7549 14.8691C21.6734 14.8384 21.5862 14.8258 21.4994 14.832L20.2984 14.9371C20.1593 14.9459 20.0275 15.0026 19.9255 15.0975C19.8235 15.1925 19.7575 15.3199 19.7389 15.458L19.4006 18.6462L16.1579 18.9299L16.5046 15.977C16.5119 15.8922 16.5011 15.8068 16.4729 15.7265C16.4448 15.6462 16.4 15.5727 16.3414 15.511C16.2828 15.4492 16.2119 15.4005 16.1331 15.3682C16.0544 15.3358 15.9698 15.3205 15.8847 15.3233L14.6837 15.4283C14.5446 15.4371 14.4128 15.4938 14.3108 15.5888C14.2088 15.6837 14.1428 15.8111 14.1242 15.9493L13.7979 19.1364L10.1949 19.4516C10.0542 19.4611 9.9212 19.5196 9.81915 19.617C9.71711 19.7144 9.65245 19.8444 9.63642 19.9846L9.49529 21.2071C9.48497 21.297 9.495 21.388 9.52464 21.4734C9.55427 21.5588 9.60274 21.6365 9.66646 21.7007C9.73019 21.7648 9.80753 21.8138 9.89276 21.844C9.97799 21.8742 10.0689 21.8849 10.1588 21.8752L13.4916 21.5836L12.9271 26.4738L9.32408 26.789C9.18335 26.7985 9.05039 26.857 8.94834 26.9544C8.8463 27.0517 8.78164 27.1818 8.76561 27.322L8.62448 28.5445C8.61416 28.6343 8.62419 28.7253 8.65383 28.8108C8.68346 28.8962 8.73193 28.9739 8.79565 29.038C8.85938 29.1022 8.93672 29.1512 9.02195 29.1814C9.10718 29.2116 9.19812 29.2223 9.28802 29.2126L12.6448 28.9189L12.2981 31.8718C12.2907 31.9601 12.3028 32.0489 12.3336 32.132C12.3645 32.215 12.4132 32.2903 12.4764 32.3523C12.5396 32.4144 12.6157 32.4618 12.6993 32.4911C12.7829 32.5204 12.8719 32.5309 12.9601 32.5219L14.1611 32.4168C14.2944 32.4017 14.4189 32.3425 14.5148 32.2486C14.6108 32.1547 14.6726 32.0315 14.6906 31.8985L15.0408 28.7093L18.2835 28.4256L17.9368 31.3785C17.9296 31.4633 17.9404 31.5487 17.9685 31.629C17.9966 31.7093 18.0415 31.7828 18.1 31.8445C18.1586 31.9063 18.2296 31.955 18.3083 31.9873C18.387 32.0197 18.4717 32.035 18.5567 32.0322L19.7577 31.9272C19.8968 31.9184 20.0286 31.8617 20.1306 31.7668C20.2327 31.6718 20.2986 31.5444 20.3173 31.4062L20.6735 28.2165L24.2766 27.9012C24.4173 27.8918 24.5502 27.8333 24.6523 27.7359C24.7543 27.6385 24.819 27.5084 24.835 27.3683L24.9702 26.1463C24.9805 26.0564 24.9704 25.9654 24.9408 25.88C24.9112 25.7946 24.8627 25.7169 24.799 25.6527C24.7353 25.5886 24.6579 25.5396 24.5727 25.5094C24.4874 25.4792 24.3965 25.4685 24.3066 25.4782L20.9558 25.7714L21.5143 20.8817L25.1173 20.5665C25.2581 20.557 25.391 20.4985 25.4931 20.4011C25.5951 20.3038 25.6598 20.1737 25.6758 20.0336L25.8169 18.811C25.8277 18.7191 25.817 18.6259 25.7859 18.5387C25.7548 18.4515 25.7041 18.3727 25.6376 18.3083C25.5711 18.2439 25.4907 18.1956 25.4025 18.1673C25.3144 18.139 25.2209 18.1314 25.1294 18.145ZM18.5538 25.9815L15.3111 26.2652L15.8756 21.375L19.1183 21.0913L18.5538 25.9815Z" fill="white"/><rect x="24.4225" y="17.6761" width="30.1951" height="30" rx="2" transform="rotate(6 24.4225 17.6761)" fill="#0096CC"/><path fill-rule="evenodd" clip-rule="evenodd" d="M37.1313 27.2006C38.6084 27.0649 40.0926 27.3702 41.3962 28.078C42.6998 28.7858 43.7643 29.8643 44.4549 31.177C45.1456 32.4898 45.4315 33.9779 45.2764 35.4531L45.1603 36.5581C45.0859 37.2664 44.7331 37.9161 44.1797 38.3642C43.6262 38.8124 42.9174 39.0224 42.2092 38.9479C41.5009 38.8735 40.8512 38.5208 40.403 37.9673C40.3195 37.8642 40.2443 37.7556 40.1776 37.6427C39.3995 38.2603 38.4097 38.5486 37.4207 38.4447C36.4194 38.3394 35.5009 37.8407 34.8673 37.0582C34.2336 36.2758 33.9368 35.2737 34.042 34.2723C34.1473 33.271 34.646 32.3525 35.4284 31.7189C36.2109 31.0853 37.213 30.7884 38.2144 30.8937C39.2157 30.9989 40.1342 31.4976 40.7678 32.2801C41.4014 33.0625 41.6983 34.0646 41.593 35.066L41.4769 36.171C41.4487 36.4396 41.5283 36.7085 41.6983 36.9184C41.8683 37.1284 42.1147 37.2622 42.3834 37.2904C42.652 37.3186 42.9209 37.239 43.1308 37.069C43.3407 36.899 43.4745 36.6526 43.5028 36.3839L43.6189 35.2789C43.7395 34.1315 43.5172 32.9741 42.98 31.9531C42.4428 30.932 41.6149 30.0932 40.601 29.5427C39.587 28.9922 38.4327 28.7547 37.2838 28.8603C36.1349 28.9659 35.0431 29.4097 34.1465 30.1358C33.2499 30.8619 32.5887 31.8375 32.2466 32.9393C31.9045 34.0412 31.8967 35.2197 32.2244 36.3259C32.5521 37.4321 33.2004 38.4163 34.0875 39.1541C34.9745 39.8918 36.0604 40.3499 37.2078 40.4705L37.2086 40.4706C38.2268 40.5787 39.2549 40.4161 40.189 39.9985C40.6092 39.8107 41.1021 39.999 41.2899 40.4192C41.4777 40.8394 41.2894 41.3322 40.8692 41.5201C39.6661 42.0579 38.3425 42.267 37.0327 42.128L37.1207 41.2993L37.0336 42.1281C37.0333 42.128 37.033 42.128 37.0327 42.128C35.5578 41.9728 34.162 41.3838 33.0217 40.4355C31.8813 39.487 31.0477 38.2215 30.6264 36.7993C30.2051 35.377 30.215 33.8617 30.6549 32.4451C31.0948 31.0285 31.9448 29.7741 33.0976 28.8406C34.2504 27.907 35.6541 27.3363 37.1313 27.2006ZM39.9355 34.8918C39.9945 34.33 39.828 33.7679 39.4726 33.3289C39.1171 32.89 38.6019 32.6102 38.0401 32.5512C37.4784 32.4922 36.9163 32.6587 36.4773 33.0141C36.0384 33.3696 35.7586 33.8848 35.6996 34.4465C35.6405 35.0083 35.8071 35.5704 36.1625 36.0094C36.5179 36.4483 37.0332 36.7281 37.5949 36.7871C38.1566 36.8462 38.7188 36.6796 39.1578 36.3242C39.5967 35.9687 39.8765 35.4535 39.9355 34.8918Z" fill="white"/><rect x="24.4225" y="17.6761" width="30.1951" height="30" rx="2" transform="rotate(6 24.4225 17.6761)" stroke="white" stroke-width="2"/></svg>',
				'layouts'			=> '<svg width="56" height="56" viewBox="0 0 56 56" fill="none" xmlns="http://www.w3.org/2000/svg"><rect x="4.05859" y="7.01099" width="28" height="28" rx="1" transform="rotate(-4 4.05859 7.01099)" fill="#8C8F9A"/><rect x="9.32422" y="10.6523" width="8" height="11" transform="rotate(-4 9.32422 10.6523)" fill="white"/><path d="M10.2305 23.6208L18.211 23.0628L18.6993 30.0457L10.7188 30.6038L10.2305 23.6208Z" fill="white"/><rect x="19.3008" y="9.95483" width="8" height="7" transform="rotate(-4 19.3008 9.95483)" fill="white"/><path d="M19.9297 18.9329L27.9102 18.3748L28.6775 29.348L20.697 29.9061L19.9297 18.9329Z" fill="white"/><rect x="21.0097" y="19.0574" width="28" height="28" rx="2" transform="rotate(4 21.0097 19.0574)" fill="#0096CC" stroke="white" stroke-width="2"/><path d="M28.0312 26.136L40.9996 27.0428L40.028 40.9375L27.0596 40.0306L28.0312 26.136Z" fill="white"/><path d="M21.5508 25.6826L26.1823 26.0065L25.2107 39.9011L20.5792 39.5773L21.5508 25.6826Z" fill="#94E3FF"/><path d="M42.8516 27.1724L47.4831 27.4962L46.5115 41.3909L41.88 41.067L42.8516 27.1724Z" fill="#94E3FF"/></svg>',
				'types'				=> '<svg width="56" height="56" viewBox="0 0 56 56" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M34.6667 38.3334H13.3333C12.9797 38.3334 12.6406 38.1929 12.3905 37.9429C12.1405 37.6928 12 37.3537 12 37.0001V13.0001C12 12.6465 12.1405 12.3073 12.3905 12.0573C12.6406 11.8072 12.9797 11.6667 13.3333 11.6667H34.6667C35.0203 11.6667 35.3594 11.8072 35.6095 12.0573C35.8595 12.3073 36 12.6465 36 13.0001V37.0001C36 37.3537 35.8595 37.6928 35.6095 37.9429C35.3594 38.1929 35.0203 38.3334 34.6667 38.3334ZM18.6667 18.3334V21.0001H29.3333V18.3334H18.6667ZM18.6667 23.6667V26.3334H29.3333V23.6667H18.6667ZM18.6667 29.0001V31.6667H25.3333V29.0001H18.6667Z" fill="#8C8F9A"/><circle cx="37.5" cy="37.5" r="11.5" fill="white"/><path d="M37.3184 45.2982C39.0895 45.2978 40.8095 44.7049 42.2047 43.6139L46.591 48.0003L48.002 46.5894L43.6156 42.203C44.7072 40.8077 45.3005 39.0873 45.3009 37.3157C45.3009 32.9144 41.7197 29.3333 37.3184 29.3333C32.9171 29.3333 29.3359 32.9144 29.3359 37.3157C29.3359 41.7171 32.9171 45.2982 37.3184 45.2982ZM37.3184 31.3289C40.6202 31.3289 43.3053 34.014 43.3053 37.3157C43.3053 40.6175 40.6202 43.3026 37.3184 43.3026C34.0167 43.3026 31.3316 40.6175 31.3316 37.3157C31.3316 34.014 34.0167 31.3289 37.3184 31.3289Z" fill="#0096CC" stroke="#0096CC" stroke-width="0.5"/></svg>'
			],
			'cardPromoTitle' => '<svg width="194" height="134" viewBox="0 0 194 134" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_1145_45675)"><path d="M0.304137 12.4702C0.191179 9.2355 2.72183 6.52171 5.9565 6.40875L183.616 0.20476C186.85 0.0918028 189.564 2.62245 189.677 5.85713L193.695 120.903C193.808 124.138 191.277 126.852 188.042 126.965L10.3831 133.169C7.14839 133.282 4.43459 130.751 4.32164 127.516L0.304137 12.4702Z" fill="white"/><g clip-path="url(#clip1_1145_45675)"><rect width="189.488" height="79.1163" transform="translate(0.0996094 6.61328) rotate(-2)" fill="#0068A0"/><circle cx="31.9551" cy="72.2044" r="73.9884" transform="rotate(-2 31.9551 72.2044)" fill="#005B8C"/><circle cx="171.467" cy="121.086" r="95.9651" transform="rotate(-2 171.467 121.086)" fill="#004D77"/><circle cx="97.3271" cy="41.0898" r="23.1977" transform="rotate(-2 97.3271 41.0898)" fill="white"/><path fill-rule="evenodd" clip-rule="evenodd" d="M96.3874 34.8668C94.4428 34.0733 92.4038 32.8581 90.5213 32.325C91.2378 33.8793 91.8547 35.5403 92.4904 37.1812C91.3685 38.0873 90.0461 38.8064 88.7859 39.5836C89.9178 40.6269 91.4893 41.1989 92.7568 42.0968C91.8104 43.3517 89.9117 44.6878 89.4769 45.8103C91.5152 45.4823 93.8864 44.9831 95.7883 44.8322C96.2803 46.9216 96.5768 49.2207 97.2394 51.1271C98.0368 48.5457 98.7463 45.8823 99.7183 43.4638C101.37 44.0804 103.384 44.8324 104.987 45.174C103.817 43.6481 102.741 42.0213 101.694 40.3641C103.119 39.2531 104.562 38.1583 105.933 36.9959C103.934 36.8819 101.866 36.8423 99.7709 36.8323C99.363 34.6854 99.3603 32.104 98.7796 30.1425C98.0622 31.7919 97.2005 33.3066 96.3874 34.8668ZM97.9577 52.7121C97.7339 53.8473 98.6168 54.2788 98.5099 54.9659C97.6438 54.7008 96.9947 54.5718 95.8514 54.7746C95.8566 53.8948 96.6424 53.743 96.4409 52.6704C84.1483 51.7059 83.3511 29.5821 95.4775 27.7953C110.64 25.5612 112.035 51.8288 97.9577 52.7121Z" fill="#E34F0E"/><path fill-rule="evenodd" clip-rule="evenodd" d="M98.7796 30.1424C99.3603 32.1039 99.3629 34.6853 99.7709 36.8321C101.866 36.8422 103.934 36.8818 105.933 36.9958C104.562 38.1582 103.119 39.253 101.694 40.364C102.741 42.0213 103.817 43.648 104.987 45.1739C103.384 44.8323 101.37 44.0803 99.7183 43.4636C98.7463 45.8822 98.0368 48.5456 97.2393 51.127C96.5767 49.2206 96.2803 46.9215 95.7883 44.8321C93.8864 44.983 91.5152 45.4822 89.4769 45.8102C89.9117 44.6877 91.8104 43.3516 92.7567 42.0967C91.4892 41.1988 89.9178 40.6268 88.7859 39.5835C90.0461 38.8063 91.3684 38.0872 92.4904 37.1811C91.8547 35.5402 91.2378 33.8792 90.5213 32.3249C92.4037 32.858 94.4428 34.0732 96.3874 34.8667C97.2005 33.3065 98.0622 31.7918 98.7796 30.1424Z" fill="white"/></g><line x1="2.85281" y1="85.4371" x2="192.226" y2="78.8241" stroke="#D0D1D7" stroke-width="0.488372"/><path d="M7.82039 97.5739L7.89811 99.7995C7.92009 100.429 8.38985 100.867 9.01947 100.845L17.8634 100.536C18.493 100.514 18.9311 100.045 18.9091 99.415L18.8313 97.1894C18.8094 96.5598 18.3396 96.1217 17.71 96.1437L8.86607 96.4525C8.23646 96.4745 7.7984 96.9443 7.82039 97.5739ZM16.1996 97.2813L16.2774 99.5069C16.2993 100.137 16.7691 100.575 17.3987 100.553L26.2426 100.244C26.8722 100.222 27.3103 99.752 27.2883 99.1224L27.2106 96.8968C27.1886 96.2671 26.7189 95.8291 26.0892 95.8511L17.2453 96.1599C16.6157 96.1819 16.1776 96.6517 16.1996 97.2813ZM24.5789 96.9887L24.6566 99.2143C24.6786 99.8439 25.1484 100.282 25.778 100.26L34.6219 99.9511C35.2515 99.9291 35.6896 99.4594 35.6676 98.8298L35.5898 96.6041C35.5679 95.9745 35.0981 95.5365 34.4685 95.5585L25.6246 95.8673C24.995 95.8893 24.5569 96.359 24.5789 96.9887ZM32.9581 96.696L33.0359 98.9217C33.0578 99.5513 33.5276 99.9893 34.1572 99.9674L43.0011 99.6585C43.6307 99.6365 44.0688 99.1668 44.0468 98.5372L43.9691 96.3115C43.9471 95.6819 43.4774 95.2439 42.8477 95.2658L34.0038 95.5747C33.3742 95.5967 32.9361 96.0664 32.9581 96.696ZM41.3374 96.4034L41.4151 98.6291C41.4371 99.2587 41.9069 99.6967 42.5365 99.6747L51.3804 99.3659C52.01 99.3439 52.4481 98.8742 52.4261 98.2445L52.3483 96.0189C52.3264 95.3893 51.8566 94.9513 51.227 94.9732L42.3831 95.2821C41.7535 95.3041 41.3154 95.7738 41.3374 96.4034ZM49.7166 96.1108L49.7944 98.3364C49.8163 98.9661 50.2861 99.4041 50.9157 99.3821L59.7596 99.0733C60.3892 99.0513 60.8273 98.5816 60.8053 97.9519L60.7276 95.7263C60.7056 95.0967 60.2359 94.6586 59.6062 94.6806L50.7623 94.9895C50.1327 95.0115 49.6946 95.4812 49.7166 96.1108ZM58.0959 95.8182L58.1736 98.0438C58.1956 98.6735 58.6654 99.1115 59.295 99.0895L68.1389 98.7807C68.7685 98.7587 69.2066 98.2889 69.1846 97.6593L69.1068 95.4337C69.0849 94.8041 68.6151 94.366 67.9855 94.388L59.1416 94.6969C58.512 94.7188 58.0739 95.1886 58.0959 95.8182ZM66.4756 95.5403L66.5529 97.7512C66.5748 98.3808 67.0446 98.8189 67.6742 98.7969L72.3305 98.6343C72.9601 98.6123 73.3981 98.1426 73.3761 97.513L73.2989 95.302C73.2769 94.6724 72.8072 94.2343 72.1776 94.2563L67.5213 94.4189C66.8917 94.4409 66.4537 94.9106 66.4756 95.5403ZM70.6653 95.3939L70.7425 97.6049C70.7645 98.2345 71.2342 98.6726 71.8638 98.6506L76.5201 98.488C77.1497 98.466 77.5878 97.9963 77.5658 97.3667L77.4886 95.1557C77.4666 94.5261 76.9968 94.088 76.3672 94.11L71.711 94.2726C71.0813 94.2946 70.6433 94.7643 70.6653 95.3939ZM74.8544 95.233L74.9321 97.4586C74.9541 98.0882 75.4239 98.5263 76.0535 98.5043L84.8974 98.1955C85.527 98.1735 85.9651 97.7037 85.9431 97.0741L85.8653 94.8485C85.8434 94.2189 85.3736 93.7808 84.744 93.8028L75.9001 94.1116C75.2705 94.1336 74.8324 94.6034 74.8544 95.233ZM83.2336 94.9404L83.3114 97.166C83.3333 97.7956 83.8031 98.2337 84.4327 98.2117L93.2766 97.9029C93.9062 97.8809 94.3443 97.4111 94.3223 96.7815L94.2446 94.5559C94.2226 93.9263 93.7528 93.4882 93.1232 93.5102L84.2793 93.819C83.6497 93.841 83.2116 94.3108 83.2336 94.9404ZM91.6129 94.6478L91.6906 96.8734C91.7126 97.503 92.1824 97.9411 92.812 97.9191L101.656 97.6103C102.285 97.5883 102.724 97.1185 102.702 96.4889L102.624 94.2633C102.602 93.6337 102.132 93.1956 101.502 93.2176L92.6586 93.5264C92.029 93.5484 91.5909 94.0182 91.6129 94.6478ZM99.9927 94.3698L100.07 96.5808C100.092 97.2104 100.562 97.6485 101.191 97.6265L105.847 97.4639C106.477 97.4419 106.915 96.9721 106.893 96.3425L106.816 94.1315C106.794 93.5019 106.324 93.0639 105.695 93.0858L101.038 93.2484C100.409 93.2704 99.9707 93.7402 99.9927 94.3698ZM104.182 94.2235L104.259 96.4345C104.281 97.0641 104.751 97.5022 105.381 97.4802L110.037 97.3176C110.667 97.2956 111.105 96.8258 111.083 96.1962L111.006 93.9852C110.984 93.3556 110.514 92.9176 109.884 92.9395L105.228 93.1021C104.598 93.1241 104.16 93.5939 104.182 94.2235ZM108.371 94.0626L108.449 96.2882C108.471 96.9178 108.941 97.3559 109.57 97.3339L118.414 97.025C119.044 97.003 119.482 96.5333 119.46 95.9037L119.382 93.678C119.36 93.0484 118.891 92.6104 118.261 92.6324L109.417 92.9412C108.787 92.9632 108.349 93.4329 108.371 94.0626ZM116.751 93.77L116.828 95.9956C116.85 96.6252 117.32 97.0632 117.95 97.0413L126.794 96.7324C127.423 96.7104 127.861 96.2407 127.839 95.6111L127.762 93.3854C127.74 92.7558 127.27 92.3178 126.64 92.3398L117.796 92.6486C117.167 92.6706 116.729 93.1403 116.751 93.77Z" fill="#141B38"/><path d="M8.24031 109.567L8.31803 111.792C8.34001 112.422 8.80977 112.86 9.43939 112.838L18.2833 112.529C18.9129 112.507 19.351 112.037 19.329 111.408L19.2513 109.182C19.2293 108.552 18.7595 108.114 18.1299 108.136L9.286 108.445C8.65638 108.467 8.21832 108.937 8.24031 109.567ZM16.6196 109.274L16.6973 111.5C16.7193 112.129 17.189 112.567 17.8186 112.545L26.6626 112.236C27.2922 112.214 27.7302 111.745 27.7082 111.115L27.6305 108.889C27.6085 108.26 27.1388 107.822 26.5092 107.844L17.6652 108.153C17.0356 108.175 16.5976 108.644 16.6196 109.274ZM24.9988 108.981L25.0765 111.207C25.0985 111.837 25.5683 112.275 26.1979 112.253L35.0418 111.944C35.6714 111.922 36.1095 111.452 36.0875 110.822L36.0098 108.597C35.9878 107.967 35.518 107.529 34.8884 107.551L26.0445 107.86C25.4149 107.882 24.9768 108.352 24.9988 108.981ZM33.3781 108.689L33.4558 110.914C33.4778 111.544 33.9475 111.982 34.5771 111.96L43.4211 111.651C44.0507 111.629 44.4887 111.159 44.4667 110.53L44.389 108.304C44.367 107.675 43.8973 107.237 43.2677 107.259L34.4237 107.567C33.7941 107.589 33.3561 108.059 33.3781 108.689ZM41.7573 108.396L41.835 110.622C41.857 111.251 42.3268 111.689 42.9564 111.667L51.8003 111.359C52.4299 111.337 52.868 110.867 52.846 110.237L52.7683 108.012C52.7463 107.382 52.2765 106.944 51.6469 106.966L42.803 107.275C42.1734 107.297 41.7353 107.766 41.7573 108.396ZM50.1366 108.104L50.2143 110.329C50.2363 110.959 50.706 111.397 51.3356 111.375L60.1796 111.066C60.8092 111.044 61.2472 110.574 61.2252 109.945L61.1475 107.719C61.1255 107.089 60.6558 106.651 60.0262 106.673L51.1822 106.982C50.5526 107.004 50.1146 107.474 50.1366 108.104ZM58.5158 107.811L58.5935 110.037C58.6155 110.666 59.0853 111.104 59.7149 111.082L68.5588 110.773C69.1884 110.751 69.6265 110.282 69.6045 109.652L69.5268 107.426C69.5048 106.797 69.035 106.359 68.4054 106.381L59.5615 106.69C58.9319 106.712 58.4938 107.181 58.5158 107.811ZM66.8956 107.533L66.9728 109.744C66.9948 110.374 67.4645 110.812 68.0941 110.79L72.7504 110.627C73.38 110.605 73.818 110.135 73.7961 109.506L73.7189 107.295C73.6969 106.665 73.2271 106.227 72.5975 106.249L67.9413 106.412C67.3116 106.434 66.8736 106.903 66.8956 107.533ZM71.0852 107.387L71.1624 109.598C71.1844 110.227 71.6542 110.665 72.2838 110.643L76.94 110.481C77.5696 110.459 78.0077 109.989 77.9857 109.359L77.9085 107.148C77.8865 106.519 77.4167 106.081 76.7871 106.103L72.1309 106.265C71.5013 106.287 71.0632 106.757 71.0852 107.387ZM75.2743 107.226L75.352 109.451C75.374 110.081 75.8438 110.519 76.4734 110.497L85.3173 110.188C85.9469 110.166 86.385 109.696 86.363 109.067L86.2853 106.841C86.2633 106.212 85.7935 105.773 85.1639 105.795L76.32 106.104C75.6904 106.126 75.2523 106.596 75.2743 107.226ZM83.6536 106.933L83.7313 109.159C83.7533 109.788 84.223 110.226 84.8526 110.204L93.6965 109.896C94.3262 109.874 94.7642 109.404 94.7422 108.774L94.6645 106.549C94.6425 105.919 94.1728 105.481 93.5432 105.503L84.6992 105.812C84.0696 105.834 83.6316 106.303 83.6536 106.933ZM92.0328 106.64L92.1105 108.866C92.1325 109.496 92.6023 109.934 93.2319 109.912L102.076 109.603C102.705 109.581 103.143 109.111 103.121 108.482L103.044 106.256C103.022 105.626 102.552 105.188 101.922 105.21L93.0785 105.519C92.4489 105.541 92.0108 106.011 92.0328 106.64ZM100.413 106.362L100.49 108.573C100.512 109.203 100.982 109.641 101.611 109.619L106.267 109.457C106.897 109.435 107.335 108.965 107.313 108.335L107.236 106.124C107.214 105.495 106.744 105.057 106.114 105.079L101.458 105.241C100.829 105.263 100.391 105.733 100.413 106.362ZM104.602 106.216L104.679 108.427C104.701 109.057 105.171 109.495 105.801 109.473L110.457 109.31C111.087 109.288 111.525 108.819 111.503 108.189L111.425 105.978C111.403 105.348 110.934 104.91 110.304 104.932L105.648 105.095C105.018 105.117 104.58 105.587 104.602 106.216ZM108.791 106.055L108.869 108.281C108.891 108.91 109.361 109.349 109.99 109.327L118.834 109.018C119.464 108.996 119.902 108.526 119.88 107.896L119.802 105.671C119.78 105.041 119.311 104.603 118.681 104.625L109.837 104.934C109.207 104.956 108.769 105.426 108.791 106.055ZM117.171 105.763L117.248 107.988C117.27 108.618 117.74 109.056 118.37 109.034L127.214 108.725C127.843 108.703 128.281 108.233 128.259 107.604L128.182 105.378C128.16 104.748 127.69 104.31 127.06 104.332L118.216 104.641C117.587 104.663 117.149 105.133 117.171 105.763ZM125.55 105.47L125.628 107.696C125.65 108.325 126.119 108.763 126.749 108.741L135.593 108.432C136.222 108.411 136.66 107.941 136.638 107.311L136.561 105.086C136.539 104.456 136.069 104.018 135.439 104.04L126.595 104.349C125.966 104.371 125.528 104.84 125.55 105.47ZM133.93 105.192L134.007 107.403C134.029 108.033 134.499 108.471 135.128 108.449L139.784 108.286C140.414 108.264 140.852 107.794 140.83 107.165L140.753 104.954C140.731 104.324 140.261 103.886 139.631 103.908L134.975 104.071C134.346 104.093 133.908 104.562 133.93 105.192ZM138.119 105.046L138.196 107.257C138.218 107.886 138.688 108.324 139.318 108.302L143.974 108.14C144.604 108.118 145.042 107.648 145.02 107.018L144.942 104.807C144.92 104.178 144.451 103.74 143.821 103.762L139.165 103.924C138.535 103.946 138.097 104.416 138.119 105.046Z" fill="#D0D1D7"/><path d="M13.3828 120.693C13.107 120.436 12.7403 120.299 12.3635 120.312C11.9867 120.325 11.6305 120.488 11.3733 120.763L10.4034 121.803C10.1461 122.079 10.009 122.446 10.0222 122.823C10.0354 123.2 10.1977 123.556 10.4735 123.813C10.7494 124.07 11.1161 124.207 11.493 124.194C11.8699 124.181 12.2261 124.019 12.4833 123.743L12.9681 123.223" stroke="#8C8F9A" stroke-width="0.651159" stroke-linecap="round" stroke-linejoin="round"/><path d="M12.4128 121.733C12.6886 121.99 13.0552 122.127 13.432 122.113C13.8089 122.1 14.165 121.938 14.4222 121.662L15.3922 120.623C15.6494 120.347 15.7865 119.98 15.7734 119.603C15.7602 119.226 15.5978 118.87 15.322 118.613C15.0462 118.356 14.6795 118.218 14.3026 118.232C13.9257 118.245 13.5694 118.407 13.3122 118.683L12.8274 119.203" stroke="#8C8F9A" stroke-width="0.651159" stroke-linecap="round" stroke-linejoin="round"/><path d="M19.3966 121.184L19.4743 123.41C19.4963 124.039 19.966 124.478 20.5956 124.456L29.4396 124.147C30.0692 124.125 30.5072 123.655 30.4852 123.025L30.4075 120.8C30.3855 120.17 29.9158 119.732 29.2862 119.754L20.4422 120.063C19.8126 120.085 19.3746 120.555 19.3966 121.184ZM27.7758 120.892L27.8535 123.117C27.8755 123.747 28.3453 124.185 28.9749 124.163L37.8188 123.854C38.4484 123.832 38.8865 123.362 38.8645 122.733L38.7868 120.507C38.7648 119.877 38.295 119.439 37.6654 119.461L28.8215 119.77C28.1919 119.792 27.7538 120.262 27.7758 120.892ZM36.1551 120.599L36.2328 122.825C36.2548 123.454 36.7245 123.892 37.3541 123.87L46.1981 123.561C46.8277 123.539 47.2657 123.07 47.2437 122.44L47.166 120.214C47.144 119.585 46.6743 119.147 46.0447 119.169L37.2007 119.478C36.5711 119.5 36.1331 119.969 36.1551 120.599ZM44.5343 120.306L44.612 122.532C44.634 123.162 45.1038 123.6 45.7334 123.578L54.5773 123.269C55.2069 123.247 55.645 122.777 55.623 122.148L55.5453 119.922C55.5233 119.292 55.0535 118.854 54.4239 118.876L45.58 119.185C44.9504 119.207 44.5123 119.677 44.5343 120.306ZM52.9136 120.014L52.9913 122.239C53.0133 122.869 53.483 123.307 54.1126 123.285L62.9566 122.976C63.5862 122.954 64.0242 122.485 64.0022 121.855L63.9245 119.629C63.9025 119 63.4328 118.562 62.8032 118.584L53.9592 118.892C53.3296 118.914 52.8916 119.384 52.9136 120.014Z" fill="#D0D1D7"/></g><path d="M5.97355 6.89683L183.633 0.692834L183.599 -0.283315L5.93946 5.92068L5.97355 6.89683ZM189.189 5.87417L193.207 120.92L194.183 120.886L190.165 5.84008L189.189 5.87417ZM188.025 126.477L10.366 132.681L10.4001 133.657L188.059 127.453L188.025 126.477ZM4.80971 127.499L0.792211 12.4531L-0.183938 12.4872L3.83356 127.533L4.80971 127.499ZM10.366 132.681C7.4009 132.784 4.91326 130.464 4.80971 127.499L3.83356 127.533C3.95593 131.038 6.89587 133.779 10.4001 133.657L10.366 132.681ZM193.207 120.92C193.31 123.885 190.99 126.373 188.025 126.477L188.059 127.453C191.563 127.33 194.305 124.39 194.183 120.886L193.207 120.92ZM183.633 0.692834C186.598 0.58929 189.085 2.90905 189.189 5.87417L190.165 5.84008C190.043 2.33585 187.103 -0.405685 183.599 -0.283315L183.633 0.692834ZM5.93946 5.92068C2.43523 6.04305 -0.306308 8.98299 -0.183938 12.4872L0.792211 12.4531C0.688667 9.48802 3.00843 7.00037 5.97355 6.89683L5.93946 5.92068Z" fill="#D0D1D7"/><defs><clipPath id="clip0_1145_45675"><path d="M0.304137 12.4702C0.191179 9.2355 2.72183 6.52171 5.9565 6.40875L183.616 0.20476C186.85 0.0918028 189.564 2.62245 189.677 5.85713L193.695 120.903C193.808 124.138 191.277 126.852 188.042 126.965L10.3831 133.169C7.14839 133.282 4.43459 130.751 4.32164 127.516L0.304137 12.4702Z" fill="white"/></clipPath><clipPath id="clip1_1145_45675"><rect width="189.488" height="79.1163" fill="white" transform="translate(0.0996094 6.61328) rotate(-2)"/></clipPath></defs></svg>',
            'twitterSmash' =>'<svg width="51" height="52" viewBox="0 0 51 52" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M47.0738 12.6252C45.4856 13.3471 43.7738 13.8215 42 14.0484C43.815 12.9552 45.2175 11.2227 45.8775 9.13962C44.1656 10.1709 42.2681 10.8927 40.2675 11.3052C38.6381 9.53149 36.3488 8.50024 33.75 8.50024C28.9031 8.50024 24.9431 12.4602 24.9431 17.3484C24.9431 18.0496 25.0256 18.7302 25.17 19.3696C17.8275 18.9984 11.2894 15.4715 6.93752 10.1296C6.17439 11.429 5.74127 12.9552 5.74127 14.564C5.74127 17.6371 7.28814 20.3596 9.68064 21.9065C8.21627 21.9065 6.85502 21.494 5.65877 20.8752V20.9371C5.65877 25.2271 8.71127 28.8159 12.7538 29.6202C11.4559 29.9754 10.0933 30.0248 8.77315 29.7646C9.33333 31.5229 10.4304 33.0613 11.9102 34.1638C13.39 35.2662 15.1781 35.8772 17.0231 35.9109C13.8956 38.3868 10.0189 39.7251 6.03002 39.7059C5.32877 39.7059 4.62752 39.6646 3.92627 39.5821C7.84502 42.0984 12.5063 43.5627 17.4975 43.5627C33.75 43.5627 42.6807 30.074 42.6807 18.3796C42.6807 17.9877 42.6806 17.6165 42.66 17.2246C44.3925 15.9871 45.8775 14.4196 47.0738 12.6252Z" fill="#1B90EF"/><mask id="path-2-outside-1_3011_78642" maskUnits="userSpaceOnUse" x="29.4255" y="24.0344" width="22" height="27" fill="black"><rect fill="white" x="29.4255" y="24.0344" width="22" height="27"/><path fill-rule="evenodd" clip-rule="evenodd" d="M40.2137 26.0344C45.067 26.0344 49 30.8985 49 36.8987C49 42.5376 45.5272 47.1716 41.0837 47.7108L42.1514 49.0159L39.1082 49.2747L39.5935 47.7378C35.0296 47.3442 31.4255 42.6508 31.4255 36.8987C31.4255 30.8985 35.3603 26.0344 40.2137 26.0344ZM42.7327 34.2171L42.3099 29.854L39.611 33.2377L35.7396 31.0306L36.8687 35.0618L32.7766 36.3779L36.5993 38.306L35.0915 42.3546L39.0111 40.5586L40.9523 44.3351L42.2308 40.1178L46.448 40.9106L43.9447 37.2899L47.1141 34.343L42.7327 34.2171Z"/></mask><path fill-rule="evenodd" clip-rule="evenodd" d="M40.2137 26.0344C45.067 26.0344 49 30.8985 49 36.8987C49 42.5376 45.5272 47.1716 41.0837 47.7108L42.1514 49.0159L39.1082 49.2747L39.5935 47.7378C35.0296 47.3442 31.4255 42.6508 31.4255 36.8987C31.4255 30.8985 35.3603 26.0344 40.2137 26.0344ZM42.7327 34.2171L42.3099 29.854L39.611 33.2377L35.7396 31.0306L36.8687 35.0618L32.7766 36.3779L36.5993 38.306L35.0915 42.3546L39.0111 40.5586L40.9523 44.3351L42.2308 40.1178L46.448 40.9106L43.9447 37.2899L47.1141 34.343L42.7327 34.2171Z" fill="#FE544F"/><path d="M41.0837 47.7108L40.8746 45.9883C40.2465 46.0646 39.7097 46.4767 39.4737 47.0638C39.2377 47.6509 39.3401 48.3199 39.7407 48.8096L41.0837 47.7108ZM42.1514 49.0159L42.2985 50.7447C42.9401 50.6902 43.4987 50.2848 43.7495 49.6917C44.0003 49.0987 43.9021 48.4155 43.4943 47.9171L42.1514 49.0159ZM39.1082 49.2747L37.4536 48.7522C37.2793 49.3042 37.3917 49.9068 37.7534 50.3588C38.1151 50.8108 38.6784 51.0527 39.2552 51.0036L39.1082 49.2747ZM39.5935 47.7378L41.2481 48.2603C41.4067 47.7581 41.3286 47.211 41.0359 46.7733C40.7431 46.3355 40.2673 46.0543 39.7426 46.0091L39.5935 47.7378ZM42.3099 29.854L44.0369 29.6866C43.9687 28.9832 43.4808 28.3915 42.8033 28.1905C42.1257 27.9896 41.394 28.2196 40.9534 28.7721L42.3099 29.854ZM42.7327 34.2171L41.0057 34.3845C41.09 35.2548 41.8089 35.9264 42.6829 35.9515L42.7327 34.2171ZM39.611 33.2377L38.7516 34.7451C39.4942 35.1685 40.4345 34.988 40.9675 34.3197L39.611 33.2377ZM35.7396 31.0306L36.5989 29.5232C35.9876 29.1747 35.2264 29.2299 34.6718 29.6629C34.1171 30.096 33.8789 30.8211 34.0687 31.4986L35.7396 31.0306ZM36.8687 35.0618L37.4 36.7136C38.2878 36.4281 38.7911 35.4919 38.5396 34.5938L36.8687 35.0618ZM32.7766 36.3779L32.2454 34.7261C31.5752 34.9416 31.1016 35.5408 31.0468 36.2427C30.9919 36.9446 31.3666 37.61 31.9952 37.9271L32.7766 36.3779ZM36.5993 38.306L38.2254 38.9116C38.534 38.083 38.1702 37.155 37.3808 36.7568L36.5993 38.306ZM35.0915 42.3546L33.4655 41.749C33.2205 42.4069 33.3961 43.1476 33.9103 43.6255C34.4245 44.1035 35.1761 44.2244 35.8143 43.932L35.0915 42.3546ZM39.0111 40.5586L40.5543 39.7653C40.1302 38.9401 39.1318 38.5946 38.2883 38.9811L39.0111 40.5586ZM40.9523 44.3351L39.4091 45.1283C39.7323 45.7571 40.4046 46.1269 41.1087 46.0632C41.8127 45.9995 42.4077 45.515 42.6128 44.8385L40.9523 44.3351ZM42.2308 40.1178L42.5514 38.4126C41.6811 38.249 40.8272 38.767 40.5703 39.6145L42.2308 40.1178ZM46.448 40.9106L46.1274 42.6158C46.8188 42.7458 47.5202 42.4455 47.9033 41.8554C48.2864 41.2654 48.2753 40.5024 47.8752 39.9238L46.448 40.9106ZM43.9447 37.2899L42.7632 36.0192C42.1319 36.6062 42.0272 37.5676 42.5175 38.2767L43.9447 37.2899ZM47.1141 34.343L48.2956 35.6137C48.8114 35.1342 48.9861 34.3905 48.7379 33.7315C48.4898 33.0724 47.8679 32.6288 47.164 32.6086L47.1141 34.343ZM50.7351 36.8987C50.7351 30.3026 46.3513 24.2993 40.2137 24.2993V27.7696C43.7827 27.7696 47.2649 31.4945 47.2649 36.8987H50.7351ZM41.2927 49.4333C46.8681 48.7567 50.7351 43.1242 50.7351 36.8987H47.2649C47.2649 41.9509 44.1862 45.5865 40.8746 45.9883L41.2927 49.4333ZM43.4943 47.9171L42.4266 46.6121L39.7407 48.8096L40.8085 50.1146L43.4943 47.9171ZM39.2552 51.0036L42.2985 50.7447L42.0044 47.287L38.9611 47.5458L39.2552 51.0036ZM37.9389 47.2153L37.4536 48.7522L40.7628 49.7972L41.2481 48.2603L37.9389 47.2153ZM29.6904 36.8987C29.6904 43.242 33.7052 48.9715 39.4444 49.4665L39.7426 46.0091C36.354 45.7168 33.1607 42.0596 33.1607 36.8987H29.6904ZM40.2137 24.2993C34.0764 24.2993 29.6904 30.3021 29.6904 36.8987H33.1607C33.1607 31.495 36.6442 27.7696 40.2137 27.7696V24.2993ZM40.5828 30.0214L41.0057 34.3845L44.4598 34.0497L44.0369 29.6866L40.5828 30.0214ZM40.9675 34.3197L43.6664 30.936L40.9534 28.7721L38.2545 32.1558L40.9675 34.3197ZM34.8802 32.538L38.7516 34.7451L40.4704 31.7304L36.5989 29.5232L34.8802 32.538ZM38.5396 34.5938L37.4104 30.5626L34.0687 31.4986L35.1979 35.5298L38.5396 34.5938ZM33.3079 38.0297L37.4 36.7136L36.3375 33.41L32.2454 34.7261L33.3079 38.0297ZM37.3808 36.7568L33.558 34.8286L31.9952 37.9271L35.8179 39.8552L37.3808 36.7568ZM36.7176 42.9602L38.2254 38.9116L34.9733 37.7004L33.4655 41.749L36.7176 42.9602ZM38.2883 38.9811L34.3687 40.7771L35.8143 43.932L39.7339 42.136L38.2883 38.9811ZM42.4955 43.5419L40.5543 39.7653L37.4679 41.3518L39.4091 45.1283L42.4955 43.5419ZM40.5703 39.6145L39.2918 43.8317L42.6128 44.8385L43.8913 40.6212L40.5703 39.6145ZM46.7685 39.2053L42.5514 38.4126L41.9103 41.8231L46.1274 42.6158L46.7685 39.2053ZM42.5175 38.2767L45.0208 41.8973L47.8752 39.9238L45.3719 36.3031L42.5175 38.2767ZM45.9326 33.0723L42.7632 36.0192L45.1262 38.5606L48.2956 35.6137L45.9326 33.0723ZM42.6829 35.9515L47.0643 36.0774L47.164 32.6086L42.7826 32.4827L42.6829 35.9515Z" fill="white" mask="url(#path-2-outside-1_3011_78642)"/><path fill-rule="evenodd" clip-rule="evenodd" d="M42.3099 29.854L42.7328 34.2171L47.1142 34.343L43.9448 37.2899L46.448 40.9106L42.2309 40.1178L40.9524 44.3351L39.0112 40.5585L35.0916 42.3546L36.5994 38.306L32.7767 36.3779L36.8688 35.0618L35.7396 31.0306L39.611 33.2377L42.3099 29.854Z" fill="white"/></svg>',
			'footer_twitter_logo' => '<svg width="32" height="26" viewBox="0 0 32 26" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M31.6905 3.5C30.5355 4.025 29.2905 4.37 28.0005 4.535C29.3205 3.74 30.3405 2.48 30.8205 0.965C29.5755 1.715 28.1955 2.24 26.7405 2.54C25.5555 1.25 23.8905 0.5 22.0005 0.5C18.4755 0.5 15.5955 3.38 15.5955 6.935C15.5955 7.445 15.6555 7.94 15.7605 8.405C10.4205 8.135 5.66555 5.57 2.50055 1.685C1.94555 2.63 1.63055 3.74 1.63055 4.91C1.63055 7.145 2.75555 9.125 4.49555 10.25C3.43055 10.25 2.44055 9.95 1.57055 9.5V9.545C1.57055 12.665 3.79055 15.275 6.73055 15.86C5.78664 16.1183 4.79569 16.1543 3.83555 15.965C4.24296 17.2437 5.04085 18.3626 6.11707 19.1644C7.19329 19.9662 8.49372 20.4105 9.83555 20.435C7.56099 22.2357 4.74154 23.209 1.84055 23.195C1.33055 23.195 0.820547 23.165 0.310547 23.105C3.16055 24.935 6.55055 26 10.1805 26C22.0005 26 28.4955 16.19 28.4955 7.685C28.4955 7.4 28.4955 7.13 28.4805 6.845C29.7405 5.945 30.8205 4.805 31.6905 3.5Z" fill="#1B90EF"/></svg>',
			'footer_x_logo' => '<svg width="32" height="27" viewBox="0 0 29 27" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M22.2328 0.548828H26.5616L17.1045 11.3577L28.23 26.0661H19.5188L12.6958 17.1456L4.88886 26.0661H0.557455L10.6728 14.5048L0 0.548828H8.93237L15.0997 8.70259L22.2328 0.548828ZM20.7135 23.4752H23.1121L7.62902 3.00372H5.05505L20.7135 23.4752Z" fill="black"/></svg>'
        ];
		return $builder_svg_icons;
	}



	/**
	 * Gets a list of info
	 * Used in multiple places in the feed creator
	 * Other Platforms + Social Links
	 * Upgrade links
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public static function get_smashballoon_info() {
		$smash_info = [
			'colorSchemes' => [
				'facebook' => '#006BFA',
				'twitter' => '#1B90EF',
				'instagram' => '#BA03A7',
				'youtube' => '#EB2121',
				'linkedin' => '#007bb6',
				'mail' => '#666',
				'smash' => '#EB2121'
			],
			'upgrade' => [
					'name' => __( 'Upgrade to Pro', 'custom-twitter-feeds' ),
					'icon' => 'twitter',
					'link' => 'https://smashballoon.com/twitter-feed/'
			],
			'platforms' => [
				[
					'name' => __( 'Facebook Feed', 'custom-twitter-feeds' ),
					'icon' => 'facebook',
					'link' => 'https://smashballoon.com/facebook-feed/?utm_campaign=twitter-free&utm_source=balloon&utm_medium=facebook'
				],
				[
					'name' => __( 'Instagram Feed', 'custom-twitter-feeds' ),
					'icon' => 'instagram',
					'link' => 'https://smashballoon.com/instagram-feed/?utm_campaign=twitter-free&utm_source=balloon&utm_medium=instagram'
				],
				[
					'name' => __( 'YouTube Feed', 'custom-twitter-feeds' ),
					'icon' => 'youtube',
					'link' => 'https://smashballoon.com/youtube-feed/?utm_campaign=twitter-free&utm_source=balloon&utm_medium=youtube'
				],
				[
					'name' => __( 'Social Wall Plugin', 'custom-twitter-feeds' ),
					'icon' => 'smash',
					'link' => 'https://smashballoon.com/social-wall/?utm_campaign=twitter-free&utm_source=balloon&utm_medium=social-wall ',
				]
			],
			'socialProfiles' => [
				'facebook' => 'https://www.facebook.com/SmashBalloon/',
				//'instagram' => 'https://www.instagram.com/SmashBalloon/',
				//'youtube' => 'https://www.youtube.com/smashballoon',
				'twitter' => 'https://twitter.com/smashballoon',
			],
			'morePlatforms' => ['instagram','youtube','twitter']
		];

		return $smash_info;
	}

	/**
	 * Text specific to onboarding. Will return an associative array 'active' => false
	 * if onboarding has been dismissed for the user or there aren't any legacy feeds.
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public function get_onboarding_text() {
		// TODO: return if no legacy feeds
		$ctf_statuses_option = get_option( 'ctf_statuses', array() );

		if ( ! isset( $ctf_statuses_option['legacy_onboarding'] ) ) {
			return array( 'active' => false );
		}

		if ( $ctf_statuses_option['legacy_onboarding']['active'] === false
		     || self::onboarding_status() === 'dismissed' ) {
			return array( 'active' => false );
		}

		$type = $ctf_statuses_option['legacy_onboarding']['type'];

		$text = array(
			'active' => true,
			'type' => $type,
			'legacyFeeds' => array(
				'heading' => __( 'Legacy Feed Settings', 'custom-twitter-feeds' ),
				'description' => sprintf( __( 'These settings will impact %s legacy feeds on your site. You can learn more about what legacy feeds are and how they differ from new feeds %shere%s.', 'custom-twitter-feeds' ), '<span class="cff-fb-count-placeholder"></span>', '<a href="https://smashballoon.com/doc/facebook-legacy-feeds/" target="_blank" rel="nofollow noopener">', '</a>' ),
			),
			'getStarted' => __( 'You can now create and customize feeds individually. Click "Add New" to get started.', 'custom-twitter-feeds' ),
		);

		if ($type === 'single') {
			$text['tooltips'] = array(
				array(
					'step' => 1,
					'heading' => __( 'How you create a feed has changed', 'custom-twitter-feeds' ),
					'p' => __( 'You can now create and customize feeds individually without using shortcode options.', 'custom-twitter-feeds' ) . ' ' . __( 'Click "Add New" to get started.', 'custom-twitter-feeds' ),
					'pointer' => 'top'
				),
				array(
					'step' => 2,
					'heading' => __( 'Your existing feed is here', 'custom-twitter-feeds' ),
					'p' => __( 'You can edit your existing feed from here, and all changes will only apply to this feed.', 'custom-twitter-feeds' ),
					'pointer' => 'top'
				)
			);
		} else {
			$text['tooltips'] = array(
				array(
					'step' => 1,
					'heading' => __( 'How you create a feed has changed', 'custom-twitter-feeds' ),
					'p' => __( 'You can now create and customize feeds individually without using shortcode options.', 'custom-twitter-feeds' ) . ' ' . __( 'Click "Add New" to get started.', 'custom-twitter-feeds' ),
					'pointer' => 'top'
				),
				array(
					'step' => 2,
					'heading' => __( 'Your existing feeds are under "Legacy" feeds', 'custom-twitter-feeds' ),
					'p' => __( 'You can edit the settings for any existing "legacy" feed (i.e. any feed created prior to this update) here.', 'custom-twitter-feeds' ) . ' ' . __( 'This works just like the old settings page and affects all legacy feeds on your site.', 'custom-twitter-feeds' )
				),
				array(
					'step' => 3,
					'heading' => __( 'Existing feeds work as normal', 'custom-twitter-feeds' ),
					'p' => __( 'You don\'t need to update or change any of your existing feeds. They will continue to work as usual.', 'custom-twitter-feeds' ) . ' ' . __( 'This update only affects how new feeds are created and customized.', 'custom-twitter-feeds' )
				)
			);
		}

		return $text;
	}

	public function get_customizer_onboarding_text() {

		if ( self::onboarding_status( 'customizer' ) === 'dismissed' ) {
			return array( 'active' => false );
		}

		$text = array(
			'active' => true,
			'type' => 'customizer',
			'tooltips' => array(
				array(
					'step' => 1,
					'heading' => __( 'Embedding a Feed', 'custom-twitter-feeds' ),
					'p' => __( 'After you are done customizing the feed, click here to add it to a page or a widget.', 'custom-twitter-feeds' ),
					'pointer' => 'top'
				),
				array(
					'step' => 2,
					'heading' => __( 'Customize', 'custom-twitter-feeds' ),
					'p' => __( 'Change your feed layout, color scheme, or customize individual feed sections here.', 'custom-twitter-feeds' ),
					'pointer' => 'top'
				),
				array(
					'step' => 3,
					'heading' => __( 'Settings', 'custom-twitter-feeds' ),
					'p' => __( 'Update your feed source, filter your posts, or change advanced settings here.', 'custom-twitter-feeds' ),
					'pointer' => 'top'
				)
			)
		);

		return $text;
	}

	/**
	 * Text related to the feed customizer
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public function get_customize_screens_text() {
		$text =  [
			'common' => [
				'preview' => __( 'Preview', 'custom-twitter-feeds' ),
				'help' => __( 'Help', 'custom-twitter-feeds' ),
				'embed' => __( 'Embed', 'custom-twitter-feeds' ),
				'save' => __( 'Save', 'custom-twitter-feeds' ),
				'sections' => __( 'Sections', 'custom-twitter-feeds' ),
				'enable' => __( 'Enable', 'custom-twitter-feeds' ),
				'background' => __( 'Background', 'custom-twitter-feeds' ),
				'text' => __( 'Text', 'custom-twitter-feeds' ),
				'inherit' => __( 'Inherit from Theme', 'custom-twitter-feeds' ),
				'size' => __( 'Size', 'custom-twitter-feeds' ),
				'color' => __( 'Color', 'custom-twitter-feeds' ),
				'height' => __( 'Height', 'custom-twitter-feeds' ),
				'placeholder' => __( 'Placeholder', 'custom-twitter-feeds' ),
				'select' => __( 'Select', 'custom-twitter-feeds' ),
				'enterText' => __( 'Enter Text', 'custom-twitter-feeds' ),
				'hoverState' => __( 'Hover State', 'custom-twitter-feeds' ),
				'sourceCombine'	=>  __( 'Combine sources from multiple platforms using our Social Wall plugin', 'custom-twitter-feeds' ),
			],

			'tabs' => [
				'customize' => __( 'Customize', 'custom-twitter-feeds' ),
				'settings' => __( 'Settings', 'custom-twitter-feeds' ),
			],
			'overview' => [
				'feedLayout' => __( 'Feed Layout', 'custom-twitter-feeds' ),
				'colorScheme' => __( 'Color Scheme', 'custom-twitter-feeds' ),
				'header' => __( 'Header', 'custom-twitter-feeds' ),
				'posts' => __( 'Posts', 'custom-twitter-feeds' ),
				'likeBox' => __( 'Like Box', 'custom-twitter-feeds' ),
				'loadMore' => __( 'Load More Button', 'custom-twitter-feeds' ),
			],
			'feedLayoutScreen' => [
				'layout' => __( 'Layout', 'custom-twitter-feeds' ),
				'list' => __( 'List', 'custom-twitter-feeds' ),
				'grid' => __( 'Grid', 'custom-twitter-feeds' ),
				'masonry' => __( 'Masonry', 'custom-twitter-feeds' ),
				'carousel' => __( 'Carousel', 'custom-twitter-feeds' ),
				'feedHeight' => __( 'Feed Height', 'custom-twitter-feeds' ),
				'number' => __( 'Number of Posts', 'custom-twitter-feeds' ),
				'columns' => __( 'Columns', 'custom-twitter-feeds' ),
				'desktop' => __( 'Desktop', 'custom-twitter-feeds' ),
				'tablet' => __( 'Tablet', 'custom-twitter-feeds' ),
				'mobile' => __( 'Mobile', 'custom-twitter-feeds' ),
				'bottomArea' => [
					'heading' => __( 'Tweak Post Styles', 'custom-twitter-feeds' ),
					'description' => __( 'Change post background, border radius, shadow etc.', 'custom-twitter-feeds' ),
				]
			],
			'colorSchemeScreen' => [
				'scheme' => __( 'Scheme', 'custom-twitter-feeds' ),
				'light' => __( 'Light', 'custom-twitter-feeds' ),
				'dark' => __( 'Dark', 'custom-twitter-feeds' ),
				'custom' => __( 'Custom', 'custom-twitter-feeds' ),
				'customPalette' => __( 'Custom Palette', 'custom-twitter-feeds' ),
				'background2' => __( 'Background 2', 'custom-twitter-feeds' ),
				'text2' => __( 'Text 2', 'custom-twitter-feeds' ),
				'link' => __( 'Link', 'custom-twitter-feeds' ),
				'bottomArea' => [
					'heading' => __( 'Overrides', 'custom-twitter-feeds' ),
					'description' => __( 'Colors that have been overridden from individual post element settings will not change. To change them, you will have to reset overrides.', 'custom-twitter-feeds' ),
					'ctaButton' => __( 'Reset Overrides.', 'custom-twitter-feeds' ),
				]
			],
			'headerScreen' => [
				'headerType' => __( 'Header Type', 'custom-twitter-feeds' ),
				'visual' => __( 'Visual', 'custom-twitter-feeds' ),
				'coverPhoto' => __( 'Cover Photo', 'custom-twitter-feeds' ),
				'nameAndAvatar' => __( 'Name and avatar', 'custom-twitter-feeds' ),
				'about' => __( 'About (bio and Likes)', 'custom-twitter-feeds' ),
				'displayOutside' => __( 'Display outside scrollable area', 'custom-twitter-feeds' ),
				'icon' => __( 'Icon', 'custom-twitter-feeds' ),
				'iconImage' => __( 'Icon Image', 'custom-twitter-feeds' ),
				'iconColor' => __( 'Icon Color', 'custom-twitter-feeds' ),
			],
			// all Lightbox in common
			// all Load More in common
			'likeBoxScreen' => [
				'small' => __( 'Small', 'custom-twitter-feeds' ),
				'large' => __( 'Large', 'custom-twitter-feeds' ),
				'coverPhoto' => __( 'Cover Photo', 'custom-twitter-feeds' ),
				'customWidth' => __( 'Custom Width', 'custom-twitter-feeds' ),
				'defaultSetTo' => __( 'By default, it is set to auto', 'custom-twitter-feeds' ),
				'width' => __( 'Width', 'custom-twitter-feeds' ),
				'customCTA' => __( 'Custom CTA', 'custom-twitter-feeds' ),
				'customCTADescription' => __( 'This toggles the custom CTA like "Show now" and "Contact"', 'custom-twitter-feeds' ),
				'showFans' => __( 'Show Fans', 'custom-twitter-feeds' ),
				'showFansDescription' => __( 'Show visitors which of their friends follow your page', 'custom-twitter-feeds' ),
				'displayOutside' => __( 'Display outside scrollable area', 'custom-twitter-feeds' ),
				'displayOutsideDescription' => __( 'Make the like box fixed by moving it outside the scrollable area', 'custom-twitter-feeds' ),
			],
			'postsScreen' => [
				'thumbnail' => __( 'Thumbnail', 'custom-twitter-feeds' ),
				'half' => __( 'Half width', 'custom-twitter-feeds' ),
				'full' => __( 'Full width', 'custom-twitter-feeds' ),
				'useFull' => __( 'Use full width layout when post width is less than 500px', 'custom-twitter-feeds' ),
				'postStyle' => __( 'Post Style', 'custom-twitter-feeds' ),
				'editIndividual' => __( 'Edit Individual Elements', 'custom-twitter-feeds' ),
				'individual' => [
					'description' => __( 'Hide or show individual elements of a post or edit their options', 'custom-twitter-feeds' ),
					'name' => __( 'Name', 'custom-twitter-feeds' ),
					'edit' => __( 'Edit', 'custom-twitter-feeds' ),
					'postAuthor' => __( 'Author', 'custom-twitter-feeds' ),
					'postText' => __( 'Post Text', 'custom-twitter-feeds' ),
					'date' => __( 'Date', 'custom-twitter-feeds' ),
					'photosVideos' => __( 'Photos/Videos', 'custom-twitter-feeds' ),
					'likesShares' => __( 'Likes, Shares and Comments', 'custom-twitter-feeds' ),
					'eventTitle' => __( 'Event Title', 'custom-twitter-feeds' ),
					'eventDetails' => __( 'Event Details', 'custom-twitter-feeds' ),
					'postAction' => __( 'Post Action Links', 'custom-twitter-feeds' ),
					'sharedPostText' => __( 'Shared Post Text', 'custom-twitter-feeds' ),
					'sharedLinkBox' => __( 'Shared Link Box', 'custom-twitter-feeds' ),
					'postTextDescription' => __( 'The main text of the Instagram post', 'custom-twitter-feeds' ),
					'maxTextLength' => __( 'Maximum Text Length', 'custom-twitter-feeds' ),
					'characters' => __( 'Characters', 'custom-twitter-feeds' ),
					'linkText' => __( 'Link text to Instagram post', 'custom-twitter-feeds' ),
					'postDateDescription' => __( 'The date of the post', 'custom-twitter-feeds' ),
					'format' => __( 'Format', 'custom-twitter-feeds' ),
					'custom' => __( 'Custom', 'custom-twitter-feeds' ),
					'learnMoreFormats' => '<a href="https://smashballoon.com/doc/date-formatting-reference/" target="_blank" rel="nofollow noopener">' . __( 'Learn more about custom formats', 'custom-twitter-feeds' ) . '</a>',
					'addTextBefore' => __( 'Add text before date', 'custom-twitter-feeds' ),
					'addTextBeforeEG' => __( 'E.g. Posted', 'custom-twitter-feeds' ),
					'addTextAfter' => __( 'Add text after date', 'custom-twitter-feeds' ),
					'addTextAfterEG' => __( 'E.g. - posted date', 'custom-twitter-feeds' ),
					'timezone' => __( 'Timezone', 'custom-twitter-feeds' ),
					'tzDescription' => __( 'Timezone settings are global across all feeds. To update it use the global settings.', 'custom-twitter-feeds' ),
					'tzCTAText' => __( 'Go to Global Settings', 'custom-twitter-feeds' ),
					'photosVideosDescription' => __( 'Any photos or videos in your posts', 'custom-twitter-feeds' ),
					'useOnlyOne' => __( 'Use only one image per post', 'custom-twitter-feeds' ),
					'postActionLinksDescription' => __( 'The "View on Instagram" and "Share" links at the bottom of each post', 'custom-twitter-feeds' ),
					'viewOnFBLink' => __( 'View on Instagram link', 'custom-twitter-feeds' ),
					'viewOnFBLinkDescription' => __( 'Toggle "View on Instagram" link below each post', 'custom-twitter-feeds' ),
					'customizeText' => __( 'Customize Text', 'custom-twitter-feeds' ),
					'shareLink' => __( 'Share Link', 'custom-twitter-feeds' ),
					'shareLinkDescription' => __( 'Toggle "Share" link below each post', 'custom-twitter-feeds' ),
					'likesSharesDescription' => __( 'The comments box displayed at the bottom of each timeline post', 'custom-twitter-feeds' ),
					'iconTheme' => __( 'Icon Theme', 'custom-twitter-feeds' ),
					'auto' => __( 'Auto', 'custom-twitter-feeds' ),
					'light' => __( 'Light', 'custom-twitter-feeds' ),
					'dark' => __( 'Dark', 'custom-twitter-feeds' ),
					'expandComments' => __( 'Expand comments box by default', 'custom-twitter-feeds' ),
					'hideComment' => __( 'Hide comment avatars', 'custom-twitter-feeds' ),
					'showLightbox' => __( 'Show comments in lightbox', 'custom-twitter-feeds' ),
					'eventTitleDescription' => __( 'The title of an event', 'custom-twitter-feeds' ),
					'eventDetailsDescription' => __( 'The information associated with an event', 'custom-twitter-feeds' ),
					'textSize' => __( 'Text Size', 'custom-twitter-feeds' ),
					'textColor' => __( 'Text Color', 'custom-twitter-feeds' ),
					'sharedLinkBoxDescription' => __( "The link info box that's created when a link is shared in a Instagram post", 'custom-twitter-feeds' ),
					'boxStyle' => __( 'Box Style', 'custom-twitter-feeds' ),
					'removeBackground' => __( 'Remove background/border', 'custom-twitter-feeds' ),
					'linkTitle' => __( 'Link Title', 'custom-twitter-feeds' ),
					'linkURL' => __( 'Link URL', 'custom-twitter-feeds' ),
					'linkDescription' => __( 'Link Description', 'custom-twitter-feeds' ),
					'chars' => __( 'chars', 'custom-twitter-feeds' ),
					'sharedPostDescription' => __( 'The description text associated with shared photos, videos, or links', 'custom-twitter-feeds' ),
				],
				'postType' => __( 'Post Type', 'custom-twitter-feeds' ),
				'boxed' => __( 'boxed', 'custom-twitter-feeds' ),
				'regular' => __( 'Regular', 'custom-twitter-feeds' ),
				'indvidualProperties' => __( 'Indvidual Properties', 'custom-twitter-feeds' ),
				'backgroundColor' => __( 'Background Color', 'custom-twitter-feeds' ),
				'borderRadius' => __( 'Border Radius', 'custom-twitter-feeds' ),
				'boxShadow' => __( 'Box Shadow', 'custom-twitter-feeds' ),
			],
			'shoppableFeedScreen' => [
				'heading1' => __( 'Make your Instagram Feed Shoppable', 'custom-twitter-feeds' ),
				'description1' => __( 'This feature links the post to the one specificed in your caption.<br/><br/>Don’t want to add links to the caption? You can add links manually to each post.<br/><br/>Enable it to get started.', 'custom-twitter-feeds' ),
				'heading2' => __( 'Tap “Add” or “Update” on an<br/>image to add/update it’s URL', 'custom-twitter-feeds' ),

			]
		];

		$text['onboarding'] = $this->get_customizer_onboarding_text();

		return $text;
	}


	/**
	 * Get Links with UTM
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public static function get_links_with_utm() {
		$license_key = null;
		if ( get_option('ctf_license_key') ) {
			$license_key = get_option('ctf_license_key');
		}
		$all_access_bundle = sprintf('https://smashballoon.com/all-access/?edd_license_key=%s&upgrade=true&utm_campaign=twitter-free&utm_source=all-feeds&utm_medium=footer-banner&utm_content=learn-more', $license_key);
		$all_access_bundle_popup = sprintf('https://smashballoon.com/all-access/?edd_license_key=%s&upgrade=true&utm_campaign=twitter-free&utm_source=balloon&utm_medium=all-access', $license_key);
		$sourceCombineCTA = sprintf('https://smashballoon.com/social-wall/?edd_license_key=%s&upgrade=true&utm_campaign=twitter-free&utm_source=customizer&utm_medium=sources&utm_content=social-wall', $license_key);

		return array(
			'allAccessBundle' => 'https://smashballoon.com/pricing/twitter-feed/?utm_campaign=twitter-free&utm_source=balloon&utm_medium=upgrade-button',
			'popup' => array(
				'allAccessBundle' => 'https://smashballoon.com/pricing/twitter-feed/?utm_campaign=twitter-free&utm_source=balloon&utm_medium=upgrade-button',
				'fbProfile' => 'https://www.facebook.com/SmashBalloon/',
				'twitterProfile' => 'https://twitter.com/smashballoon',
			),
			'sourceCombineCTA' => $sourceCombineCTA,
			'multifeedCTA' => 'https://smashballoon.com/extensions/multifeed/?utm_campaign=twitter-free&utm_source=customizer&utm_medium=sources&utm_content=multifeed',
			'doc' => 'https://smashballoon.com/docs/twitter/?utm_campaign=twitter-free&utm_source=support&utm_medium=view-documentation-button&utm_content=view-documentation',
			'blog' => 'https://smashballoon.com/blog/?utm_campaign=twitter-free&utm_source=support&utm_medium=view-blog-button&utm_content=view-blog',
			'gettingStarted' => 'https://smashballoon.com/docs/getting-started/?utm_campaign=twitter-free&utm_source=support&utm_medium=getting-started-button&utm_content=getting-started',
		);
	}

	public static function get_social_wall_links() {
		return array(
			'<a href="'. esc_url( admin_url( 'admin.php?page=ctf-feed-builder' ) ) . '">' . __( 'All Feeds', 'custom-twitter-feeds' ) . '</a>',
			'<a href="'. esc_url( admin_url( 'admin.php?page=ctf-settings' ) ) . '">' . __( 'Settings', 'custom-twitter-feeds' ) . '</a>',
			'<a href="'. esc_url( admin_url( 'admin.php?page=ctf-about-us' ) ) . '">' . __( 'About Us', 'custom-twitter-feeds' ) . '</a>',
			'<a href="'. esc_url( admin_url( 'admin.php?page=ctf-support' ) ) . '">' . __( 'Support', 'custom-twitter-feeds' ) . '</a>',
		);
	}

	/**
	 * Returns an associate array of all existing feeds along with their data
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public static function get_feed_list( $feeds_args = array() ) {
		$feeds_data = CTF_Db::feeds_query( $feeds_args );
        $ctf_options = get_option( 'ctf_options', array() );
		$i = 0;
		foreach ( $feeds_data as $single_feed ) {
			$args = array(
				'feed_id' => '*' . $single_feed['id'],
				'html_location' => array( 'content' ),
			);
			$count = CTF_Feed_Locator::count( $args );

			$content_locations = CTF_Feed_Locator::twitter_feed_locator_query( $args );

			// if this is the last page, add in the header footer and sidebar locations
			if ( count( $content_locations ) < CTF_Db::RESULTS_PER_PAGE ) {

				$args = array(
					'feed_id' => '*' . $single_feed['id'],
					'html_location' => array( 'header', 'footer', 'sidebar' ),
					'group_by' => 'html_location'
				);
				$other_locations = CTF_Feed_Locator::twitter_feed_locator_query( $args );

				$locations = array();

				$combined_locations = array_merge( $other_locations, $content_locations );
			} else {
				$combined_locations = $content_locations;
			}

			foreach ( $combined_locations as $location ) {
				$page_text = get_the_title( $location['post_id'] );
				if ( $location['html_location'] === 'header' ) {
					$html_location = __( 'Header', 'custom-twitter-feeds' );
				} elseif ( $location['html_location'] === 'footer' ) {
					$html_location = __( 'Footer', 'custom-twitter-feeds' );
				} elseif ( $location['html_location'] === 'sidebar' ) {
					$html_location = __( 'Sidebar', 'custom-twitter-feeds' );
				} else {
					$html_location = __( 'Content', 'custom-twitter-feeds' );
				}
				$shortcode_atts = json_decode( $location['shortcode_atts'], true );
				$shortcode_atts = is_array( $shortcode_atts ) ? $shortcode_atts : array();

				$full_shortcode_string = '[custom-twitter-feeds';
				foreach ( $shortcode_atts as $key => $value ) {
					if ( ! empty( $value ) ) {
						$full_shortcode_string .= ' ' . esc_html( $key ) . '="' . esc_html( $value ) . '"';
					}
				}
				$full_shortcode_string .= ']';

				$locations[] = [
					'link' => esc_url( get_the_permalink( $location['post_id'] ) ),
					'page_text' => $page_text,
					'html_location' => $html_location,
					'shortcode' => $full_shortcode_string
				];
			}
			$feeds_data[ $i ]['instance_count'] = $count;
			$feeds_data[ $i ]['location_summary'] = $locations;
			$settings = json_decode( $feeds_data[ $i ]['settings'], true );

			$settings['feed'] = $single_feed['id'];

			$twitter_feed_settings = new CTF_Settings( $settings, CTF_Feed_Saver::settings_defaults() );

			$feeds_data[ $i ]['settings'] = $twitter_feed_settings->get_settings();
			$feeds_data[ $i ]['feed_enabled'] = self::auto_enable_disable_feed( $feeds_data[ $i ]['settings'], $single_feed['id'] );
			$i++;
		}
		return $feeds_data;
	}

    /**
	 * Check if Feed should be enabled or not
	 *
	 * @return bool
	 *
	 * @since 2.0
	 */
	public static function auto_enable_disable_feed($settings, $feed_id) {
        $ctf_options = get_option('ctf_options', array());
        $saved_enabled_feed = isset($ctf_options['enabled_feed_id']);
        $feed_saver = new CTF_Feed_Saver($saved_enabled_feed);
        $is_usertimeline = strtolower($settings['type']) === 'usertimeline';

        if($saved_enabled_feed && $ctf_options['enabled_feed_id'] === $feed_id){
            return true;
        }else if($saved_enabled_feed && $feed_saver->exists_in_database() ){
            return false;
        }else{
            if(!$is_usertimeline){
                return false;
            }else{
                $ctf_options['enabled_feed_id'] = $feed_id;
                update_option( 'ctf_options', $ctf_options );
                return true;
            }
        }


    }

	/**
	 * Returns an associate array of all existing sources along with their data
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public function get_legacy_feed_list() {
		$ctf_statuses = get_option( 'ctf_statuses', array() );

		if ( empty( $ctf_statuses['support_legacy_shortcode'] ) ) {
			return [];
		}

		$args = array(
			'html_location' => array( 'header', 'footer', 'sidebar', 'content' ),
			'group_by' => 'shortcode_atts',
			'page' => 1
		);
		$feeds_data = CTF_Feed_Locator::legacy_twitter_feed_locator_query( $args );


		if ( empty( $feeds_data ) ) {
			$args = array(
				'html_location' => array( 'header', 'footer', 'sidebar', 'content' ),
				'group_by' => 'shortcode_atts',
				'page' => 1
			);
			$feeds_data = CTF_Feed_Locator::legacy_twitter_feed_locator_query( $args );
		}

		$feed_saver = new CTF_Feed_Saver( 'legacy' );
		$settings = $feed_saver->get_feed_settings();
		$default_type = 'usertimeline';

		if ( isset( $settings['feedtype'] ) ) {
			$default_type = $settings['feedtype'];

		} elseif ( isset( $settings['type'] ) ) {
			if ( strpos( $settings['type'], ',' ) === false ) {
				$default_type = $settings['type'];
			}
		}
		$i = 0;
		$reindex = false;
		foreach ( $feeds_data as $single_feed ) {
			$args = array(
				'shortcode_atts' => $single_feed['shortcode_atts'],
				'html_location' => array( 'content' ),
			);
			$content_locations = CTF_Feed_Locator::twitter_feed_locator_query( $args );

			$count = CTF_Feed_Locator::count( $args );
			if ( count( $content_locations ) < CTF_Db::RESULTS_PER_PAGE ) {

				$args = array(
					'feed_id' => $single_feed['feed_id'],
					'html_location' => array( 'header', 'footer', 'sidebar' ),
					'group_by' => 'html_location'
				);
				$other_locations = CTF_Feed_Locator::twitter_feed_locator_query( $args );

				$combined_locations = array_merge( $other_locations, $content_locations );
			} else {
				$combined_locations = $content_locations;
			}

			$locations = array();
			foreach ( $combined_locations as $location ) {
				$page_text = get_the_title( $location['post_id'] );
				if ( $location['html_location'] === 'header' ) {
					$html_location = __( 'Header', 'custom-twitter-feeds' );
				} elseif ( $location['html_location'] === 'footer' ) {
					$html_location = __( 'Footer', 'custom-twitter-feeds' );
				} elseif ( $location['html_location'] === 'sidebar' ) {
					$html_location = __( 'Sidebar', 'custom-twitter-feeds' );
				} else {
					$html_location = __( 'Content', 'custom-twitter-feeds' );
				}
				$shortcode_atts = json_decode( $location['shortcode_atts'], true );
				$shortcode_atts = is_array( $shortcode_atts ) ? $shortcode_atts : array();

				$full_shortcode_string = '[custom-twitter-feeds';
				foreach ( $shortcode_atts as $key => $value ) {
					if ( ! empty( $value ) ) {
						$full_shortcode_string .= ' ' . esc_html( $key ) . '="' . esc_html( $value ) . '"';
					}
				}
				$full_shortcode_string .= ']';

				$locations[] = [
					'link' => esc_url( get_the_permalink( $location['post_id'] ) ),
					'page_text' => $page_text,
					'html_location' => $html_location,
					'shortcode' => $full_shortcode_string
				];
			}
			$shortcode_atts = json_decode( $feeds_data[ $i ]['shortcode_atts'], true );
			$shortcode_atts = is_array( $shortcode_atts ) ? $shortcode_atts : array();

			$full_shortcode_string = '[custom-twitter-feeds';
			foreach ( $shortcode_atts as $key => $value ) {
				if ( ! empty( $value ) ) {
					$full_shortcode_string .= ' ' . esc_html( $key ) . '="' . esc_html( $value ) . '"';
				}
			}
			$full_shortcode_string .= ']';

			$feeds_data[ $i ]['shortcode'] = $full_shortcode_string;
			$feeds_data[ $i ]['instance_count'] = $count;
			$feeds_data[ $i ]['location_summary'] = $locations;
			$feeds_data[ $i ]['feed_name'] = $feeds_data[ $i ]['feed_id'];
			$feeds_data[ $i ]['feed_type'] = $default_type;

			if ( isset( $shortcode_atts['feedtype'] ) ) {
				$feeds_data[ $i ]['feed_type'] = $shortcode_atts['feedtype'];

			} elseif ( isset( $shortcode_atts['type'] ) ) {
				if ( strpos( $shortcode_atts['type'], ',' ) === false ) {
					$feeds_data[ $i ]['feed_type'] = $shortcode_atts['type'];
				}
			}

			if ( isset( $feeds_data[ $i ]['id'] ) ) {
				unset( $feeds_data[ $i ]['id'] );
			}

			if ( isset( $feeds_data[ $i ]['html_location'] ) ) {
				unset( $feeds_data[ $i ]['html_location'] );
			}

			if ( isset( $feeds_data[ $i ]['last_update'] ) ) {
				unset( $feeds_data[ $i ]['last_update'] );
			}

			if ( isset( $feeds_data[ $i ]['post_id'] ) ) {
				unset( $feeds_data[ $i ]['post_id'] );
			}

			if ( ! empty( $shortcode_atts['feed'] ) ) {
				$reindex = true;
				unset( $feeds_data[ $i ] );
			}

			if ( isset( $feeds_data[ $i ]['shortcode_atts'] ) ) {
				unset( $feeds_data[ $i ]['shortcode_atts'] );
			}

			$i++;
		}

		if ( $reindex ) {
			$feeds_data = array_values( $feeds_data );
		}

		// if there were no feeds found in the locator table we still want the legacy settings to be available
		// if it appears as though they had used version 3.x or under at some point.
		if ( empty( $feeds_data )
		     && ! is_array( $ctf_statuses['support_legacy_shortcode'] )
		     && ( $ctf_statuses['support_legacy_shortcode'] ) ) {

			$feeds_data = array(
				array(
					'feed_id' => __( 'Legacy Feed', 'custom-twitter-feeds' ) . ' ' . __( '(unknown location)', 'custom-twitter-feeds' ),
					'feed_name' => __( 'Legacy Feed', 'custom-twitter-feeds' ) . ' ' . __( '(unknown location)', 'custom-twitter-feeds' ),
					'shortcode' => '[custom-twitter-feeds]',
					'feed_type' => '',
					'instance_count' => false,
					'location_summary' => array()
				)
			);
		}

		return $feeds_data;
	}

	/**
	 * Status of the onboarding sequence for specific user
	 *
	 * @return string|boolean
	 *
	 * @since 2.0
	 */
	public static function onboarding_status( $type = 'newuser' ) {
		$onboarding_statuses = get_user_meta( get_current_user_id(), 'ctf_onboarding', true );
		$status = false;
		if ( ! empty( $onboarding_statuses ) ) {
			$statuses = maybe_unserialize( $onboarding_statuses );
			$status = isset( $statuses[ $type ] ) ? $statuses[ $type ] : false;
		}

		return $status;
	}

	/**
	 * Update status of onboarding sequence for specific user
	 *
	 * @return string|boolean
	 *
	 * @since 2.0
	 */
	public static function update_onboarding_meta( $value, $type = 'newuser' ) {
		$onboarding_statuses = get_user_meta( get_current_user_id(), 'ctf_onboarding', true );
		if ( ! empty( $onboarding_statuses ) ) {
			$statuses = maybe_unserialize( $onboarding_statuses );
			$statuses[ $type ] = $value;
		} else {
			$statuses = array(
				$type => $value
			);
		}

		$statuses = maybe_serialize( $statuses );

		update_user_meta( get_current_user_id(), 'ctf_onboarding', $statuses );
	}

	/**
	 * Used to dismiss onboarding using AJAX
	 *
	 * @since 2.0
	 */
	public static function after_dismiss_onboarding() {
		$cap = current_user_can( 'manage_twitter_feed_options' ) ? 'manage_twitter_feed_options' : 'manage_options';
		$cap = apply_filters( 'ctf_settings_pages_capability', $cap );

		if ( current_user_can( $cap ) ) {
			$type = 'newuser';
			if ( isset( $_POST['was_active'] ) ) {
				$type = sanitize_text_field( $_POST['was_active'] );
			}
			CTF_Feed_Builder::update_onboarding_meta( 'dismissed', $type );
		}
		wp_die();
	}

	public static function add_customizer_att( $atts ) {
   	    if ( ! is_array( $atts ) ) {
   	    	$atts = [];
        }
   	    $atts['feedtype'] = 'customizer';
   	    return $atts;
	}

	/**
	 * Feed Builder Wrapper.
	 *
	 * @since 2.0
	 */
	public function feed_builder(){
		include_once CTF_BUILDER_DIR . 'templates/builder.php';
	}

	/**
	 * For types listed on the top of the select feed type screen
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public function get_feed_types() {
    	$feed_types = [
		    [
			    'type' => 'usertimeline',
			    'title'=> __( 'User Timeline', 'custom-twitter-feeds' ),
			    'description'=> __( 'A timeline of Tweets from any Twitter user', 'custom-twitter-feeds' ),
			    'icon'	=>  'usertimelineIcon'
		    ],
		    [
			    'type' => 'hashtag',
			    'title' => __( 'Hashtag', 'custom-twitter-feeds' ),
			    'description'=> __( 'Public Tweets which contain a specific hashtag', 'custom-twitter-feeds' ),
			    'icon'	=>  'hashtagIcon'
		    ],
		    [
			    'type' => 'search',
			    'title' => __( 'Search', 'custom-twitter-feeds' ),
			    'description'=> __( 'List of Tweets from a specific search query', 'custom-twitter-feeds' ),
			    'icon'	=>  'searchIcon'
		    ],
		    [
			    'type' => 'lists',
			    'title' => __( 'Lists', 'custom-twitter-feeds' ),
			    'description'=> __( 'Curated Tweets from a speicific Twitter list', 'custom-twitter-feeds' ),
			    'icon'	=>  'listsIcon'
		    ],
			[
			    'type' => 'socialwall',
			    'title' => __( 'Social Wall', 'custom-twitter-feeds' ),
			    'description'=> __( 'A brief description about the feed type', 'custom-twitter-feeds' ),
			    'icon'	=>  'socialwall1Icon'
		    ]
		    /*
		    [
			    'type' => 'hometimeline',
			    'title' => __( 'Home Timeline', 'custom-twitter-feeds' ),
			    'description'=> __( 'The home timeline from your Twitter account', 'custom-twitter-feeds' ),
			    'icon'	=>  'homeTimelineIcon'
		    ],
		    [
			    'type' => 'mentionstimeline',
			    'title' => __( 'Mentions', 'custom-twitter-feeds' ),
			    'description'=> __( 'A brief description about the feed type', 'custom-twitter-feeds' ),
			    'icon'	=>  'mentionsIcon'
		    ],
		    */
	    ];

    	return $feed_types;
	}


	/**
	 * For types listed on the top of the select feed type screen
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public function get_advanced_feed_types() {
    	$advanced_feed_types = [
		    [
			    'type' => 'search',
			    'title' => __( 'Search', 'custom-twitter-feeds' ),
			    'description'=> __( 'A brief description about the feed type', 'custom-twitter-feeds' ),
			    'icon'	=>  'searchIcon'
		    ],
		    [
			    'type' => 'mentionstimeline',
			    'title' => __( 'Mentions*', 'custom-twitter-feeds' ),
			    'description'=> __( 'Not available currently', 'custom-twitter-feeds' ),
			    'icon'	=>  'mentionsIcon'
		    ],
		    [
			    'type' => 'lists',
			    'title' => __( 'Lists*', 'custom-twitter-feeds' ),
			    'description'=> __( 'Not available currently', 'custom-twitter-feeds' ),
			    'icon'	=>  'listsIcon'
		    ],
		    [
			    'type' => 'socialwall',
			    'title' => __( 'Social Wall', 'custom-twitter-feeds' ),
			    'description'=> __( 'A brief description about the feed type', 'custom-twitter-feeds' ),
			    'icon'	=>  'socialwall1Icon'
		    ]
	    ];

    	return $advanced_feed_types;
	}

	/**
	 * Connect Account Screen
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public static function connect_account_secreen(){
		return array(
			'heading' => __( 'Connect To Smash Balloon', 'custom-twitter-feeds' ),
			'description' => __( 'In order to create a feed and retrieve tweets, you need to verify an email address with Smash Balloon.', 'custom-twitter-feeds' ),
			'preferManually' => __( 'Prefer to use your own Twitter Developer App?', 'custom-twitter-feeds' ),
			'addAppCred' => __( 'Add App Credentials', 'custom-twitter-feeds' ),
			'connectNewAccount' => __( 'Add New Account', 'custom-twitter-feeds' ),
			'heading_2' => __( 'Add Your Own Twitter App Credentials', 'custom-twitter-feeds' ),
			'manualModal' => [
				'name' => __( 'Name', 'custom-twitter-feeds' ),
				'namePlhdr' => __( 'Enter App Name', 'custom-twitter-feeds' ),
				'consumerKey' => __( 'Consumer Key', 'custom-twitter-feeds' ),
				'consumerKeyPlhdr' => __( 'Enter Consumer Key', 'custom-twitter-feeds' ),
				'consumerSecret' => __( 'Consumer Secret', 'custom-twitter-feeds' ),
				'consumerSecretPlhdr' => __( 'Enter Consumer Secret', 'custom-twitter-feeds' ),
				'accessToken' => __( 'Access Token', 'custom-twitter-feeds' ),
				'accessTokenPlhdr' => __( 'Enter Access Token', 'custom-twitter-feeds' ),
				'accessTokenSecret' => __( 'Access Token Secret', 'custom-twitter-feeds' ),
				'accessTokenSecretPlhdr' => __( 'Enter Access Token Secret', 'custom-twitter-feeds' ),
			]
		);
	}

	/**
	 * For types listed on the top of the select feed type screen
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public function get_feed_templates(){
		$feed_templates = [
		    [
			    'type' => 'default',
			    'title'=> __( 'Default', 'custom-twitter-feeds' ),
			    'description'=> '',
			    'icon'	=>  'defaultTemplate'
		    ],
		    [
			    'type' => 'masonry_cards',
			    'title'=> __( 'Masonry Cards', 'custom-twitter-feeds' ),
			    'description'=> '',
			    'icon'	=>  'masonryCardsTemplate'
		    ],
		    [
			    'type' => 'simple_carousel',
			    'title'=> __( 'Simple Carousel', 'custom-twitter-feeds' ),
			    'description'=> '',
			    'icon'	=>  'simpleCarouselTemplate'
		    ],
		    [
			    'type' => 'simple_cards',
			    'title'=> __( 'Simple Cards', 'custom-twitter-feeds' ),
			    'description'=> '',
			    'icon'	=>  'simpleCardsTemplate'
		    ],
		    [
			    'type' => 'showcase_carousel',
			    'title'=> __( 'Showcase Carousel', 'custom-twitter-feeds' ),
			    'description'=> '',
			    'icon'	=>  'showcaseCarouselTemplate'
		    ],
		    [
			    'type' => 'latest_tweet',
			    'title'=> __( 'Latest Tweet', 'custom-twitter-feeds' ),
			    'description'=> '',
			    'icon'	=>  'latestTweetTemplate'
		    ],
		    [
			    'type' => 'widget',
			    'title'=> __( 'Widget', 'custom-twitter-feeds' ),
			    'description'=> '',
			    'icon'	=>  'widgetTemplate'
		    ]
		];

		return $feed_templates;
	}


	/**
	 * Get App Credentials
	 *
	 * @return array
	 *
	 * @since 2.0
	 */
	public static function get_app_credentials( $ctf_options ) {
		#return [
		#	'app_name' 					=> '',
		#	'consumer_key' 				=> '',
		#	'consumer_secret' 			=> '',
		#	'access_token' 				=> '',
		#	'access_token_secret' 		=> ''
		#];
		return [
			'app_name' 					=> isset( $ctf_options['app_name'] ) && $ctf_options['app_name'] !== '' ? $ctf_options['app_name'] : '',
			'consumer_key' 				=> isset( $ctf_options['consumer_key'] ) && $ctf_options['consumer_key'] !== '' ? $ctf_options['consumer_key'] : '',
			'consumer_secret' 			=> isset( $ctf_options['consumer_secret'] ) && $ctf_options['consumer_secret'] !== '' ? $ctf_options['consumer_secret'] : '',
			'access_token' 				=> isset( $ctf_options['access_token'] ) && $ctf_options['access_token'] !== '' ? $ctf_options['access_token'] : '',
			'access_token_secret' 		=> isset( $ctf_options['access_token_secret'] ) && $ctf_options['access_token_secret'] !== '' ? $ctf_options['access_token_secret'] : ''
		];

	}

	/**
	 * Check if On
	 * Function to check if a shortcode options is set to ON or TRUE
	 *
	 * @access public
  	 * @static
	 * @since 2.0
	 * @return boolean
	 */
	static function check_if_on( $value ){
		return ( isset( $value ) && !empty( $value ) && ( $value == 'true' || $value == 'on') ) ?  true : false;
	}

	/**
	 * Get Translated text Set in the Settings Page
	 *
	 * @return array
	 *
	 * @since 4.0
	 */
	function get_translated_text(){
		$options  = get_option( 'ctf_options' );
		$text =  [
			'secondText' 	=> __( 'second', 'custom-twitter-feeds' ),
			'secondsText' 	=> __( 'seconds', 'custom-twitter-feeds' ),
			'minuteText' 	=> __( 'minute', 'custom-twitter-feeds' ),
			'minutesText' 	=> __( 'minutes', 'custom-twitter-feeds' ),
			'hourText' 		=> __( 'hour', 'custom-twitter-feeds' ),
			'hoursText' 	=> __( 'hours', 'custom-twitter-feeds' ),
			'dayText' 		=> __( 'day', 'custom-twitter-feeds' ),
			'daysText' 		=> __( 'days', 'custom-twitter-feeds' ),
			'weekText' 		=> __( 'week', 'custom-twitter-feeds' ),
			'weeksText' 	=> __( 'weeks', 'custom-twitter-feeds' ),
			'monthText' 	=> __( 'month', 'custom-twitter-feeds' ),
			'monthsText' 	=> __( 'months', 'custom-twitter-feeds' ),
			'yearText' 		=> __( 'year', 'custom-twitter-feeds' ),
			'yearsText' 	=> __( 'years', 'custom-twitter-feeds' ),
			'agoText' 		=> __( 'ago', 'custom-twitter-feeds' ),
			'ctf_minute' 	=> ! empty( $options['mtime'] ) ? $options['mtime'] : 'm',
            'ctf_hour' 		=> ! empty( $options['htime'] ) ? $options['htime'] : 'h',
            'ctf_now_str' 	=> ! empty( $options['nowtime'] ) ? $options['nowtime'] : 'now'
		];

		return $text;
	}

	public static function sb_other_plugins_modal() {
		check_ajax_referer( 'ctf-smash-balloon' , 'sb_nonce' );

		if ( ! current_user_can( 'activate_plugins' ) || ! current_user_can( 'install_plugins' ) ) {
			wp_send_json_error();
		}
		$plugin = isset( $_POST['plugin'] ) ? sanitize_key( $_POST['plugin'] ) : '';
		$sb_other_plugins = self::install_plugins_popup();
		$plugin = isset( $sb_other_plugins[ $plugin ] ) ? $sb_other_plugins[ $plugin ] : false;
		if ( ! $plugin ) {
			wp_send_json_error();
		}

		// Build the content for modals
		$output = '<div class="ctf-fb-source-popup ctf-fb-popup-inside ctf-install-plugin-modal">
		<div class="ctf-fb-popup-cls"><svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg">
		<path d="M14 1.41L12.59 0L7 5.59L1.41 0L0 1.41L5.59 7L0 12.59L1.41 14L7 8.41L12.59 14L14 12.59L8.41 7L14 1.41Z" fill="#141B38"></path>
		</svg></div>
		<div class="ctf-install-plugin-body ctf-fb-fs">
		<div class="ctf-install-plugin-header">
		<div class="sb-plugin-image">'. $plugin['svgIcon'] .'</div>
		<div class="sb-plugin-name">
		<h3>'. $plugin['name'] .'<span>Free</span></h3>
		<p><span class="sb-author-logo">
		<svg width="13" height="17" viewBox="0 0 13 17" fill="none" xmlns="http://www.w3.org/2000/svg">
		<path fill-rule="evenodd" clip-rule="evenodd" d="M5.72226 4.70098C4.60111 4.19717 3.43332 3.44477 2.34321 3.09454C2.73052 4.01824 3.05742 5.00234 3.3957 5.97507C2.72098 6.48209 1.93286 6.8757 1.17991 7.30453C1.82065 7.93788 2.72809 8.3045 3.45109 8.85558C2.87196 9.57021 1.73414 10.3129 1.45689 10.9606C2.65579 10.8103 4.05285 10.5668 5.16832 10.5174C5.41343 11.7495 5.53984 13.1002 5.88845 14.2288C6.40758 12.7353 6.87695 11.192 7.49488 9.79727C8.44849 10.1917 9.61069 10.6726 10.5416 10.9052C9.88842 9.98881 9.29237 9.01536 8.71356 8.02465C9.57007 7.40396 10.4364 6.79309 11.2617 6.14122C10.0952 6.03375 8.88647 5.96834 7.66107 5.91968C7.46633 4.65567 7.5175 3.14579 7.21791 1.98667C6.76462 2.93671 6.2297 3.80508 5.72226 4.70098ZM6.27621 15.1705C6.12214 15.8299 6.62974 16.1004 6.55318 16.5C6.052 16.3273 5.67498 16.2386 5.00213 16.3338C5.02318 15.8194 5.48587 15.7466 5.3899 15.1151C-1.78016 14.3 -1.79456 1.34382 5.3345 0.546422C14.2483 -0.450627 14.528 14.9414 6.27621 15.1705Z" fill="#FE544F"></path><path fill-rule="evenodd" clip-rule="evenodd" d="M7.21769 1.98657C7.51728 3.1457 7.46611 4.65557 7.66084 5.91955C8.88625 5.96824 10.0949 6.03362 11.2615 6.14113C10.4362 6.79299 9.56984 7.40386 8.71334 8.02454C9.29215 9.01527 9.8882 9.98869 10.5414 10.9051C9.61046 10.6725 8.44827 10.1916 7.49466 9.79716C6.87673 11.1919 6.40736 12.7352 5.88823 14.2287C5.53962 13.1001 5.41321 11.7494 5.16809 10.5173C4.05262 10.5667 2.65558 10.8102 1.45666 10.9605C1.73392 10.3128 2.87174 9.57012 3.45087 8.85547C2.72786 8.30438 1.82043 7.93778 1.17969 7.30443C1.93264 6.8756 2.72074 6.482 3.39547 5.97494C3.05719 5.00224 2.73031 4.01814 2.34299 3.09445C3.43308 3.44467 4.60089 4.19707 5.72204 4.70088C6.22947 3.80499 6.7644 2.93662 7.21769 1.98657Z" fill="white"></path>
		</svg>
		</span>
		<span class="sb-author-name">'. $plugin['author'] .'</span>
		</p></div></div>
		<div class="ctf-install-plugin-content">
		<p>'. $plugin['description'] .'</p>';

		$plugin_install_data = array(
			'step' => 'install',
			'action' => 'ctf_install_addon',
			'nonce' => wp_create_nonce('ctf-admin'),
			'plugin' => $plugin['plugin'],
			'download_plugin' => $plugin['download_plugin'],
		);

		if ( ! $plugin['installed'] ) {
			$output .= sprintf(
				"<button class='ctf-install-plugin-btn ctf-btn-orange' id='ctf_install_op_btn' data-plugin-atts='%s'>%s</button></div></div></div>",
				ctf_json_encode( $plugin_install_data ),
				__('Install', 'custom-twitter-feeds')
			);
		}
		if ( $plugin['installed'] && ! $plugin['activated'] ) {
			$plugin_install_data['step'] = 'activate';
			$plugin_install_data['action'] = 'ctf_activate_addon';
			$output .= sprintf(
				"<button class='ctf-install-plugin-btn ctf-btn-orange' id='ctf_install_op_btn' data-plugin-atts='%s'>%s</button></div></div></div>",
				ctf_json_encode( $plugin_install_data ),
				__('Activate', 'custom-twitter-feeds')
			);
		}
		if ( $plugin['installed'] && $plugin['activated'] ) {
			$output .= sprintf(
				"<button class='ctf-install-plugin-btn ctf-btn-orange' id='ctf_install_op_btn' disabled='disabled'>%s</button></div></div></div>",
				__('Plugin installed & activated', 'custom-twitter-feeds')
			);
		}

		new \TwitterFeed\Admin\CTF_Response( true, array(
			'output' => $output
		) );
	}

	/**
	 * Plugins information for plugin install modal in all feeds page on select source flow
	 *
	 * @since 6.0
	 *
	 * @return array
	 */
	public static function install_plugins_popup() {
		// get the WordPress's core list of installed plugins
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$installed_plugins = get_plugins();

		$is_facebook_installed = false;
        $facebook_plugin = 'custom-facebook-feed/custom-facebook-feed.php';
        if ( isset( $installed_plugins['custom-facebook-feed-pro/custom-facebook-feed.php'] ) ) {
            $is_facebook_installed = true;
            $facebook_plugin = 'custom-facebook-feed/custom-facebook-feed.php';
        } else if ( isset( $installed_plugins['custom-facebook-feed/custom-facebook-feed.php'] ) ) {
            $is_facebook_installed = true;
        }

        $is_instagram_installed = false;
        $instagram_plugin = 'instagram-feed/instagram-feed.php';
        if ( isset( $installed_plugins['instagram-feed-pro/instagram-feed.php'] ) ) {
            $is_instagram_installed = true;
            $instagram_plugin = 'instagram-feed-pro/instagram-feed.php';
        } else if ( isset( $installed_plugins['instagram-feed/instagram-feed.php'] ) ) {
            $is_instagram_installed = true;
        }

        $is_youtube_installed = false;
        $youtube_plugin = 'feeds-for-youtube/youtube-feed.php';
        if ( isset( $installed_plugins['youtube-feed-pro/youtube-feed.php'] ) ) {
            $is_youtube_installed = true;
            $youtube_plugin = 'youtube-feed-pro/youtube-feed.php';
        } else if ( isset( $installed_plugins['feeds-for-youtube/youtube-feed.php'] ) ) {
            $is_youtube_installed = true;
        }

		return array(
			'facebook' => array(
				'displayName' => __( 'Facebook', 'custom-twitter-feeds' ),
				'name' => __( 'Facebook Feed', 'custom-twitter-feeds' ),
				'author' => __( 'By Smash Balloon', 'custom-twitter-feeds' ),
				'description' => __('To display a Facebook feed, our Facebook plugin is required. </br> It provides a clean and beautiful way to add your Facebook posts to your website. Grab your visitors attention and keep them engaged with your site longer.', 'custom-twitter-feeds'),
				'dashboard_permalink' => admin_url( 'admin.php?page=cff-feed-builder' ),
				'svgIcon' => '<svg viewBox="0 0 14 15"  width="36" height="36"><path d="M7.00016 0.860001C3.3335 0.860001 0.333496 3.85333 0.333496 7.54C0.333496 10.8733 2.7735 13.64 5.96016 14.14V9.47333H4.26683V7.54H5.96016V6.06667C5.96016 4.39333 6.9535 3.47333 8.48016 3.47333C9.20683 3.47333 9.96683 3.6 9.96683 3.6V5.24667H9.12683C8.30016 5.24667 8.04016 5.76 8.04016 6.28667V7.54H9.8935L9.5935 9.47333H8.04016V14.14C9.61112 13.8919 11.0416 13.0903 12.0734 11.88C13.1053 10.6697 13.6704 9.13043 13.6668 7.54C13.6668 3.85333 10.6668 0.860001 7.00016 0.860001Z" fill="rgb(0, 107, 250)"/></svg>',
				'installed' => $is_facebook_installed,
				'activated' => is_plugin_active( $facebook_plugin ),
				'plugin' => $facebook_plugin,
				'download_plugin' => 'https://downloads.wordpress.org/plugin/custom-facebook-feed.zip',
			),
			'instagram' => array(
				'displayName' => __( 'Instagram', 'custom-twitter-feeds' ),
				'name' => __( 'Instagram Feed', 'custom-twitter-feeds' ),
				'author' => __( 'By Smash Balloon', 'custom-twitter-feeds' ),
				'description' => __('To display an Instagram feed, our Instagram plugin is required. </br> It provides a clean and beautiful way to add your Instagram posts to your website. Grab your visitors attention and keep them engaged with your site longer.', 'custom-twitter-feeds'),
				'dashboard_permalink' => admin_url( 'admin.php?page=sb-instagram-feed' ),
				'svgIcon' => '<svg width="36" height="36" viewBox="0 0 36 36" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M18 9.91406C13.5 9.91406 9.91406 13.5703 9.91406 18C9.91406 22.5 13.5 26.0859 18 26.0859C22.4297 26.0859 26.0859 22.5 26.0859 18C26.0859 13.5703 22.4297 9.91406 18 9.91406ZM18 23.2734C15.1172 23.2734 12.7266 20.9531 12.7266 18C12.7266 15.1172 15.0469 12.7969 18 12.7969C20.8828 12.7969 23.2031 15.1172 23.2031 18C23.2031 20.9531 20.8828 23.2734 18 23.2734ZM28.2656 9.63281C28.2656 8.57812 27.4219 7.73438 26.3672 7.73438C25.3125 7.73438 24.4688 8.57812 24.4688 9.63281C24.4688 10.6875 25.3125 11.5312 26.3672 11.5312C27.4219 11.5312 28.2656 10.6875 28.2656 9.63281ZM33.6094 11.5312C33.4688 9 32.9062 6.75 31.0781 4.92188C29.25 3.09375 27 2.53125 24.4688 2.39062C21.8672 2.25 14.0625 2.25 11.4609 2.39062C8.92969 2.53125 6.75 3.09375 4.85156 4.92188C3.02344 6.75 2.46094 9 2.32031 11.5312C2.17969 14.1328 2.17969 21.9375 2.32031 24.5391C2.46094 27.0703 3.02344 29.25 4.85156 31.1484C6.75 32.9766 8.92969 33.5391 11.4609 33.6797C14.0625 33.8203 21.8672 33.8203 24.4688 33.6797C27 33.5391 29.25 32.9766 31.0781 31.1484C32.9062 29.25 33.4688 27.0703 33.6094 24.5391C33.75 21.9375 33.75 14.1328 33.6094 11.5312ZM30.2344 27.2812C29.7422 28.6875 28.6172 29.7422 27.2812 30.3047C25.1719 31.1484 20.25 30.9375 18 30.9375C15.6797 30.9375 10.7578 31.1484 8.71875 30.3047C7.3125 29.7422 6.25781 28.6875 5.69531 27.2812C4.85156 25.2422 5.0625 20.3203 5.0625 18C5.0625 15.75 4.85156 10.8281 5.69531 8.71875C6.25781 7.38281 7.3125 6.32812 8.71875 5.76562C10.7578 4.92188 15.6797 5.13281 18 5.13281C20.25 5.13281 25.1719 4.92188 27.2812 5.76562C28.6172 6.25781 29.6719 7.38281 30.2344 8.71875C31.0781 10.8281 30.8672 15.75 30.8672 18C30.8672 20.3203 31.0781 25.2422 30.2344 27.2812Z" fill="url(#paint0_linear)"/><defs><linearGradient id="paint0_linear" x1="13.4367" y1="62.5289" x2="79.7836" y2="-5.19609" gradientUnits="userSpaceOnUse"><stop stop-color="white"/><stop offset="0.147864" stop-color="#F6640E"/><stop offset="0.443974" stop-color="#BA03A7"/><stop offset="0.733337" stop-color="#6A01B9"/><stop offset="1" stop-color="#6B01B9"/></linearGradient></defs></svg>',
				'installed' => $is_instagram_installed,
				'activated' => is_plugin_active( $instagram_plugin ),
				'plugin' => $instagram_plugin,
				'download_plugin' => 'https://downloads.wordpress.org/plugin/instagram-feed.zip',
			),
			'youtube' => array(
				'displayName' => __( 'YouTube', 'custom-twitter-feeds' ),
				'name' => __( 'Feeds for YouTube', 'custom-twitter-feeds' ),
				'author' => __( 'By Smash Balloon', 'custom-twitter-feeds' ),
				'description' => __( 'To display a YouTube feed, our YouTube plugin is required. It provides a simple yet powerful way to display videos from YouTube on your website, Increasing engagement with your channel while keeping visitors on your website.', 'custom-twitter-feeds' ),
				'dashboard_permalink' => admin_url( 'admin.php?page=youtube-feed' ),
				'svgIcon' => '<svg width="36" height="36" viewBox="0 0 36 36" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M15 22.5L22.785 18L15 13.5V22.5ZM32.34 10.755C32.535 11.46 32.67 12.405 32.76 13.605C32.865 14.805 32.91 15.84 32.91 16.74L33 18C33 21.285 32.76 23.7 32.34 25.245C31.965 26.595 31.095 27.465 29.745 27.84C29.04 28.035 27.75 28.17 25.77 28.26C23.82 28.365 22.035 28.41 20.385 28.41L18 28.5C11.715 28.5 7.8 28.26 6.255 27.84C4.905 27.465 6.035 26.595 3.66 25.245C3.465 24.54 3.33 23.595 3.24 22.395C3.135 21.195 3.09 20.16 3.09 19.26L3 18C3 14.715 3.24 12.3 3.66 10.755C6.035 9.405 4.905 8.535 6.255 8.16C6.96 7.965 8.25 7.83 10.23 7.74C12.18 7.635 13.965 7.59 15.615 7.59L18 7.5C24.285 7.5 28.2 7.74 29.745 8.16C31.095 8.535 31.965 9.405 32.34 10.755Z" fill="#EB2121"/></svg>',
				'installed' => $is_youtube_installed,
				'activated' => is_plugin_active( $youtube_plugin ),
				'plugin' => $youtube_plugin,
				'download_plugin' => 'https://downloads.wordpress.org/plugin/feeds-for-youtube.zip',
			),
		);
	}
}

